/*
 *  Specialized shader for Enermodal version
 *
 *      10/28/93 Greg Ward
 */

#include  "ray.h"
#include  "source.h"
#include  "func.h"

#define  MAXLAYER       8       /* maximum number of glazing layers */

/*
 *  The specification for glazing is as follows:
 *
 *      void glazing id
 *      3+ trans afunc funcfile xf ..
 *      0
 *      1+ #layers A2 A3 ..
 *
 *  Trans is the (monochromatic) transmission value.
 *  Afunc is the (monochromatic) absorption value function.
 *  Its only argument is the layer number, 1 through #layers.
 *  Absorption values (in watts/m2) are accumulated for each layer
 *  in the gla_absorbed[] array.  Note that this mechanism permits
 *  only one such surface in the ray tree, and gla_nlays must be
 *  zeroed before each calculation.
 *
 *  The first layer is the one closest to the positive surface normal.
 */

extern int  ambdiv;                     /* ambient divisions */
extern int  ambssamp;                   /* ambient super-samples (should be zero!) */

extern RAY  thisray;

double  gla_absorbed[MAXLAYER]; /* absorbed values */
int  gla_nlays = 0;             /* number of glazing layers */


m_glazing(m, r)         /* color a ray which hit an Enermodal glazing surface */
OBJREC  *m;
register RAY  *r;
{
	static int  nambsamps = 0;
	double  trans;
	MFUNC  *mf;
	RAY  p;
	double  d, sumv;
	register int  i;
					    /* check arguments */
	if (m->oargs.nfargs < 1 |
			m->oargs.farg[0] < 0.5 |
			m->oargs.farg[0] >= MAXLAYER+.5)
		objerror(m, USER, "bad number of layers");
	if (r->rod < 0.0)
		flipsurface(r);
						/* get function file */
	mf = getfunc(m, 2, 1, 0);
						/* compute transmittance */
	setfunc(m, r);
	trans = evalue(mf->ep[0]);
						/* transmitted ray */
	rayorigin(&p, r, TRANS, 1.0);
	VCOPY(p.rdir, r->rdir);
	rayvalue(&p);
	sumv = bright(p.rcol);
	scalecolor(p.rcol, trans);
	addcolor(r->rcol, p.rcol);
						/* compute absorption */
	if (r->parent != &thisray)
		return;			/* avoid overcounting */
	i = m->oargs.farg[0] + .5;
	if (i != gla_nlays) {                   /* clear absorption array */
		gla_nlays = i;
		while (i--)
			gla_absorbed[i] = 0.0;
	}
	switch (r->rtype) {
	case AMBIENT:
		if (nambsamps == 0) {
			nambsamps = sqrt(ambdiv/PI) + 0.5;
			nambsamps *= (int)(PI*nambsamps + 0.5);
			nambsamps += ambssamp;
		}
		sumv *= PI/nambsamps;
#ifdef DEBUG
		putc('A', stderr);
#endif
		break;
	case SHADOW:
		if (!(source[r->rsrc].sflags & SDISTANT))/* only for sun! */
			return;
		sumv *= r->rod*source[r->rsrc].ss2;
#ifdef DEBUG
		putc('S', stderr);
#endif
		break;
	default:
#ifdef DEBUG
		putc('I', stderr);
#endif
		return;
	}
	setfunc(m, r);
	for (i = 0; i < gla_nlays; i++) {
		d = i+1;
		gla_absorbed[i] += sumv * funvalue(m->oargs.sarg[1], 1, &d);
	}
}
