# util.tcl - Tcl utilities
#
# $Id: util.tcl,v 1.2 96/02/11 21:34:01 leech Exp $
#
# Copyright (C) 1996, Jonathan P. Leech
#
# This software may be freely copied, modified, and redistributed,
# provided that this copyright notice is preserved on all copies.
#
# There is no warranty or other guarantee of fitness for this software,
# it is provided solely "as is". Bug reports or fixes may be sent
# to the author, who may or may not act on them as he desires.
#
# You may not include this software in a program or other software product
# without supplying the source, or without informing the end-user that the
# source is available for no extra charge.
#
# If you modify this software, you should include a notice giving the
# name of the person performing the modification, the date of modification,
# and the reason for such modification.
#
set __rcslog {
$Log:	util.tcl,v $
Revision 1.2  96/02/11  21:34:01  leech
Added random number calls, since TclX isn't being used.

Revision 1.1  96/02/09	16:41:58  leech
Initial revision

}

global util_debug
set util_debug 0

# Constrain arg between 'min' and 'max'
proc constrain { arg min max } {
    if { $arg < $min } {
	return $min
    } elseif { $arg > $max } {
	return $max
    } else {
	return $arg
    }
}

# Round each element of a list, returning the new list
proc list_round { list } {
    foreach elem $list {
	lappend ret [expr round($elem)]
    }
    return $ret
}

# Multiply each element of a list by a scalar, returning the new list
proc list_mult { s list } {
    foreach elem $list {
	lappend ret [expr $s * $elem]
    }
    return $ret
}

# list_nextval - return the next value following val in the list (wraps).
#   list - list of values
#   val  - current value
proc list_nextval { list val } {
    set len [llength $list]
    set pos [expr [lsearch $list $val] + 1]
    if { $pos == $len } {
	set pos 0
    }
    return [lindex $list $pos]
}

# Return the argument with leading character removed
proc stripchar { s } {
    return [string range $s 1 end]
}

# Return the argument capitalized
proc capitalize { s } {
    set head [string index $s 0]
    set tail [string range $s 1 end]

    return [string toupper $head]$tail
}

# Print arguments iff global $util_debug is true
proc debug { args } {
    global util_debug
    if { $util_debug } {
	puts stdout "$args"
    }
}

# Destroy the specified list of windows
proc delwin { args } {
    foreach w $args {
	if [winfo exists $w] { destroy $w }
    }
}

# util_save_dir
#   When called with no arguments, returns a directory (usually that
#	invoked in).
#   When called with an argument, sets the directory to that argument.
proc util_save_dir { {dir ""} } {
    global util_save_path

    if { [string length $dir] > 0 } {
	set util_save_path $dir
    } else {
	return $util_save_path
    }
}

# selector - create a selector widget
#   widget  - widget to create
#   name    - label to place on the widget
#   choices - possible choices to select (list of value or display/value pairs)
#   state   - initial value to select
#   command - command prefix with new choice appended a selection is made
proc selector { widget name choices state command } {
    frame $widget -relief raised -border 3 ;# -bg gray
    label $widget.l -text "$name" ;# -bg gray
    pack $widget.l -side top

    set win 0
    foreach choice $choices {
	if { [llength $choice] < 2 } {
	    set callback $choice
	    set label [capitalize $callback]
	} else {
	    set label [lindex $choice 0]
	    set callback [lindex $choice 1]
	}

	set window $widget.choice$win
	radiobutton $window -value $callback -text $label \
	    -variable $widget.value -command "$command $callback"
	pack $window -side top -anchor w

	incr win
    }

    # Set initial selector state
    global $widget.value
    set $widget.value $state
}

# Like selector, but create a menubutton instead of a frame
proc selectmenu { widget name choices state command } {
    menubutton $widget -text $name -menu $widget.menu
    menu $widget.menu

    set win 0
    foreach choice $choices {
	if { [llength $choice] < 2 } {
	    set callback $choice
	    set label [capitalize $callback]
	} else {
	    set label [lindex $choice 0]
	    set callback [lindex $choice 1]
	}

	set window $widget.choice$win
	$widget.menu add radiobutton -value $callback -label $label \
	    -variable $widget.value -command "eval $command $callback"

	incr win
    }

    # Set initial selector state
    global $widget.value
    set $widget.value $state
}

# Return log (base 10) of passed value
proc log10 { x } {
    return [expr log($x) / log(10.)]
}

proc echo { args } {
    puts "echo: $args"
}

# Like TclX for_file command, but reads from a file descriptor
#   instead of opening a passed file. This helps construct
#   Tcl filters
proc for_fd { var fd command } {
    upvar $var buf

    set buf [gets $fd]
    while {! [eof $fd]} {
	uplevel $command
	set buf [gets $fd]
    }
}

# Cheesy random number generator, since we're not using TclX
global randx
set randx 1

proc srand { seed } {
    global randx ; set randx $seed
}

proc rand { } {
    global randx
    set randx [expr $randx * 1103515245 + 12345]
    return [expr $randx & 0x7fffffff]
}

# Returns [0,$range-1]
proc random { range } {
    return [expr (double([rand]) / 0x7fffffff) * $range]
}

# Returns [0,1)
proc frand { } {
    return [expr double([rand]) / 0x7fffffff]
}
