# Varedit - an [incr Tcl] 1.5 variable editor widget
#
# $Id: Varedit.tcl,v 1.1 96/02/09 16:42:06 leech Exp $
#
# Copyright (C) 1996, Jonathan P. Leech
#
# This software may be freely copied, modified, and redistributed,
# provided that this copyright notice is preserved on all copies.
#
# There is no warranty or other guarantee of fitness for this software,
# it is provided solely "as is". Bug reports or fixes may be sent
# to the author, who may or may not act on them as he desires.
#
# You may not include this software in a program or other software product
# without supplying the source, or without informing the end-user that the
# source is available for no extra charge.
#
# If you modify this software, you should include a notice giving the
# name of the person performing the modification, the date of modification,
# and the reason for such modification.
#
set __rcslog {
$Log:	Varedit.tcl,v $
Revision 1.1  96/02/09  16:42:06  leech
Initial revision

}

#
##############################################################################
#
# Configuration options:
#   -label	    Label of editor widget
#   -min	    Minimum allowed value
#   -max	    Maximum allowed value
#   -value	    Current value
#   -command	    Command prefix to be used when value changes. Callback is:
#		    $command $value
#   -integral	    Whether editor is constrained to integer values or not.
#		    Default is no constraint.
#   -state	    normal or disabled; if disabled, scrollbar disappears
#
# Public methods:
#   get

itcl_class Varedit {
    constructor {config} {
	#  Create a window with the same name as this object
	set class [$this info class]
	::rename $this $this-tmp-
	frame $this -class $class -relief raised
	::rename $this $this-win-
	::rename $this-tmp- $this

	# Menu button with the variable's name (so we can control parameters)
	button $this.l -width 20 -anchor w -text "$label" -relief flat \
		-command "$this control"

	# Entry controlling the variable
	global $this.value
	entry $this.e -width 10 -textvariable $this.value -relief sunken
	bind $this.e <Return> "$this returnKey $this.e"

	# Scrollbar controlling the variable
	scrollbar $this.s -orient horizontal -command "$this scroll"
	$this.s set $scrollend $scrollsize 0 $scrollsize

	# Calling var_editor_set sets the current value and updates the entry
	#  and scrollbar to match.
	set_value $value

	# Pack the widget
	repack

	# Object is now valid
	set created 1

	#  Explicitly handle config's that may have been ignored earlier
	foreach attr $config {
	    config -$attr [set $attr]
	}
    }

    method set_value { newvalue } {
## puts "$this set_value: $newvalue"
	# Constrain the variable to its allowed range
	set newvalue [constrain $newvalue $min $max]
# puts "  constrain: $newvalue"

	if { $integral } {
	    set newvalue [expr round($newvalue)]
# puts "  integral: $newvalue"
	}

	# Compute scaled scrollbar position and adjust the slider
	set pos [value_to_scroll $newvalue]
## puts "  value_to_scroll: $newvalue"
	$this.s set $scrollend $scrollsize $pos [expr $pos + $scrollsize]

	# Update the value and corresponding entry text variable
	set value $newvalue

	global $this.value
	::set $this.value $value
## puts "  at end: value = $value $this.value = [::set $this.value]"
    }

    method get {} {
# puts "$this get: $value"
	return $value
    }

    # control - pop up a control to modify the range of the editor
    method control { } {
	set text [lindex [$this.l configure -text] 4]
# puts "$this control: text $text"

	if [winfo exists $this.control] {
	    return
	}

	toplevel $this.control
	wm title $this.control "Range controller - $text"

	label $this.control.l -text "Range controller - $text"
	foreach f { min max } {
	    set w $this.control.$f
	    frame $w

	    label $w.l -text "$f" -width 4

	    global $w.val
	    entry $w.e -textvariable $w.val -relief sunken
	    bind $w.e <Return> "$this control_return $f"
	    pack $w.l $w.e -side left

	    ::set $w.val [set $f]
	}
	button $this.control.dismiss -text "Dismiss" -command "destroy $this.control"

	pack $this.control.l $this.control.min $this.control.max \
	     $this.control.dismiss -side top
    }

    # Called back when a return key is pressed in the varedit control widget
    method control_return { minmax } {
	global $this.control.$minmax.val
	upvar #0 $this.control.$minmax.val val

	switch $minmax {
	    min {
		$this range $val $max
	    }
	    max {
		$this range $min $val
	    }
	}
    }

    # range - modify the range of a variable editor; clamp
    #	the current value if needed.
    #
    #	newmin - new minimum value
    #	newmax - new maximum value
    method range { newmin newmax } {
# puts "$this range: $newmin $newmax"
	if { $newmin >= $newmax } {
	    error "$this range: min must be <= max"
	}
	if { $logscale && $newmin <= 0 } {
	    error "$this range: min must be > 0 for logarithmic scale"
	}

	set min $newmin
	set max $newmax
	# Recompute log scaling coefficients, if needed
	if { $logscale } {
	    compute_logcoeffs
	}

	# Reposition slider and constrain value
# puts "  calling set_value"
	set_value $value
    }

    # scroll - update the editor, changing the value in the entry
    #	and and adjusting the slider. The new value is specified
    #	as a scrollbar position.
    #
    #	pos  - scrollbar position to adjust to
    method scroll { pos } {
## puts "$this scroll: pos $pos"
	# Keep slider visible
	set pos [constrain $pos 0 $scrollrange]
## puts "  constrain: $pos"

	# Adjust the slider
	$this.s set $scrollend $scrollsize $pos [expr $pos + $scrollsize]

	# Update the entry and underlying variable
	set value [scroll_to_value $pos]
## puts "  scroll_to_value: value = $value"
## puts "  setting global $this.value = $value"
	global $this.value
	::set $this.value $value

	executeCommand
    }

    # executeCommand - call back a variable editor widget's command
    #	procedure (if non-empty) with the widget's value
    method executeCommand {} {
# puts "$this executeCommand: $command $value"
	if { [string length $command] > 0} {
	    ::eval $command $value
	}
    }

    # returnKey - update a variable editor widget from its
    #	entry. The new value is obtained from the contents of the entry.
    #
    #	entry - name of the entry widget return was typed in
    method returnKey { entry } {
# puts "$this returnKey: entry $entry"
	set field [$entry get]
	if { [scan $field "%f" value] == 1 } {
# puts "  good field: set_value $value"
	    set_value $value
	    executeCommand
	} else {
	    puts "$this: bad entry $field"
	}
    }

    # scroll_to_value - convert from a scrollbar position to
    #	a variable value. Used to scale between the slider and the
    #	underlying variable value displayed in the entry.
    #
    #	pos - slider position to convert from
    method scroll_to_value { pos } {
	if { $logscale } {
	    set s [expr double($pos) / $scrollrange]	;# 0..1
	    set val [expr pow(10,($s-$b)/$a)]	;# min..max
	    return $val
	} else {
	    set val [expr $min + $pos * double($max - $min) / $scrollrange]
	    if { $integral } {
		return [expr round($val)]
	    } else {
		return $val
	    }
	}
    }

    # value_to_scroll - convert from a variable value to a
    #	scrollbar position. Used to scale between the slider and the
    #	underlying variable value displayed in the entry.
    #
    #	name - name of variable editor referred to
    #	val  - value to convert from
    method value_to_scroll { val } {
	if { $logscale } {
	    set s [expr $a * [log10 $val] + $b]  ;# 0..1
	    return [expr round($s * $scrollrange)]
	} else {
	    return [expr round(double($val - $min) / ($max - $min) * $scrollrange)]
	}
    }

    # logarithmic - change to/from a logarithmic scale
    method logarithmic { flag } {
	if { $flag && $min <= 0 } {
	    error "$this logarithmic: min must be > 0 for logarithmic scale"
	}
	set logscale $flag
	if $flag {
	    compute_logcoeffs
	}
	set_value $value    ;# Change scrollbar, if needed
    }

    # compute_logcoeffs - recompute scaling coefficients
    method compute_logcoeffs { } {
	# Compute scaling coefficients
	set lmin [log10 $min]
	set lmax [log10 $max]
	set a [expr 1.0 / ($lmax - $lmin)]
	set b [expr -1.0 * $a * $lmin]
    }

    method change_state {} {
	if { $created } {
	    switch $state {
		disabled {
		    $this.l config -state disabled
		    $this.e config -state disabled
		    repack
		}
		normal {
		    $this.l config -state normal
		    $this.e config -state normal
		    repack
		}
		default {
		    ::error "$this: unrecognized state $state"
		    $this config -state normal
		}
	    }
	}
    }

    # Repack child widgets according to state
    method repack {} {
	pack forget $this.l $this.e $this.s
	pack $this.l $this.e -side left -anchor w
	if { ! [string compare $state normal] } {
	    pack $this.s -ipadx 1i -side left -anchor w -fill both
	}
    }

    # config - change public attributes
    method config {config} {}
    method configure {config} {}

    # destructor - destroy window containing widget
    destructor {
	::rename $this-win- {}	; # destroy $this - no longer needed
    }

    # Common data
    common scrollrange 1000	;# Discretization of the variable range (scrollbar size)
    common scrollsize  100	;# Size of the slider in the scrollbar
    common scrollend   1100	;# Scrollbar limit (so it doesn't run off end)

    # Public data
    public label "" {
	if { $created } {
	    $this.l configure -text "$label"
	}
    }
    public command ""	    ;# Command callback
    public value 0	    ;# Current value
    public integral 0	    ;# True if constrained to integers
    public min 0	    ;# Minimum value
    public max 100	    ;# Maximum value
    public state normal change_state	;# Widget state

    # Protected data
    protected created 0     ;# true if components of widget have been created
    protected logscale 0    ;# true if logarithmic scale
    protected a 1	    ;# scale for logarithmic
    protected b 0	    ;# offset for logarithmic
}
