/* predict.c, motion compensated prediction                                 */

/* Copyright (C) 1994, MPEG Software Simulation Group. All Rights Reserved. */

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any license fee or
 * royalty on an "as is" basis.  The MPEG Software Simulation Group disclaims
 * any and all warranties, whether express, implied, or statuary, including any
 * implied warranties or merchantability or of fitness for a particular
 * purpose.  In no event shall the copyright-holder be liable for any
 * incidental, punitive, or consequential damages of any kind whatsoever
 * arising from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs and user's
 * customers, employees, agents, transferees, successors, and assigns.
 *
 * The MPEG Software Simulation Group does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any third-party
 * patents.
 *
 * Commercial implementations of MPEG-1 and MPEG-2 video, including shareware,
 * are subject to royalty fees to patent holders.  Many of these patents are
 * general enough such that they are unavoidable regardless of implementation
 * design.
 *
 */

#include <stdio.h>
#include "config.h"
#include "global.h"

static void predict_mb _ANSI_ARGS_((
  unsigned char *oldref[], unsigned char *newref[], unsigned char *cur[],
  int lx, int bx, int by, int pict_type, int pict_struct, int mb_type,
  int motion_type, int secondfield,
  int PMV[2][2][2], int mv_field_sel[2][2], int dmvector[2]));

static void pred _ANSI_ARGS_((unsigned char *src[], int sfield,
  unsigned char *dst[], int dfield,
  int lx, int w, int h, int x, int y, int dx, int dy, int addflag));

static void pred_comp _ANSI_ARGS_((unsigned char *src, unsigned char *dst,
  int lx, int w, int h, int x, int y, int dx, int dy, int addflag));

void clearblock _ANSI_ARGS_((unsigned char *cur[], int lx, int i0, int j0));

/* only temporarily */
static void calc_DMV(){};

/* form prediction for a complete frame
 *
 * reff: reference frame for forward prediction
 * refb: reference frame for backward prediction
 * cur:  destination (current) frame
 * mbi:  macroblock info
 */

void predict(reff,refb,cur,mbi)
unsigned char *reff[],*refb[],*cur[3];
struct mbinfo *mbi;
{
  int i, j, k;

  k = 0;

  for (j=0; j<height; j+=16)
    for (i=0; i<width; i+=16)
    {
      if (mbi[k].mb_type&MB_INTRA)
        clearblock(cur,width,i,j);
      else
        predict_mb(reff,refb,cur,width,i,j,pict_type,FRAME_PICTURE,
                   mbi[k].mb_type,mbi[k].motion_type,0,
                   mbi[k].MV,mbi[k].mv_field_sel,mbi[k].dmvector);

      k++;
    }
}

/* form prediction for one macroblock
 *
 * already covers field pictures and dual prime (not yet used)
 */

static void predict_mb(oldref,newref,cur,lx,bx,by,pict_type,pict_struct,
  mb_type,motion_type,secondfield,PMV,mv_field_sel,dmvector)
unsigned char *oldref[],*newref[],*cur[];
int lx;
int bx,by;
int pict_type;
int pict_struct;
int mb_type;
int motion_type;
int secondfield;
int PMV[2][2][2], mv_field_sel[2][2], dmvector[2];
{
  int addflag, currentfield;
  unsigned char **predframe;
  int DMV[2][2];

  addflag = 0;
  if ((mb_type & MB_FORWARD) || (pict_type==P_TYPE))
  {
    if (pict_struct==FRAME_PICTURE)
    {
      if ((motion_type==MC_FRAME) || !(mb_type & MB_FORWARD))
      {
        /* frame-based prediction */
        pred(oldref,0,cur,0,
          lx,16,16,bx,by,PMV[0][0][0],PMV[0][0][1],addflag);
      }
      else if (motion_type==MC_FIELD) /* field-based prediction */
      {
        /* top field prediction */
        pred(oldref,mv_field_sel[0][0],cur,0,
          lx<<1,16,8,bx,by>>1,PMV[0][0][0],PMV[0][0][1]>>1,addflag);

        /* bottom field prediction */
        pred(oldref,mv_field_sel[1][0],cur,1,
          lx<<1,16,8,bx,by>>1,PMV[1][0][0],PMV[1][0][1]>>1,addflag);
      }
      else if (motion_type==MC_DMV) /* dual prime prediction */
      {
        /* calculate derived motion vectors */
        calc_DMV(DMV,dmvector,PMV[0][0][0],PMV[0][0][1]>>1);

        /* predict top field from top field */
        pred(oldref,0,cur,0,
          lx<<1,16,8,bx,by>>1,PMV[0][0][0],PMV[0][0][1]>>1,0);

        /* predict bottom field from bottom field */
        pred(oldref,1,cur,1,
          lx<<1,16,8,bx,by>>1,PMV[0][0][0],PMV[0][0][1]>>1,0);

        /* predict and add to top field from bottom field */
        pred(oldref,1,cur,0,
          lx<<1,16,8,bx,by>>1,DMV[0][0],DMV[0][1],1);

        /* predict and add to bottom field from top field */
        pred(oldref,0,cur,1,
          lx<<1,16,8,bx,by>>1,DMV[1][0],DMV[1][1],1);
      }
      else
      {
        /* invalid motion_type */
        if (!quiet)
          fprintf(stderr,"invalid motion_type\n");
      }
    }
    else /* TOP_FIELD or BOTTOM_FIELD */
    {
      /* field picture */
      currentfield = (pict_struct==BOTTOM_FIELD);
      /* determine which frame to use for prediction */
      if ((pict_type==P_TYPE) && secondfield
         && (currentfield!=mv_field_sel[0][0]))
        predframe = cur; /* same frame */
      else
        predframe = oldref; /* previous frame */
      if ((motion_type==MC_FIELD) || !(mb_type & MB_FORWARD))
      {
        /* field-based prediction */
        pred(predframe,mv_field_sel[0][0],cur,currentfield,
          lx<<1,16,16,bx,by,PMV[0][0][0],PMV[0][0][1],addflag);
      }
      else if (motion_type==MC_16X8)
      {
        pred(predframe,mv_field_sel[0][0],cur,currentfield,
          lx<<1,16,8,bx,by,PMV[0][0][0],PMV[0][0][1],addflag);

        /* determine which frame to use for lower half prediction */
        if ((pict_type==P_TYPE) && secondfield
           && (currentfield!=mv_field_sel[1][0]))
          predframe = cur; /* same frame */
        else
          predframe = oldref; /* previous frame */

        pred(predframe,mv_field_sel[1][0],cur,currentfield,
          lx<<1,16,8,bx,by+8,PMV[1][0][0],PMV[1][0][1],addflag);
      }
      else if (motion_type==MC_DMV) /* dual prime prediction */
      {
        /* calculate derived motion vectors */
        calc_DMV(DMV,dmvector,PMV[0][0][0],PMV[0][0][1]);

        /* predict from field of same parity */
        pred(oldref,currentfield,cur,currentfield,
          lx<<1,16,16,bx,by,PMV[0][0][0],PMV[0][0][1],0);

        /* predict from field of opposite parity */
        pred(predframe,!currentfield,cur,currentfield,
          lx<<1,16,16,bx,by,DMV[0][0],DMV[0][1],1);
      }
      else
      {
        /* invalid motion_type */
        if (!quiet)
          fprintf(stderr,"invalid motion_type\n");
      }
    }
    addflag = 1;
  }

  if (mb_type & MB_BACKWARD)
  {
    if (pict_struct==FRAME_PICTURE)
    {
      if (motion_type==MC_FRAME)
      {
        /* frame-based prediction */
        pred(newref,0,cur,0,
          lx,16,16,bx,by,PMV[0][1][0],PMV[0][1][1],addflag);
      }
      else /* field-based prediction */
      {
        /* top field prediction */
        pred(newref,mv_field_sel[0][1],cur,0,
          lx<<1,16,8,bx,by>>1,PMV[0][1][0],PMV[0][1][1]>>1,addflag);

        /* bottom field prediction */
        pred(newref,mv_field_sel[1][1],cur,1,
          lx<<1,16,8,bx,by>>1,PMV[1][1][0],PMV[1][1][1]>>1,addflag);
      }
    }
    else /* TOP_FIELD or BOTTOM_FIELD */
    {
      /* field picture */
      currentfield = (pict_struct==BOTTOM_FIELD);
      if (motion_type==MC_FIELD)
      {
        /* field-based prediction */
        pred(newref,mv_field_sel[0][1],cur,currentfield,
          lx<<1,16,16,bx,by,PMV[0][1][0],PMV[0][1][1],addflag);
      }
      else if (motion_type==MC_16X8)
      {
        pred(newref,mv_field_sel[0][1],cur,currentfield,
          lx<<1,16,8,bx,by,PMV[0][1][0],PMV[0][1][1],addflag);

        pred(newref,mv_field_sel[1][1],cur,currentfield,
          lx<<1,16,8,bx,by+8,PMV[1][1][0],PMV[1][1][1],addflag);
      }
      else
      {
        /* invalid motion_type */
        if (!quiet)
          fprintf(stderr,"invalid motion_type\n");
      }
    }
  }
}

static void pred(src,sfield,dst,dfield,lx,w,h,x,y,dx,dy,addflag)
unsigned char *src[];
int sfield;
unsigned char *dst[];
int dfield;
int lx;
int w, h;
int x, y;
int dx, dy;
int addflag;
{
  /* Y */
  pred_comp(src[0]+(sfield?lx>>1:0),dst[0]+(dfield?lx>>1:0),
    lx,w,h,x,y,dx,dy,addflag);

  /* Cb */
  pred_comp(src[1]+(sfield?lx>>2:0),dst[1]+(dfield?lx>>2:0),
    lx>>1,w>>1,h>>1,x>>1,y>>1,dx/2,dy/2,addflag);

  /* Cr */
  pred_comp(src[2]+(sfield?lx>>2:0),dst[2]+(dfield?lx>>2:0),
    lx>>1,w>>1,h>>1,x>>1,y>>1,dx/2,dy/2,addflag);
}

/* low level prediction routine
 *
 * src:     prediction source
 * dst:     prediction destination
 * lx:      line width (for both src and dst)
 * x,y:     destination coordinates
 * dx,dy:   half pel motion vector
 * w,h:     size of prediction block
 * addflag: store or add prediction
 */

static void pred_comp(src,dst,lx,w,h,x,y,dx,dy,addflag)
unsigned char *src;
unsigned char *dst;
int lx;
int w, h;
int x, y;
int dx, dy;
int addflag;
{
  int xint, xh, yint, yh;
  int i, j;
  unsigned char *s, *d;

  /* half pel scaling */
  xint = dx>>1;
  xh = dx & 1;
  yint = dy>>1;
  yh = dy & 1;

  /* origins */
  s = src+lx*(y+yint)+x+xint;
  d = dst+lx*y+x;

  if (!xh && !yh)
    if (addflag)
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
          d[i] = (unsigned int)(d[i]+s[i]+1)>>1;
        s+= lx;
        d+= lx;
      }
    else
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
          d[i] = s[i];
        s+= lx;
        d+= lx;
      }
  else if (!xh && yh)
    if (addflag)
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
          d[i] = (d[i] + ((unsigned int)(s[i]+s[i+lx]+1)>>1)+1)>>1;
        s+= lx;
        d+= lx;
      }
    else
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
          d[i] = (unsigned int)(s[i]+s[i+lx]+1)>>1;
        s+= lx;
        d+= lx;
      }
  else if (xh && !yh)
    if (addflag)
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
          d[i] = (d[i] + ((unsigned int)(s[i]+s[i+1]+1)>>1)+1)>>1;
        s+= lx;
        d+= lx;
      }
    else
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
          d[i] = (unsigned int)(s[i]+s[i+1]+1)>>1;
        s+= lx;
        d+= lx;
      }
  else /* if (xh && yh) */
    if (addflag)
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
          d[i] = (d[i] + ((unsigned int)(s[i]+s[i+1]+s[i+lx]+s[i+lx+1]+2)>>2)+1)>>1;
        s+= lx;
        d+= lx;
      }
    else
      for (j=0; j<h; j++)
      {
        for (i=0; i<w; i++)
          d[i] = (unsigned int)(s[i]+s[i+1]+s[i+lx]+s[i+lx+1]+2)>>2;
        s+= lx;
        d+= lx;
      }
}

void clearblock(cur,lx,i0,j0)
unsigned char *cur[];
int lx, i0, j0;
{
  int i, j;
  unsigned char *p;

  p = cur[0]+i0+lx*j0;

  for (j=0; j<16; j++)
  {
    for (i=0; i<16; i++)
      p[i] = 128;
    p+= lx;
  }

  p = cur[1]+(i0>>1)+(lx>>1)*(j0>>1);

  for (j=0; j<8; j++)
  {
    for (i=0; i<8; i++)
      p[i] = 128;
    p+= lx>>1;
  }

  p = cur[2]+(i0>>1)+(lx>>1)*(j0>>1);

  for (j=0; j<8; j++)
  {
    for (i=0; i<8; i++)
      p[i] = 128;
    p+= lx>>1;
  }
}
