/* mpeg2enc.c, main() and parameter file reading                            */

/* Copyright (C) 1994, MPEG Software Simulation Group. All Rights Reserved. */

/*
 * Disclaimer of Warranty
 *
 * These software programs are available to the user without any license fee or
 * royalty on an "as is" basis.  The MPEG Software Simulation Group disclaims
 * any and all warranties, whether express, implied, or statuary, including any
 * implied warranties or merchantability or of fitness for a particular
 * purpose.  In no event shall the copyright-holder be liable for any
 * incidental, punitive, or consequential damages of any kind whatsoever
 * arising from the use of these programs.
 *
 * This disclaimer of warranty extends to the user of these programs and user's
 * customers, employees, agents, transferees, successors, and assigns.
 *
 * The MPEG Software Simulation Group does not represent or warrant that the
 * programs furnished hereunder are free of infringement of any third-party
 * patents.
 *
 * Commercial implementations of MPEG-1 and MPEG-2 video, including shareware,
 * are subject to royalty fees to patent holders.  Many of these patents are
 * general enough such that they are unavoidable regardless of implementation
 * design.
 *
 */

#include <stdio.h>
#include <stdlib.h>

#define GLOBAL /* used by global.h */
#include "config.h"
#include "global.h"

/* private prototypes */
static void init _ANSI_ARGS_((void));
static void readparmfile _ANSI_ARGS_((char *fname));
static void readquantmat _ANSI_ARGS_((void));


int main(argc,argv)
int argc;
char *argv[];
{
  if (argc!=3)
  {
    printf("\n%s, %s\n",version,author);
    printf("Usage: mpeg2encode in.par out.m2v\n");
    exit(0);
  }

  /* read parameter file */
  readparmfile(argv[1]);

  /* read quantization matrices */
  readquantmat();

  /* open output file */
  if (!(outfile=fopen(argv[2],"wb")))
  {
    sprintf(errortext,"Couldn't create output file %s",argv[1]);
    error(errortext);
  }

  init();
  putseq();

  fclose(outfile);
  fclose(statfile);

  return 0;
}

static void init()
{
  int i;

  initbits();
  init_fdct();
  init_idct();

  /* field pictures not yet implemented */
  pict_struct=FRAME_PICTURE;
  
  /* Non-linear qscale not yet implemented */
  q_scale_type=0;  

  /* round picture dimensions to nearest multiple of 16 or 32 */
  mb_width = (horizontal_size+15)/16;
  mb_height = prog_seq ? (vertical_size+15)/16 : 2*((vertical_size+31)/32);
  width = 16*mb_width;
  height = 16*mb_height;

  init_vbv_delay();

  /* clip table */
  if (!(clp = (unsigned char *)malloc(1024)))
    error("malloc failed\n");
  clp+= 384;
  for (i=-384; i<640; i++)
    clp[i] = (i<0) ? 0 : ((i>255) ? 255 : i);

  for (i=0; i<3; i++)
  {
    if (!(newrefframe[i] = (unsigned char *)malloc(width*height/(i?4:1))))
      error("malloc failed\n");
    if (!(oldrefframe[i] = (unsigned char *)malloc(width*height/(i?4:1))))
      error("malloc failed\n");
    if (!(auxframe[i] = (unsigned char *)malloc(width*height/(i?4:1))))
      error("malloc failed\n");
    if (!(neworgframe[i] = (unsigned char *)malloc(width*height/(i?4:1))))
      error("malloc failed\n");
    if (!(oldorgframe[i] = (unsigned char *)malloc(width*height/(i?4:1))))
      error("malloc failed\n");
    if (!(auxorgframe[i] = (unsigned char *)malloc(width*height/(i?4:1))))
      error("malloc failed\n");
    if (!(predframe[i] = (unsigned char *)malloc(width*height/(i?4:1))))
      error("malloc failed\n");
  }

  mbinfo = (struct mbinfo *)malloc(mb_width*mb_height*sizeof(struct mbinfo));

  if (!mbinfo)
    error("malloc failed\n");

  blocks = (short (*)[6][64])malloc(mb_width*mb_height*sizeof(short [6][64]));

  /* open statistics output file */
  if (statname[0]=='-')
    statfile = stdout;
  else if (!(statfile = fopen(statname,"w")))
  {
    sprintf(errortext,"Couldn't create statistics output file %s",statname);
    error(errortext);
  }
}

void error(text)
char *text;
{
  fprintf(stderr,text);
  putc('\n',stderr);
  exit(1);
}

static void readparmfile(fname)
char *fname;
{
  int i;
  int h,m,s,f;
  double sample_rate;
  FILE *fd;
  char line[256];
  static double ratetab[8]=
    {24000.0/1001.0,24.0,25.0,30000.0/1001.0,30.0,50.0,60000.0/1001.0,60.0};
  extern double r,avg_act,Xi,Xb,Xp,d0i,d0p,d0b; /* rate control */

  if (!(fd = fopen(fname,"r")))
  {
    sprintf(errortext,"Couldn't open parameter file %s",fname);
    error(errortext);
  }

  fgets(id_string,254,fd);
  fgets(line,254,fd); sscanf(line,"%s",tplorg);
  fgets(line,254,fd); sscanf(line,"%s",tplref);
  fgets(line,254,fd); sscanf(line,"%s",iqname);
  fgets(line,254,fd); sscanf(line,"%s",niqname);
  fgets(line,254,fd); sscanf(line,"%s",statname);
  fgets(line,254,fd); sscanf(line,"%d",&inputtype);
  fgets(line,254,fd); sscanf(line,"%d",&nframes);
  fgets(line,254,fd); sscanf(line,"%d",&frame0);
  fgets(line,254,fd); sscanf(line,"%d:%d:%d:%d",&h,&m,&s,&f);
  fgets(line,254,fd); sscanf(line,"%d",&N);
  fgets(line,254,fd); sscanf(line,"%d",&M);
  fgets(line,254,fd); sscanf(line,"%d",&mpeg1);
  fgets(line,254,fd); sscanf(line,"%d",&horizontal_size);
  fgets(line,254,fd); sscanf(line,"%d",&vertical_size);
  fgets(line,254,fd); sscanf(line,"%d",&aspectratio);
  fgets(line,254,fd); sscanf(line,"%d",&frame_rate_code);
  fgets(line,254,fd); sscanf(line,"%lf",&bit_rate);
  fgets(line,254,fd); sscanf(line,"%d",&vbv_buffer_size);   
  fgets(line,254,fd); sscanf(line,"%d",&low_delay);     
  fgets(line,254,fd); sscanf(line,"%d",&constrparms);
  fgets(line,254,fd); sscanf(line,"%d",&profile);
  fgets(line,254,fd); sscanf(line,"%d",&level);
  fgets(line,254,fd); sscanf(line,"%d",&prog_seq);
  fgets(line,254,fd); sscanf(line,"%d",&video_format);
  fgets(line,254,fd); sscanf(line,"%d",&color_primaries);
  fgets(line,254,fd); sscanf(line,"%d",&transfer_characteristics);
  fgets(line,254,fd); sscanf(line,"%d",&matrix_coefficients);
  fgets(line,254,fd); sscanf(line,"%d",&display_horizontal_size);
  fgets(line,254,fd); sscanf(line,"%d",&display_vertical_size);
  fgets(line,254,fd); sscanf(line,"%d",&dc_prec);
  fgets(line,254,fd); sscanf(line,"%d",&topfirst);
  fgets(line,254,fd); sscanf(line,"%d %d %d",
    frame_pred_dct_tab,frame_pred_dct_tab+1,frame_pred_dct_tab+2);
  
  fgets(line,254,fd); sscanf(line,"%d %d %d",
    conceal_tab,conceal_tab+1,conceal_tab+2);
  
  fgets(line,254,fd); sscanf(line,"%d %d %d",
    qscale_tab,qscale_tab+1,qscale_tab+2);

  fgets(line,254,fd); sscanf(line,"%d %d %d",
    intravlc_tab,intravlc_tab+1,intravlc_tab+2);
  fgets(line,254,fd); sscanf(line,"%d %d %d",
    altscan_tab,altscan_tab+1,altscan_tab+2);
  fgets(line,254,fd); sscanf(line,"%d",&repeatfirst);
  fgets(line,254,fd); sscanf(line,"%d",&prog_frame);
/* intra slice interval refresh period */  
  fgets(line,254,fd); sscanf(line,"%d",&P);
  fgets(line,254,fd); sscanf(line,"%lf",&r);
  fgets(line,254,fd); sscanf(line,"%lf",&avg_act);
  fgets(line,254,fd); sscanf(line,"%lf",&Xi);
  fgets(line,254,fd); sscanf(line,"%lf",&Xp);
  fgets(line,254,fd); sscanf(line,"%lf",&Xb);
  fgets(line,254,fd); sscanf(line,"%lf",&d0i);
  fgets(line,254,fd); sscanf(line,"%lf",&d0p);
  fgets(line,254,fd); sscanf(line,"%lf",&d0b);

  if (N<1)
    error("N must be positive");
  if (M<1)
    error("M must be positive");
  if (N%M != 0)
    error("N must be an integer multiple of M");

  motion_data = (struct motion_data *)malloc(M*sizeof(struct motion_data));
  if (!motion_data)
    error("Couldn't allocate motion_data\n");

  for (i=0; i<M; i++)
  {
    fgets(line,254,fd);
    sscanf(line,"%d %d %d %d",
      &motion_data[i].forw_hor_f_code, &motion_data[i].forw_vert_f_code,
      &motion_data[i].sxf, &motion_data[i].syf);

    if (i!=0)
    {
      fgets(line,254,fd);
      sscanf(line,"%d %d %d %d",
        &motion_data[i].back_hor_f_code, &motion_data[i].back_vert_f_code,
        &motion_data[i].sxb, &motion_data[i].syb);
    }
  }

  fclose(fd);

  if (frame_rate_code<1 || frame_rate_code>8)
    error("frame_rate code must be between 1 and 8");

  picture_rate = ratetab[frame_rate_code-1];

  /* timecode -> frame number */
  tc0 = h;
  tc0 = 60*tc0 + m;
  tc0 = 60*tc0 + s;
  tc0 = (int)(picture_rate+0.5)*tc0 + f;

  /* make flags boolean (x!=0 -> x=1) */
  constrparms = !!constrparms;
  prog_seq = !!prog_seq;
  topfirst = !!topfirst;

  for (i=0; i<3; i++)
  {
    frame_pred_dct_tab[i] = !!frame_pred_dct_tab[i];
    intravlc_tab[i] = !!intravlc_tab[i];
    altscan_tab[i] = !!altscan_tab[i];
  }
  repeatfirst = !!repeatfirst;
  prog_frame = !!prog_frame;

  /* range checks */
  if ((horizontal_size&4095)==0)
    error("horizontal_size must not be a multiple of 4096");
  if (!(horizontal_size>0 && horizontal_size<16384))
    error("horizontal_size must be between 1 and 16383");
  if (mpeg1 && horizontal_size>4095)
    error("horizontal_size must be less than 4096");
  if (horizontal_size%2 != 0)
    error("horizontal_size must be a even (4:2:0)");
  if ((vertical_size&4095)==0)
    error("vertical_size must not be a multiple of 4096");
  if (!(vertical_size>0 && vertical_size<16384))
    error("vertical_size must be between 1 and 16383");
  if (mpeg1 && vertical_size>4095)
    error("vertical size must be less than 4096");
  if (vertical_size%2 != 0)
    error("vertical_size must be a even (4:2:0)");
  if (mpeg1)
  {
    if (aspectratio<1 || aspectratio>14)
      error("pel_aspect_ratio must be between 1 and 14");
  }
  else
  {
    if (aspectratio<1 || aspectratio>4)
      error("aspect_ratio_information must be 1, 2, 3 or 4");
  }
  if (bit_rate<=0.0)
    error("bit_rate must be positive");
  if (!(vbv_buffer_size>0 && vbv_buffer_size<0x40000))
    error("vbv_buffer_size must be in range 1..(2^18-1)");
  if (mpeg1 && vbv_buffer_size>=1024)
    error("vbv_buffer_size must be less than 1024");
  if (dc_prec<0 || dc_prec>3)
    error("intra_dc_precision must be in range 0...3");

  for (i=0; i<M; i++)
  {
    if (motion_data[i].forw_hor_f_code<1 || motion_data[i].forw_hor_f_code>9)
      error("f_code must be between 1 and 9");
    if (motion_data[i].forw_vert_f_code<1 || motion_data[i].forw_vert_f_code>9)
      error("f_code must be between 1 and 9");
    if (mpeg1 && motion_data[i].forw_hor_f_code>7)
      error("f_code must be le less than 8");
    if (mpeg1 && motion_data[i].forw_vert_f_code>7)
      error("f_code must be le less than 8");
    if (motion_data[i].sxf<=0)
      error("search window must be positive");
    if (motion_data[i].syf<=0)
      error("search window must be positive");
    if (i!=0)
    {
      if (motion_data[i].back_hor_f_code<1 || motion_data[i].back_hor_f_code>9)
        error("f_code must be between 1 and 9");
      if (motion_data[i].back_vert_f_code<1 || motion_data[i].back_vert_f_code>9)
        error("f_code must be between 1 and 9");
      if (mpeg1 && motion_data[i].back_hor_f_code>7)
        error("f_code must be le less than 8");
      if (mpeg1 && motion_data[i].back_vert_f_code>7)
        error("f_code must be le less than 8");
      if (motion_data[i].sxb<=0)
        error("search window must be positive");
      if (motion_data[i].syb<=0)
        error("search window must be positive");
    }
  }

  if (!mpeg1)
  {
    /* profile and level checks */
    if (profile<1 || profile>5)
      error("undefined Profile");
    if (level<4 || level>10 || level&1)
      error("undefined Level");

    if (profile<4)
      error("This encoder currently generates only Simple and Main profile bitstreams");
    if (profile>5 || profile<1)
      error("Profile out of legal range");
    if (profile==5 && M!=1)
      error("Simple profile does not allow B pictures");

    /* Table 8-8 */
    if (frame_rate_code>5 && level>=8)
      error("Picture rate greater than permitted in specified Level");

/* need to implement progressive_sequence and repeat first field check
   for picture rates of 25, 29.97, and 30 fps */

    for (i=0; i<M; i++)
    {
      if ((motion_data[i].forw_hor_f_code>7 && level==10)
          || (motion_data[i].forw_hor_f_code>8 && level==8)
          || (motion_data[i].forw_hor_f_code>9 && level<=6))
        error("forward vertical f_code greater than permitted in specified Level");

      if ((motion_data[i].forw_vert_f_code>4 && level==10)
          || (motion_data[i].forw_vert_f_code>4 && level<=8))
        error("forward vertical f_code greater than permitted in specified Level");

      if (i!=0)
      {
        if ((motion_data[i].back_hor_f_code>7 && level==10)
            || (motion_data[i].back_hor_f_code>8 && level==8)
            || (motion_data[i].back_hor_f_code>9 && level<=6))
          error("backward horizontal f_code greater than permitted in specified Level");

        if ((motion_data[i].back_vert_f_code>4 && level==10)
            || (motion_data[i].back_vert_f_code>4 && level<=8))
          error("backward vertical f_code greater than permitted in specified Level");
      }
    }

    /* Table 8-10 */
    if (horizontal_size>1920 && level==4)
      error("Horizontal size is greater than High Level limit");
    if (horizontal_size>1440 && level==6)
      error("Horizontal size is greater than High 1440 Level limit");
    if (horizontal_size>720 && level==8)
      error("Horizontal size is greater than Main Level limit");
    if (horizontal_size>352 && level==10)
      error("Horizontal size is greater than Low Level limit");

    if (vertical_size>1152 && level==4)
      error("Vertical size is greater than High Level limit");
    if (vertical_size>1152 && level==6)
      error("Vertical size is greater than High 1440 Level limit");
    if (vertical_size>576 && level==8)
      error("Vertical size is greater than Main Level limit");
    if (vertical_size>288 && level==10)
      error("Vertical size is greater than Low Level limit");

    /* Table 8-11 */
    sample_rate = horizontal_size*vertical_size*picture_rate;
    if (sample_rate>62668800.0 && level == 4)
      error("Sample rate is greater than High Level limit");
    if (sample_rate>47001600.0 && level == 6)
      error("Sample rate is greater than High 1440 Level limit");
    if (sample_rate>10368000.0 && level == 8)
      error("Sample rate is greater than Main Level limit");
    if (sample_rate>3041280.0 && level == 10)
      error("Sample rate is greater than Low Level limit");

/* need to add checks for illegal combinations of levels and profiles
   (e.g. SP@HL)  */

    /* Table 8-12 */
    if (bit_rate>80.0E6 && level==4)
      error("Bit rate exceeds High Level limit");
    if (bit_rate>60.0E6 && level==6)
      error("Bit rate exceeds High 1440 Level limit");
    if (bit_rate>15.0E6 && level==8)
      error("Bit rate exceeds Main Level limit");
    if (bit_rate>4.0E6 && level==10)
      error("Bit rate exceeds Low Level limit");

    /* Table 8-13 */
    if ((vbv_buffer_size*16*1024)>9781248 && level==4)
      error("vbv_buffer_size exceeds High Level limit");
    if ((vbv_buffer_size*16*1024)>7340032 && level==6)
      error("vbv_buffer_size exceeds High 1440 Level limit");
    if ((vbv_buffer_size*16*1024)>1835008 && level==8)
      error("vbv_buffer_size exceeds Main Level limit");
    if ((vbv_buffer_size*16*1024)>475136 && level==10)
      error("vbv_buffer_size exceeds Low Level limit");
  }
  else
  {
    /* MPEG-1 */
    if (constrparms)
    {
      if (horizontal_size>768
          || vertical_size>576
          || ((horizontal_size+15)/16)*((vertical_size+15)/16)>396
          || ((horizontal_size+15)/16)*((vertical_size+15)/16)*picture_rate>396*25.0
          || picture_rate>30.0)
      {
        if (!quiet)
          fprintf(stderr,"Warning: setting constrained_parameters_flag = 0\n");
        constrparms = 0;
      }
    }

    if (constrparms)
    {
      for (i=0; i<M; i++)
      {
        if (motion_data[i].forw_hor_f_code>4)
        {
          if (!quiet)
            fprintf(stderr,"Warning: setting constrained_parameters_flag = 0\n");
          constrparms = 0;
          break;
        }

        if (motion_data[i].forw_vert_f_code>4)
        {
          if (!quiet)
            fprintf(stderr,"Warning: setting constrained_parameters_flag = 0\n");
          constrparms = 0;
          break;
        }

        if (i!=0)
        {
          if (motion_data[i].back_hor_f_code>4)
          {
            if (!quiet)
              fprintf(stderr,"Warning: setting constrained_parameters_flag = 0\n");
            constrparms = 0;
            break;
          }

          if (motion_data[i].back_vert_f_code>4)
          {
            if (!quiet)
              fprintf(stderr,"Warning: setting constrained_parameters_flag = 0\n");
            constrparms = 0;
            break;
          }
        }
      }
    }
  }

  /* relational checks */

  if (mpeg1)
  {
    if (!prog_seq)
    {
      if (!quiet)
        fprintf(stderr,"Warning: setting progressive_sequence = 1\n");
      prog_seq = 1;
    }

    if (dc_prec!=0)
    {
      if (!quiet)
        fprintf(stderr,"Warning: setting intra_dc_precision = 0\n");
      dc_prec = 0;
    }

    for (i=0; i<3; i++)
      if (intravlc_tab[i])
      {
        if (!quiet)
          fprintf(stderr,"Warning: setting intravlc_tab[%d] = 0\n",i);
        intravlc_tab[i] = 0;
      }
    for (i=0; i<3; i++)
      if (altscan_tab[i])
      {
        if (!quiet)
          fprintf(stderr,"Warning: setting altscan_tab[%d] = 0\n",i);
        altscan_tab[i] = 0;
      }
  }

  if (!mpeg1 && constrparms)
  {
    if (!quiet)
      fprintf(stderr,"Warning: setting constrained_parameters_flag = 0\n");
    constrparms = 0;
  }

  if (prog_seq && !prog_frame)
  {
    if (!quiet)
      fprintf(stderr,"Warning: setting progressive_frame = 1\n");
    prog_frame = 1;
  }

  if (!prog_frame && repeatfirst)
  {
    if (!quiet)
      fprintf(stderr,"Warning: setting repeat_first_field = 0\n");
    repeatfirst = 0;
  }

  if (prog_frame)
  {
    for (i=0; i<3; i++)
      if (!frame_pred_dct_tab[i])
      {
        if (!quiet)
          fprintf(stderr,"Warning: setting frame_pred_frame_dct[%d] = 1\n",i);
        frame_pred_dct_tab[i] = 1;
      }
  }

  if (prog_seq && !repeatfirst && topfirst)
  {
    if (!quiet)
      fprintf(stderr,"Warning: setting top_field_first = 0\n");
    topfirst = 0;
  }

  /* search windows */
  for (i=0; i<M; i++)
  {
    if (motion_data[i].sxf > (4<<motion_data[i].forw_hor_f_code)-1)
    {
      if (!quiet)
        fprintf(stderr,
          "Warning: reducing forward horizontal search width to %d\n",
          (4<<motion_data[i].forw_hor_f_code)-1);
      motion_data[i].sxf = (4<<motion_data[i].forw_hor_f_code)-1;
    }

    if (motion_data[i].syf > (4<<motion_data[i].forw_vert_f_code)-1)
    {
      if (!quiet)
        fprintf(stderr,
          "Warning: reducing forward vertical search width to %d\n",
          (4<<motion_data[i].forw_vert_f_code)-1);
      motion_data[i].syf = (4<<motion_data[i].forw_vert_f_code)-1;
    }

    if (i!=0)
    {
      if (motion_data[i].sxb > (4<<motion_data[i].back_hor_f_code)-1)
      {
        if (!quiet)
          fprintf(stderr,
            "Warning: reducing backward horizontal search width to %d\n",
            (4<<motion_data[i].back_hor_f_code)-1);
        motion_data[i].sxb = (4<<motion_data[i].back_hor_f_code)-1;
      }

      if (motion_data[i].syb > (4<<motion_data[i].back_vert_f_code)-1)
      {
        if (!quiet)
          fprintf(stderr,
            "Warning: reducing backward vertical search width to %d\n",
            (4<<motion_data[i].back_vert_f_code)-1);
        motion_data[i].syb = (4<<motion_data[i].back_vert_f_code)-1;
      }
    }
  }

}

static void readquantmat()
{
  int i,v;
  FILE *fd;

  if (iqname[0]=='-')
  {
    /* use default intra matrix */
    load_iquant = 0;
    for (i=0; i<64; i++)
      intra_q[i] = default_intra_quantizer_matrix[i];
  }
  else
  {
    /* read customized intra matrix */
    load_iquant = 1;
    if (!(fd = fopen(iqname,"r")))
    {
      sprintf(errortext,"Couldn't open quant matrix file %s",iqname);
      error(errortext);
    }

    for (i=0; i<64; i++)
    {
      fscanf(fd,"%d",&v);
      if (v<1 || v>255)
        error("invalid value in quant matrix");
      intra_q[i] = v;
    }

    fclose(fd);
  }

  if (niqname[0]=='-')
  {
    /* use default non-intra matrix */
    load_niquant = 0;
    for (i=0; i<64; i++)
      inter_q[i] = 16;
  }
  else
  {
    /* read customized non-intra matrix */
    load_niquant = 1;
    if (!(fd = fopen(niqname,"r")))
    {
      sprintf(errortext,"Couldn't open quant matrix file %s",niqname);
      error(errortext);
    }

    for (i=0; i<64; i++)
    {
      fscanf(fd,"%d",&v);
      if (v<1 || v>255)
        error("invalid value in quant matrix");
      inter_q[i] = v;
    }

    fclose(fd);
  }
}
