/* nurbs.c */

/*
 * Mesa 3-D graphics library
 * Version:  1.2
 * Copyright (C) 1995  Brian Paul  (brianp@ssec.wisc.edu)
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/*
$Id: nurbs.c,v 1.12 1995/07/31 16:29:49 brianp Exp $

$Log: nurbs.c,v $
 * Revision 1.12  1995/07/31  16:29:49  brianp
 * applied Bogdan's patch from July 31, 1995
 *
 * Revision 1.11  1995/07/28  21:36:36  brianp
 * changed all GLUenum to GLenum
 *
 * Revision 1.10  1995/07/28  14:44:14  brianp
 * incorporated Bogdan's July 27 revisions
 *
 * Revision 1.9  1995/05/30  13:12:39  brianp
 * added gluNurbsCallback() stub
 *
 * Revision 1.8  1995/05/29  20:09:11  brianp
 * added gluGetNurbsProperty()
 *
 * Revision 1.7  1995/05/24  13:44:11  brianp
 * added gluBeginTrim, gluEndTrim, gluPwlCurve stubs
 *
 * Revision 1.6  1995/05/22  16:56:20  brianp
 * Release 1.2
 *
 * Revision 1.5  1995/05/16  19:17:21  brianp
 * minor changes to allow compilation with real OpenGL headers
 *
 * Revision 1.4  1995/04/28  20:06:23  brianp
 * print an error message when trying to use gluNewNurbsRenderer()
 *
 * Revision 1.3  1995/04/28  14:37:32  brianp
 * moved GLUnurbsObj struct from .h to .c file
 *
 * Revision 1.2  1995/03/04  19:39:18  brianp
 * version 1.1 beta
 *
 * Revision 1.1  1995/02/24  15:45:01  brianp
 * Initial revision
 *
 */


/*
 * NURBS implementation written by Bogdan Sikorski (gstbs@io.coi.pw.edu.pl)
 * See README-nurbs for more info.
 */


#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "gluP.h"
#include "nurbs.h"


void
call_user_error( GLUnurbsObj *nobj, GLenum error )
{
	nobj->error=error;
	if(nobj->error_callback != NULL) {
		(*(nobj->error_callback))(error);
	}
	else {
	   printf("NURBS error %d %s\n", error, gluErrorString(error) );
	}
}



GLUnurbsObj *gluNewNurbsRenderer( void )
{
   GLUnurbsObj *n;
   GLfloat tmp_viewport[4];
   GLint i,j;

   n = (GLUnurbsObj *) malloc( sizeof(GLUnurbsObj) );
   if (n) {
      /* init */
      n->culling=GL_FALSE;
      n->nurbs_type=GLU_NURBS_NONE;
      n->error=GLU_NO_ERROR;
      n->error_callback=NULL;
      n->auto_load_matrix=GL_TRUE;
      for(i=0;i<4;i++)
      	for(j=0;j<4;j++)
      		if(i==j)
      		{
				n->sampling_matrices.model[i*4+j]=1.0;
				n->sampling_matrices.proj[i*4+j]=1.0;
			}
			else
      		{
				n->sampling_matrices.model[i*4+j]=0.0;
				n->sampling_matrices.proj[i*4+j]=0.0;
			}
	  glGetFloatv(GL_VIEWPORT,tmp_viewport);
	  for(i=0;i<4;i++)
		  n->sampling_matrices.viewport[i]=tmp_viewport[i];
      n->sampling_tolerance=50.0;
      n->display_mode=GLU_FILL;
   }
   return n;
}



void gluDeleteNurbsRenderer( GLUnurbsObj *nobj )
{
   if (nobj) {
      free( nobj );
   }
}



void gluLoadSamplingMatrices( GLUnurbsObj *nobj,
			      const GLfloat modelMatrix[16],
			      const GLfloat projMatrix[16],
			      const GLint viewport[4] )
{
	GLint	i;

	for(i=0;i<16;i++)
	{
		nobj->sampling_matrices.model[i]=modelMatrix[i];
		nobj->sampling_matrices.proj[i]=projMatrix[i];
	}
	for(i=0;i<4;i++)
		nobj->sampling_matrices.viewport[i]=viewport[i];
}


void gluNurbsProperty( GLUnurbsObj *nobj, GLenum property, GLfloat value )
{
   GLenum val;

   switch (property) {
      case GLU_SAMPLING_TOLERANCE:
      	 if(value <= 0.0)
      	 {
      	 	call_user_error(nobj,GLU_INVALID_VALUE);
      	 	return;
		 }
         nobj->sampling_tolerance=value;
         break;
      case GLU_DISPLAY_MODE:
         val=(GLenum)value;
         if(val!=GLU_FILL && val!=GLU_OUTLINE_POLYGON && val!=GLU_OUTLINE_PATCH)
         {
         	call_user_error(nobj,GLU_INVALID_ENUM);
         	return;
		 }
		 if(nobj->nurbs_type==GLU_NURBS_CURVE)
         {
         	call_user_error(nobj,GLU_NURBS_ERROR26);
         	return;
		 }
if(val==GLU_OUTLINE_POLYGON)
fprintf(stderr,"for the moment NURBS can display only in GLU_OUTLINE_PATCH mode\n");
         nobj->display_mode=val;
         break;
      case GLU_CULLING:
         val=(GLenum)value;
         if(val!=GL_TRUE && val!=GL_FALSE)
         {
         	call_user_error(nobj,GLU_INVALID_ENUM);
         	return;
		 }
         nobj->culling = (GLboolean) value;
         break;
      case GLU_AUTO_LOAD_MATRIX:
         val=(GLenum)value;
         if(val!=GL_TRUE && val!=GL_FALSE)
         {
         	call_user_error(nobj,GLU_INVALID_ENUM);
         	return;
		 }
         nobj->auto_load_matrix = (GLboolean) value;
         break;
      default:
         call_user_error(nobj,GLU_NURBS_ERROR26);
   }
}


void gluGetNurbsProperty( GLUnurbsObj *nobj, GLenum property, GLfloat *value )
{
   switch (property) {
      case GLU_SAMPLING_TOLERANCE:
         *value = nobj->sampling_tolerance;
         break;
      case GLU_DISPLAY_MODE:
         *value = (GLfloat) nobj->display_mode;
         break;
      case GLU_CULLING:
	 *value = nobj->culling ? 1.0 : 0.0;
         break;
      case GLU_AUTO_LOAD_MATRIX:
         *value = nobj->auto_load_matrix ? 1.0 : 0.0;
	 break;
      default:
         call_user_error(nobj,GLU_INVALID_ENUM);
	 ;
   }
}



void gluBeginCurve( GLUnurbsObj *nobj )
{
	if(nobj->nurbs_type==GLU_NURBS_CURVE)
	{
		call_user_error(nobj,GLU_NURBS_ERROR6);
		return;
	}
	nobj->nurbs_type=GLU_NURBS_CURVE;
}


void gluEndCurve( GLUnurbsObj * nobj )
{
	if(nobj->nurbs_type==GLU_NURBS_NONE)
	{
		call_user_error(nobj,GLU_NURBS_ERROR7);
		return;
	}
	nobj->nurbs_type=GLU_NURBS_NONE;
}


void gluNurbsCurve( GLUnurbsObj *nobj, GLint nknots, GLfloat *knot,
		    GLint stride, GLfloat *ctlarray, GLint order, GLenum type )
{
	if(nobj->nurbs_type==GLU_NURBS_TRIM)
	{
		return;
	}
	if(type==GLU_MAP1_TRIM_2 || type==GLU_MAP1_TRIM_3)
	{
		call_user_error(nobj,GLU_NURBS_ERROR22);
		return;
	}
	if(nobj->nurbs_type!=GLU_NURBS_CURVE)
	{
		call_user_error(nobj,GLU_NURBS_ERROR10);
		return;
	}
switch(type)
{
	case GL_MAP1_VERTEX_3:
	case GL_MAP1_VERTEX_4:
		glPushAttrib(GL_EVAL_BIT | GL_ENABLE_BIT);
		glDisable(GL_MAP1_VERTEX_3);
		glDisable(GL_MAP1_VERTEX_4);
		do_nurbs_curve(nobj, nknots, knot, stride, ctlarray, order, type );
		glPopAttrib();
		break;
	case GL_MAP1_INDEX:
	case GL_MAP1_COLOR_4:
	case GL_MAP1_NORMAL:
	case GL_MAP1_TEXTURE_COORD_1:
	case GL_MAP1_TEXTURE_COORD_2:
	case GL_MAP1_TEXTURE_COORD_3:
	case GL_MAP1_TEXTURE_COORD_4:
		fprintf(stderr,"for the moment NURBS works only on geometric type\n");
		break;
	default:
         call_user_error(nobj,GLU_INVALID_ENUM);
}
}


void gluBeginSurface( GLUnurbsObj *nobj )
{
	nobj->nurbs_type=GLU_NURBS_SURFACE;
}


void gluEndSurface( GLUnurbsObj * nobj )
{
	if(nobj->nurbs_type==GLU_NURBS_TRIM)
	{
		fprintf(stderr,"trimmed NURBS not yet implemented\n");
		return;
	}
	else
	if(nobj->nurbs_type==GLU_NURBS_NO_TRIM)
	{
		glPushAttrib(GL_EVAL_BIT | GL_ENABLE_BIT);
		glDisable(GL_MAP2_VERTEX_3);
		glDisable(GL_MAP2_VERTEX_4);
		do_nurbs_surface(nobj);
		glPopAttrib();
	}
	nobj->nurbs_type=GLU_NURBS_NONE;
}


void gluNurbsSurface( GLUnurbsObj *nobj,
		      GLint sknot_count, GLfloat *sknot,
		      GLint tknot_count, GLfloat *tknot,
		      GLint s_stride, GLint t_stride,
		      GLfloat *ctrlarray,
		      GLint sorder, GLint torder,
		      GLenum type )
{
	switch(nobj->nurbs_type)
	{
		case GLU_NURBS_TRIM:
			fprintf(stderr,"trimmed NURBS not yet implemented\n");
			return;
		case GLU_NURBS_NO_TRIM:
			glPushAttrib(GL_EVAL_BIT | GL_ENABLE_BIT);
			glDisable(GL_MAP2_VERTEX_3);
			glDisable(GL_MAP2_VERTEX_4);
			do_nurbs_surface(nobj);
			glPopAttrib();
			nobj->nurbs_type=GLU_NURBS_SURFACE;
			/* no break - fall-through */
		case GLU_NURBS_SURFACE:
switch(type)
{
	case GL_MAP2_VERTEX_3:
	case GL_MAP2_VERTEX_4:
			nobj->surface.sknot_count=sknot_count;
			nobj->surface.sknot=sknot;
			nobj->surface.tknot_count=tknot_count;
			nobj->surface.tknot=tknot;
			nobj->surface.s_stride=s_stride;
			nobj->surface.t_stride=t_stride;
			nobj->surface.ctrlarray=ctrlarray;
			nobj->surface.sorder=sorder;
			nobj->surface.torder=torder;
			nobj->surface.type=type;
			nobj->nurbs_type=GLU_NURBS_NO_TRIM;
			break;
	case GL_MAP1_INDEX:
	case GL_MAP1_COLOR_4:
	case GL_MAP1_NORMAL:
	case GL_MAP1_TEXTURE_COORD_1:
	case GL_MAP1_TEXTURE_COORD_2:
	case GL_MAP1_TEXTURE_COORD_3:
	case GL_MAP1_TEXTURE_COORD_4:
		fprintf(stderr,"for the moment NURBS works only on geometric type\n");
		break;
	default:
         call_user_error(nobj,GLU_INVALID_ENUM);
}
			break;
		default:
			call_user_error(nobj,GLU_NURBS_ERROR11);
	}
}


void
gluNurbsCallback( GLUnurbsObj *nobj, GLenum which, void (*fn)())
{
	nobj->error_callback=fn;
	if(which!=GLU_ERROR)
		call_user_error(nobj,GLU_INVALID_ENUM);
}

void
gluBeginTrim( GLUnurbsObj *nobj )
{
	fprintf(stderr,"trimmed NURBS not yet implemented\n");
	nobj->nurbs_type=GLU_NURBS_TRIM;
}

void
gluPwlCurve( GLUnurbsObj *nobj, GLint cout, GLfloat *array, GLint stride,
	GLenum type)
{
	if(nobj->nurbs_type==GLU_NURBS_CURVE)
	{
		call_user_error(nobj,GLU_NURBS_ERROR9);
		return;
	}
	if(nobj->nurbs_type==GLU_NURBS_NONE)
	{
		call_user_error(nobj,GLU_NURBS_ERROR19);
		return;
	}
}

void
gluEndTrim( GLUnurbsObj *nobj )
{
	if(nobj->nurbs_type!=GLU_NURBS_TRIM)
	{
		call_user_error(nobj,GLU_NURBS_ERROR16);
		return;
	}
}
