/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/cdEdit/RCS/remoteControl.c,v 2.0 91/10/06 21:01:55 chua Exp $ */
/* $Log:	remoteControl.c,v $
 * Revision 2.0  91/10/06  21:01:55  chua
 * Update to version 2.0
 * 
 * Revision 1.35  91/09/10  13:35:38  chua
 * In eject, if file is untitled and modified, set the edit times to the
 * disc times.
 * 
 * Revision 1.34  91/09/05  18:19:00  chua
 * In Pause, set to pause mode only if the player is currently in play mode.
 * 
 * Revision 1.33  91/09/05  17:56:08  chua
 * In Play, if the player is already in play mode, do not do anything.
 * 
 * Revision 1.32  91/09/04  11:38:24  chua
 * In Eject, remove the status variable (not used).
 * 
 * Revision 1.31  91/09/03  15:26:00  chua
 * Added the copyright header.
 * 
 * The Options popup window has been removed, so the functions for opening and closing it
 * (Options, Done) are also deleted.
 * 
 * In Eject, do not wait till a new disc has been ejected.  Simply eject the disc and dim
 * the buttons.
 * 
 * A new function, InfoHandler, takes the place of the previous Info (notify procedure for
 * the Info button, which has been deleted).  This InfoHandler function will popup the
 * info popup window.
 * 
 * Revision 1.3  91/07/10  11:33:41  chua
 * In the Quit procedure, just call xv_destroy_safe (window).  This will
 * cause the QuitNotify interpose procedure to be called, which will handle
 * the other graceful quitting procedure (like checking if there are unsaved
 * changes etc).
 * 
 * Revision 1.2  91/07/10  11:04:21  chua
 * Removed a debugging printf statement.
 * 
 * Revision 1.1  91/07/09  16:48:09  chua
 * In the quit procedure, set the quit variable to 1 if quit is to be done, indicating that
 * there is no need to call quit anymore after xv_main_loop exits.
 * 
 * Revision 1.0  91/07/08  13:46:35  chua
 * Initial revision
 *  */

static char remoteControlrcsid[] = "$Header: /Source/Media/collab/cdEdit/RCS/remoteControl.c,v 2.0 91/10/06 21:01:55 chua Exp $";

#include "main.h"

int numTracks;							    /* number of tracks in disc */
int startTrack;
int endTrack;
int displayType;						    /* display time as absolute or relative */
static int clearframe = 0;
static int search = 0;

/*
 * Notify callback function for `OpenEditPopupButton'.
 * Opens the edit popup window.
 */
void Edit(item, event)
     Panel_item	item;
     Event		*event;
{
  xv_set(cdEdit_EditPopup->EditPopup, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
  xv_set(cdEdit_EditPopup->EditPopup, XV_SHOW, TRUE, NULL);
}

/*
 * Notify callback function for `QuitButton'.
 * Quit the application.  The CD is also ejected.
 */
void Quit(item, event)
     Panel_item	item;
     Event		*event;
{
  xv_destroy_safe(cdEdit_window1->window1);
}

/* 
 * This function will accept number input on the numeric pad using the mouse.  the current value on the search textfield is shifted to
 * the left as input is made (if 2 digits are exceeded) 
 */
void Num(n)
     int n;
{
  int value;
  
  if (clearframe == 1) 
  {
    xv_set(cdEdit_window1->SearchTrackText, PANEL_VALUE, 0,
	   NULL);
  }
  clearframe = 0;
  value = xv_get(cdEdit_window1->SearchTrackText, PANEL_VALUE);
  value = value*10 + n;
  if (value > 99) value = value % 100;				    /* Remove most significant digit */
  xv_set (cdEdit_window1->SearchTrackText, PANEL_VALUE, value, NULL);
}

/*
 * Notify callback function for `SearchButton'.
 * This function will cause the CD player to search to the track specified in the search track textfield.
 */
void Search(item, event)
     Panel_item	item;
     Event		*event;
{
  int searchNum;
  
  searchNum = xv_get(cdEdit_window1->SearchTrackText, PANEL_VALUE); /* Get the track to seek to */
  clearframe = 1;
  if (searchNum > endTrack || searchNum < 1)			    /* Check if the search track is valid */
  {
    AlertMessage("Illegal track number.", NULL);
    return;
  }
  playerState = PlayMode;
  cdrom_play_track(fd, searchNum, endTrack);
}

/*
 * Notify callback function for `SearchTrackText'.
 * If the return key is hit in the search track textfield, call the Search procedure above.  A flag is necessary to keep track of when Search is called 
 * because when the return key is hit, this function is called twice, but we only want to do the search once.
 */
Panel_setting SearchKbd(item, event)
     Panel_item	item;
     Event		*event;
{
  if (event_action(event) == '\r')				    /* Return key is pressed, go to search mode */
  {
    if (search == 0) 
    {
      Search(item, event);
      search = 1;
    }
    else 
    {
      search = 0;
    }
    return PANEL_NONE;
  }
  return PANEL_INSERT;
}

/*
 * Notify callback function for `DurationSlider'.
 * This function will calculate the offset into the CD to start play from, depending on where the user released the duration slider, and
 * whether the application was in absolute or relative mode.  It will then seek to that offset and start play from there.
 */
int Duration(item, value, event)
     Panel_item      item;
     int             value;
     Event           *event;
{
  int track, min, sec;
  struct cdrom_msf msf;
  Msf relative, result;
  char s[5];
  
  if (playerState != StopMode)					    /* Only do the seek if the CD is already playing */
  {
    playerState = DurationMode;					    /* So that the timer notify procedure will not set the slider to the current play time */
    sec = value;						    /* Concert the value on the slider to minutes and seconds */
    min = 0;
    while (sec >= 60) 
    {
      sec -= 60;
      min ++;
    }
    sprintf(s, "%02d:%02d", min, sec);
    xv_set(cdEdit_window1->TempDurationMsg,			    /* Update the time display on top of the duration slider */
	   PANEL_LABEL_STRING, s,
	   NULL);
    if (event_is_up(event))					    /* If mouse button is released, start play from the time at the release point */
    {
      if (displayType == Relative)				    /* Check if it is relative or absolute mode */
      {
	track = GetCurrentTrack();				    /* If it is in relative mode, convert the time to absolute time first */
	relative = (Msf) malloc(sizeof(struct msf));
	relative->min = min;
	relative->sec = sec;
	relative->frame = 1;
	result = (Msf) ConvertRelativeToAbsolute(track, relative);
	free (relative);
	msf.cdmsf_min0 = result->min;
	msf.cdmsf_sec0 = result->sec;
	msf.cdmsf_frame0 = result->frame;
      }
      else 
      {
	msf.cdmsf_min0 = min;
	msf.cdmsf_sec0 = sec;
	msf.cdmsf_frame0 = 1;
      }
      msf.cdmsf_min1 = toc->total_msf->min;			    /* Play till end of disc */
      msf.cdmsf_sec1 = toc->total_msf->sec; 
      msf.cdmsf_frame1 = toc->total_msf->frame;
      playerState = PlayMode;
      while (cdrom_play_msf(fd, msf) == 0) 
      {
	if (msf.cdmsf_frame1 == 0)				    /* If play doesn't work, reduce the maximum frame number by one until it does. */
	{							    /* This number to be reduced varies from disc to disc.  The reason is because */
	  if (msf.cdmsf_sec1 == 0)				    /* the value given by toc->total_msf->frame is not always accurate. */
	  {
	    msf.cdmsf_sec1 = 59;
	    msf.cdmsf_min1--;
	  }
	  else 
	  {
	    msf.cdmsf_sec1--;
	    msf.cdmsf_frame1 = 74;
	  }
	}
	else 
	{
	  msf.cdmsf_frame1 --;
	}
      }
    }
  }
}

/*
 * Notify callback function for `OneButton'.
 */
void One(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(1);
}

/*
 * Notify callback function for `TwoButton'.
 */
void Two(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(2);
}

/*
 * Notify callback function for `ThreeButton'.
 */
void Three(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(3);
}

/*
 * Notify callback function for `FourButton'.
 */
void Four(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(4);
}

/*
 * Notify callback function for `FiveButton'.
 */
void Five(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(5);
}

/*
 * Notify callback function for `SixButton'.
 */
void Six(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(6);
}

/*
 * Notify callback function for `SevenButton'.
 */
void Seven(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(7);
}

/*
 * Notify callback function for `EightButton'.
 */
void Eight(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(8);
}

/*
 * Notify callback function for `NineButton'.
 */
void Nine(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(9);
}

/*
 * Notify callback function for `ZeroButton'.
 */
void Zero(item, event)
     Panel_item	item;
     Event		*event;
{
  Num(0);
}

/*
 * Notify callback function for `PlayButton'.
 * Puts the CD player in play mode.  If the CD player was originally in stop or play mode, start play from track 1. 
 * If the CD player was in pause mode, resume play from the point where the pause was made.
 */
void Play(item, event)
     Panel_item	item;
     Event		*event;
{
  if (playerState == PlayMode)					    /* Do not do anything if the player is already in play mode */
    return;
  if (playerState == PauseMode) 
  {
    cdrom_resume(fd);
  }
  else 
  {
    cdrom_play_track(fd, startTrack, endTrack);
  }
  playerState = PlayMode;
}

/*
 * Notify callback function for `ReverseSkipButton'.
 * Skips to the previous track.  If it is at track 1, just play track 1 again.
 * If the player was in stop mode, play track 1.
 */
void ReverseSkip(item, event)
     Panel_item	item;
     Event		*event;
{
  int currentTrack;
  
  currentTrack = GetCurrentTrack() - 1;				   
  playerState = PlayMode;
  if (currentTrack <= 0) 
  {
    currentTrack = 1;
  }
  cdrom_play_track(fd, currentTrack, endTrack);
}

/*
 * Notify callback function for `PauseButton'.
 * Set the CD player in pause mode.
 */
void Pause(item, event)
     Panel_item	item;
     Event		*event;
{
  if (playerState == PauseMode) 
  {
    cdrom_resume(fd);
    playerState = PlayMode;
  }
  else if (playerState == PlayMode)
  {
    playerState = PauseMode;
    cdrom_pause(fd);
  }
}

/*
 * Notify callback function for `ForwardSkipButton'.
 * Skips to the next track.  If it is at the last track, play the last track again..
 * If the player was in stop mode, play track 1.
 */
void ForwardSkip(item, event)
     Panel_item	item;
     Event		*event;
{
  int currentTrack;
  
  currentTrack = GetCurrentTrack() + 1;
  playerState = PlayMode;
  if (currentTrack == (endTrack + 1)) 
  {
    currentTrack = endTrack;
  }
  cdrom_play_track(fd, currentTrack, endTrack);
}

/*
 * Notify callback function for `StopButton'.
 * Put the CD player in stop mode, and update the time display to show the entire time for the whole CD.
 */
void Stop(item, event)
     Panel_item	item;
     Event		*event;
{
  playerState = StopMode;
  cdrom_stop(fd);
}

/*
 * Notify callback function for `EjectButton'.
 * Ejects the CD.  As playerState is set to Nothing, the time display will automatically be updated to all zeros when the timer notify procedure
 * is next called.
 * If filename is 'untitled' and there are unsaved changes, update the edit times to be the same as the disc times, so that the times of this disc will
 * be saved should the user save the untitled file.
 */
void Eject(item, event)
     Panel_item      item;
     Event           *event;
{
  playerState = Nothing;
  cdrom_stop(fd);						    /* Stop the CD, then eject it */
  cdrom_eject(fd);
  destroy_toc(toc);						    /* Clear the table of contents */
  discInPlayer = 0;
  DimButtons(TRUE);
  xv_set(cdEdit_window1->window1, 
	 FRAME_RIGHT_FOOTER, "No disc in player",
	 NULL);
  if (CheckforUntitled(cdfilename) == OK && change == 1 &&	    /* If no file loaded, update the edit times with the disc times */
      editTotalTracks == 0)
  {
    editmin = toc->total_msf->min;
    editsec = toc->total_msf->sec;
    editTotalTracks = toc->size;
  }
}

/*
 * Sets the volume level according to the current values in the volume and balance sliders.
 */
void SetVolume(volume, balance)
     int volume, balance;
{
  int left, right;
  
  if (balance == -15)						    /* Balance on extreme left */
  {
    right = 0;
  }
  else 
  {
    right = volume + balance;
    if (right > 250) 
    {
      right = 250;
    }
  }
  if (balance == 15)						    /* balance on extreme right */
  {
    left = 0;
  }
  else 
  {
    left = volume - balance;
    if (left > 250) 
    {
      left = 250;
    }
  }
  cdrom_volume (fd, left, right);
}

/*
 * Notify callback function for `VolumeSlider'.
 * Call SetVolume to set the appropriate volume with respect to the volume and balance sliders.
 */
int Volume(item, value, event)
     Panel_item      item;
     int             value;
     Event           *event;
{
  int balance;
  
  balance = xv_get (cdEdit_window1->BalanceSlider, PANEL_VALUE);
  SetVolume(value, balance);
}

/*
 * Notify callback function for `BalanceSlider'.
 * Call SetVolume to set the appropriate volume with respect to the volume and balance sliders.
 */
int Balance(item, value, event)
     Panel_item      item;
     int             value;
     Event           *event;
{
  int volume;
  
  volume = xv_get (cdEdit_window1->VolumeSlider, PANEL_VALUE);
  SetVolume(volume, value);
  return 0;
}

/*
 * Menu handler for `OptionsMenu (About cdEdit ...)'.
 */
Menu_item InfoHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    xv_set(cdEdit_InfoPopup->InfoPopup, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
    xv_set(cdEdit_InfoPopup->InfoPopup, XV_SHOW, TRUE, NULL);
    
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

/*
 * Notify callback function for `CloseInfoPopupButton'.
 * Closes the Info popup window.
 */
void InfoDone(item, event)
     Panel_item	item;
     Event		*event;
{
  xv_set(cdEdit_InfoPopup->InfoPopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(cdEdit_InfoPopup->InfoPopup, XV_SHOW, FALSE, NULL);
}
