/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/TimeLine/RCS/pause.c,v 1.0 91/09/30 17:02:22 chua Exp Locker: drapeau $ */
/* $Log:	pause.c,v $
 * Revision 1.0  91/09/30  17:02:22  chua
 * Update to version 1.0
 * 
 * Revision 0.48  91/09/23  17:14:28  chua
 * The Insert/Delete Pause Marker menu item is now found under the Options
 * menu, instead of the Edit menu.
 * 
 * Revision 0.47  91/09/19  17:29:02  chua
 * Make sure that variables are initialized properly.  Change formatting slightly,
 * so that (if, for, while) statements with only one statement in them will not have
 * braces.
 * 
 * Revision 0.46  91/08/05  16:53:28  chua
 * Deleted the RepaintCanvas routine, as it is no longer necessary.  In places where it
 * is called, just call the ScrollToFirstQuarter routine, which will do the necessary
 * repaint as well.
 * 
 * Revision 0.45  91/08/05  13:05:15  chua
 * Changed the name of the function call, ScrollToMiddle to ScrollToFirstQuarter.
 * 
 * Revision 0.44  91/07/30  17:32:56  chua
 * When pause markers are inserted or deleted, set the change flag to 1.
 * 
 * Revision 0.43  91/07/26  17:27:35  chua
 * In UpdatePauseList, include an extra parameter, refresh, to determine if a 
 * canvas refresh is to be done after the updating.
 * 
 * Revision 0.42  91/07/24  10:55:10  chua
 * Split the InsertPause procedure into two, with InsertNewPause being the other.
 * InsertNewPause is called with a pointer to a new pause node as its parameter.
 * The reason the split is done is so that this function can be called by the OpenHandler
 * as well.
 * There is also a FreePause procedure to free all nodes in a pause list.
 * 
 * Revision 0.41  91/07/22  15:22:22  chua
 * 
 * 
 * Revision 0.40  91/07/22  15:03:12  chua
 * This file contains the routines that deals with inserting and deleting pause markers from the
 * TimeLine document.  The routines are:
 * 
 * InsertPauseMarkerHandler - opens the Pause popup window.
 * FindPause - returns a pointer to a pause node, given its position in the pause list.
 * PauseListNotify - notify procedure for the pause panel list.
 * UpdatePauseList - function which will update the panel list from information in the pause linked list.
 * InsertPause - inserts a new pause marker.
 * DeletePause - deletes a selected pause marker.
 * ClosePausePopup - closes the Pause popup window.
 * ClearAllPause - clears all the pause markers.
 *  */

static char gridrcsid[] = "$Header: /Source/Media/collab/TimeLine/RCS/pause.c,v 1.0 91/09/30 17:02:22 chua Exp Locker: drapeau $";

#include "main.h"

/*
 * Menu handler for `OptionsMenu (Insert/Delete Pause Marker)'.
 * This function will open the Pause popup window.
 */
Menu_item InsertPauseMarkerHandler(item, op)
     Menu_item	item;
     Menu_generate	op;
{
  TimeLineFramePtr tlFrame;
  TimeLine_window_objects * ip = (TimeLine_window_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(ip->controls, PANEL_CLIENT_DATA)];
  switch (op) 
  {
   case MENU_DISPLAY:
    break;
   case MENU_DISPLAY_DONE:
    break;
   case MENU_NOTIFY:
    xv_set(tlFrame->PausePopup->PausePopup,
	   FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
    xv_set(tlFrame->PausePopup->PausePopup,
	   XV_SHOW, TRUE, NULL);
    break;
   case MENU_NOTIFY_DONE:
    break;
  }
  return item;
}

/*
 * Function that will return a pointer to the appropriate pause node given its relative position in the pause list.
 */
Pause *FindPause(tlFrame)
     TimeLineFramePtr tlFrame;
{
  int 		i;
  Pause *currentPause;
  
  currentPause = tlFrame->pauseHead;					   
  for (i=0; i < tlFrame->pauseEdit && currentPause != NULL; i++)
    currentPause = currentPause->next;
  return currentPause;
}

/*
 * Notify callback function for `PauseList'.
 * On a PANEL_LIST_OP_DESELECT operation, this function will reset the selected pause message to zero, and also the pauseEdit and selectedPause
 * variables to indicate that no edit has been selected.
 * On a PANEL_LIST_OP_SELECT operation, this function will set the pauseEdit and selectedPause variables to point to the selected node and scroll
 * the canvas so that the pause marker will be visible.
 */
int PauseListNotify(item, string, client_data, op, event)
     Panel_item	item;
     char		*string;
     Xv_opaque	client_data;
     Panel_list_op	op;
     Event		*event;
{
  int selection;
  char selected[4];
  TimeLineFramePtr tlFrame;
  Pause_PausePopup_objects	*ip = (Pause_PausePopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  Window	owner = xv_get(ip->PausePopup, XV_OWNER);
  TimeLine_window_objects * tlip = (TimeLine_window_objects *) xv_get(owner, XV_KEY_DATA, INSTANCE);
  
  tlFrame = TimeLineWindow[xv_get(tlip->controls, PANEL_CLIENT_DATA)];
  switch(op) 
  {
   case PANEL_LIST_OP_DESELECT:
    tlFrame->pauseEdit = -1;
    tlFrame->selectedPause = NULL;
    sprintf(selected, "%d", tlFrame->pauseEdit + 1);
    xv_set(tlFrame->PausePopup->PauseSelectValueMsg, PANEL_LABEL_STRING, selected, NULL);
    break;
   case PANEL_LIST_OP_SELECT:
    sscanf(string, "%d", &selection);				    /* Get the number of the edit which has been selected */
    tlFrame->pauseEdit = selection - 1;
    tlFrame->selectedPause = (Pause *) FindPause(tlFrame);
    ScrollToFirstQuarter(tlFrame, tlFrame->selectedPause->position, 0);
    sprintf(selected, "%d", tlFrame->pauseEdit + 1);
    xv_set(tlFrame->PausePopup->PauseSelectValueMsg, PANEL_LABEL_STRING, selected, NULL);
    break;
   case PANEL_LIST_OP_VALIDATE:
    break;
   case PANEL_LIST_OP_DELETE:
    break;
  }
  return XV_OK;
}

/*
 * This function will update the pause panel list which indicates where all the pause markers are.
 * It goes through the pause list for the frame and updates the panel list accordingly.
 * The parameter, refresh, indicates if we need to do a canvas redraw.   This is usually necessary unless this function is called from
 * the OpenHandler, in which case the refresh is done at the end of the loading of file.
 */
void UpdatePauseList(tlFrame, refresh)
     TimeLineFramePtr tlFrame;
     int refresh;
{
  Pause *currentPause;
  char 	buf[100], numpauses[5], selected[4];
  int 	count = 0, i;
  int 	oldlines, replace;
  
  sprintf(numpauses, "%d", tlFrame->numPause);			    /* Update the display for the number of pauses */
  xv_set(tlFrame->PausePopup->PauseCountMsg,
	 PANEL_LABEL_STRING, numpauses, NULL);
  oldlines = xv_get(tlFrame->PausePopup->PauseList,		    /* Get the number of rows currently in the panel list */
		    PANEL_LIST_NROWS);
  replace = oldlines;
  if (tlFrame->numPause <= oldlines) 
    replace = tlFrame->numPause;
  currentPause = tlFrame->pauseHead;
  xv_set(tlFrame->PausePopup->PauseList,			    /* Hide the panel list while updating is done */
	 XV_SHOW, FALSE, NULL);
  for (i=0; i < replace; i++)					    /* Replace the old strings by the new ones. */
  {
    sprintf(buf, " %4d.          %4d:%02d",
	    count+1, currentPause->min, currentPause->sec); 
    xv_set (tlFrame->PausePopup->PauseList,
	    PANEL_LIST_STRING, i, buf,
	    PANEL_LIST_FONT, i, font,
	    NULL);
    count++;
    currentPause = currentPause->next;
  }
  if (tlFrame->numPause > oldlines) 
  {
    for (i=oldlines; i < tlFrame->numPause; i++)		    /* Insert the additional new strings */
    {
      sprintf(buf, " %4d.          %4d:%02d",
	      count+1, currentPause->min, currentPause->sec); 
      xv_set (tlFrame->PausePopup->PauseList,
	      PANEL_LIST_INSERT, i,
	      PANEL_LIST_STRING, i, buf,
	      PANEL_LIST_FONT, i, font,
	      NULL);
      count++;
      currentPause = currentPause->next;
    }
  }
  else							    /* Delete the excess old strings */
  {
    for (i=tlFrame->numPause; i < oldlines; i++)
      xv_set(tlFrame->PausePopup->PauseList,
	     PANEL_LIST_DELETE, tlFrame->numPause,
	     NULL);
  }
  if (tlFrame->pauseEdit >= 0) 
    xv_set(tlFrame->PausePopup->PauseList,			    /* Select the newly added entry on the panel list */
	   PANEL_LIST_SELECT, tlFrame->pauseEdit, 
	   TRUE, NULL);
  sprintf(selected, "%d", tlFrame->pauseEdit + 1);		    /* Update the selected pause field */
  xv_set(tlFrame->PausePopup->PauseSelectValueMsg, PANEL_LABEL_STRING, selected, NULL);
  xv_set(tlFrame->PausePopup->PauseList,
	 XV_SHOW, TRUE, NULL);
  if (refresh == 1) 
    DrawCanvasRepaintHandler(tlFrame->TimeLine_window->DrawCanvas, tlFrame->paintWinDraw, 
			     tlFrame->dpyDraw, tlFrame->xidDraw, NULL);
}

/*
 * This procedure will insert a new pause node in the pause list.  If the insert is successful, it will return the position of the new node in the
 * list.  If not, it will return -1.
 */
int InsertNewPause(tlFrame, newPause)
     TimeLineFramePtr tlFrame;
     Pause *newPause;
{
  int selected;
  int found;
  Pause *currentPause;
  
  selected = 0;
  if (tlFrame->pauseHead == NULL)				    /* Pause list is empty.  Insert at the beginning. */
    tlFrame->pauseHead = newPause;
  else								    /* Go down the note list to find the appropriate position to insert */
  {
    currentPause = tlFrame->pauseHead;
    if (currentPause->position == newPause->position)		    /* We do not want to insert two pause markers at the same place */
    {
      free (newPause);
      return -1;
    }
    found = 0;
    if (currentPause->position > newPause->position) 
    {
      found = 1;
      newPause->next = currentPause;
      tlFrame->pauseHead = newPause;
    }
    else while (currentPause->next != NULL && !found)
    {
      selected ++;
      if (currentPause->next->position == newPause->position)	    /* We do not want to insert two pause markers at the same place */
      {
	free (newPause);
	return -1;
      }
      if (currentPause->next->position < newPause->position) 
	currentPause = currentPause->next;
      else 
      {
	newPause->next = currentPause->next;
	currentPause->next = newPause;
	found = 1;
      }
    }
    if (!found)							    /* Insert at the end of list */
    {
      selected ++;
      currentPause->next = newPause;
    }
  }
  return selected;
}

/*
 * Notify callback function for `InsertPauseButton'.
 * Inserts a new pause marker at either the insertion point or specified time, depending on which checkbox the user has chosen.
 * A new pause marker would not be inserted in the pause marker list if it starts at zero, or there is another pause marker that is already in the list
 * having the same time.
 */
void InsertPause(item, event)
     Panel_item	item;
     Event		*event;
{
  Pause *newPause;
  int selected;
  TimeLineFramePtr tlFrame;
  Pause_PausePopup_objects	*ip = (Pause_PausePopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  Window	owner = xv_get(ip->PausePopup, XV_OWNER);
  TimeLine_window_objects * tlip = (TimeLine_window_objects *) xv_get(owner, XV_KEY_DATA, INSTANCE);

  tlFrame = TimeLineWindow[xv_get(tlip->controls, PANEL_CLIENT_DATA)];
  newPause = (Pause *) malloc (sizeof(struct Pause));
  newPause->next = NULL;
  if (xv_get(ip->PauseChoice, PANEL_VALUE) == 0)		    /* Insert at insertion point */
  {
    if (tlFrame->lastX < 0)					    /* If insertion point is not on the canvas, return */
    {
      free (newPause);
      return;
    }
    newPause->position = tlFrame->lastX;
    newPause->min = atoi(xv_get(ip->IPMinText, PANEL_VALUE));
    newPause->sec = atoi((int) xv_get(ip->IPSecText, PANEL_VALUE));
  }
  else								    /* Insert at the specified time */
  {
    newPause->sec = xv_get(ip->SPSecText, PANEL_VALUE);
    newPause->min = xv_get(ip->SPMinText, PANEL_VALUE);
    while (newPause->sec >= 60) 
    {
      newPause->min ++;
      newPause->sec -= 60;
    }
    newPause->position = newPause->min * 60 * PixelsPerSecond +	    /* Calculate the position in terms of pixels */
      newPause->sec * PixelsPerSecond;
  }
  if (newPause->position == 0)					    /* No point in inserting a pause marker at zero time */
  {
    free (newPause);
    return;
  }
  if ((selected = InsertNewPause(tlFrame, newPause)) == -1) 
  {
    free (newPause);
    return;
  }
  tlFrame->numPause++;
  tlFrame->pauseEdit = selected;
  tlFrame->selectedPause = newPause;
  ScrollToFirstQuarter(tlFrame, newPause->position, 1);		    /* Scroll to make the newly inserted pause marker visible */
  UpdatePauseList(tlFrame, 1);
  tlFrame->change = 1;						    /* Set the change flag to 1 */
  UpdateHeader(tlFrame, 1);
}

/*
 * Notify callback function for `DeletePauseButton'.
 * Deletes a pause marker (only if one has been selected from the panel list).
 */
void DeletePause(item, event)
     Panel_item	item;
     Event		*event;
{
  TimeLineFramePtr tlFrame;
  Pause *currentPause, *prevPause;
  Pause_PausePopup_objects	*ip = (Pause_PausePopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  int found;
  Window	owner = xv_get(ip->PausePopup, XV_OWNER);
  TimeLine_window_objects * tlip = (TimeLine_window_objects *) xv_get(owner, XV_KEY_DATA, INSTANCE);
  
  tlFrame = TimeLineWindow[xv_get(tlip->controls, PANEL_CLIENT_DATA)];
  if (tlFrame->pauseHead == NULL || tlFrame->pauseEdit == -1)	    /* Pause list is empty, or there is no edit chosen */
    return;	
  xv_set(tlFrame->PausePopup->PauseList,			    /* Deselect the selected entry on the panel list */
	 PANEL_LIST_SELECT, tlFrame->pauseEdit, 
	 FALSE, NULL);
  currentPause = tlFrame->pauseHead;
  if (currentPause == tlFrame->selectedPause) 
    tlFrame->pauseHead = currentPause->next;
  else 
  {
    prevPause = currentPause;
    currentPause = currentPause->next;
    found = 0;
    while (currentPause != NULL && !found) 
    {
      if (currentPause == tlFrame->selectedPause) 
      {
	prevPause->next = currentPause->next;
	found = 1;
      }
      prevPause = currentPause;
      if (!found) 
	currentPause = currentPause->next;
    }
  }
  tlFrame->numPause --;
  tlFrame->pauseEdit = -1;
  tlFrame->selectedPause = NULL;
  free (currentPause);
  UpdatePauseList(tlFrame, 1);
  tlFrame->change = 1;						    /* Set the change flag to 1 */
  UpdateHeader(tlFrame, 1);
}

/*
 * Notify callback function for `PauseDoneProc'.
 * Closes the pause popup window.
 */
void ClosePausePopup(item, event)
     Panel_item	item;
     Event		*event;
{
  Pause_PausePopup_objects	*ip = (Pause_PausePopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);

  xv_set(ip->PausePopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
  xv_set(ip->PausePopup, XV_SHOW, FALSE, NULL);
}

/*
 * Notify callback function for `ClearAllPauseButton'.
 * This function will clear all the pause markers.  
 */
void ClearAllPause(item, event)
     Panel_item	item;
     Event		*event;
{
  TimeLineFramePtr tlFrame;
  Pause_PausePopup_objects	*ip = (Pause_PausePopup_objects *) xv_get(item, XV_KEY_DATA, INSTANCE);
  Window	owner = xv_get(ip->PausePopup, XV_OWNER);
  TimeLine_window_objects * tlip = (TimeLine_window_objects *) xv_get(owner, XV_KEY_DATA, INSTANCE);
  
  tlFrame = TimeLineWindow[xv_get(tlip->controls, PANEL_CLIENT_DATA)];
  if (tlFrame->numPause > 0) 
  {
    FreePause(tlFrame);
    UpdatePauseList(tlFrame, 1);
    tlFrame->change = 1;					    /* Set the change flag to 1 */
    UpdateHeader(tlFrame, 1);
  }
}

/*
 * This procedure will free all the pause nodes in a pause list.
 */
void FreePause(tlFrame)
     TimeLineFramePtr tlFrame;
{
  Pause *pause, *freePause;
  
  pause = tlFrame->pauseHead;					    /* Free all the pause nodes */
  while (pause != NULL) 
  {
    freePause = pause;
    pause = pause->next;
    free (freePause);
  }
  tlFrame->pauseHead = NULL;
  tlFrame->pauseEdit = -1;
  tlFrame->selectedPause = NULL;
  tlFrame->numPause = 0;
}
