/*
 * Copyright (c) 1990, 1991 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */

/* $Header: /Source/Media/collab/TimeLine/RCS/TimeLine.c,v 1.15 92/10/30 16:20:48 drapeau Exp $ */
/* $Log:	TimeLine.c,v $
 * Revision 1.15  92/10/30  16:20:48  drapeau
 * Modified the UpdateHeader() function to indicate whether synchronization
 * hints are available for the current document.
 * 
 * Revision 1.14  92/10/07  15:48:33  drapeau
 * Added function Usage() to describe command-line options to the application.
 * 
 * Revision 1.13  92/10/07  15:14:38  drapeau
 * New version of TimeLine adds two command-line options: "-autoPlay" and
 * "-autoQuit".  "-autoPlay" indicates that TimeLine should immediately begin
 * performance of the document given on the command line (if there is one).
 * "-autoQuit" indicates that TimeLine should quit immediately after
 * completion of performance of the document given on the command line.
 * The "-autoQuit" flag is only valid if "-autoPlay" is also specified.
 * These two options were added to support automated demos using TimeLine.
 * 
 * Revision 1.12  92/09/24  17:01:11  drapeau
 * One minor code change; the rest are cosmetic.
 * * A new document format has been added, to accommodate documents with
 *   synchronization information.  The new header for these documents
 *   is "#TimeLine Edit Document, Version 2#".  Now TimeLine will be able to
 *   load both older-style documents and the new style.
 * * The cosmetic changes are to improve readability.
 * 
 * Revision 1.11  92/09/15  14:12:00  drapeau
 * Removed extraneous include file: "<gdd.h>", since it does not seem to be used
 * by anything in the application.
 * 
 * Revision 1.10  92/01/03  17:25:50  drapeau
 * Minor change to GetUnknownIcon() to reflect new location of the
 * include file containing the icon data.
 * 
 * Revision 1.0  91/09/30  16:46:47  chua
 * Update to version 1.0
 * 
 * Add TimeLinePort as a member of the tlFrame structure.
 * 
 * In CheckChanges, include a new case, ConnectPM (when connecting with a 
 * new port manager), and delete the UpdateApps case.
 * 
 * Revision 0.81  91/09/25  13:45:08  chua
 * In QuitNotify, proceed to destroy other TimeLine frames if quitAll was set to 1.
 * 
 * Revision 0.80  91/09/23  16:59:42  chua
 * Introduced a new variable, noteInstrument, which keeps track of which instrument
 * contains the currently selected note (there can only be ONE currently selected note).
 * 
 * Got rid of the stop menu and replaced it by a stop and pause button represented
 * by icon symbols.
 * 
 * Revision 0.79  91/09/20  15:04:14  chua
 * Initialize the cursor type to normal.
 * 
 * Revision 0.78  91/09/19  18:32:32  chua
 * Changed the parent of the Region popup to the TimeLine frame.
 * 
 * Revision 0.77  91/09/19  17:27:49  chua
 * Make sure that variables are initialized properly.  Change formatting slightly,
 * so that (if, for, while) statements with only one statement in them will not have
 * braces.
 * 
 * Revision 0.76  91/09/17  17:11:24  chua
 * In TimeLineInit, in the call to OpenAppsInitialize, add a new parameter, to indicate
 * if unassigned apps in the list from the port manager should be added to the 
 * instrument list.
 * 
 * Revision 0.75  91/09/16  14:35:36  chua
 * Include the Browse.h file and do the gethostname here instead of in
 * OpenAppsInitialize (need only do it once).
 * 
 * Revision 0.74  91/09/06  17:31:14  chua
 * In TimeLineInit, initialize TimeLineSender to NULL.
 * 
 * Revision 0.73  91/09/04  17:59:05  chua
 * Add a new function, GetUnknownIcon, which will create the server image
 * for the unknown icon from the icon file.
 * 
 * Revision 0.72  91/09/04  15:07:16  chua
 * Include the getopt.h file from the ~collab/GetOpt directory rather than
 * in the local directory.
 * 
 * Revision 0.71  91/09/04  15:01:12  chua
 * Removed the TIMELINE_HOME environment necessity, since all the media 
 * applications now support the GetAppIcon protocol message.
 * 
 * In TimeLineInit, check that a sender is not NULL before attempting to
 * create a new receiver.
 * 
 * Use frame_set_rect to position the TimeLine frame so that it is always
 * at the same position at startup.
 * 
 * Added a new function, AlertMessage, which simply calls a notice_prompt
 * to display the messages that are passed in as parameters.
 * 
 * Revision 0.70  91/08/26  14:20:50  chua
 * Deleted the actualApps variable from the tlFrame structure.
 * 
 * Revision 0.69  91/08/22  14:59:16  chua
 * In TimeLineInit (line 428), substitute the call to OpenHandler with a call to
 * Browse (with parameter BrowseCheckOpen), which will do the job of checking if the
 * file is valid.
 * 
 * Make the declaration of startFilename static.
 * 
 * Revision 0.68  91/08/16  16:54:22  chua
 * Changed the name of the CreateOpenPanel call to CreateBrowse.
 * 
 * Revision 0.67  91/08/12  15:51:19  chua
 * In TimeLineInit (line 390), change Untitled to untitled.
 * In the CheckChanges routine, add a new option, CloseFile.
 * 
 * Revision 0.66  91/08/08  14:20:16  chua
 * In TimeLineInit, as the object handler to OpenPopup has been removed, just make a
 * call to CreateOpenPanel and not assign the return value to anything.
 * 
 * Revision 0.65  91/07/30  11:19:45  chua
 * Incorporated the file browser.  In TimeLineInit, create the file browser popup
 * window.
 * 
 * Revision 0.64  91/07/26  17:19:55  chua
 * In TimeLineInit, include the initialization of the Region popup window and the
 * variables associated with the annotated regions.
 * 
 * Revision 0.63  91/07/22  15:13:49  chua
 * In TimeLineInit, include the initialization of the Pause popup window and the variables
 * associated with the pause markers.
 * 
 * Revision 0.62  91/07/18  15:03:14  chua
 * In the TimeLineInit routine, initialize the new variable, gridSpacing to 0.
 * 
 * Revision 0.61  91/07/17  16:29:19  chua
 * In the TimeLineInit routine, also initialize the Info popup window.
 * 
 * Revision 0.60  91/07/17  10:12:54  chua
 * Changed the canvas length to 1800 pixels instead of 6000 previously.  The capability
 * to handle infinitely long TimeLine documents is now incorporated.
 * 
 * Revision 0.59  91/07/09  16:52:02  chua
 * Added a new procedure, QuitNotify, which is called when we are trying to destroy a 
 * frame, either using the frame's quit menu, or by choosing the quit menu button from
 * the TimeLine.
 * 
 * Revision 0.58  91/06/26  16:49:18  chua
 * Added a new function CheckOptions, which will take the command line and parse the commands.
 * The user can specify the hostname and portnumber (the latter is not done at the moment).
 * 
 * Revision 0.57  91/06/24  16:28:29  chua
 * Changed the error message for reminding the user to set the TIMELINE_HOME environment. (It was 
 * previously TimeLineHome instead).
 * 
 * The TimeLineLength field in the TimeLineFrame structure now refers to length in pixels, instead of
 * in seconds.
 * 
 * In the TimeLineInit function, add code to disable to stop button (until play is activated).
 * 
 * Minor wording changes to the UpdateHeader function.
 * 
 * Revision 0.56  91/06/05  16:11:25  chua
 * Added some additional comments.  No new code.
 * 
 * Revision 0.55  91/06/04  17:36:47  chua
 * Added the copyright comments in the beginning of the file.
 * 
 * Revision 0.54  91/06/04  17:16:14  chua
 * Added code to differentiate between a Clipboard frame and a Document frame in the 
 * TimeLineInit and UpdateHeader routines.
 * 
 * Revision 0.53  91/06/04  10:40:12  chua
 * Added a new function, UpdateHeader to set the header of a frame.  This
 * is to distinguish between the multiple documents of a TimeLine.
 * The filename currently being edited will be displayed in the header, as
 * well as whether it has been modified.
 * 
 * Revision 0.52  91/05/31  16:05:54  chua
 * Created a new function TimeLineInit to facilitate the forming of multiple documents.
 * Make changes to the data structure in main.h so that all the data required by the
 * TimeLine document is encapsulated in a struct.
 * 
 * Revision 0.51  91/05/28  12:08:22  chua
 * 
 * 
 * Revision 0.50  91/05/24  16:31:23  chua
 * In the CheckChanges procedure, check also that the number of applications
 * is greater than zero before displaying the notice prompt.
 * 
 * Revision 0.49  91/05/23  17:27:42  chua
 * Included the initialization for the Insert Blank Time pop-up window.
 * Also changed the length of the TimeLine to one hour instead of ten minutes.
 * 
 * Revision 0.48  91/05/22  16:37:26  chua
 * *** empty log message ***
 * 
 * Revision 0.47  91/05/22  13:53:45  chua
 * *** empty log message ***
 * 
 * Revision 0.46  91/05/22  11:35:13  chua
 * *** empty log message ***
 * 
 * Revision 0.45  91/05/17  16:58:39  chua
 * *** empty log message ***
 * 
 * Revision 0.44  91/05/17  16:57:33  chua
 * *** empty log message ***
 * 
 * Revision 0.43  91/05/17  16:56:33  chua
 * *** empty log message ***
 * 
 * Revision 0.42  91/05/16  15:18:59  chua
 * Moved the Quit procedure to file.c  (file.c contains all the menu handlers for the Document
 * menu button).
 * 
 * Revision 0.41  1991/04/24  00:46:49  chua
 * Added a new line to initialize the global variable 'areaSelected'.
 *
 * Revision 0.40  1991/03/29  01:23:08  chua
 * This file contains the main procedure of the TimeLine application and two short procedures.
 * The main procedure will initialize XView, check that the TIMELINE_HOME environment variable has been set, and
 * that the user has passed in a hostname for the PortManager.
 * It will then connect with the PortManager and initialize both a sender and receiver.
 * Next, the XView user interface components are initialized as well as some global variables.  The TimeLine
 * application is next initialized by calling OpenAppsInitialize to get the applications currently registered with
 * the PortManager and calling CanvasInit to initialize the canvases.
 * After enabling the TimeLine application to detect RPC events, the xv_main_loop function is called which
 * essentially turns over control of the program to XView.  Now it is up to the user to 'trigger' off any of the
 * functions in the TimeLine application depending on which button was pressed, where the mouse was clicked etc.
 *
 * The two other functions in this file are CheckChanges and Quit.
 * */

static char TimeLinercsid[] = "$Header: /Source/Media/collab/TimeLine/RCS/TimeLine.c,v 1.15 92/10/30 16:20:48 drapeau Exp $";

#include <sys/types.h>
#include <xview/notify.h>
#include <xview/xv_xrect.h>
#include <xview/textsw.h>
#include "main.h"
#include <Browse.h>
#include <getopt.h>

#ifdef MAIN

int			WindowFree[MaxWindows + 1];
int			numberOfWindows;						    
int			ReceiverPortNumber;	
int			quitAll;
int			autoPlay;
int			autoQuit;
char			TimeLineHostname[MAXHOSTNAMELEN];				    
TimeLineFramePtr	TimeLineWindow[MaxWindows + 1];
Server_image		unknownIcon;

static int initialStart;					    /* Indicates whether the first frame is to be created */
static char *startFilename;

/*
 * Instance XV_KEY_DATA key.  An instance is a set of related
 * user interface objects.  A pointer to an object's instance
 * is stored under this key in every object.  This must be a
 * global variable.
 */
Attr_attribute	INSTANCE;

void main(int argc, char** argv)
{
  int i;

  xv_init(XV_INIT_ARGC_PTR_ARGV, &argc, argv, 0);		    /* Initialize XView */
  INSTANCE = xv_unique_key();
  gethostname(TimeLineHost, MAXHOSTNAMELEN);			    /* Get name of host on which the TimeLine Editor is running */
  strcpy (TimeLineHostname, "localhost");			    /* Set the default PortManager hostname to localhost */
  ReceiverPortNumber = AnyPort;					    /* Set the default port number to AnyPort */
  startFilename = NULL;
  autoPlay = False;
  autoQuit = False;
  CheckOptions(argc, argv);					    /* Check for any options on the command line */
  numberOfWindows = 0;						    /* Initialization of variables */
  scrollHorStart = 0;
  scrollHorEnd = 0;
  scrollVerStart = 0;
  scrollVerEnd = 0;
  paletteName = 0;
  initialStart = 1;
  clipDuration = 0;
  quitAll = 1;
  for (i=0; i < MaxWindows; i++) 
  {
    TimeLineWindow[i] = (TimeLineFramePtr) malloc (sizeof (struct TimeLineFrame));
    WindowFree[i] = 0;
  }
  GetUnknownIcon();
  CreateCursor();
  TimeLineInit(TimeLineWindow[1], 1);				    /* Create the first TimeLine window */
  exit(0);
}
#endif

/* 
 * This function parses the command line and retrieves all the known options and their arguments.
 * Currently, the two options are hostname and portnumber.
 * After parsing the options, the variable optind will point to the start of those non-option arguments.  In this case, it will be the filename to be
 * loaded.  At present, only one filename will be handled.  So if there are multiple filenames typed, the last one will be loaded.
 */
void CheckOptions(argc, argv)
     int 	argc;
     char 	**argv;
{
  int optionChar;  
  int option_index = 0;
  static struct option long_options[] =
  {
    {"hostname", 1, 0, 'h'},
    {"portnumber", 1, 0, 'p'},
    {"autoPlay", 0, 0, 'P'},
    {"autoQuit", 0, 0, 'Q'},
    {0, 0, 0, 0}
  };

  while (1)							    /* Start parsing all known options */
  {
    optionChar = getopt_long_only (argc, argv, "h:p:PQ",
			      long_options, &option_index);
    if (optionChar == EOF)					    /* Done with all known options */
      break;
    switch (optionChar)
    {
     case 'h':
      if (optarg) 
	strcpy (TimeLineHostname, optarg);
      break;
     case 'p':
      if (optarg) 
	ReceiverPortNumber = atoi(optarg);
      break;
     case 'P':
      autoPlay = True;
      break;
     case 'Q':
      autoQuit = True;
      break;
     default:
      Usage();
      exit(1);
      break;
    }
  }
  if (optind < argc)						    /* Check if a filename has been specified */
  {
    startFilename = (char *) malloc (256);
    strcpy (startFilename, argv[optind]);
  }
}

/* 
 * This is the interpose function for the frame destroy procedure.  It is called when the quit button is pressed, or when the frame's "Quit"
 * menu selection is chosen.
 * The frame destroying process happens in two phases (meaning this function will be called twice).  The first time it is called, status =
 * DESTROY_CHECKING (this is done automatically by XView).  At this point, the interpose function can choose to veto the destruction, or 
 * let it proceed to phase two, where the actual destruction occurs.
 * If it is the clipboard, veto the destruction (just hide it).
 * If there are unsaved changes and the user chooses to abort the quit operation, also veto the destruction.
 * If a destruction is to happen, update the window count variable (numberOfWindows) and check if it is the last existing window to be destroyed.
 * If so, destroy the clipboard as well.
 * If quitAll == 1, destroy all other frames as well.  This is done 'recursively'.  The frame being destroyed will destroy the next frame.  This
 * next frame will then destroy the following one and so on.
 *
 */
Notify_value QuitNotify(client, status)
     Notify_client client;
     Destroy_status status;
{
  int whichWindow;
  TimeLineFramePtr tlFrame;
  TimeLine_window_objects * ip = (TimeLine_window_objects *) xv_get(client, XV_KEY_DATA, INSTANCE);

  whichWindow = xv_get(ip->controls, PANEL_CLIENT_DATA);
  tlFrame = TimeLineWindow[whichWindow];
  if (status == DESTROY_CHECKING)				    /* Check to see if we really want to destroy the frame */
  {
    if (whichWindow == 0 && numberOfWindows > 0)		    /* If it is the clipboard, do not destroy, just hide it */
    {
      xv_set(tlFrame->TimeLine_window->window, XV_SHOW, FALSE, NULL);
      return notify_veto_destroy(client);
    }
    if (CheckChanges(QuitTimeLine, tlFrame) == NOTICE_YES)   /* check if unsaved changes exist */
      return notify_veto_destroy(client);
  }
  else								    /* Second phase. Proceed to destroy */
  {
    if (whichWindow != 0)					    /* Update the count of windows still existing */
    {
      numberOfWindows --;
      WindowFree[whichWindow] = 0;
    }
    if (numberOfWindows > 0 && quitAll == 1)			    /* Destroy the next window in use. */
    {
      while (WindowFree[whichWindow] == 0) 
      {
	if (whichWindow == MaxWindows - 1)
	  whichWindow = 1;
	else
	  whichWindow ++;
      }
      xv_destroy_safe(TimeLineWindow[whichWindow]->TimeLine_window->window);
    }
    else 
      quitAll = 1;
    if (numberOfWindows == 0 && whichWindow != 0)		    /* Destroy the clipboard if it is the only window left open */
      xv_destroy_safe(TimeLineWindow[0]->TimeLine_window->window);  
    if (tlFrame->TimeLineSender && tlFrame->TimeLineReceiver)	    /* Check that sender exists before attempting to disconnect */
      SenderDisconnectFromPortMgr(tlFrame->TimeLineSender, 
				  &(tlFrame->TimeLineReceiver->receivePort)); /* Disconnect the TimeLine frame from the PortManager */
    return notify_next_destroy_func(client, status);
  }
  return NOTIFY_DONE;
} 
    

/*
 * This function initializes a TimeLine frame window.  It first forms
 * a connection with the Port Manager and creates a sender and receiver.
 * The Insert Blank Time, Info, Pause, and Region popup windows are then
 * initialized.  Next, the variables used for controlling the TimeLine
 * are initialized.  The stop menu button is made inactive.  The
 * OpenAppsInitialize function is called to get the list of open
 * applications from the Port Manager.  This is done only if the frame is
 * not the Clipboard, that is, whichFrame != 0.  If the frame is the
 * clipboard, call the DimButtons procedure twice to disable some of the
 * menu functions.  Canvas Init is now called to set the size of the
 * canvases and also the colormap.  Check if a file is to be loaded if
 * this is the first TimeLine frame to be opened.  xv_main_loop is called
 * if it is the first frame created, else it is just necessary to make
 * XV_SHOW to TRUE for the frame, as the Notifier would already have been
 * started.  If it is the first frame created, call the TimeLineInit
 * function again to create the clipboard frame.  Called by main
 * (TimeLine.c) and NewHandler (file.c)
 */
void TimeLineInit(tlFrame, whichFrame)
     TimeLineFramePtr tlFrame;
     int whichFrame;
{
  Rect rect;
  char buf[40];

  tlFrame->TimeLine_window = TimeLine_window_objects_initialize(NULL, NULL); /* Initialize user interface components. */
  xv_set(tlFrame->TimeLine_window->controls,			    /* To distinguish this frame from the others */
	 PANEL_CLIENT_DATA, whichFrame, NULL);
  notify_interpose_destroy_func(tlFrame->TimeLine_window->window,   /* Interpose the destroy function */
				QuitNotify);
  tlFrame->TimeLineSender = NULL;
  if (whichFrame != 0)						    /* Register with Port Manager if this is not the clipboard */
  {
    tlFrame->TimeLinePort.hostName = TimeLineHostname;		    /* Initialize link with Port Manager */
    tlFrame->TimeLinePort.portNumber = PortMgrPortNumber;
    tlFrame->TimeLineSender = NewSender(&(tlFrame->TimeLinePort));
    if (!tlFrame->TimeLineSender) 
    {
      AlertMessage(tlFrame, "The PortManager is not running.", 
		   "Please check that it is running and try again.", NULL);
      return;
    }
    tlFrame->TimeLineReceiver = NewReceiver(tlFrame->TimeLineSender, TimeLineName, ReceiverPortNumber);
  }
  tlFrame->InsertBlankTimePopup = 
    BlankTime_InsertBlankTimePopup_objects_initialize(NULL,	    /* Initialize insert blank time popup window */
						      tlFrame->TimeLine_window->window);
  tlFrame->InfoPopup = Info_InfoPopup_objects_initialize(NULL, tlFrame->TimeLine_window->window);
  tlFrame->PausePopup = Pause_PausePopup_objects_initialize(NULL, tlFrame->TimeLine_window->window);
  tlFrame->RegionPopup = Region_RegionPopup_objects_initialize(NULL, tlFrame->TimeLine_window->window);
  tlFrame->PortManagerPopup = ConnectPortManager_PortManagerPopup_objects_initialize(NULL, tlFrame->TimeLine_window->window);
  if (whichFrame != 0)						
  {
    sprintf(buf, "Current Port Manager : %s\n", tlFrame->TimeLinePort.hostName);
    xv_set(tlFrame->PortManagerPopup->CurrentPortManagerMsg, PANEL_LABEL_STRING, buf, NULL);
  }
  CreateBrowse(OpenHandler, SaveHandler, tlFrame->TimeLine_window->window);
  NormalCursor(tlFrame);
  rect.r_left = 100;						    /* Set the position of the main frame */
  rect.r_top = 250;
  rect.r_width = (int) xv_get(tlFrame->TimeLine_window->window, XV_WIDTH);
  rect.r_height = (int) xv_get(tlFrame->TimeLine_window->window, XV_HEIGHT) ;
  frame_set_rect(tlFrame->TimeLine_window->window, &rect);		
  xv_set(tlFrame->InfoPopup->InfoPopup,				    /* Set the position of the info popup window */
	 XV_X, 250,
	 XV_Y, 250,
	 NULL);
  xv_set(tlFrame->PortManagerPopup->PortManagerPopup,		    /* Set the position of the port manager popup window */
	 XV_X, 250,
	 XV_Y, 250,
	 NULL);
  xv_set(tlFrame->InsertBlankTimePopup->InsertBlankTimeChoice,	    /* Set choice item in insert blank time panel to exclusive */
	 PANEL_CHOOSE_NONE, FALSE,
	 PANEL_CHOOSE_ONE, TRUE, 
	 PANEL_VALUE, 1, NULL);
  xv_set(tlFrame->PausePopup->PauseChoice,			    /* Set choice item in pause marker info window to exclusive */
	 PANEL_CHOOSE_NONE, FALSE,
	 PANEL_CHOOSE_ONE, TRUE, 
	 PANEL_VALUE, 0, NULL);
  if (whichFrame == 0) 
    sprintf(buf, "Clipboard");
  else 
    sprintf (buf, "Document %d\n", whichFrame);
  xv_set(tlFrame->InsertBlankTimePopup->TimeLineDocMsg,		    /* Indicate which TimeLine document popup windows belong to */
	 PANEL_LABEL_STRING, buf, NULL);
  xv_set(tlFrame->PausePopup->TimeLineDocMsg,	
	 PANEL_LABEL_STRING, buf, NULL);
  xv_set(tlFrame->RegionPopup->TimeLineDocMsg,	
	 PANEL_LABEL_STRING, buf, NULL);
  tlFrame->TimeLineLength = 1800;				    /* Set length of canvas to 1800 pixels long */
  tlFrame->canvasStart = 0;					    /* Initialization of tlFrame variables */
  tlFrame->areaSelected = 0;
  tlFrame->noteSelected = 0;
  tlFrame->gridSpacing = 0;
  tlFrame->change = 0;
  tlFrame->numberOfApps = 0;
  tlFrame->chosenApp = -1;
  tlFrame->lastX = -1;
  tlFrame->startX = tlFrame->endX = 0;
  tlFrame->startY = tlFrame->endY = 0;
  tlFrame->startnoteX = tlFrame->endnoteX = 0;
  tlFrame->startnoteY = tlFrame->endnoteY = 0;
  tlFrame->zoomLevel = 1;
  tlFrame->selectedInstrument = NULL;
  tlFrame->noteInstrument = NULL;
  tlFrame->instHead = NULL;
  tlFrame->pauseHead = NULL;
  tlFrame->selectedPause = NULL;
  tlFrame->donePause = NULL;
  tlFrame->numPause = 0;
  tlFrame->pauseEdit = -1;
  tlFrame->status = StopMode;
  tlFrame->regionHead = NULL;
  tlFrame->selectedRegion = NULL;
  tlFrame->numRegion = 0;
  tlFrame->regionEdit = -1;
  tlFrame->syncHints = NoSyncHintsAvailable;			    /* No synchronization hints are available for this document */
  xv_set(tlFrame->TimeLine_window->stopButton, PANEL_INACTIVE, TRUE, NULL);
  xv_set(tlFrame->TimeLine_window->pauseButton, PANEL_INACTIVE, TRUE, NULL);
  strcpy (tlFrame->filename, "untitled");
  UpdateHeader(tlFrame, 0);
  CanvasInit(tlFrame);						    /* Initialize the canvases */
  if (whichFrame != 0) 
    OpenAppsInitialize(tlFrame, 1);				    /* Check with the PortManager for the open applications */
  else 
  {
    DimButtons(1, TimeLineWindow[0]);
    DimButtons(0, TimeLineWindow[0]);				    /* Disable some of the menu functions */
  }    
  WindowFree[whichFrame] = 1;
  if (initialStart == 1)
  {
    initialStart = 0;
    numberOfWindows++;
    TimeLineInit(TimeLineWindow[0], 0);				    /* Create the clipboard frame */
    (void) notify_enable_rpc_svc (TRUE);			    /* Enable XView to detect RPC events */
    if (startFilename != NULL)					    /* Check if a file is to be loaded */
    {
      Browse(startFilename, BrowseCheckOpen, 1,
	     "#TimeLine Edit Document", "TimeLine Editor");
      if (autoPlay == True)					    /* Was a command-line option set to begin playback... */
	Play(tlFrame->TimeLine_window->controls, MENU_NOTIFY);	    /* ...immediately?  If so, begin playback now. */
    }
    xv_main_loop(tlFrame->TimeLine_window->window);		    /* Turn control over to XView */
  }
  else if (whichFrame != 0)
  {
    numberOfWindows++;
    xv_set(tlFrame->TimeLine_window->window, XV_SHOW, TRUE, NULL);  /* Notify already started, just show the frame */
  }
}

/*
 * Checks if there are any unsaved changes to the current TimeLine document.  If so, it warns the user by using a notice prompt, and
 * gives the user the choice of either going ahead with the operation or cancelling the operation.
 * Called by Quit (TimeLine.c), ClearAllHandler (edit.c), Load (file.c) and UpdateAppsHandler (openApps.c) 
 */
int CheckChanges (optype, tlFrame)
     int	 optype;
     TimeLineFramePtr tlFrame;
{
  int 	result, whichWindow;
  char 	buf[100], buf1[100], buf2[100];

  result = NOTICE_NO;
  if (tlFrame->change && tlFrame->numberOfApps > 0)		    /* Perform the following only if there are unsaved changes and there are apps open */
  {  
    switch (optype) 
    {
     case LoadFile :						    /* Loading a new document */
      sprintf (buf, "Loading a file will cause all changes to be lost.");
      sprintf (buf1, "Go ahead and load the file?");
      break;
     case ClearAll :						    /* Clearing the TimeLine */
      sprintf (buf, "This clear all operation will cause all changes to be lost.");
      sprintf (buf1, "Go ahead and clear the TimeLine?");
      break;
     case ConnectPM :						    /* Check in with a new port manager */
      sprintf (buf, "Checking in with a new port manager will cause all changes to be lost.");
      sprintf (buf1, "Go ahead and check in with new port manager?");
      break;
     case QuitTimeLine :					    /* Quit the TimeLine application */
      sprintf (buf, "Quiting the TimeLine Editor will cause all changes to be lost.");
      sprintf (buf1, "Go ahead and quit?");
      break;
     case CloseFile :
      sprintf (buf, "Closing the file will cause all changes to be lost.");
      sprintf (buf1, "Go ahead and close?");
      break;       
     default : 
      break;
    }
    whichWindow = xv_get(tlFrame->TimeLine_window->controls, PANEL_CLIENT_DATA);
    sprintf (buf2, "Unsaved changes exist in TimeLine Document %d.", whichWindow);
    result = notice_prompt(tlFrame->TimeLine_window->window, NULL,
			   NOTICE_MESSAGE_STRINGS,
			   buf2, buf, buf1, NULL,
			   NOTICE_BUTTON_NO,       "Yes",
			   NOTICE_BUTTON_YES,      "No",
			   NULL);
  }
  return (result);
}

/*
 * This function will update the header of the TimeLine Editor frame to either the current document name, or untitled.
 * It will also specify the status of the file, if it has been modified.
 * Called by TimeLineInit (TimeLine.c)
 */
void UpdateHeader(TimeLineFramePtr	tlFrame,
		  int			modified)
{
  char	label[100];
  char	synchronizedLabel[32];
  char	modifiedLabel[32];
  int	whichFrame;
  
  whichFrame = xv_get(tlFrame->TimeLine_window->controls, PANEL_CLIENT_DATA);
  if (whichFrame == 0)						    /* Request to show clipboard */
    sprintf(label, "TimeLine Editor Clipboard");
  else
  {
    if (tlFrame->syncHints == SyncHintsAvailable)
      sprintf(synchronizedLabel, " (synchronized)");
    else
      sprintf(synchronizedLabel, "");
    if (modified != 0)
      sprintf(modifiedLabel, " (modified)");
    else
      sprintf(modifiedLabel, "");

      sprintf(label, "TimeLine Editor Document %d :  \"%s\"%s%s",
	      whichFrame, tlFrame->filename,
	      modifiedLabel, synchronizedLabel);
  }
  xv_set(tlFrame->TimeLine_window->window,			    /* Set the frame header */
	 XV_LABEL, label,
	 NULL);
}								    /* end function UpdateHeader */



/*
 * This function will displace a notice prompt, which is used to alert the user that something is wrong.
 */
void AlertMessage(tlFrame, msg1, msg2, msg3)
     TimeLineFramePtr tlFrame;
     char *msg1;
     char *msg2;
     char *msg3;
{
  notice_prompt(tlFrame->TimeLine_window->window, NULL,
		NOTICE_MESSAGE_STRINGS,
		msg1, msg2, msg3,
		NULL,
		NOTICE_BUTTON, "OK", 100,
		NULL);
  
}

/*
 * This function will get the bits of the unknown icon, create a server image of it and store it in the unknownIcon variable.
 */
void GetUnknownIcon()
{
  static unsigned short iconBits[] = 
  {
#include "icons/UnknownIcon"
  };

  unknownIcon = (Server_image) xv_create(NULL, SERVER_IMAGE,
				  SERVER_IMAGE_BITS, iconBits,
				  SERVER_IMAGE_DEPTH, 1,
				  XV_WIDTH, 64,
				  XV_HEIGHT, 64,
				  NULL);
}


void Usage()
{
  fprintf(stderr, "Usage: TimeLine <documentName> <options>\n");
  fprintf(stderr, "\tOptions include:\n");
  fprintf(stderr, "\t-autoPlay : When a documentName is specified, this option\n");
  fprintf(stderr, "\t            tells TimeLine to begin playback of the document\n");
  fprintf(stderr, "\t            immediately after it has been opened.\n");
  fprintf(stderr, "\t-autoQuit : When a documentName is specified, this option\n");
  fprintf(stderr, "\t            tells TimeLine to quit immediately after playback\n");
  fprintf(stderr, "\t            of that document has ended.\n");
  fprintf(stderr, "\t            Note: This option is only valid if the -autoPlay\n");
  fprintf(stderr, "\t                  option has also been set.\n");
  fprintf(stderr, "\t-hostname : name of the computer with which TimeLine will\n");
  fprintf(stderr, "\t            register itself and ask the PortManager for\n");
  fprintf(stderr, "\t            other media editors.\n");
  return;
}								    /* end function Usage */
