/*
 * Copyright (c) 1990, 1991, 1992 Stanford University
 *
 * Permission to use, copy, modify, and distribute this software and 
 * its documentation for any purpose is hereby granted without fee, provided
 * that (i) the above copyright notices and this permission notice appear in
 * all copies of the software and related documentation, and (ii) the name
 * Stanford may not be used in any advertising or publicity relating to
 * the software without the specific, prior written permission of
 * Stanford.
 * 
 * THE SOFTWARE IS PROVIDED "AS-IS" AND WITHOUT WARRANTY OF ANY KIND, 
 * EXPRESS, IMPLIED OR OTHERWISE, INCLUDING WITHOUT LIMITATION, ANY 
 * WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  
 *
 * IN NO EVENT SHALL STANFORD BE LIABLE FOR ANY SPECIAL, INCIDENTAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OF ANY KIND, OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER OR NOT
 * ADVISED OF THE POSSIBILITY OF DAMAGE, AND ON ANY THEORY OF LIABILITY,
 * ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
 * SOFTWARE.
 */


/* $Header: /Source/Media/collab/Browse/RCS/Browse.c,v 1.15 92/09/14 15:40:43 drapeau Exp $ */
/* $Log:	Browse.c,v $
 * Revision 1.15  92/09/14  15:40:43  drapeau
 * Two changes:
 * * In AddDirToList(), fixed a memory allocation error.  One statement was
 *   allocating space for a filename, plus the string terminator, but not
 *   for a trailing slash ("/") that was to be added to the string.  On Suns,
 *   this memory allocation error is not easily caught, but the error is caught
 *   on other machines.  Now the correct number of bytes is allocated.
 * * Reformatted code to better conform to coding standards.
 * 
 * Revision 1.14  92/09/12  18:15:05  drapeau
 *  Made code more portable by using portable defined constants for the stat()
 * system call instead of non-portable macros.  In particular, used
 * "S_IFREG" instead of "S_ISREG", and "S_IFDIR" instead of "S_ISDIR".
 * 
 * Revision 1.13  92/06/15  13:32:41  drapeau
 * Changed  #include <sys/dirent.h>   to  #include <dirent.h>  to get the
 * typedefed structure DIR. Took out the outdated   #include <sys/dir.h>
 * Changed struct direct  to  struct dirent  --> **Important: this change
 * should also be made to the BSD version. (it is, as of this revision.)
 * 
 * Revision 1.12  92/05/29  14:31:07  drapeau
 * Made several changes:
 * * Added copyright notice to the code.
 * * Made functions used only by this module "static", so other files don't
 *   have access to them.
 * * Prepended the word "Browse" to all functions that might be used by code
 *   outside this module.
 * 
 * Revision 1.11  92/05/26  13:15:28  drapeau
 * Made a number of changes, to add functionality and to fix errors:
 * User can now type "~", "/~", or "~/" to quickly go to the home
 * directory.
 * Made changes to the event handling notify functions to account for
 * differences in the way XView 2.0 and XView 3.0 processes events for
 * text panels.  This browser should now work correctly with both.
 * Implemented a double-click scheme for the scrolling lists.  The user can
 * now double-click on an item to open it, instead of selecting the item
 * then clicking on the "Open" button or pressing Return.
 * Spacebar can now be used in addition to Escape and Tab for file
 * completion.
 * Memory allocation is now much improved.  Earlier versions of the Browser
 * never released the memory they allocated.  This version should have
 * plugged all memory leaks.  In general, this version of the code is
 * much cleaner.
 * 
 * Revision 1.0  92/01/06  12:52:59  drapeau
 * Completely revised the implementation of Browse code.  Additions and changes
 * too numerous to list here.
 * 
 * Revision 0.14  91/05/30  11:36:23  warren
 * Took out alphabetizing code in order to get a working version going...
 * This version adds an owner field to the CreateBrowse procedure so that 
 * olwm will handle quitting from the frame menu properly.
 * 
 * Revision 0.11  1991/05/14  16:46:45  warren
 * Fixed notice bug by making notices owned by browsePopup instead of OpenButton.
 *
 * Revision 0.10  1991/05/09  01:36:37  warren
 * *** empty log message ***
 * */
static char rcsid[] = "$Header: /Source/Media/collab/Browse/RCS/Browse.c,v 1.15 92/09/14 15:40:43 drapeau Exp $";

#include <stdio.h>
#include <pwd.h>
#include <sys/param.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/errno.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/textsw.h>
#include <xview/xv_xrect.h>
#include <xview/notice.h>
#include <xview/svrimage.h>
#include <gdd.h>
#include <gutil.h>
#include <Browse.h>

#define True           1
#define False          0
#define NGroupsMax     255                                /* put these defs here and not in Browse.h since all the */
#define BrowseNone    	-1                                /* other apps will be including that file and we don't want */
#define Default         "/"                               /* definition conflict                                      */
#define BlockSize	512
#define Read            0
#define Write           1
#define Execute         2
#define ReturnKey       13
#define TabKey          9
#define EscapeKey       27
#define SpaceBar        32
#define DistanceThreshold    3                            /* Distance interval for a double click */
#define TimeThreshold   500                               /* Time interval for a double click to be detected if MultiClickTimeout
                                                             isn't specified in .Xdefaults (1/2 second) */

static char	currentPath[MAXPATHLEN];			    /* Keeps track of the entire currentPath path */
static char	curDir[MAXPATHLEN];				    /* Always holds name of directory being displayed in dirList*/
static int 	frow, drow;					    /* Row selected in fileList or dirList, -1 if none*/
int		callerID;
int		showHiddenFiles = 0;
int		showSpecificFiles;
char		dummy[MAXPATHLEN];
char*		firstLineGlobal = NULL;
char*		appNameGlobal = NULL;
int		saveFlag, functionGlobal;
Server_image	file, folder, openFile;
static void	FileCompletion();
static void	SetMultiple();
static void	BrowseDeselectAll();
Panel_setting	BrowseCurrentPathTextNotify();
void   		BrowseOpenButtonNotify();
void		BrowseSaveButtonNotify();
void		BrowseUpDirButtonNotify();
static int	CheckPath();
static int	CheckPermission();
static int	BrowseCheckDoubleClick();
static void	StringSort(char *array[], int left, int right);
static void	Swap(char *array[], int i, int j);

Browse_browsePopup_objects *browsePopup;

void
  CreateBrowse(int (*usersOpenHandler)(char *, int),  /* usersOpenHandler is a pointer to a func returning int */
	       int (*usersSaveHandler)(char *, int),  /* usersSaveHandler is a pointer to a func returning int */
	       Xv_opaque owner)  	   	    /* that takes one pointer to char arg. See K&R 5.11 */
{	
  if (!browsePopup)
  {
    browsePopup = Browse_browsePopup_objects_initialize(NULL, owner); /* let the user decide who will own this panel */
    xv_set((Xv_object)browsePopup->browsePopup, XV_X, 300, XV_Y, 170, NULL);
    /* and a couple more objects that Guide does not cover */
    file = (Server_image)xv_create(NULL, SERVER_IMAGE,
				   XV_WIDTH, fileWidth,
				   XV_HEIGHT, fileHeight,
				   SERVER_IMAGE_BITS, fileBits,
				   NULL);
    folder = (Server_image)xv_create(NULL, SERVER_IMAGE,
				     XV_WIDTH, folderWidth,
				     XV_HEIGHT, folderHeight,
				     SERVER_IMAGE_BITS, folderBits,
				     NULL);
    openFile = (Server_image)xv_create(NULL, SERVER_IMAGE,
				       XV_WIDTH, openFileWidth,
				       XV_HEIGHT, openFileHeight,
				       SERVER_IMAGE_BITS, openFileBits,
				       NULL);
    xv_set((Xv_object)browsePopup->openButton, 
	   PANEL_CLIENT_DATA, usersOpenHandler, NULL);
    xv_set((Xv_object)browsePopup->saveButton, 
	   PANEL_CLIENT_DATA, usersSaveHandler, NULL);
    frow = BrowseNone;
    drow = BrowseNone;
  }
}


void
  Browse(char *path, int function, int id, char *firstLine, char *appName)
{
  char buf[100];
  callerID = id;
  functionGlobal = function;
  if (firstLineGlobal)
    free(firstLineGlobal);
  if (appNameGlobal)
    free(appNameGlobal);
  firstLineGlobal = NULL;
  appNameGlobal = NULL;
  if (browsePopup->hiddenFilesSetting)
  {
    xv_destroy_safe(browsePopup->hiddenFilesSetting);
    browsePopup->hiddenFilesSetting = NULL; 
  }
  if (browsePopup->specificFilesSetting)
  {
    xv_destroy_safe(browsePopup->specificFilesSetting); 
    browsePopup->specificFilesSetting = NULL;  
  }
  if (firstLine)
  {
    browsePopup->specificFilesSetting = 
      Browse_browsePopup_specificFilesSetting_create(browsePopup, browsePopup->controls2);  
    showSpecificFiles = 1;
    sprintf(buf, "%s Files Only:", appName);
    xv_set((Xv_object)browsePopup->specificFilesSetting, 
	   XV_X, 96,
	   PANEL_LABEL_STRING, buf, NULL); 
    xv_set((Xv_object)browsePopup->specificFilesSetting, PANEL_VALUE, 1, NULL);
    firstLineGlobal = (char *)strdup(firstLine);
    appNameGlobal = (char *)strdup(appName);
  }
  else
  {
    showSpecificFiles = 0;
    browsePopup->hiddenFilesSetting = 
      Browse_browsePopup_hiddenFilesSetting_create(browsePopup, browsePopup->controls2);  
    xv_set((Xv_object)browsePopup->hiddenFilesSetting, PANEL_VALUE, 2, NULL);
  }
  if (function == BrowseSave)
  {
    xv_set((Xv_object)browsePopup->fileList, PANEL_CHOOSE_ONE, TRUE, NULL); 
    xv_set((Xv_object)browsePopup->browsePopup, XV_LABEL, "Save", NULL);
    xv_set((Xv_object)browsePopup->openButton, PANEL_INACTIVE, TRUE, NULL);
    xv_set((Xv_object)browsePopup->saveButton, PANEL_INACTIVE, FALSE, NULL);
    saveFlag = 1;
  }
  else if (function == BrowseOpen)
  {
    xv_set((Xv_object)browsePopup->fileList, PANEL_CHOOSE_ONE, TRUE, NULL);  
    xv_set((Xv_object)browsePopup->browsePopup, XV_LABEL, "Open", NULL);
    xv_set((Xv_object)browsePopup->openButton, PANEL_INACTIVE, FALSE, NULL);
    xv_set((Xv_object)browsePopup->saveButton, PANEL_INACTIVE, TRUE, NULL);
    saveFlag = 0;
  }
  else if (function == BrowseCheckOpen)
  {
    xv_set((Xv_object)browsePopup->fileList, PANEL_CHOOSE_ONE, TRUE, NULL);  
    xv_set((Xv_object)browsePopup->browsePopup, XV_LABEL, "Open", NULL);
    xv_set((Xv_object)browsePopup->openButton, PANEL_INACTIVE, FALSE, NULL);
    xv_set((Xv_object)browsePopup->saveButton, PANEL_INACTIVE, TRUE, NULL);
    saveFlag = 0;
    SetCurrentText(path);
    BrowseOpenButtonNotify(NULL, NULL);
    return;
  }
  else if (function == BrowseCheckSave)
  {
    xv_set((Xv_object)browsePopup->fileList, PANEL_CHOOSE_ONE, TRUE, NULL); 
    xv_set((Xv_object)browsePopup->browsePopup, XV_LABEL, "Save", NULL);
    xv_set((Xv_object)browsePopup->openButton, PANEL_INACTIVE, TRUE, NULL);
    xv_set((Xv_object)browsePopup->saveButton, PANEL_INACTIVE, FALSE, NULL);
    saveFlag = 1;
    SetCurrentText(path);
    BrowseSaveButtonNotify(NULL, NULL);
    return;
  }
  else if (function == BrowseMultiple)
  {
    xv_set((Xv_object)browsePopup->fileList, PANEL_CHOOSE_ONE, FALSE, NULL);  
    xv_set((Xv_object)browsePopup->browsePopup, XV_LABEL, "Open Several Files", NULL);
    xv_set((Xv_object)browsePopup->openButton, PANEL_INACTIVE, FALSE, NULL);
    xv_set((Xv_object)browsePopup->saveButton, PANEL_INACTIVE, TRUE, NULL);
    saveFlag = 0;
  }
  if(!path)
  {
    if (strcmp(currentPath,"")==0)
    {
      ClearList(browsePopup->fileList);
      ClearList(browsePopup->dirList);   
      if(!getwd(currentPath))				    /* First time initialization when no path is path */
	notice_prompt(browsePopup->browsePopup, NULL,
		      NOTICE_MESSAGE_STRINGS, "Can't get the working directory",  /* currentPath, */
		      NULL,
		      NOTICE_BUTTON,  "OK", 100,
		      NULL);
      strncpy(curDir, currentPath, MAXPATHLEN);            /* curDir always contains the dirList displayed directory */
      AddDirToList(curDir, browsePopup->dirList);
      xv_set((Xv_object)browsePopup->currentPathText, PANEL_VALUE, currentPath, NULL);
    }
    else if (!DirTest(currentPath))
    {
      RemoveLastComponent(currentPath);
      SetCurrentText(currentPath);
    }
  }
  else							    /* user gave a path, check it then use it */
  {
    if(strlen(path) > MAXPATHLEN)
    {
      sprintf(buf, "Path longer than maximum allowed %d characters.\n", MAXPATHLEN);
      notice_prompt(browsePopup->browsePopup, NULL,
		    NOTICE_MESSAGE_STRINGS, buf,
		    NULL,
		    NOTICE_BUTTON,  "OK", 100,
		    NULL);
      /* perhaps we should call usersOpenHandler here and give 'em a null */
      strcpy(currentPath, Default);
    }
    else
    {
      ClearList(browsePopup->fileList);
      ClearList(browsePopup->dirList);    
      strncpy(currentPath, path, MAXPATHLEN);
      strncpy(curDir, path, MAXPATHLEN);
      if (!DirTest(currentPath))
	RemoveLastComponent(curDir);		  /* curDir always contains the dirList displayed directory */
    }
    AddDirToList(curDir, browsePopup->dirList);
    xv_set((Xv_object)browsePopup->currentPathText, PANEL_VALUE, currentPath, NULL);
  }
  BrowseCurrentPathTextNotify(NULL, NULL);
  xv_set((Xv_object)browsePopup->browsePopup, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
  xv_set((Xv_object)browsePopup->browsePopup, XV_SHOW, TRUE, NULL);
}


/*
 * Notify callback function for `currentPathText'.
 */
Panel_setting
  BrowseCurrentPathTextNotify(Panel_item item, Event *event)
{
  char value[MAXPATHLEN];
  char buf[50];
  char realPath[MAXPATHLEN];
  struct stat statbuf;
  
  if (event != NULL && event_id(event) != ReturnKey)
    return panel_text_notify(item, event);
  
  strcpy(value, (char *)xv_get((Xv_object)browsePopup->currentPathText, PANEL_VALUE));
  if (((strlen(value)==1) && value[0]=='~') || ((strlen(value)==2) && !strcmp(value, "~/"))
      || !strcmp(&value[strlen(value)-3], "/~/") || !strcmp(&value[strlen(value)-2], "/~"))
    strcpy(value, (char *)getenv("HOME"));
  if(strlen(value) > MAXPATHLEN - 3)
  {
    sprintf(buf, "Path longer than maximum allowed %d characters.\n", MAXPATHLEN);
    notice_prompt(browsePopup->browsePopup, NULL,
		  NOTICE_MESSAGE_STRINGS, buf,
		  NULL,
		  NOTICE_BUTTON,  "OK", 100,
		  NULL);
    /* perhaps we should call usersOpenHandler here and give 'em a null */
    strcpy(currentPath, Default);
    strcpy(curDir, Default);	
  }
  else
  {
    strcpy(realPath, value);
    if (realpath(value, dummy) != NULL)
      realpath(value, realPath);
    strncpy(currentPath, realPath, MAXPATHLEN);
    SetCurrentText(currentPath);
    strncpy(curDir, realPath, MAXPATHLEN);
    if (!DirTest(currentPath))
      RemoveLastComponent(curDir);		  /* curDir always contains the dirList displayed directory */
  }
  if (item != NULL)
  {
    if (CheckPath(curDir))
    {
      if (!saveFlag && !DirTest(currentPath))
      {
	if (CheckPath(currentPath))
	{                               /* file exists and we are trying to open it */
	  if (stat(currentPath, &statbuf) != -1)
	  {
	    /* check read permissions */
	    if (CheckPermission(&statbuf, Read, currentPath))
	      BrowseOpenButtonNotify(-1, NULL); 
	    else
	      BrowseUpDirButtonNotify(NULL, NULL);
	  }
	  else
	    printf("stat failure on %s\n", currentPath);
	}
      }
      else if (saveFlag && !DirTest(currentPath))
      {                               
	if (realpath(currentPath, dummy) != NULL)
	{                               /* file exists and we are trying to save to it */
	  if (stat(currentPath, &statbuf) != -1)
	  {		  /* check write permissions for file (but not for the directory) */
	    if (CheckPermission(&statbuf, Write, currentPath))
	      BrowseSaveButtonNotify(-1, NULL);  
	    else
	      BrowseUpDirButtonNotify(NULL, NULL);
	  }
	  else
	    printf("stat failure on %s\n", currentPath);
	}
	else
	{                               /* file doesn't exist and we are trying to save to it */
	  if (stat(curDir, &statbuf) != -1)
	  {		  /* check write permissions for file (but not for the directory) */
	    if (CheckPermission(&statbuf, Write, curDir))
	      /* check write permissions for directory */
	      BrowseSaveButtonNotify(-1, NULL);  
	    else
	      BrowseUpDirButtonNotify(NULL, NULL);
	  }
	  else
	    printf("stat failure on %s\n", curDir);
	}
      }
      else if (DirTest(currentPath))
	BrowseUpDirButtonNotify(NULL, NULL);
    }
  }
  else if (DirTest(currentPath))
    BrowseUpDirButtonNotify(NULL, NULL);
  if (strcmp(curDir, "/") != 0 && strlen(curDir) > 0)
    xv_set((Xv_object)browsePopup->upDirButton, PANEL_INACTIVE, FALSE, NULL);
  if (event != NULL)
    return panel_text_notify(item, event);
}

static int
  CheckPermission(statbuf, type, path)
struct stat *statbuf;
int type;
char *path;
{
  int groupList[NGroupsMax];
  int i, numGroups;
  int othersCheck, groupCheck, userCheck;
  char buf[MAXPATHLEN];
  char mode[10];
  
  if (type == Read)
  {
    othersCheck = S_IROTH;
    groupCheck = S_IRGRP;
    userCheck = S_IRUSR;
    strcpy(mode, "Read");
  }
  else if (type == Write)
  {
    othersCheck = S_IWOTH;
    groupCheck = S_IWGRP;
    userCheck = S_IWUSR;
    strcpy(mode, "Write");
  }
  if (type == Execute)
  {
    othersCheck = S_IXOTH;
    groupCheck = S_IXGRP;
    userCheck = S_IXUSR;
    strcpy(mode, "Execute");
  }
  if (statbuf->st_mode & othersCheck)  /* check others read bit first */
    return 1;
  numGroups = getgroups(NGroupsMax, groupList);
  for (i=0; i<numGroups && groupList[i]!=statbuf->st_gid; i++)
    ;
  /* either end of groups was reached or a match was found */
  if (i<numGroups) 
  {        /* match */
    if ((statbuf->st_mode & S_ISGID) || (statbuf->st_mode & groupCheck))
      return 1;
  }
  /* check effective user id */
  if (statbuf->st_uid==geteuid())
    if ((statbuf->st_mode & S_ISUID) || (statbuf->st_mode & userCheck))
      return 1;
  sprintf(buf, "%s permission denied for %s.\n", mode, path);
  notice_prompt(browsePopup->browsePopup, NULL,		   
		NOTICE_MESSAGE_STRINGS, buf,
		NULL,
		NOTICE_BUTTON,  "OK", 100,
		NULL);
  xv_set((Xv_object)browsePopup->browsePopup, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
  xv_set((Xv_object)browsePopup->browsePopup, XV_SHOW, TRUE, NULL);
  return 0;
}


static int
  CheckPath(char *list)
{
  char buf[MAXPATHLEN];
  if (realpath(list, dummy) == NULL)
  {
    sprintf(buf, "%s does not exist or cannot be accessed.\n", list);
    notice_prompt(browsePopup->browsePopup, NULL,		   
		  NOTICE_MESSAGE_STRINGS, buf,
		  NULL,
		  NOTICE_BUTTON,  "OK", 100,
		  NULL);
    while (realpath(list, dummy) == NULL)
    {
      if (*(list+strlen(list)-1)=='/')
	*(list+strlen(list)-1)='\0';
      if (RemoveLastComponent(list)==FALSE || !strcmp(list, "/"))
	if(!getwd(list))	
	  notice_prompt(browsePopup->browsePopup, NULL,
			NOTICE_MESSAGE_STRINGS, "Can't get the working directory",  /* currentPath, */
			NULL,
			NOTICE_BUTTON,  "OK", 100,
			NULL);
    }
    strcpy(curDir, list);
    strcpy(currentPath, list);
    SetCurrentText(currentPath);
    BrowseUpDirButtonNotify(NULL, NULL);
    xv_set((Xv_object)browsePopup->browsePopup, FRAME_CMD_PUSHPIN_IN, TRUE, NULL);
    xv_set((Xv_object)browsePopup->browsePopup, XV_SHOW, TRUE, NULL);
    return 0;
  }
  return 1;
}

/*
 * Notify callback function for `dirList'.
 */
int
  BrowseDirListNotify(Panel_item	item,
		      char		*string,
		      Xv_opaque	client_data,
		      Panel_list_op	op,
		      Event		*event)
{
  char temp[MAXPATHLEN];
  switch(op) 
  {
   case PANEL_LIST_OP_DESELECT:
    if (BrowseCheckDoubleClick(event))
    {
      BrowseCurrentPathTextNotify(browsePopup->browsePopup, NULL);
      break;
    }
    if (functionGlobal == BrowseMultiple)
    {
      ;
    }
    else
    {
      ClearList(browsePopup->fileList);
      strcpy(currentPath, curDir);
      SetCurrentText(curDir);
    }
    drow = -1;
    break;
    
   case PANEL_LIST_OP_SELECT:      
    strcpy(temp, currentPath);
    AddComponent(currentPath, string);
    if ( DirTest(currentPath))
    {
      ClearList(browsePopup->fileList);
      AddDirToList(currentPath, browsePopup->fileList);
      BrowseDeselectAll(browsePopup->dirList);
      xv_set((Xv_object)browsePopup->dirList, PANEL_LIST_SELECT, client_data, TRUE, NULL);  
    }
    else
      if (functionGlobal == BrowseMultiple)
      {
	strcpy(currentPath, curDir);
	AddComponent(currentPath, string);
	if ( DirTest(currentPath))
	{
	  ClearList(browsePopup->fileList);
	  AddDirToList(currentPath, browsePopup->fileList);
	  BrowseDeselectAll(browsePopup->dirList);
	  xv_set((Xv_object)browsePopup->dirList, PANEL_LIST_SELECT, client_data, TRUE, NULL);  
	}
	else
	  strcpy(currentPath, temp);
      }
    drow = client_data;
    SetCurrentText(currentPath);
    if (BrowseCheckDoubleClick(event))
      BrowseCurrentPathTextNotify(browsePopup->browsePopup, NULL);
    break;
    
   case PANEL_LIST_OP_VALIDATE:
    break;
    
   case PANEL_LIST_OP_DELETE:
    break;
  }
  return XV_OK;
}

/*
 * Notify callback function for `fileList'.
 */
int
  BrowseFileListNotify(Panel_item	item,
		       char		*string,
		       Xv_opaque	client_data,
		       Panel_list_op	op,
		       Event		*event)
{
  switch(op) 
  {
   case PANEL_LIST_OP_DESELECT:
    if (BrowseCheckDoubleClick(event))
    {
      BrowseCurrentPathTextNotify(browsePopup->browsePopup, NULL);
      break;
    }
    if (strcmp(curDir, "/")!=0)
      sprintf(currentPath, "%s/%s", realpath(curDir, dummy), 
	      (char *)xv_get((Xv_object)browsePopup->dirList, PANEL_LIST_STRING, AnySelected(browsePopup->dirList)));
    else
      sprintf(currentPath, "/%s",
	      (char *)xv_get((Xv_object)browsePopup->dirList, PANEL_LIST_STRING, AnySelected(browsePopup->dirList)));
    SetCurrentText(currentPath);
    frow = -1;
    break;
    
   case PANEL_LIST_OP_SELECT:
    AddComponent(currentPath, string);
    if ( DirTest(currentPath) )
    {
      if (AnySelected(browsePopup->dirList) != BrowseNone)
	AddComponent(curDir, (char *)xv_get((Xv_object)browsePopup->dirList, PANEL_LIST_STRING, 
					    AnySelected(browsePopup->dirList)));
      else
	xv_set((Xv_object)browsePopup->upDirButton, PANEL_INACTIVE, TRUE, NULL);
      xv_set((Xv_object)browsePopup->dirList, XV_SHOW, FALSE, NULL);
      BrowseDeselectAll(browsePopup->fileList);
      xv_set((Xv_object)browsePopup->fileList, PANEL_LIST_SELECT, client_data, TRUE, NULL);  
      xv_set((Xv_object)browsePopup->fileList, 
	     XV_X, (int)xv_get((Xv_object)browsePopup->dirList, XV_X),
	     XV_Y, (int)xv_get((Xv_object)browsePopup->dirList, XV_Y),
	     PANEL_NOTIFY_PROC, BrowseDirListNotify,
	     NULL);
      drow = client_data;
      xv_destroy_safe(browsePopup->dirList);		    /* Make SURE that the file panel forgets the selection */
      browsePopup->dirList = browsePopup->fileList;
      browsePopup->fileList = Browse_browsePopup_fileList_create(browsePopup, browsePopup->controls2);  
      SetMultiple();
      AddDirToList(currentPath, browsePopup->fileList);
      if (strcmp(curDir, "/") != 0 && strlen(curDir) > 0)
	xv_set((Xv_object)browsePopup->upDirButton, PANEL_INACTIVE, FALSE, NULL);
      frow = -1;
    }
    else
    {
      frow = client_data;
      if (functionGlobal == BrowseMultiple)
	RemoveLastComponent(currentPath);
    }
    SetCurrentText(currentPath);
    if (BrowseCheckDoubleClick(event))
      BrowseCurrentPathTextNotify(browsePopup->browsePopup, NULL);
    break;
    
   case PANEL_LIST_OP_VALIDATE:
    break;
    
   case PANEL_LIST_OP_DELETE:
    break;
  }
  return XV_OK;
}

/*
 * Notify callback function for `openButton'.
 */
void
  BrowseOpenButtonNotify(Panel_item item, Event *event)
{
  int (*usersOpenHandler)(char *, int), i, rows;
  char temp[MAXPATHLEN], path[MAXPATHLEN];
  
  if ((int)item != -1 && functionGlobal!=BrowseMultiple)
  {
    BrowseCurrentPathTextNotify(-1, NULL);
    return;
  }
  
  if (DirTest(currentPath) && functionGlobal!=BrowseMultiple)
  {
    ClearList(browsePopup->fileList);
    AddDirToList(currentPath, browsePopup->fileList);
  }
  else
  {                   
    usersOpenHandler = (int (*)(char *, int))xv_get((Xv_object)browsePopup->openButton, PANEL_CLIENT_DATA);
    if (functionGlobal == BrowseMultiple)
    {
      rows = (int)xv_get((Xv_object)browsePopup->fileList, PANEL_LIST_NROWS);
      for(i=0; i<rows; i++)
	if ((int)xv_get((Xv_object)browsePopup->fileList, PANEL_LIST_SELECTED, i))
	{
	  strcpy(path, (char *)xv_get((Xv_object)browsePopup->currentPathText, PANEL_VALUE));
	  sprintf(temp, "%s%s", path, (char *)xv_get((Xv_object)browsePopup->fileList, PANEL_LIST_STRING, i));
	  (*usersOpenHandler)(temp, callerID); 
	  /*		printf("%s\n", temp);  */
	  xv_set((Xv_object)browsePopup->fileList, PANEL_LIST_SELECT, i, FALSE, NULL); 
	  xv_set((Xv_object)browsePopup->fileList, PANEL_LIST_GLYPH, i, openFile, NULL); 
	}
      rows = (int)xv_get((Xv_object)browsePopup->dirList, PANEL_LIST_NROWS);
      for(i=0; i<rows; i++)
	if ((int)xv_get((Xv_object)browsePopup->dirList, PANEL_LIST_SELECTED, i))
	{
	  sprintf(temp, "%s%s", curDir, (char *)xv_get((Xv_object)browsePopup->dirList, PANEL_LIST_STRING, i));
	  if (*(temp+strlen(temp)-1)!='/')
	  {
	    (*usersOpenHandler)(temp, callerID); 
	    /*		    printf("%s\n", temp); */
	    xv_set((Xv_object)browsePopup->dirList, PANEL_LIST_SELECT, i, FALSE, NULL); 
	    xv_set((Xv_object)browsePopup->dirList, PANEL_LIST_GLYPH, i, openFile, NULL); 
	  }
	}
    }
    else
    {
      strcpy(temp, (char *)xv_get((Xv_object)browsePopup->currentPathText, PANEL_VALUE));
      if (!(*usersOpenHandler)(temp, callerID))		    /* usersOpenHandler gets to show error message */
      {
	xv_set((Xv_object)browsePopup->browsePopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);		   
	xv_set((Xv_object)browsePopup->browsePopup, XV_SHOW, FALSE, NULL);
      }
      else
	BrowseUpDirButtonNotify(NULL, NULL);
    }
  }   
}

/*
 * Notify callback function for `saveButton'.
 */
void
  BrowseSaveButtonNotify(Panel_item item, Event *event)
{
  int (*usersSaveHandler)(char *, int);
  char temp[MAXPATHLEN];
  if ((int)item != -1)
  {
    BrowseCurrentPathTextNotify(-1, NULL);
    return;
  }
  
  usersSaveHandler = (int (*)(char *, int))xv_get((Xv_object)browsePopup->saveButton, PANEL_CLIENT_DATA);
  strcpy(temp, (char *)xv_get((Xv_object)browsePopup->currentPathText, PANEL_VALUE));
  if (!(*usersSaveHandler)(temp, callerID))			    /* usersSaveHandler gets to show error message */
  {
    xv_set((Xv_object)browsePopup->browsePopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);		
    xv_set((Xv_object)browsePopup->browsePopup, XV_SHOW, FALSE, NULL);
  }
  else
    BrowseUpDirButtonNotify(NULL, NULL);
}



/*
 * Notify callback function for `updir_button'.
 */
void
  BrowseUpDirButtonNotify(Panel_item	item, Event *event)
{
  char temp[MAXPATHLEN], *i;
  int row = BrowseNone;
  xv_set((Xv_object)browsePopup->fileList, PANEL_INACTIVE, FALSE, NULL);
  xv_set((Xv_object)browsePopup->dirList, PANEL_INACTIVE, FALSE, NULL);
  xv_destroy(browsePopup->fileList);
  browsePopup->fileList = Browse_browsePopup_fileList_create(browsePopup, browsePopup->controls2);
  AddDirToList(curDir, browsePopup->fileList);
  frow = BrowseNone;
  i = (char *)rindex(realpath(curDir, dummy), '/');
  strcpy(temp, ++i);
  if (curDir[strlen(curDir)-1]=='/')
    curDir[strlen(curDir)-1]='\0';
  RemoveLastComponent(curDir);
  xv_destroy(browsePopup->dirList);
  browsePopup->dirList = Browse_browsePopup_dirList_create(browsePopup, browsePopup->controls2);
  SetMultiple();
  AddDirToList(curDir, browsePopup->dirList);
  strcpy(&temp[strlen(temp)], "/");
  if (strcmp(temp, "/") != 0)
  {
    row = FindRow(browsePopup->dirList, temp);
    if (row != BrowseNone)
    {
      xv_set((Xv_object)browsePopup->dirList, PANEL_LIST_SELECT, row, TRUE, NULL);  
      drow = row;        
    }
    else
    {
      printf("Error -- cannot find %s\n", temp);
      ClearList(browsePopup->fileList);
      ClearList(browsePopup->dirList);   
      if(!getwd(currentPath))				    /* First time initialization when no path is path */
	notice_prompt(browsePopup->browsePopup, NULL,
		      NOTICE_MESSAGE_STRINGS, "Can't get the working directory",  /* currentPath, */
		      NULL,
		      NOTICE_BUTTON,  "OK", 100,
		      NULL);
      strncpy(curDir, currentPath, MAXPATHLEN);            /* curDir always contains the dirList displayed directory */
      AddDirToList(curDir, browsePopup->dirList);
      xv_set((Xv_object)browsePopup->currentPathText, PANEL_VALUE, currentPath, NULL);
      BrowseUpDirButtonNotify(NULL, NULL);
    }
    if (strcmp(curDir, "/") != 0)				    /* strcmp returns 0 for equality */
      sprintf(currentPath,"%s/%s", 
	      realpath(curDir, dummy), (char *)xv_get((Xv_object)browsePopup->dirList, PANEL_LIST_STRING, drow));
    else if (drow != BrowseNone)
    {
      sprintf(currentPath,"/%s", (char *)xv_get((Xv_object)browsePopup->dirList, PANEL_LIST_STRING, drow)); 
      xv_set((Xv_object)browsePopup->upDirButton, PANEL_INACTIVE, TRUE, NULL);
    }
    if (saveFlag == 0 || realpath(currentPath, dummy) != NULL)
      realpath(currentPath, temp);
    SetCurrentText(currentPath);
  }
  else
    xv_set((Xv_object)browsePopup->upDirButton, PANEL_INACTIVE, TRUE, NULL);
}

/*
 * Notify callback function for `cancel_button'.
 */
void
  BrowseCancelButtonNotify(item, event)
Panel_item	item;
Event		*event;
{
  xv_set((Xv_object)browsePopup->browsePopup, FRAME_CMD_PUSHPIN_IN, FALSE, NULL);
}


/***********************************************************************
  return an allocated char * that points to the last item in a path.
  ***********************************************************************/
static char *
  GetFileName(char *path)
{
  char *p;
  
  if (p = (char*)strrchr(path, '/'))
    p++;
  else
    p = path;
  return (char *)strdup(p) ;
}


/***********************************************************************
 * This routine tests to see if a file name is a directory entry
 ***********************************************************************/
static int
  DirTest(char *filename)
{
  struct stat statbuf;
  extern int errno;
  
  if (stat(filename, &statbuf) != -1)
    return(S_IFDIR == (statbuf.st_mode & S_IFMT));
  else
    return(FALSE);
}



/*******************************************************************************
  The path passed in is scanned via readdir().  For each file in the
  path, a list item is created and inserted into a attribute list.  This
  attribute list should be cached to make browsing faster. The attribute list
  is then sent to the Panel.  Also, could cache dirent in our own stucture.
  ****************************************************************************/

static int
  AddDirToList(char *path, Panel panelList)
{
  DIR*			dirp; 
  struct dirent*	dp;
  struct stat		s_buf;
  int			item = 0;
  int			numFiles = 0;
  int			j;
  int			blocks = 1;
  extern int		errno;
  static char**		names;
  static char**		tempnames;  
  char			fullName[MAXPATHLEN];
  char*			tempName;
  char			buf[100];
  FILE*			fp;
  Server_image		glyph;
  
  if( !path || !panelList )
    return(FALSE);
  dirp = opendir(path);                                           /* opendir returns NULL if path cannot be accessed or */
  if (dirp == NULL)                                               /* if it isn't a directory                            */
    if (saveFlag == 0)
      return(FALSE);
  xv_set((Xv_object)panelList, XV_SHOW, FALSE, NULL);
  names = (char **)malloc((blocks * BlockSize) * sizeof(char *));
  if (stat(path, &s_buf) != -1 && s_buf.st_mode & S_IREAD)          /* Don't add a folder to the list if user can't read it */
    while (dp = readdir(dirp))					    /* Modify this later to put a cool x'ed folder glyph  */
    {
      tempName = GetFileName(dp->d_name);    /* GetFileName() allocates the memory */
      if (strncmp(".", tempName, 1) || showHiddenFiles)
      {
	sprintf(fullName, "%s/%s", path, tempName);  
	if (stat(fullName, &s_buf) != -1)
	  if (S_IFDIR == (s_buf.st_mode & S_IFMT))
	  {
	    names[numFiles] = malloc(strlen(tempName)+2);	    /* Allocate space for string, plus terminator, plus slash */
	    sprintf(names[numFiles], "%s/", tempName);
	    numFiles++;
	  }
	  else
	  {
	    if (showSpecificFiles)
	    {
	      if (S_IFREG == (s_buf.st_mode & S_IFMT) && (fp=fopen(fullName, "r")) != NULL)
	      {
		fgets(dummy, strlen(firstLineGlobal)+1, fp);
		dummy[strlen(firstLineGlobal)+1] = '\0';
		if (!strncmp(dummy, firstLineGlobal, strlen(firstLineGlobal)))
		{
		  names[numFiles] = strdup(tempName);
		  numFiles++;
		}
		fclose(fp);
	      }
	    }
	    else
	    {
	      names[numFiles] = strdup(tempName);
	      numFiles++;
	    }
	  }
	if (numFiles > (blocks * BlockSize)-1 ) 
	{
	  if (tempnames = (char **)realloc(names, (++blocks * BlockSize) * sizeof(char *)))
	    names = tempnames;					    /* Successful reallocation */
	  else
	    notice_prompt(browsePopup->browsePopup, NULL,		    /* reallocation failed */
			  NOTICE_MESSAGE_STRINGS, "Incomplete list, ran out of memory to hold directory names.",
			  NULL,
			  NOTICE_BUTTON,  "OK", 100,
			  NULL);
	}	
      }
      free(tempName);
    }
  StringSort(names, 0, numFiles-1);
  for (j = 0; j < numFiles; j++)
  {
    if ( (strcmp( names[j], "./")!=0) 
	&& (strcmp( names[j], "../")!=0) )
    {
      if (*(names[j]+strlen(names[j])-1)=='/')
	glyph = folder;
      else
	glyph = file;
      xv_set((Xv_object)panelList,
	     PANEL_LIST_INSERT, item,
	     PANEL_LIST_GLYPH, item, glyph,
	     PANEL_LIST_STRING, item, names[j],
	     PANEL_LIST_CLIENT_DATA, item, item,
	     NULL);				    /* modify this later to put in glyphs  */
      free(names[j]);
      item++;
    }
    else
      free(names[j]);
  }
  free(names);
  if( item == 0)
  {
    if (firstLineGlobal)
    {
      if (showSpecificFiles)
	sprintf(buf, "%s", appNameGlobal);
      else
	strcpy(buf, "Readable");
      xv_set((Xv_object)browsePopup->message1, PANEL_LABEL_STRING, "No", PANEL_LABEL_BOLD, TRUE, NULL);
      xv_set((Xv_object)browsePopup->message2, PANEL_LABEL_STRING, buf, PANEL_LABEL_BOLD, TRUE, NULL);
      xv_set((Xv_object)browsePopup->message3, PANEL_LABEL_STRING, "Files", PANEL_LABEL_BOLD, TRUE, NULL);
    }
    else
      xv_set((Xv_object)browsePopup->message1, PANEL_LABEL_STRING, "Empty Directory", PANEL_LABEL_BOLD, TRUE, NULL);
  }
  else
  {
    xv_set((Xv_object)panelList, PANEL_INACTIVE, FALSE, NULL);
    xv_set((Xv_object)browsePopup->message1, PANEL_LABEL_STRING, "", PANEL_LABEL_BOLD, TRUE, NULL);
    xv_set((Xv_object)browsePopup->message2, PANEL_LABEL_STRING, "", PANEL_LABEL_BOLD, TRUE, NULL);
    xv_set((Xv_object)browsePopup->message3, PANEL_LABEL_STRING, "", PANEL_LABEL_BOLD, TRUE, NULL);
  }
  xv_set((Xv_object)panelList, XV_SHOW, TRUE, NULL);
  closedir(dirp);
  return(1);
}


/**********************************************************************
  Copy strings from p1 to p2
  **********************************************************************/
static int			
  CopyList(Panel *p1, Panel *p2)
{
  int p1rows, i;
  
  if (p1 == NULL || p2 == NULL)
    return(FALSE);
  p1rows = (int)xv_get((Xv_object)p1, PANEL_LIST_NROWS);
  xv_set((Xv_object)p2, XV_SHOW, FALSE, NULL);
  for(i = 0; i < p1rows ; i++)					    /* row numbering starts at zero */
  {
    xv_set((Xv_object)p2,
	   PANEL_LIST_INSERT, i,
	   PANEL_LIST_STRING, i, (char *)xv_get((Xv_object)p1, PANEL_LIST_STRING, i),
	   PANEL_LIST_CLIENT_DATA, i, i,
	   NULL);
  }
  xv_set((Xv_object)p2, XV_SHOW, TRUE, NULL);
}

/***********************************************************************
  Add s2 to s1 inserting a /
  Possible conventions:	always put a / at end
  never put a / at end
  put a / at end of dirs only
  I chose never.  Assume string is already null terminated. Assume s2 does
  not have a leading or trailing '/'.
  **********************************************************************/
static int
  AddComponent(char *s1, char *s2)
{
  if ( s2 == NULL || s2 == NULL)
    return(TRUE);						    /* Don't do anything, got a null argument, or a ref to already currentPath */
  if ( strlen(s1)+strlen(s2)+3 > MAXPATHLEN )			    /* +2 for each "\0" and +1 for "/" */
  {
    fprintf(stderr, "AddComponent: New path would be too long (greater than %d characters)\n", MAXPATHLEN);
    return(FALSE);
  }
  else
  {
    if (!(s1[strlen(s1)-1] == '/'))                   /* don't add a '/' if there already is one */
      strcat(s1, "/");
    strncat(s1, s2, MAXPATHLEN-strlen(currentPath)-2);		    /* Use strncat incase a non null terminated s2 comes in */
    return(TRUE);
  }
}


/**********************************************************************
  Take the last component off of the pathname string. If try to remove /
  just return /.
  ie string = "/adir/afile"		returns string ="/adir"
  string = "/adir"			returns string ="/"
  string = "aname"                     returns string ="aname"
  **********************************************************************/
static int
  RemoveLastComponent(char *string)
{
  char *p;
  
  if (!string)
  {
    fprintf(stderr, "RemoveLastComponent: Ack! I got a null string \n");
    return(FALSE);						    /* ACK, null string sent in! */
  }
  p = (char *)strrchr(string, '/');
  if (p)
  {
    bzero(p+1, strlen(p));
    return (TRUE) ;
  }
  else
    return(FALSE);						    /* oops, strrchr failed, single component name */
}


static int
  SetCurrentText(char *string)
{
  xv_set((Xv_object)browsePopup->currentPathText, PANEL_VALUE, string, NULL);
}


/***********************************************************************
  Clear this list (and anything to the right of this list ? <- add later)
  **********************************************************************/
static int
  ClearList(Panel *panelList)
{
  int i;
  
  if (panelList)
  {
    xv_set((Xv_object)panelList, XV_SHOW, FALSE, NULL);
    for(i = (int)xv_get((Xv_object)panelList, PANEL_LIST_NROWS) - 1; i >= 0; i--) /* row numbering starts at zero */
      xv_set((Xv_object)panelList, PANEL_LIST_DELETE, i, NULL);
    xv_set((Xv_object)panelList, XV_SHOW, TRUE, NULL);
    return(TRUE);
  }
  else
    return(FALSE);
}


/**********************************************************************
  Scan a panel list and return the number of the selected row.
  If no row selected, return BrowseNone ( defined in guts.h )
  **********************************************************************/
static int
  AnySelected(Panel *panelList)
{
  int i;
  char name[MAXPATHLEN];
  if (!panelList)
    return(FALSE);
  for(i=(int)xv_get((Xv_object)panelList, PANEL_LIST_NROWS)-1; i >= 0; i--)
    if ((int)xv_get((Xv_object)panelList, PANEL_LIST_SELECTED, i))
    {
      strcpy(name, (char *)xv_get((Xv_object)panelList, PANEL_LIST_STRING, i));
      if (*(name+strlen(name)-1)=='/')
	return(i);
    }
  return(BrowseNone);
}

/**********************************************************************
  truncate s1 at the last occurence of s2, & remove trailing /.
  if s1 = "string th/ing"
  and s2 = "ing"
  this will return "string th" 
  **********************************************************************/

static int
  RemoveComponent(char *s1, char *s2)
{
  char *p, *q;
  if(s1!=NULL && s2!=NULL)							    /* No NULLs allowed */
  {
    if(strlen(s1) < strlen(s2))
      return(FALSE);
    q = NULL;
    for(p = (char *)strstr(s1, s2); p != NULL; p =(char *)strstr(p, s2)) /*March down the string until strstr finds no more s2 */
    {
      q = p;
      p++;							    /* so that we don't keep returning the same occurence of s2 */
    }
    if (q)
    {
      bzero(--q, strlen(q));				    /* decrement q to get rid of '/' */
      return(TRUE);
    }
    else
      return(FALSE);					    /* did not find s2 in s1 */
  }
  else
    return(FALSE);
}

static int
  FindRow(Panel *panelList, char *string)
{
  int i;
  char name[MAXPATHLEN];
  if (!panelList)
    return(BrowseNone);
  for(i=(int)xv_get((Xv_object)panelList, PANEL_LIST_NROWS)-1; i >= 0; i--)
  {
    strcpy(name, (char *)xv_get((Xv_object)panelList, PANEL_LIST_STRING, i));
    if ( strcmp(name, string) == 0 )
      return(i);
  }
  return(BrowseNone);
}

/* sort array[left]...array[right] into increasing order */

static void
  StringSort(char *array[], int left, int right)
{
  int i, last;
  if (left >= right) return;
  Swap (array, left, (left+right)/2);
  last = left;
  for (i = left+1; i<=right; i++)
    if (strcmp(array[i], array[left]) < 0)
      Swap(array, ++last, i);
  Swap(array, left, last);
  StringSort(array, left, last-1);  
  StringSort(array, last+1, right);
}

static void
  Swap(char *array[], int i, int j)
{
  char *temp;
  temp = array[i];
  array[i] = array[j];
  array[j] = temp;
}

static int
  PrintList(Panel *panelList)                /*  for debugging purposes only  */
{
  int i;
  for(i=0; i<(int)xv_get((Xv_object)panelList, PANEL_LIST_NROWS); i++)
    printf("%s  ", (char *)xv_get((Xv_object)panelList, PANEL_LIST_STRING, i));
  printf("\n");
  return 1;
}


/*
 * Notify callback function for `hiddenFilesSetting'.
 */
void
  BrowseShowHiddenFiles(item, value, event)
Panel_item	item;
int		value;
Event		*event;
{
  if (showHiddenFiles==0)
  {
    xv_set((Xv_object)browsePopup->hiddenFilesSetting, PANEL_VALUE, 1, NULL);
    showHiddenFiles = 1;
    BrowseCurrentPathTextNotify(NULL, NULL);
  }
  else
  {
    xv_set((Xv_object)browsePopup->hiddenFilesSetting, PANEL_VALUE, 2, NULL);
    showHiddenFiles = 0;
    BrowseCurrentPathTextNotify(NULL, NULL);
  }
}

/*
 * Notify callback function for `specificFilesSetting'.
 */
void
  BrowseShowSpecificFiles(item, value, event)
Panel_item	item;
int		value;
Event		*event;
{
  if (showSpecificFiles==0)
  {
    xv_set((Xv_object)browsePopup->specificFilesSetting, PANEL_VALUE, 1, NULL);
    showSpecificFiles = 1;
    BrowseCurrentPathTextNotify(NULL, NULL);
  }
  else
  {
    xv_set((Xv_object)browsePopup->specificFilesSetting, PANEL_VALUE, 2, NULL);
    showSpecificFiles = 0;
    if (browsePopup->hiddenFilesSetting == NULL)
    {
      browsePopup->hiddenFilesSetting = 
	Browse_browsePopup_hiddenFilesSetting_create(browsePopup, browsePopup->controls2);  
      xv_set((Xv_object)browsePopup->hiddenFilesSetting, PANEL_VALUE, 2, NULL);
    }
    BrowseCurrentPathTextNotify(NULL, NULL);
  }
}

/*
 * Event callback function for `currentPathText'.
 */
void
  BrowseCurrentPathEventHandler(item, event)
Panel_item	item;
Event		*event;
{
  if ((event_id(event) == TabKey || event_id(event) == EscapeKey || 
       event_id(event) == SpaceBar) && event_is_down(event))    
    FileCompletion();
  else
    panel_default_handle_event(item, event);  
}


static void
  FileCompletion()
{
  char pattern[MAXPATHLEN];
  char path[MAXPATHLEN];
  char newPath[MAXPATHLEN];
  char bestMatch[MAXPATHLEN];
  char **fileMatches;
  DIR *dirp; 
  struct dirent *dp;
  int numFiles = 0;
  int matchFound = 0;
  int i, firstFlag, stop= 0;
  char oldc, c;
  struct passwd *pw;
  
  fileMatches = (char **)malloc(BlockSize*sizeof(char *));
  strcpy(path,(char *)xv_get((Xv_object)browsePopup->currentPathText, PANEL_VALUE));
  strcpy(pattern, "");
  if (strstr(path, "//"))
  {
    strcpy(pattern, strstr(path, "//")+2);
    strcpy(path, "/");
  }
  else if (strstr(path, "~/"))
  {
    strcpy(pattern, strstr(path, "~/")+2);
    strcpy(path, (char *)getenv("HOME"));
    strcpy(&path[strlen(path)], "/");
  }
  else if (strstr(path, "/~") || path[0]=='~')
  {
    strcpy(newPath, strstr(path, "~")+1);
    if (strstr(newPath, "/"))
      strcpy(pattern, strstr(newPath, "/")+1);
    newPath[strlen(newPath)-strlen(pattern)] = '\0';
    if (newPath[strlen(newPath)-1]=='/')
      newPath[strlen(newPath)-1]='\0';
    if (pw=(struct passwd *)getpwnam(newPath))
    {
      strcpy(path, (char *)pw->pw_dir);
      strcpy(&path[strlen(path)], "/");
    }
    else
    {
      strcpy(path, (char *)getenv("HOME"));
      RemoveLastComponent(path);
      strcpy(pattern, newPath);
    }
  }
  strcpy(newPath, path);
  sprintf(path, "%s%s", newPath, pattern);
  if (strrchr(path, '/'))
    strcpy(pattern, (char *)(strrchr(path, '/')+1));
  path[strlen(path)-strlen(pattern)] = '\0';
  
  if (pattern)
  {
    dirp = opendir(path); 
    if (dirp == NULL)         
      return;
    /* check the first BlockSize files for a match */
    while ((dp = readdir(dirp)) && numFiles < BlockSize)  
    {
      if (!strncmp(pattern, dp->d_name, strlen(pattern)))
      {
	fileMatches[numFiles] = malloc(strlen(dp->d_name)+1);
	strcpy(fileMatches[numFiles], dp->d_name);
	numFiles++;
      }
    }
    if (numFiles > 0)
    {
      strcpy(bestMatch, pattern);
      while(!matchFound)
      {
	firstFlag = 1;
	for (i=0; i<numFiles; i++)
	{
	  c=fileMatches[i][strlen(bestMatch)];
	  if (!firstFlag)
	  {
	    if (c != oldc)
	      matchFound = 1;
	  }
	  else
	    firstFlag = 0;
	  oldc = c;
	  free(fileMatches[i]);
	}
	if (stop++>15)
	  matchFound = 1;
	if (!matchFound)
	{
	  i = strlen(bestMatch);
	  bestMatch[i] = c;
	  bestMatch[i+1] = '\0';
	}
      }
      sprintf(newPath, "%s%s", path, bestMatch);
      if (DirTest(newPath) && newPath[strlen(newPath)-1]!='/')
	strcpy(&newPath[strlen(newPath)], "/");	    
      xv_set((Xv_object)browsePopup->currentPathText, PANEL_VALUE, newPath, NULL);
    }
  }
  free(fileMatches);
}

static void
  SetMultiple()
{
  if (functionGlobal == BrowseMultiple)
  {
    xv_set((Xv_object)browsePopup->fileList, PANEL_CHOOSE_ONE, FALSE, NULL);  
    xv_set((Xv_object)browsePopup->dirList, PANEL_CHOOSE_ONE, FALSE, NULL);  
  }
  else
  {
    xv_set((Xv_object)browsePopup->fileList, PANEL_CHOOSE_ONE, TRUE, NULL);  
    xv_set((Xv_object)browsePopup->dirList, PANEL_CHOOSE_ONE, TRUE, NULL);  
  }
}

static void 
  BrowseDeselectAll(Panel *panelList)
{
  int i;
  for(i = (int)xv_get((Xv_object)panelList, PANEL_LIST_NROWS) - 1; i >= 0; i--) 
  {
    if ((int)xv_get((Xv_object)panelList, PANEL_LIST_SELECTED, i))
      xv_set((Xv_object)panelList, PANEL_LIST_SELECT, i, FALSE, NULL);
  }
}

/*
 * This function checks if a double click event has occurred.  
 * Both the time and mouse distance are taken into account.
 */
static int BrowseCheckDoubleClick(Event *event)
{
  static Event lastEvent;
  static short initial = True;
  static int timeThreshold;
  short	returnValue = False;
  char *clickTimeout;
  int deltaTime;
  int deltaX, deltaY;
  struct timeval thisTime, lastTime;
  
  if (initial == True) 
  {
    timeThreshold = TimeThreshold;
    clickTimeout = XGetDefault((Display *) xv_get(browsePopup->browsePopup, XV_DISPLAY), 
			       "OpenWindows", "MultiClickTimeout");	    /* Get the timeout value from the Xdefaults database */
    if (clickTimeout != NULL) 
      timeThreshold = atoi(clickTimeout) * 100;
    initial = False;
  }
  else
  {
    if (event_is_up(event))					    /* only deal with mouse down events */
      return returnValue;
    if (event_action(event) == ACTION_SELECT && event_action(&lastEvent) == ACTION_SELECT ) 
    {
      thisTime = event_time(event);
      lastTime = event_time(&lastEvent);
      deltaTime  = (thisTime.tv_sec - lastTime.tv_sec) * 1000
	+ (thisTime.tv_usec - lastTime.tv_usec) / 1000;
      if (deltaTime > 0 && deltaTime <= timeThreshold )				    /* Is the time within bounds? */
      {
	deltaX	= abs(event_x(&lastEvent) - event_x(event));	    /* Check if the distance is within DistanceThreshold */
	deltaY	= abs(event_y(&lastEvent) - event_y(event));	
	if (deltaX <= DistanceThreshold && deltaY <= DistanceThreshold)
	  returnValue = True;
      }
    }
  }
  lastEvent = *event;
  return returnValue;
}
