 /*
  * Khoros: $Id: attributes.c,v 1.4 1992/03/20 22:42:13 dkhoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: attributes.c,v 1.4 1992/03/20 22:42:13 dkhoros Exp $";
#endif

 /*
  * $Log: attributes.c,v $
 * Revision 1.4  1992/03/20  22:42:13  dkhoros
 * VirtualPatch5
 *
  */ 

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1990, University of New Mexico.  All rights reserved.
 *
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"	 /* Copyright 1990 by UNM */
#include "cantata.h"


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>								<<<<
   >>>>	    file name:  attributes.c				<<<<
   >>>>								<<<<
   >>>>   description:						<<<<
   >>>>								<<<<
   >>>>      routines:  xvl_set_grid()				<<<<
   >>>>			xvl_set_mapping()			<<<<
   >>>>			xvl_set_variables()			<<<<
   >>>>			xvl_set_dav()				<<<<
   >>>>			xvl_set_modified()			<<<<
   >>>>			xvl_set_clipboard()			<<<<
   >>>>			xvl_set_gridsize()			<<<<
   >>>>			xvl_set_glyphsize()			<<<<
   >>>>			xvl_set_distributed()			<<<<
   >>>>			xvl_set_negotiator()			<<<<
   >>>>								<<<<
   >>>>			xvl_get_attributes()			<<<<
   >>>>			xvl_set_attributes()			<<<<
   >>>>			xvl_load_attributes()			<<<<
   >>>>			xvl_load_wresource()			<<<<
   >>>>								<<<<
   >>>> modifications:						<<<<
   >>>>								<<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */



/************************************************************
*
* Routine Name: xvl_set_grid
*
*      Purpose: This routine is used to set the workspace's glyph
*		grid to be visible or not.
*
*        Input: workspace    -  the workspace to set the mapping
*
*       Output: makes the workspace grid visible or not
*
*    CALLED BY:
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


xvl_set_grid(workspace)

Workspace *workspace;
{
	Glyph	  *glyph;
	GlyphList *glyphlist;


	/*
	 *  Change the grid for the current workspace and see if any of it's
	 *  children want to be changed.
	 */
	xvl_create_grid(workspace);
	xvl_redraw_connections(workspace);

	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   if (glyph->type == PROCEDURE)
	   {
	      if (glyph->val.macro->inherit_attributes == True)
		 xvl_set_grid(glyph->val.macro);
	   }
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name: xvl_set_mapping
*
*      Purpose: This routine is used to set the workspace's glyph
*		mapping (leave_glyph).  This variable is used
*		to indicate how we are supposed to handle the mapping
*		and unmapping of glyphs when the user opens the
*		form or subform, xvl_form_cb().
*
*        Input: workspace    -  the workspace to set the mapping
*
*       Output: 
*
*    CALLED BY:
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


xvl_set_mapping(workspace)

Workspace *workspace;
{
	Glyph		*glyph;
	GlyphList	*glyphlist;
	Workspace	*macro;


	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   if (glyph->type == PROCEDURE)
	   {
	      macro = glyph->val.macro;
	      if (macro->inherit_attributes == True)
		 xvl_set_mapping(macro);
	   }
	   xvl_update_mapping(glyph);
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name: xvl_set_variables
*
*      Purpose: This routine is used to set the workspace's variables
*		to be global or not
*
*        Input: workspace    -  the workspace to set the mapping
*
*       Output: makes the workspace variables global
*
*    CALLED BY:
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


xvl_set_variables(workspace)

Workspace *workspace;
{
	Glyph	  *glyph;
	GlyphList *glyphlist;


	/*
	 *  Change the workspace's variable to look for global variables
	 *  (search up the parent).
	 */
	xvl_global_variables(workspace);

	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   if (glyph->type == PROCEDURE)
	   {
	      if (glyph->val.macro->inherit_attributes == True)
		 xvl_set_variables(glyph->val.macro);
	   }
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name: xvl_set_dav
*
*      Purpose: This routine is used to set the workspace's glyphs
*		display data availables or not (dav).
*
*        Input: workspace    -  the workspace to set the mapping
*
*       Output: makes the workspace dav connections visible or not
*
*    CALLED BY:
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


xvl_set_dav(workspace)

Workspace *workspace;
{
	Glyph	  *glyph;
	GlyphList *glyphlist;


	/*
	 *  Change the dav for the current workspace and see if any of it's
	 *  children want to be changed.
	 */
	xvl_update_all_dav(workspace);

	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   if (glyph->type == PROCEDURE)
	   {
	      if (glyph->val.macro->inherit_attributes == True)
		 xvl_set_dav(glyph->val.macro);
	   }
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name: xvl_set_modified
*
*      Purpose: This routine is used to set the workspace's glyphs
*		display modified or not (modified).
*
*        Input: workspace    -  the workspace to set the mapping
*
*       Output: makes the workspace modified glyphs visible or not
*
*    CALLED BY:
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


xvl_set_modified(workspace)

Workspace *workspace;
{
	Glyph	  *glyph;
	GlyphList *glyphlist;


	/*
	 *  Change the dav for the current workspace and see if any of it's
	 *  children want to be changed.
	 */
	xvl_update_all_modified(workspace);

	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   if (glyph->type == PROCEDURE)
	   {
	      if (glyph->val.macro->inherit_attributes == True)
		 xvl_set_modified(glyph->val.macro);
	   }
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name: xvl_set_clipboard
*
*      Purpose: This routine is used to set the workspace's glyph
*		clipboard icon
*
*        Input: workspace    -  the workspace to set the mapping
*
*       Output: 
*
*    CALLED BY:
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


xvl_set_clipboard(workspace)

Workspace *workspace;
{
	Glyph	    *glyph;
	GlyphList   *glyphlist;
	Workspace   *attributes;


	/*
	 *  The show clipboard has changed.  So we need to map or unmap the
	 *  clipboard icon with each workspace.
	 */
	if (workspace->clipboard != NULL)
	{
	   attributes = xvl_get_attributes(workspace);
	   if (attributes->show_clipboard)
	      XtMapWidget(workspace->clipboard);
	   else
	      XtUnmapWidget(workspace->clipboard);
	}

	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   if (glyph->type == PROCEDURE)
	   {
	      if (glyph->val.macro->inherit_attributes == True)
		 xvl_set_clipboard(glyph->val.macro);
	   }
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name: xvl_set_gridsize
*
*      Purpose: This routine is used to set the workspace's glyph
*		grid size.
*
*        Input: workspace    -  the workspace to set the mapping
*
*       Output: 
*
*    CALLED BY:
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


xvl_set_gridsize(workspace)

Workspace *workspace;
{
	Glyph	    *glyph;
	GlyphList   *glyphlist;
	Workspace   *attributes;


	/*
	 *  The grid size has changed.  So we need to call
	 *  xvl_snap_glyph to snap the glyphs to the new
	 *  grid size.
	 */
	attributes = xvl_get_attributes(workspace);
	if (attributes->show_grid)
	   xvl_create_grid(workspace);

	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   xvl_snap_glyph(glyph, False);

	   if (glyph->type == PROCEDURE)
	   {
	      if (glyph->val.macro->inherit_attributes == True)
		 xvl_set_gridsize(glyph->val.macro);
	   }
	   glyphlist = glyphlist->next;
	}
	XClearWindow(XtDisplay(workspace->draw), XtWindow(workspace->draw));
	xvl_redraw_connections(workspace);
}



/************************************************************
*
* Routine Name: xvl_set_glyphsize
*
*      Purpose: This routine is used to set the workspace's glyph
*		size.
*
*        Input: workspace    -  the workspace to set the mapping
*
*       Output: 
*
*    CALLED BY:
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


xvl_set_glyphsize(workspace)

Workspace *workspace;
{
	Glyph	    *glyph;
	GlyphList   *glyphlist;


	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   xvl_create_glyph(glyph);

	   if (glyph->type == PROCEDURE)
	   {
	      if (glyph->val.macro->inherit_attributes == True)
		 xvl_set_glyphsize(glyph->val.macro);
	   }
	   glyphlist = glyphlist->next;
	}
	XClearWindow(XtDisplay(workspace->draw), XtWindow(workspace->draw));
	xvl_redraw_connections(workspace);
}



/************************************************************
*
* Routine Name: xvl_set_distributed
*
*      Purpose: This routine is used to set the workspace's glyph
*		distributed attribute.  Essentially it enables
*		or disables the user from specifying where a
*		glyph is to be executed.
*
*        Input: workspace    -  the workspace to set the mapping
*
*       Output: 
*
*    CALLED BY:
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


xvl_set_distributed(workspace)

Workspace *workspace;
{
	Glyph	    *glyph;
	GlyphList   *glyphlist;


	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   xvl_update_distributed(glyph);

	   if (glyph->type == CONTROL || glyph->type == GLYPH &&
                 glyph->type == COMMAND)
	   {
	      xvl_update_glyph_nodelist(glyph);
	   }
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name: xvl_set_negotiator
*
*      Purpose: This routine is used to set the workspace's glyph
*		stream execution attribute.  Essentially it enables
*		or disables the user from specifying a stream based
*		transport from being used.
*
*        Input: workspace    -  the workspace to set the mapping
*
*       Output: 
*
*    CALLED BY:
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


xvl_set_negotiator(workspace)

Workspace *workspace;
{
	Glyph	    *glyph;
	GlyphList   *glyphlist;


	glyphlist = workspace->glyphs;
	while (glyphlist != NULL)
	{
	   glyph = glyphlist->glyph;
	   xvl_update_negotiator(glyph);

	   if (glyph->type == CONTROL || glyph->type == GLYPH &&
                 glyph->type == COMMAND)
	   {
	      xvl_update_glyph_nodelist(glyph);
	   }
	   glyphlist = glyphlist->next;
	}
}



/************************************************************
*
* Routine Name: xvl_get_attributes
*
*      Purpose: This routine is used to get the attributes for
*		the current workspace.
*
*        Input: workspace    -  the workspace to get the attributes for
*
*       Output: returns the workspace that contains the current
*		workspace's attributes.
*
*    CALLED BY:
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


Workspace *xvl_get_attributes(workspace)

Workspace *workspace;
{
	if (workspace == clipboard)
	   return(main_workspace);
	else if (workspace->inherit_attributes == False)
	   return(workspace);

	while (workspace->parent != NULL)
	{
	   workspace = workspace->parent->workspace;
	   if (workspace->inherit_attributes == False)
	      return(workspace);
	}
	return(workspace);
}



/************************************************************
*
* Routine Name: xvl_set_attributes
*
*      Purpose: This routine is used to set the attributes for
*		the current workspace.
*
*        Input: workspace    -  the workspace to get the attributes for
*
*       Output: returns the workspace that contains the current
*		workspace's attributes.
*
*    CALLED BY:
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/


xvl_set_attributes(workspace)

Workspace *workspace;
{
	xvl_set_grid(workspace);
	xvl_set_mapping(workspace);
	xvl_set_variables(workspace);
	xvl_set_dav(workspace);
	xvl_set_modified(workspace);
	xvl_set_clipboard(workspace);
	xvl_set_gridsize(workspace);
	xvl_set_glyphsize(workspace);
}



/**************************************************************
*
*Routine Name: xvl_load_attributes
*
*     Purpose: Load the user's workspace preferences. The type
*	       of attributes are grid size, show grid, use viewport,
*	       workspace geometry, workspace foreground and background
*	       colors.
*
*       Input:  none
*
*      Output:  loads the "workspace" attributes part of the structure.
*
* CALLED FROM: main
*
*  Written By: Mark Young
*
*
***************************************************************/


xvl_load_attributes(workspace)

Workspace	*workspace;
{
	XGCValues   xgcv;
	XFontStruct *font;
	char        *value;
	unsigned    long mask;
	Display	    *display = xvf_display;
	Window      rootwindow = XDefaultRootWindow(display);


	/*
	 *  Check to see if the user wishes to turn the grid on.
	 */
	if (((value = XGetDefault(display, "cantata", "grid")) != NULL) ||
	    ((value = XGetDefault(display,"cantata","Grid")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->show_grid = True;
	   else
	      workspace->show_grid = False;

	   free(value);
	}
	else
	   workspace->show_grid = False;

	/*
	 *  Check to see if the user wishes to turn the grid on.
	 */
	if (((value = XGetDefault(display, "cantata", "echo_routine")) != NULL)
	    || ((value = XGetDefault(display,"cantata","EchoRoutine")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->echo_exec = True;
	   else
	      workspace->echo_exec = False;

	   free(value);
	}
	else
	   workspace->echo_exec = False;

	/*
	 *  Check to see if the user place the glyph's or to place it
	 *  where the cursor is at creation time.
	 */
	if (((value = XGetDefault(display, "cantata",
	    "glyph_placement")) != NULL) || ((value = XGetDefault(display,
	    "cantata","GlyphPlacement")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->glyph_placement = True;
	   else
	      workspace->glyph_placement = False;

	   free(value);
	}
	else
	   workspace->glyph_placement = False;

	/*
	 *  Check to see if the user place the workspace or to place it
	 *  according to the workspace and macro geometry
	 */
	if (((value = XGetDefault(display, "cantata",
	    "auto_placement")) != NULL) || ((value = XGetDefault(display,
	    "cantata","AutoPlacement")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->auto_placement = True;
	   else
	      workspace->auto_placement = False;

	   free(value);
	}
	else
	   workspace->auto_placement = False;

	/*
	 *  Check to see if the user place the glyph's or to place it
	 *  where the cursor is at creation time.
	 */
	if (((value = XGetDefault(display, "cantata", "rubberband")) != NULL) ||
	    ((value = XGetDefault(display, "cantata", "Rubberband")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->rubberband = True;
	   else
	      workspace->rubberband = False;

	   free(value);
	}
	else
	   workspace->rubberband = False;

	/*
	 *  Check to see if the user wants the small glyph or they want
	 *  the normal ones
	 */
	if (((value = XGetDefault(display, "cantata",
	    "small_glyph")) != NULL) || ((value = XGetDefault(display,
	    "cantata","SmallGlyph")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->small_glyphs = True;
	   else
	      workspace->small_glyphs = False;

	   free(value);
	}
	else
	   workspace->small_glyphs = False;

	/*
	 *  Check to see if the user wants to have a seperate canvas for a
	 *  workspace rather than attching it to the forms.
	 */
	if (((value = XGetDefault(display, "cantata",
	    "attach_canvas")) != NULL) || ((value = XGetDefault(display,
	    "cantata","AttachCanvas")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"false", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"off") == 0) ||
	       (strcmp(value,"0") == 0))
	      workspace->attach_canvas = False;
	   else
	      workspace->attach_canvas = True;

	   free(value);
	}
	else
	   workspace->attach_canvas = True;

	/*
	 *  Kludge to keep the clipboard as one workspace.
	 */
	if (workspace == clipboard)
	   workspace->attach_canvas = True;

	/*
	 *  Check to see what grid size we should snap to
	 */
	if (((value = XGetDefault(display, "cantata", "grid_size")) != NULL) ||
	    ((value = XGetDefault(display, "cantata", "GridSize")) != NULL))
	{
	   workspace->grid_size = atoi(value);
	   if (workspace->grid_size < 1)
	      workspace->grid_size = 20;
	}
	else
	   workspace->grid_size = 20;

	/*
	 *  Check to see if the user wants to snap to a grid.
	 */
	if (((value = XGetDefault(display, "cantata", "width_factor")) != NULL)
	  || ((value = XGetDefault(display, "cantata", "WidthFactor")) != NULL))
	{
	   workspace->width_factor = atoi(value);
	   if (workspace->width_factor < 1)
	      workspace->width_factor = 3;
	}
	else
	   workspace->width_factor = 3;

	/*
	 *  Check to see if the user wants to snap to a grid.
	 */
	if (((value = XGetDefault(display, "cantata", "height_factor")) != NULL)
	 || ((value = XGetDefault(display, "cantata", "HeightFactor")) != NULL))
	{
	   workspace->height_factor = atoi(value);
	   if (workspace->height_factor < 1)
	      workspace->height_factor = 3;
	}
	else
	   workspace->height_factor = 3;

	/*
	 *  Check to see if the user wants a viewport.
	 */
	if (((value = XGetDefault(display, "cantata", "viewport")) != NULL) ||
	    ((value = XGetDefault(display, "cantata", "Viewport")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->view_port = True;
	   else
	      workspace->view_port = False;

	   free(value);
	}
	else
	   workspace->view_port = True;

	/*
	 *  Check to see if the user wants to dispatch glyphs in parallel.
	 */
	if (((value = XGetDefault(display, "cantata",
	    "parallel_exec")) != NULL) || ((value = XGetDefault(display,
	    "cantata","ParallelExec")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->parallel = True;
	   else
	      workspace->parallel = False;

	   free(value);
	}
	else
	   workspace->parallel = True;

	/*
	 *  Check to see if the user wants to use the demand driven scheduling
	 *  versus data driven.
	 */
	if (((value = XGetDefault(display, "cantata",
	    "demand_driven")) != NULL) || ((value = XGetDefault(display,
	    "cantata","DemandDriven")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->demand_driven = True;
	   else
	      workspace->demand_driven = False;

	   free(value);
	}
	else
	   workspace->demand_driven = False;

	/*
	 *  Check to see if the user wants to use the demand driven scheduling
	 *  versus data driven.
	 */
	if (((value = XGetDefault(display, "cantata",
	    "leave_glyph")) != NULL) || ((value = XGetDefault(display,
	    "cantata","LeaveGlyph")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->leave_glyph = True;
	   else
	      workspace->leave_glyph = False;

	   free(value);
	}
	else
	   workspace->leave_glyph = False;

	/*
	 *  Check to see if the user wants to use global variables or make
	 *  them only local to that workspace.
	 */
	if (((value = XGetDefault(display, "cantata",
	    "global_variables")) != NULL) || ((value = XGetDefault(display,
	    "cantata","GlobalVariables")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->global_variables = True;
	   else
	      workspace->global_variables = False;

	   free(value);
	}
	else
	   workspace->global_variables = True;

	/*
	 *  Check to see if the user wishes to show data available (dav)
	 *  widgets.
	 */
	if (((value = XGetDefault(display, "cantata", "show_dav")) != NULL) ||
	    ((value = XGetDefault(display, "cantata", "ShowDav")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->show_dav = True;
	   else
	      workspace->show_dav = False;

	   free(value);
	}
	else
	   workspace->show_dav = False;

	/*
	 *  Check to see if the user wishes to show modified glyphs widgets.
	 */
	if (((value = XGetDefault(display, "cantata",
	    "show_modified")) != NULL) || ((value = XGetDefault(display,
	    "cantata","ShowModified")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->show_modified = True;
	   else
	      workspace->show_modified = False;

	   free(value);
	}
	else
	   workspace->show_modified = False;

	/*
	 *  Check to see if the user wishes to show the clipboard icon
	 *  widget.
	 */
	if (((value = XGetDefault(display, "cantata",
	    "show_clipboard")) != NULL) || ((value = XGetDefault(display,
	    "cantata", "ShowClipboard")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->show_clipboard = True;
	   else
	      workspace->show_clipboard = False;

	   free(value);
	}
	else
	   workspace->show_clipboard = False;

	/*
	 *  Check to see if the user wishes to have the workspace's inherit
	 *  the environment variables
	 */
	if (((value = XGetDefault(display, "cantata",
	    "inherit_attributes")) != NULL) || ((value = XGetDefault(display,
	    "cantata","InheritAttributes")) != NULL))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->inherit_attributes = True;
	   else
	      workspace->inherit_attributes = False;

	   free(value);
	}
	else
	   workspace->inherit_attributes = True;

	/*
	 *  If this is the main workspace then set the inheritance
	 *  of attributes to false.
	 */
	if (workspace == main_workspace)
	   workspace->inherit_attributes = False;


	/*
	 *  Get the macro workspace geometry.
	 */
	if (workspace == main_workspace)
	{
	   if (((value = XGetDefault(display,"cantata","geometry")) != NULL) ||
	       ((value = XGetDefault(display,"cantata","Geometry")) != NULL))
	      workspace->geometry = xvf_strcpy(value);
	   else
	      workspace->geometry = xvf_strcpy("950x600+10+140");
	}
	else
	{
	   if (((value = XGetDefault(display, "cantata",
	       "procedure_geometry")) != NULL) || ((value = XGetDefault(display,
	       "cantata","ProcedureGeometry")) != NULL))
	      workspace->geometry = xvf_strcpy(value);
	   else
	      workspace->geometry = xvf_strcpy("900x300+10+140");
	}

	/*
	 *  Get the default font
	 */
	if (((value = XGetDefault(display,"cantata",
	    "workspace_font")) != NULL) || ((value = XGetDefault(display,
	    "cantata","WorkspaceFont")) != NULL))
	{
	   if (!xvl_create_font(value, &font))
	   {
	      if (font != NULL)
	      {
                 fprintf(stderr,"\ncantata: unknown font (%s), using ", value);
                 fprintf(stderr,"a default font instead\n");
	      }
	      workspace->font = font;
	   }
	   else workspace->font = font;
	}
	else
	{
	   (void) xvl_create_font(NULL, &font);
	   workspace->font = font;
	}

	/*
	 *  Get the name font, if not set then set it to the workspace font.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "name_font")) != NULL) || ((value = XGetDefault(display,
	    "cantata","NameFont")) != NULL))
	{
	   if (!xvl_create_font(value, &font))
	   {
	      if (font != NULL)
	      {
                 fprintf(stderr,"\ncantata: unknown font (%s), using ", value);
                 fprintf(stderr,"workspace font instead\n");
	      }
	      workspace->name_font = workspace->font;
	   }
	   else workspace->name_font = font;
	}
	else
	{
	   workspace->name_font = workspace->font;
	}

	/*
	 *  Get the modified pixmap
	 */
	if (((value = XGetDefault(display,"cantata",
	    "modified_pixmap")) != NULL) || ((value = XGetDefault(display,
	    "cantata","ModifiedPixmap")) != NULL))
	{
	   if ((workspace->modified_pix = xvl_create_bitmap(value))
			== None)
	   {
              fprintf(stderr,"\ncantata: unknown pixmap (%s), using ", value);
              fprintf(stderr,"a default stippled pixmap.\n");
	      workspace->modified_pix = xvl_create_stippled();
	   }
	}
	else
	{
	   workspace->modified_pix = xvl_create_stippled();
	}

	/*
	 *  Check to see if the user wants to enable remote execution 
	 */
	if ((value = XGetDefault(display, "cantata", "enable_remote_exec")) ||
	    (value = XGetDefault(display, "cantata","EnableRemoteExec"))  ||
	    (value = XGetDefault(display, "cantata", "remote_exec")) ||
	    (value = XGetDefault(display, "cantata","RemoteExec")))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->remote_execution = True;
	   else
	      workspace->remote_execution = False;

	   free(value);
	}
	else
	   workspace->remote_execution = False;

	/*
	 *  Check to see if the user wants to enable stream based execution 
	 */
	if ((value = XGetDefault(display, "cantata", "enable_stream_exec")) ||
	    (value = XGetDefault(display, "cantata","EnableStreamExec")) ||
	    (value = XGetDefault(display, "cantata", "stream_exec")) ||
	    (value = XGetDefault(display, "cantata","StreamExec")))
	{
	   value = xvf_lower(value);
	   if ((strncmp(value,"true", xvf_strlen(value)) == 0) ||
	       (strcmp(value,"on") == 0) ||
	       (strcmp(value,"1") == 0))
	      workspace->stream_execution = True;
	   else
	      workspace->stream_execution = False;

	   free(value);
	}
	else
	   workspace->stream_execution = False;

	/*
	 *  Create GC for the drawing widget.
	 */
	xgcv.function	    = GXcopy;
	xgcv.line_width	    = 5;
	xgcv.line_style	    = LineSolid;
	xgcv.join_style	    = JoinRound;
	xgcv.cap_style      = CapNotLast;
	xgcv.subwindow_mode = ClipByChildren;
	xgcv.background	    = wresource.bg;
	xgcv.foreground	    = wresource.fg;
	xgcv.fill_rule	    = WindingRule;

	mask = GCFunction | GCLineWidth | GCCapStyle | GCLineStyle | GCJoinStyle
	       | GCSubwindowMode | GCFillRule | GCBackground | GCForeground;
	workspace->gc = XCreateGC(display, rootwindow, mask, &xgcv);
	if (workspace->font != NULL)
	   XSetFont(display, workspace->gc, workspace->font->fid);

	/*
	 *  Create GC for the drawing the label widget.
	 */
	xgcv.function	    = GXcopy;
	xgcv.line_width	    = 5;
	xgcv.line_style	    = LineSolid;
	xgcv.join_style	    = JoinRound;
	xgcv.cap_style      = CapNotLast;
	xgcv.subwindow_mode = ClipByChildren;
	xgcv.fill_rule	    = WindingRule;
	xgcv.background	    = wresource.bg;
	if (wresource.name_fg != ~0L)
	   xgcv.foreground = wresource.name_fg;
	else
	   xgcv.foreground = wresource.fg;

	mask = GCFunction | GCLineWidth | GCCapStyle | GCLineStyle | GCJoinStyle
	       | GCSubwindowMode | GCFillRule | GCBackground | GCForeground;
	workspace->gc_name = XCreateGC(display, rootwindow, mask, &xgcv);
	if (workspace->name_font != NULL)
	   XSetFont(display, workspace->gc_name, workspace->name_font->fid);

	xgcv.function	    = GXxor;
	xgcv.line_width	    = 2;
	xgcv.foreground	    = ~0;
	xgcv.background	    = wresource.fg ^ wresource.bg;
	mask = GCFunction | GCLineWidth | GCBackground | GCForeground;
	workspace->gc_xor = XCreateGC(display, rootwindow, mask, &xgcv);
	if (workspace->font != NULL)
	   XSetFont(display, workspace->gc_xor, workspace->font->fid);
}

xvl_load_wresource(wresource)

WorkspaceResources *wresource;
{
	XColor      xcolor;
	char        *value;
	Display	    *display = xvf_display;


	/*
	 *  Get the workspace foreground color.
	 */
	if (((value = XGetDefault(display,"cantata","line_color")) != NULL) ||
	    ((value = XGetDefault(display,"cantata","LineColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown line (connection) color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->fg = ~0L;
	   }
	   else wresource->fg = xcolor.pixel;
	}
	else wresource->fg = ~0L;

	/*
	 *  Get the workspace background color.
	 */
	if (((value = XGetDefault(display,"cantata","canvas_color")) != NULL) ||
	    ((value = XGetDefault(display,"cantata","CanvasColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown canvas color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->bg = ~0L;
	   }
	   else wresource->bg = xcolor.pixel;
	}
	else wresource->bg = ~0L;

	/*
	 *  Get the workspace foreground grid color.
	 */
	if (((value = XGetDefault(display,"cantata","grid_color")) != NULL) ||
	    ((value = XGetDefault(display,"cantata","GridColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown grid workspace color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->grid_color = ~0L;
	   }
	   else wresource->grid_color = xcolor.pixel;
	}
	else wresource->grid_color = ~0L;

	/*
	 *  Get the glyph foreground color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "glyph_foreground")) != NULL) || ((value = XGetDefault(display,
	    "cantata","GlyphForeground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown glyph foreground color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->glyph_fg = ~0L;
	   }
	   else wresource->glyph_fg = xcolor.pixel;
	}
	else wresource->glyph_fg = ~0L;

	/*
	 *  Get the glyph background color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "glyph_background")) != NULL) || ((value = XGetDefault(display,
	    "cantata","GlyphBackground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown glyph background color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->glyph_bg = ~0L;
	   }
	   else wresource->glyph_bg = xcolor.pixel;
	}
	else wresource->glyph_bg = ~0L;

	/*
	 *  Get the glyph border color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "glyph_border_color")) != NULL) || ((value = XGetDefault(display,
	    "cantata","GlyphBorderColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown glyph border color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->glyph_bd = ~0L;
	   }
	   else wresource->glyph_bd = xcolor.pixel;
	}
	else wresource->glyph_bd = ~0L;

	/*
	 *  Get the destroy foreground color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "destroy_foreground")) != NULL) || ((value = XGetDefault(display,
	    "cantata","DestroyForeground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown destroy foreground color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->destroy_fg = ~0L;
	   }
	   else wresource->destroy_fg = xcolor.pixel;
	}
	else wresource->destroy_fg = ~0L;

	/*
	 *  Get the destroy background color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "destroy_background")) != NULL) || ((value = XGetDefault(display,
	    "cantata","DestroyBackground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown destroy background color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->destroy_bg = ~0L;
	   }
	   else wresource->destroy_bg = xcolor.pixel;
	}
	else wresource->destroy_bg = ~0L;

	/*
	 *  Get the destroy border color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "destroy_border_color")) != NULL) || ((value = XGetDefault(display,
	    "cantata","DestroyBorderColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown destroy border color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->destroy_bd = ~0L;
	   }
	   else wresource->destroy_bd = xcolor.pixel;
	}
	else wresource->destroy_bd = ~0L;

	/*
	 *  Get the run foreground color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "run_foreground")) != NULL) || ((value = XGetDefault(display,
	    "cantata","RunForeground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown run foreground color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->run_fg = ~0L;
	   }
	   else wresource->run_fg = xcolor.pixel;
	}
	else wresource->run_fg = ~0L;

	/*
	 *  Get the run background color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "run_background")) != NULL) || ((value = XGetDefault(display,
	    "cantata","RunBackground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown run background color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->run_bg = ~0L;
	   }
	   else wresource->run_bg = xcolor.pixel;
	}
	else wresource->run_bg = ~0L;

	/*
	 *  Get the run border color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "run_border_color")) != NULL) || ((value = XGetDefault(display,
	    "cantata","RunBorderColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown run border color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->run_bd = ~0L;
	   }
	   else wresource->run_bd = xcolor.pixel;
	}
	else wresource->run_bd = ~0L;

	/*
	 *  Get the stop foreground color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "stop_foreground")) != NULL) || ((value = XGetDefault(display,
	    "cantata","StopForeground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown stop foreground color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->stop_fg = ~0L;
	   }
	   else wresource->stop_fg = xcolor.pixel;
	}
	else wresource->stop_fg = ~0L;

	/*
	 *  Get the stop border color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "stop_border_color")) != NULL) || ((value = XGetDefault(display,
	    "cantata","StopBorderColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown stop border color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->stop_bd = ~0L;
	   }
	   else wresource->stop_bd = xcolor.pixel;
	}
	else wresource->stop_bd = ~0L;

	/*
	 *  Get the stop background color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "stop_background")) != NULL) || ((value = XGetDefault(display,
	    "cantata","StopBackground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown stop background color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->stop_bg = ~0L;
	   }
	   else wresource->stop_bg = xcolor.pixel;
	}
	else wresource->stop_bg = ~0L;

	/*
	 *  Get the reset foreground color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "reset_foreground")) != NULL) || ((value = XGetDefault(display,
	    "cantata","ResetForeground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown reset foreground color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->reset_fg = ~0L;
	   }
	   else wresource->reset_fg = xcolor.pixel;
	}
	else wresource->reset_fg = ~0L;

	/*
	 *  Get the reset background color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "reset_background")) != NULL) || ((value = XGetDefault(display,
	    "cantata","ResetBackground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown reset background color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->reset_bg = ~0L;
	   }
	   else wresource->reset_bg = xcolor.pixel;
	}
	else wresource->reset_bg = ~0L;

	/*
	 *  Get the reset border color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "reset_border_color")) != NULL) || ((value = XGetDefault(display,
	    "cantata","ResetBorderColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown reset border color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->reset_bd = ~0L;
	   }
	   else wresource->reset_bd = xcolor.pixel;
	}
	else wresource->reset_bd = ~0L;

	/*
	 *  Get the form foreground color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "form_foreground")) != NULL) || ((value = XGetDefault(display,
	    "cantata","FormForeground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown form foreground color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->form_fg = ~0L;
	   }
	   else wresource->form_fg = xcolor.pixel;
	}
	else wresource->form_fg = ~0L;

	/*
	 *  Get the form background color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "form_background")) != NULL) || ((value = XGetDefault(display,
	    "cantata","FormBackground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown form background color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->form_bg = ~0L;
	   }
	   else wresource->form_bg = xcolor.pixel;
	}
	else wresource->form_bg = ~0L;

	/*
	 *  Get the form border color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "form_border_color")) != NULL) || ((value = XGetDefault(display,
	    "cantata","FormBorderColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown form border color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->form_bd = ~0L;
	   }
	   else wresource->form_bd = xcolor.pixel;
	}
	else wresource->form_bd = ~0L;

	/*
	 *  Get the input foreground color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "input_foreground")) != NULL) || ((value = XGetDefault(display,
	    "cantata","InputForeground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown input foreground color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->input_fg = ~0L;
	   }
	   else wresource->input_fg = xcolor.pixel;
	}
	else wresource->input_fg = ~0L;

	/*
	 *  Get the input background color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "input_background")) != NULL) || ((value = XGetDefault(display,
	    "cantata","InputBackground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown input background color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->input_bg = ~0L;
	   }
	   else wresource->input_bg = xcolor.pixel;
	}
	else wresource->input_bg = ~0L;

	/*
	 *  Get the input border color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "input_border_color")) != NULL) || ((value = XGetDefault(display,
	    "cantata","InputBorderColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown input border color");
              fprintf(stderr," (%s), using black pixel instead\n", value);
	      wresource->input_bd = ~0L;
	   }
	   else wresource->input_bd = xcolor.pixel;
	}
	else wresource->input_bd = ~0L;

	/*
	 *  Get the output foreground color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "output_foreground")) != NULL) || ((value = XGetDefault(display,
	    "cantata","OutputForeground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown output foreground color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->output_fg = ~0L;
	   }
	   else wresource->output_fg = xcolor.pixel;
	}
	else wresource->output_fg = ~0L;

	/*
	 *  Get the output background color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "output_background")) != NULL) || ((value = XGetDefault(display,
	    "cantata","OutputBackground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown output background color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->output_bg = ~0L;
	   }
	   else wresource->output_bg = xcolor.pixel;
	}
	else wresource->output_bg = ~0L;

	/*
	 *  Get the output border color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "output_border_color")) != NULL) || ((value = XGetDefault(display,
	    "cantata","OutputBorderColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown output border color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->output_bd = ~0L;
	   }
	   else wresource->output_bd = xcolor.pixel;
	}
	else wresource->output_bd = ~0L;

	/*
	 *  Get the label foreground color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "label_foreground")) != NULL) || ((value = XGetDefault(display,
	    "cantata","LabelForeground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown label foreground color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->label_fg = ~0L;
	   }
	   else wresource->label_fg = xcolor.pixel;
	}
	else wresource->label_fg = ~0L;

	/*
	 *  Get the label background color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "label_background")) != NULL) || ((value = XGetDefault(display,
	    "cantata","LabelBackground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown label background color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->label_bg = ~0L;
	   }
	   else wresource->label_bg = xcolor.pixel;
	}
	else wresource->label_bg = ~0L;

	/*
	 *  Get the label border color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "label_border_color")) != NULL) || ((value = XGetDefault(display,
	    "cantata","LabelBorderColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown label border color");
              fprintf(stderr," (%s), using black pixel instead\n", value);
	      wresource->label_bd = ~0L;
	   }
	   else wresource->label_bd = xcolor.pixel;
	}
	else wresource->label_bd = ~0L;

	/*
	 *  Get the error foreground color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "error_foreground")) != NULL) || ((value = XGetDefault(display,
	    "cantata","ErrorForeground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown error foreground color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->error_fg = ~0L;
	   }
	   else wresource->error_fg = xcolor.pixel;
	}
	else wresource->error_fg = ~0L;

	/*
	 *  Get the error background color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "error_background")) != NULL) || ((value = XGetDefault(display,
	    "cantata","ErrorBackground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown error background color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->error_bg = ~0L;
	   }
	   else wresource->error_bg = xcolor.pixel;
	}
	else wresource->error_bg = ~0L;

	/*
	 *  Get the error border color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "error_border_color")) != NULL) || ((value = XGetDefault(display,
	    "cantata","ErrorBorderColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown error border color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->error_bd = ~0L;
	   }
	   else wresource->error_bd = xcolor.pixel;
	}
	else wresource->error_bd = ~0L;

	/*
	 *  Get the remote foreground color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "remote_foreground")) != NULL) || ((value = XGetDefault(display,
	    "cantata","RemoteForeground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown remote foreground color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->remote_fg = ~0L;
	   }
	   else wresource->remote_fg = xcolor.pixel;
	}
	else wresource->remote_fg = ~0L;

	/*
	 *  Get the remote background color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "remote_background")) != NULL) || ((value = XGetDefault(display,
	    "cantata","RemoteBackground")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown remote background color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->remote_bg = ~0L;
	   }
	   else wresource->remote_bg = xcolor.pixel;
	}
	else wresource->remote_bg = ~0L;

	/*
	 *  Get the remote border color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "remote_border_color")) != NULL) || ((value = XGetDefault(display,
	    "cantata","RemoteBorderColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown remote border color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->remote_bd = ~0L;
	   }
	   else wresource->remote_bd = xcolor.pixel;
	}
	else wresource->remote_bd = ~0L;

	/*
	 *  Get the remote label name's foreground color.
	 */
	if (((value = XGetDefault(display,"cantata",
	    "name_color")) != NULL) || ((value = XGetDefault(display,
	    "cantata","NameColor")) != NULL))
	{
	   if (!xvl_create_color(value, &xcolor))
	   {
              fprintf(stderr,"\ncantata: unknown remote foreground color");
              fprintf(stderr," (%s), using default pixel instead\n", value);
	      wresource->name_fg = ~0L;
	   }
	   else wresource->name_fg = xcolor.pixel;
	}
	else wresource->name_fg = ~0L;


	/*
	 *  Get the bitmap file with all the distributed processing icons
	 */
	if (((value = XGetDefault(display,"cantata",
	    "distrib_icon_aliases")) != NULL) || ((value = XGetDefault(display,
	    "cantata","DistribIconAliases")) != NULL))
	{
	   wresource->icon_alias_file = value;
	}
	else wresource->icon_alias_file = xvf_strcpy("IconAliases");
}
