 /*
  * Khoros: $Id: lvhisto.c,v 1.2 1992/03/20 23:05:57 dkhoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: lvhisto.c,v 1.2 1992/03/20 23:05:57 dkhoros Exp $";
#endif

 /*
  * $Log: lvhisto.c,v $
 * Revision 1.2  1992/03/20  23:05:57  dkhoros
 * VirtualPatch5
 *
  */ 

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvhisto.c
 >>>>
 >>>>      Program Name: vhisto
 >>>>
 >>>> Date Last Updated: Tue Mar  5 22:33:03 1991 
 >>>>
 >>>>          Routines: lvhisto - the library call for vhisto
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvhisto - library call for vhisto
*
* Purpose:
*    
*    Finds the histogram of the gray-levels  of  a  real,  single
*    band, non-bit image.
*    
*    
* Input:
*    
*         1. image -- xvimage structure
*         2. hist -- pointer to float array.
*         3. lrange -- lower range bound
*         4. urange -- upper range bound
*         5. nbins -- number of bins
*         6. mflag -- non-zero means use mask image
*         7. mimage -- xvimage structure to use as mask image, must
*                      be structurally the same as *image.
*         8. logflg -- if nonzero, takes the log of the data before
*                      histogramming
*         9. perflg -- if nonzero then output the percentage of total
*                      count rather than the individual bin counts
*    
*    
* Output:
*    
*         1. hist -- histogram array  with filled values
*    
*         lvhisto returns 1 on success and 0 on failure.
*    
*    
*
* Written By: Ramiro Jordan, Marcelo Teran
*    
*    20-Feb-90 Added support for additional data types and
*              bin specifications. Scott Wilson
*    20-Feb-90 Added mask image support. Scott Wilson
*    21-Feb-90 Added log space support. Scott Wilson
*    21-Feb-90 Added percentage output. Scott Wilson
*    
*    
****************************************************************/


/* -library_def */
int
lvhisto(image,hist,lrange,urange,nbins,mflag,mimage,logflg,perflg)
struct xvimage *image,*mimage;
float *hist;
float lrange,urange;
int nbins,mflag,logflg,perflg;
/* -library_def_end */

/* -library_code */
{
    int  nr,nc,i,k;
    unsigned char *pb,*pbm;
    short *ps,*psm;
    int *pi,*pim;
    float *pf,*pfm,z,total;

    if (image->data_storage_type != VFF_TYP_1_BYTE &&
        image->data_storage_type != VFF_TYP_2_BYTE &&
        image->data_storage_type != VFF_TYP_4_BYTE &&
        image->data_storage_type != VFF_TYP_FLOAT)
    {
        (void) fprintf(stderr,"lvhisto: Image must be of type BYTE, SHORT, INT, or FLOAT.\n");
        return(0);
    }

    if (mflag && (mimage->data_storage_type != image->data_storage_type))
      {
        fprintf(stderr,"lvhisto: Mask image must be same data type as input image\n");
        return(0);
      }

    if (mflag && (mimage->row_size != image->row_size || mimage->col_size !=
                   image->col_size))
      {
        fprintf(stderr,"lvhisto: Mask image and input image must be same size\n");
        return(0);
      }

    nr = image->col_size;
    nc = image->row_size;

    /* Zero histogram */
    if (hist == NULL)
      {
        fprintf(stderr,"lvhisto: Null pointer to histogram array!\n");
        return(0);
      }
    bzero((char *)hist, nbins*sizeof(float));

    if (nbins <=0)
      {
        fprintf(stderr,"lvhisto: Illegal bin count - must be positive number!\n");
        return(0);
      }
    z = (urange-lrange)/nbins; /* Bin interval */
    z = 1.0/z;

    /* Form histogram */
    switch(image->data_storage_type)
      {
        case VFF_TYP_1_BYTE:
          pb = (unsigned char *)(image->imagedata);
          if (!mflag)
            {
              for (i = 0; i < nc*nr; i++)
                {
                  if (!logflg) k = (*pb-lrange)*z;
                  else k = (log10((double)(*pb))-lrange)*z;
                  if (k >= 0 && k < nbins) hist[k] += 1.0;
                  pb++;
                }
            }
          else
            {
              pbm = (unsigned char *)(mimage->imagedata);
              for (i = 0; i < nc*nr; i++)
                {
                  if (*pbm++)
                    {
                      if (!logflg) k = (*pb-lrange)*z;
                      else k = (log10((double)(*pb))-lrange)*z;
                      if (k >= 0 && k < nbins) hist[k] += 1.0;
                    }
                  pb++;
                }
            }
          break;
       case VFF_TYP_2_BYTE:
          ps = (short *)(image->imagedata);
          if (!mflag)
            {
              for (i = 0; i < nc*nr; i++)
                {
                  if (!logflg) k = (*ps-lrange)*z;
                  else k = (log10((double)(*ps))-lrange)*z;
                  if (k >= 0 && k < nbins) hist[k] += 1.0;
                  ps++;
                }
            }
          else
            {
              psm = (short *)(mimage->imagedata);
              for (i = 0; i < nc*nr; i++)
                {
                  if (*psm++)
                    {
                      if (!logflg) k = (*ps-lrange)*z;
                      else k = (log10((double)(*ps))-lrange)*z;
                      if (k >= 0 && k < nbins) hist[k] += 1.0;
                    }
                  ps++;
                }
            }
         break;
       case VFF_TYP_4_BYTE:
          pi = (int *)(image->imagedata);
          if (!mflag)
            {
              for (i = 0; i < nc*nr; i++)
                {
                  if (!logflg) k = (*pi-lrange)*z;
                  else k = (log10((double)(*pi))-lrange)*z;
                  if (k >= 0 && k < nbins) hist[k] += 1.0;
                  pi++;
                }
            }
          else
            {
              pim = (int *)(mimage->imagedata);
              for (i = 0; i < nc*nr; i++)
                {
                  if (*pim++)
                    {
                      if (!logflg) k = (*pi-lrange)*z;
                      else k = (log10((double)(*pi))-lrange)*z;
                      if (k >= 0 && k < nbins) hist[k] += 1.0;
                    }
                  pi++;
                }
            }
         break;
       case VFF_TYP_FLOAT:
          pf = (float *)(image->imagedata);
          if (!mflag)
            {
              for (i = 0; i < nc*nr; i++)
                {
                  if (!logflg) k = (*pf-lrange)*z;
                  else k = (log10((double)(*pf))-lrange)*z;
                  if (k >= 0 && k < nbins) hist[k] += 1.0;
                  pf++;
                }
            }
          else
            {
              pfm = (float *)(mimage->imagedata);
              for (i = 0; i < nc*nr; i++)
                {
                  if (*pfm++)
                    {
                      if (!logflg) k = (*pf-lrange)*z;
                      else k = (log10((double)(*pf))-lrange)*z;
                      if (k >= 0 && k < nbins) hist[k] += 1.0;
                    }
                  pf++;
                }
            }
         break;
       default:
         fprintf(stderr,"lvhisto: Unsupported data type\n");
         return(0);
         break;
     }

    if (perflg)
      {
        total = 0.0;
        for (i=0; i<nbins; i++) total += hist[i];
        for (i=0; i<nbins; i++) hist[i] /= total;
      }
       
    return(1);
}
/* -library_code_end */
