/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lFdat2viff.c
 >>>>
 >>>>      Program Name: Fdat2viff
 >>>>
 >>>> Date Last Updated: Mon Apr 15 22:31:32 1991 
 >>>>
 >>>>          Routines: lFdat2viff - the library call for Fdat2viff
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
#define  MAXCOMP       524287
#define  MAXSIZE      1048576
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lFdat2viff - library call for Fdat2viff
*
* Purpose:
*    
*    Conversion  from  Fremont   (4.7   and   7   tesla   magnet)
*    spectroscopy data (.DAT) file to VIFF format.
*    
*    
* Input:
*    
*    file           (string) Filename of the file  in  which  the  raw
*                   data is located.
*    
*    nrows          (int) number of rows
*    
*    ncols          (int) number of columns
*    
*    header_offset  (int) Number of bytes of header information in raw
*                   data file before actual data starts.
*    
*    type           (unsigned long) Data storage type of raw data.
*    
*    single_voxel   (int) Flag set to 1 if  only  a  single  pixel  or
*                   voxel is to be extracted from the file.
*    
*    row_offset     (int) If single_voxel is  set,  this  is  the  row
*                   position of the pixel to be extracted.
*    
*    col_offset     (int) If single_voxel is set, this is  the  column
*                   position of the pixel to be extracted.
*    
*    numbands       (int) The number of  data  bands  associated  with
*                   each  pixel.  In magnetic data, this refers to the
*                   spectral or time domain data for each voxel.
*    
*    
* Output:
*    
*    img1           (struct xvimage) Pointer to output (complex)  VIFF
*                   file created by this routine.
*    
*    
*
* Written By: Donna Koechner
*    
*    Thu Jun  7 17:00:15 MDT 1990 Donna  Koechner  -  updated  to
*    ghostwriter
*    Changed name from nics12viff to Fdat2viff
*    Changed program name from Fs12viff to  Fdat2viff  and  Audit
*    Done  06-OCT-90  Donna Koechner
*    
*    
****************************************************************/


/* -library_def */
int
lFdat2viff(file, img1, nrows, ncols, header_offset, type,
          single_voxel, row_offset, col_offset, numbands)

struct xvimage  **img1;         /* input & complex output image */
unsigned long   type;           /* input data storage type */
int             nrows,          /* number of rows */
                ncols,          /* number of columns */
                header_offset,  /* header offset in bytes */
                single_voxel,   /* flag set if only 1 voxel extracted */
                row_offset,     /* row offset if single_voxel true */
                col_offset,     /* column offset if single_voxel true */
                numbands;       /* number of data bands */
char            *file;
/* -library_def_end */

/* -library_code */

{
  
  char          *data,                  /* raw data array */
                *read_raw_mag();        

  int           point_offset,           /* pixel offset if single_voxel set */
                conv_to_float,          /* flag set to 1 to convert to float */
                num_pts,
                img_size,               /* calculated image size */
                tmp,
                num_pes,                /* number of phase encod steps */
                index,
                offset,
                i,j,
                *idata;

  float         *fdata,
                *newdata;

  struct xvimage        *img, *createimage();

 /*
  * Check parameter values to make sure they are resonable.
  */

  if (header_offset < 0 )
  {
    (void) fprintf(stderr,"Header offset must be >= 0\n");
    return(0);
  }

  if (numbands < 1 )
  {
    (void) fprintf(stderr,"Number of bands must be > 0\n");
    return(0);
  }

 /*
  * Calculate total image size (complex).
  */
  num_pes = nrows * ncols;
  img_size = num_pes * numbands * 2;

 /*
  * If only a single voxel is specified, calculate the position of that
  * voxel.
  */

  if (single_voxel)
  {
    /* multiply by 2 for complex, by 4 for float */
    point_offset = ( (row_offset * ncols) + col_offset ) * 2 * 4 * numbands;
  }

 /*
  * Read in raw data file.
  */
  conv_to_float = 0;

  if (!(data = read_raw_mag(file, header_offset, &num_pts, type, conv_to_float)))
  {
    return(0);
  }

  if ( num_pts < img_size )
  {
    (void)fprintf(stderr,"\nOut of bounds! Number of requested rows and \
columns is greater than \nthe number of points found in the data file.\n\
Number of points is %d. (Expected number of points is %d\n\n",
num_pts, img_size);
    return(0);
  }

 /*
  * Allocate memory for integer data array.
  */
  if((idata=(int *)(malloc((unsigned int)img_size*sizeof(int))))
      ==NULL){
    (void)fprintf(stderr,"lFdat2viff: integer array allocation failed\n");
    return(0);
  }

 /*
  * cast data to an integer array.
  */
  idata = (int *) data;

 /*
  * Convert data from 2's complement.
  */
  for (i = 0; i < img_size ; i++) {
    if ( idata[i] > MAXCOMP) {
      idata[i] = - (MAXSIZE - idata[i]);
    }
  }

 /*
  * Allocate memory for float data array.
  */
  if((fdata=(float *)(malloc((unsigned int)img_size*sizeof(float))))
      ==NULL){
    (void)fprintf(stderr,"lFdat2viff: float array allocation failed\n");
    return(0);
  }

 /*
  * cast data to an float array.
  */
  for (i=0; i<img_size; i++)
  {
    fdata[i] = (float) idata[i];
  }

 /*
  * Rearrange real/imaginary from all reals, all imags to RI pairs.
  */

 /*
  * Allocate memory for rearranged float data array.
  */
  if((newdata=(float *)(malloc((unsigned int)img_size*sizeof(float))))
      ==NULL){
    (void)fprintf(stderr,"lFdat2viff: float array allocation failed\n");
    return(0);
  }

  index = numbands *2;

  for(i=0; i<num_pes; i++)
  {
    offset = i*index;
    for (j=0; j<numbands; j++)
    {
      tmp = j*2;
      newdata[offset+tmp]   = fdata[offset+j];
      newdata[offset+tmp+1] = fdata[offset+j+numbands];
    }
  }

 /*
  * cast back to char to store 
  */
  data = (char *)newdata;

 /*
  * If only one voxel extracted, format as 1 row by numbands cols by 
  * 1 band, and return.
  */

  if (single_voxel)
  {
    data = &data[point_offset];
    nrows = 1;
    ncols = numbands;
    numbands = 1;
  }


 /*
  * Create the complex VIFF image.
  */
  img = createimage( (unsigned long) nrows,             /* number of rows */
                     (unsigned long) ncols,             /* number of columns */
                     (unsigned long) VFF_TYP_COMPLEX,   /* data_storage_type */
                     (unsigned long) 1,                 /* number_of_images */
                     (unsigned long) numbands,          /* num_data_bands */
                     "",                                /* comment */
                     (unsigned long) 0,                 /* map_row_size */
                     (unsigned long) 0,                 /* map_col_size */
                     (unsigned long) VFF_MS_NONE,       /* map_scheme */
                     (unsigned long) VFF_MAPTYP_NONE,   /* map_storage_type */
                     (unsigned long) VFF_LOC_IMPLICIT,  /* location_type */
                     (unsigned long) 0);                /* location_dimension */

  if (img == NULL)
  {
    (void) fprintf(stderr,"lFdat2viff: Unable to allocate new image!\n");
    return(0);
  }
  *img1 = img;

 /*
  * Massage data points into specified form.
  */
  load_viff_data( data,                         /* data */
                 (float *)img->imagedata,       /* array to contain bands */
                 img->location,                 /* location */
                 nrows,                         /* number of rows */
                 ncols,                         /* number of columns */
                 numbands,                      /* number of bands */
                 0,                             /* location_dimension */
                 1,                             /* interleaved = 1 */
                 0,                             /* start */
                 1,                             /* row_skip */
                 1,                             /* col_skip */
                 1 );                           /* complex data = 1 */

  free ((char *) data);
  free ((float *) fdata);
  free ((float *) newdata);
  free ((int *) idata);
  return(1);
}

/* -library_code_end */
