/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvspread.c
 >>>>
 >>>>      Program Name: vspread
 >>>>
 >>>> Date Last Updated: Mon Apr 15 22:20:19 1991 
 >>>>
 >>>>          Routines: lvspread - the library call for vspread
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvspread - library call for vspread
*
* Purpose:
*    
*    Spread an image by pixel dispersion
*    
*    
* Input:
*    
*    image          pointer to image structure to be expanded
*    
*    scale          scale  factor  by  which  to  expand   the   image
*                   (integer), must be postitive and non-zero
*    
*    
* Output:
*    
*    image          points to the image structure that has  been  pro-
*                   cessed.  The  output  data type is the same as the
*                   input data type.
*    
*    Return Value:  1 on success, 0 on failure
*    
*    
*
* Written By: Scott Wilson
****************************************************************/


/* -library_def */
int
lvspread(image,scale)
struct xvimage *image;
int  scale;
/* -library_def_end */

/* -library_code */
{
    int j,k,x,y;
    char *malloc();
    unsigned char *c1,*c2;
    short *s1,*s2;
    int *i1,*i2;
    float *f1,*f2;
    int size;
    int rows,cols,orows,ocols;
    int b,nb;

    if (scale < 1)
      {
        fprintf(stderr,"lvspread: Spread factor must be >= 1\n");
        return(0);
      }

    rows = image->col_size;
    cols = image->row_size;
    orows = rows*scale;
    ocols = cols*scale;
    nb = image->num_data_bands;
    image->pixsizx /= scale;
    image->pixsizy /= scale;

    switch (image->data_storage_type)
      {
        case VFF_TYP_BIT :
          fprintf(stderr,"lvspread: Don't know how to spread a BIT image!\n");
          return(0);
          break;
        case VFF_TYP_1_BYTE : 
          size = sizeof(unsigned char)*orows*ocols*nb;
          c1 = (unsigned char *)(image->imagedata);
          c2 = (unsigned char *)malloc(size);
          if (c2 == NULL)
            {
              fprintf(stderr,"lvspread: unable to allocate new image\n");
              return(0);
            }
          for (j=0; j<rows; j++)
            for (k=0; k<cols; k++)
              for (b=0; b<nb; b++)
                for (y=0; y<scale; y++)
                  for (x=0; x<scale; x++)
                    c2[BPIXEL(b,(k*scale+x),(j*scale+y),orows,ocols)] =
                      (float)c1[BPIXEL(b,k,j,rows,cols)]/(float)scale;
          image->row_size = ocols;
          image->col_size = orows;
          image->imagedata = (char *)c2;
          break;
        case VFF_TYP_2_BYTE :
          size = sizeof(short)*orows*ocols*nb;
          s1 = (short *)(image->imagedata);
          s2 = (short *)malloc(size);
          if (s2 == NULL)
            {
              fprintf(stderr,"lvspread: unable to allocate new image\n");
              return(0);
            }
          for (j=0; j<rows; j++)
            for (k=0; k<cols; k++)
              for (b=0; b<nb; b++)
                for (y=0; y<scale; y++)
                  for (x=0; x<scale; x++)
                    s2[BPIXEL(b,(k*scale+x),(j*scale+y),orows,ocols)] =
                      (float)s1[BPIXEL(b,k,j,rows,cols)]/(float)scale;
          image->row_size = ocols;
          image->col_size = orows;
          image->imagedata = (char *)s2;
          break;
        case VFF_TYP_4_BYTE :
          size = sizeof(int)*orows*ocols*nb;
          i1 = (int *)(image->imagedata);
          i2 = (int *)malloc(size);
          if (i2 == NULL)
            {
              fprintf(stderr,"lvspread: unable to allocate new image\n");
              return(0);
            }
          for (j=0; j<rows; j++)
            for (k=0; k<cols; k++)
              for (b=0; b<nb; b++)
                for (y=0; y<scale; y++)
                  for (x=0; x<scale; x++)
                    i2[BPIXEL(b,(k*scale+x),(j*scale+y),orows,ocols)] =
                      (float)i1[BPIXEL(b,k,j,rows,cols)]/(float)scale;
          image->row_size = ocols;
          image->col_size = orows;
          image->imagedata = (char *)i2;
          break;
        case VFF_TYP_FLOAT :
          size = sizeof(float)*orows*ocols*nb;
          f1 = (float *)(image->imagedata);
          f2 = (float *)malloc(size);
          if (f2 == NULL)
            {
              fprintf(stderr,"lvspread: unable to allocate new image\n");
              return(0);
            }
          for (j=0; j<rows; j++)
            for (k=0; k<cols; k++)
              for (b=0; b<nb; b++)
                for (y=0; y<scale; y++)
                  for (x=0; x<scale; x++)
                    f2[BPIXEL(b,(k*scale+x),(j*scale+y),orows,ocols)] =
                      (float)f1[BPIXEL(b,k,j,rows,cols)]/(float)scale;
          image->row_size = ocols;
          image->col_size = orows;
          image->imagedata = (char *)f2;
          break;
        case VFF_TYP_COMPLEX :
          fprintf(stderr,"lvspread: COMPLEX not supported (yet)\n");
          return(0);
          break;
        default:
          fprintf(stderr,"lvspread: Unknown data storage type, %d\n",image->data_storage_type);
          return(0);
          break;
      }
      return(1);
}
/* -library_code_end */
