/*
 *----------------------------------------------------------------------
 *
 * Copyright 1991, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1991 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: lvibf.c
 >>>>
 >>>>      Program Name: vibf
 >>>>
 >>>> Date Last Updated: Mon Apr 15 22:19:52 1991 
 >>>>
 >>>>          Routines: lvibf - the library call for vibf
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
/* -library_includes_end */


/****************************************************************
*
* Routine Name: lvibf - library call for vibf
*
* Purpose:
*    
*    Compute Ion Beam Figuring (IBF) Pointspread Function
*    
*    
* Input:
*    
*    surface        Pointer to surface image.
*    
*    sputter        Pointer to sputter yield image.
*    
*    distance       Distance in meters  form  the  front  of  the  ion
*                   source to the optic reference plane.
*    
*    width          Width in meters of the area that  the  beam  patch
*                   should be calculated over.
*    
*    
* Output:
*    
*    psf            Pointer to image pointer to  which  the  resultant
*                   PSF image is attached.
*    
*    
*
* Written By: Scott Wilson
*    
*    2-Feb-91 Scott Wilson - Added  geometrical  correction  for  beam
*    current density.
*    
*    4-Feb-91 Scott Wilson - Fixed insidious 100,0 bug TWICE in beam()
*    !!!!!!
*    
*    5-Feb-91 Scott Wilson - Added beam patch width parameter
*    
*    
****************************************************************/


/* -library_def */
int
lvibf(surface,sputter,distance,width,psf)
struct xvimage *surface,*sputter,**psf;
float distance,width;
/* -library_def_end */

/* -library_code */
  {
    int rows,cols,i,pixcnt;
    int x,y,xhat,yhat;
    struct xvimage *normal,*angle,*spcomp;

    cols = surface->row_size;
    rows = surface->col_size;

    pixcnt = width/(surface->pixsizx);
    *psf = createimage(rows,cols,VFF_TYP_FLOAT,1,pixcnt*pixcnt,
                       "IBF SVPSF Image",
                       0,0,VFF_MS_NONE,VFF_MAPTYP_NONE,
                       VFF_LOC_IMPLICIT,0);
    (*psf)->subrow_size = pixcnt;
    (*psf)->pixsizx = surface->pixsizx;
    (*psf)->pixsizy = surface->pixsizy;

    /* Compute angle of facets */
    if (!lvsurf(surface,&normal,&angle))
      {
        fprintf(stderr,"lvsurf failed in IBF computation\n");
        exit(1);
      }
    freeimage(normal);

    /* Compute compensated sputter yield at each point */
    if (!lvspcomp(sputter,angle,&spcomp))
      {
        fprintf(stderr,"lvspcomp failed in IBF computation\n");
        exit(1);
      }

    /* Correct for the reduction in current density as a function of angle */
    for (i=0; i<rows*cols; i++)
      ((float *)(spcomp->imagedata))[i] *=
         cos(((float *)(angle->imagedata))[i]*3.14/180.0);
    freeimage(angle);

    /* Compute removal profile psf's */
    for (y=0; y<rows; y++)
      for (x=0; x<cols; x++)
        compute_beam(x,y,rows,cols,surface,spcomp,*psf,distance,pixcnt);

    return(1);
  }

static int 
compute_beam(x,y,rows,cols,surface,spcomp,psf,distance,pixcnt)
int x,y,rows,cols,pixcnt;
struct xvimage *surface,*spcomp,*psf;
float distance;
  {
    float dx,dy,xd,yd;
    int xp,yp,xhat,yhat,x2,y2;
    float *s,*p,*c,d,f;
    float beam();

    s = (float *)(surface->imagedata);
    p = (float *)(psf->imagedata);
    c = (float *)(spcomp->imagedata);

    dx = surface->pixsizx;
    dy = surface->pixsizy;

    for (yp=0; yp<pixcnt; yp++)
      for (xp=0; xp<pixcnt; xp++)
        {
          xhat = xp-(pixcnt/2);
          yhat = yp-(pixcnt/2);
          x2 = x+xhat;
          y2 = y+yhat;
          if (x2 > (cols-1)) x2 -= cols;   /* Apply periodic boundaries */
          else if (x2 < 0) x2 += cols;
          if (y2 > (rows-1)) y2 -= rows;   /* Apply periodic boundaries */
          else if (y2 < 0) y2 += rows;
          d = distance - s[PIXEL(x2,y2,rows,cols)];
          if (d <= 0.0)
            {
              fprintf(stderr,"lvibf: Source to surface distance is zero or negative!\n");
              exit(1);
            }
          xd = xhat*dx;
          yd = yhat*dy;
          f = beam(d,(float)sqrt((double)(xd*xd+yd*yd)));
          p[BPIXEL(yp*pixcnt+xp,x,y,rows,cols)] = f*c[PIXEL(x2,y2,rows,cols)];
        }
  }

static float 
beam(distance,offset)
float distance,offset;
  {
    /* Compute ion beam removal profile at "distance" meters from the
       front of the source, at "offset" meters from the beam axis.
       This model is only for the 2.54cm source in the IBF machine
       in L231.  The beam is assumed to be azimuthally symmetric. */

    float spread,peak,r;

    /* Compute spread of beam removal profile at the given distance */
    spread = 4.8828e-5+1.367e-5*(distance*100.0);    /* Convert to cm */

    /* Compute peak depth of beam removal profile at that distance */
    peak = 6.31*pow(10.0,(double)(-0.0325*(distance*100.0)));

    /* Insert into beam model equation to get removal */
    r = peak*exp((double)(-(offset*offset)/spread));

    return(r);
  }

/* -library_code_end */
