/* copyright 1987,1988,1989, Phil Andrews, Pittsburgh Supercomputing Center */
/* all rights reserved */
/* module to take care of command processing */
#include <stdio.h>
#include "defs.h"	/* type definitions */

/* global pointers that use storage from the main program */

static struct 	mf_d_struct 	*glbl1;	/* the class 1 elements */
static struct 	pic_d_struct 	*glbl2, *dflt2, *a2;	/* class 2 elements */
static struct 	control_struct	*glbl3, *dflt3,	*a3;	/* class 3 elements */
static struct 	attrib_struct	*glbl5, *dflt5, *a5;	/* class 5 elements */
/* g stands for global, d for default, and a for active, we normally use
   the a pointer, with an = glbln except in the middle of a metafile defaults
   replacement element, when an = dfltn */

/* now the device-specific function array pointers */
static int (**delim)();		/* delimiter functions */
static int (**mfdesc)();	/* metafile descriptor functions */
static int (**pdesc)();		/* page descriptor functions */
static int (**mfctrl)();	/* mf control functions */
static int (**gprim)();		/* graphical primitives */
static int (**attr)();		/* the attribute functions */
static int (**escfun)();	/* the escape functions */
static int (**extfun)();	/* the external functions */
static int (**ctrl)();		/* external controller functions */
/* the device info structure */
static struct info_struct *dev_info;

/* the external font functions */
static int (*font_check)() = NULL;
static int (*font_text)() = NULL;

/* memory allocation */
unsigned char *allocate_mem();

/* other bits and pieces */
#define intalloc (int *) malloc /* for convenience */
static double pxl_vdc;	/* conversion from VDC units to pixels */
static double sx, sy;	/* allow separate x and scales */
static double cosr;	/* value of cos theta */
static double sinr;	/* value of sin theta */
/* indexing info */
static struct phead_struct {
	int		no_pages;
	struct p_struct	*first;
	struct p_struct	*last;
	} p_header;
static struct ad_struct last_ad;	/* address of the last command read */
static struct ad_struct first_index;	/* address of the first index read */
static int random_input, index_present, index_read;
#define use_random (random_input && (index_present == 1))
#define p_s_size sizeof(struct p_struct)
#define max_pages 10000
struct index_struct { int rec_no; int byte_offset; int total_bytes; 
int pic_l; char *pic_ptr;};
static struct index_struct p_struct[max_pages];
static char p_buffer[max_pages * 5], *p_ptr;

/* the command line options */
static struct one_opt *opt;	/* the command line options */

#define max_fonts 100

static int list_cgm;
#define get_all (opt[(int) debug].set || list_cgm)
extern double sin(), cos(), atan();
extern void exit(), free();
extern char *strcat(), *strcpy();


/* basic stuff */
#define bytes_per_word sizeof(int)
#define byte_size 8	/* 8 bits to the byte */
#define float_size (sizeof(float))
static char *version_str;	/* version string pointer */
/* macros to handle direct colours */

#define dcr(ival) dcind(0, ival, glbl1->c_v_extent)
#define dcg(ival) dcind(1, ival, glbl1->c_v_extent)
#define dcb(ival) dcind(2, ival, glbl1->c_v_extent)

/* cgm specific I/O */
#define e_size 2
#define max_cgm_str 1024
static char buffer_str[max_cgm_str + 1]; 
/* format of a CGM string is n, nC, may want to step forward the outptr */
static char *cgm_str(s_ptr, out_ptr)
unsigned char *s_ptr, **out_ptr;
{
int i;
unsigned int length;
char *cptr;
static int chars_allocated = 0;
static char *my_ptr, *use_ptr;

	length = (int) *s_ptr;
	cptr = (char *) s_ptr;
	if (length == 255) {
	    length = (s_ptr[1] << 8) + (int) (s_ptr[2]);
	    cptr += 2;
	}
	/* use compile-time array, unless not big enougth */
	if (length > max_cgm_str) {
	    if ((length + 1) > chars_allocated) {
		if (chars_allocated == 0) my_ptr = (char *) 
		    allocate_mem(length+1, 0);
		else my_ptr = (char *) realloc(my_ptr, length + 1);
		if (my_ptr) chars_allocated = length + 1;
		else {	/* failure, must truncate */
		    length = max_cgm_str;
		    my_ptr = buffer_str;
		    chars_allocated = 0;
		}
	    }
	    use_ptr = my_ptr;
	}
	else use_ptr = buffer_str;
	for (i=0; i<length; ++i) use_ptr[i] = *++cptr;
	use_ptr[length] = 0;
	if (out_ptr) *out_ptr = (unsigned char *) (cptr + 1);

	return(use_ptr);
}

/* cgm specific integers and reals */
/* grab a cgm signed integer at general precision */
static int cgm_gint(dat_ptr, precision)
unsigned char *dat_ptr;
int precision;
{
#define SIGN_MASK (-1 ^ 255)
int  ret, no_chars;
	ret = (*dat_ptr & 128) ? SIGN_MASK | *dat_ptr : *dat_ptr;
	for (no_chars = precision / byte_size; no_chars > 1; --no_chars) 
	    ret = (ret << byte_size) | *++dat_ptr;
	return(ret);
}
/* grab a cgm unsigned integer at general precision */
static unsigned int cgm_guint(dat_ptr, precision)
unsigned char *dat_ptr;
int precision;
{
unsigned int  ret, no_chars;
	ret = *dat_ptr;
	for (no_chars = precision / byte_size; no_chars > 1; --no_chars) 
	    ret = (ret << byte_size) | (*++dat_ptr & 255);

	return(ret);
}

/* grab a cgm signed integer */
static int cgm_sint(dat_ptr)
unsigned char *dat_ptr;
{
	return(cgm_gint(dat_ptr, glbl1->int_prec));
}
/* grab a cgm signed integer at index precision */
static int cgm_xint(dat_ptr)
unsigned char *dat_ptr;
{
	return(cgm_gint(dat_ptr, glbl1->ind_prec));
}
/* grab a cgm colour index */
static unsigned int cgm_cxint(dat_ptr)
unsigned char *dat_ptr;
{
	return(cgm_guint(dat_ptr, glbl1->col_i_prec));
}
/* grab a cgm direct colour */
static unsigned int cgm_dcint(dat_ptr)
unsigned char *dat_ptr;
{
	return(cgm_guint(dat_ptr, glbl1->col_prec));
}
/* grab a cgm signed integer at VDC integer precision */
static int cgm_vint(dat_ptr)
unsigned char *dat_ptr;
{
	return((int) (pxl_vdc * cgm_gint(dat_ptr, glbl3->vdc_i_prec)));
}
/* grab a cgm signed integer at fixed E (16 bit) precision */
static int cgm_eint(dat_ptr)
unsigned char *dat_ptr;
{
	return(cgm_gint(dat_ptr, e_size * byte_size));
}

static double cgm_fixed(dat_ptr, r_prec)
unsigned char *dat_ptr;
struct r_struct *r_prec;
{
double ret, exp_part, fract_part;
int cgm_gint();
	
	exp_part = cgm_gint(dat_ptr, r_prec->exp);
	dat_ptr += r_prec->exp / byte_size;
	fract_part = cgm_guint(dat_ptr, r_prec->fract);
	fract_part /= 1 << r_prec->fract;

	ret = exp_part + fract_part;

	return(ret);
}
/* an IEEE floating point */
static double cgm_float(dat_ptr, r_prec)
unsigned char *dat_ptr;
struct r_struct *r_prec;
{
int sign_bit, i, j;
unsigned int exponent;
unsigned long fract;
double ret, dfract;
unsigned char *new_ptr;

	sign_bit = (*dat_ptr >> 7) & 1;
	new_ptr = (unsigned char *) dat_ptr;

	switch (r_prec->exp + r_prec->fract) {
	/* first 32 bit precision */
case 32:    exponent = ((*new_ptr & 127)<< 1) +
		((*(new_ptr + 1) >> 7) & 1);
	    fract = ((*(new_ptr + 1) & 127) << 16) + (*(new_ptr + 2) << 8)
		+ *(new_ptr + 3);
	    if (exponent == 255) {
		if (fract == 0) {
		    fract = ~((~0) << 23);
		    exponent = 254;
		} else {
		    fprintf(stderr, "undefined IEEE number\n");
		    return(0.0);
		}
	    } 
	    if (exponent == 0) {
		if (fract == 0)ret = 0.0;
		else {
		    ret = fract;
		    for (i=0; i<149; ++i) ret /= 2.0;
		}
	    }
	    else {
		dfract = (double) fract;
		for (i=0; i<23; ++i) dfract /= 2.0;
		ret = 1.0 + dfract;
		if (exponent < 127) 
		    for (i=0; i<(127 - exponent); ++i) ret /= 2.0;
		else if (exponent > 127)
		    for (i=0; i<(exponent - 127); ++i) ret *= 2.0;
	    }
	    break;
	/* now 64 bit precision, may not fit in integer, do with FP */
case 64:    exponent = ((*new_ptr & 127) << 4) +
		((*(new_ptr + 1) >> 4) & 15);

	    dfract = (double) (*(new_ptr + 1) & 15);
	    for (j=2; j<8; ++j) {
		for (i=0; i<byte_size; ++i) dfract *= 2;
		dfract += (double) *(new_ptr + j);
	    }

	    if (exponent == 2047) {
		if (dfract == 0.0) {	/* want big number */
		    dfract = 100000000.0;
		    exponent = 2046;
		} else {
		    fprintf(stderr, "undefined IEEE number\n");
		    return(0.0);
		}
	    } 
	    if (exponent == 0) {
		if (dfract == 0.0)ret = 0.0;
		else {
		    ret = dfract;
		    for (i=0; i<1074; ++i) ret /= 2.0;
		}
	    }
	    else {
		for (i=0; i<52; ++i) dfract /= 2.0;
		ret = 1.0 + dfract;
		if (exponent < 1023)
		    for (i=0; i<(1023 - exponent); ++i) ret /= 2.0;
		else if (exponent > 1023)
		    for (i=0; i<(exponent - 1023); ++i) ret *= 2.0;
	    }
	    break;
default:    fprintf(stderr, "illegal real precisions !\n"); ret = 0.0;
	}
	return((sign_bit) ? -ret : ret);
}

/* handle the general real variable */
static double cgm_real(dat_ptr)
unsigned char *dat_ptr;
{
double ret;

	switch (glbl1->real_prec.fixed) {
case 0:	    ret = cgm_float(dat_ptr, &glbl1->real_prec); break;	/* floating */
case 1:	    ret = cgm_fixed(dat_ptr, &glbl1->real_prec); break;	/* fixed */
default:    fprintf(stderr, "illegal real flag\n"); ret = 0.0;
	}

	return(ret);
}
/* handle the vdc real variable */
static double cgm_vreal(dat_ptr)
unsigned char *dat_ptr;
{
double ret;

	switch (glbl3->vdc_r_prec.fixed) {
case 0:	    ret = cgm_float(dat_ptr, &glbl3->vdc_r_prec); break;/* floating */
case 1:	    ret = cgm_fixed(dat_ptr, &glbl3->vdc_r_prec); break;/* fixed */
default:    fprintf(stderr, "illegal vdc real flag\n"); ret = 0.0;
	}

	return(ret);
}
/* get a real vdc and change it to integer */
static int cgm_vireal(dat_ptr)
unsigned char *dat_ptr;
{
double get_real;
	get_real = cgm_vreal(dat_ptr);
	/* now convert it */
	return((int) (get_real * pxl_vdc));
}

/* our general vdc getter */
static int (*cgm_vdc)() = cgm_vint;	/* default to integer VDC's */
/* need a macro to get the step size right */
#define vdc_step ((glbl1->vdc_type == vdc_int) ? a3->vdc_i_prec :\
	a3->vdc_r_prec.exp + a3->vdc_r_prec.fract)
/* error macro, mainly to keep lint happy */
#define burp (void) fprintf
/* and a listing macro */
#define may_list if (list_cgm) (void) fprintf
/* now my globals */
static int pages_done = 0;
static char *g_in_name;	/* the full input file_name */
static char pic_name[max_str];		/* picture name */

/* stuff to take care of individual pages */
static int in_page_no = 0;		/* page no in the metafile */
static int skipping = 0;		/* are we skipping this page ? */
/* set up a couple of work arrays for polygons, etc. */
#define wk_ar_size 2024
static int x_wk_ar[wk_ar_size], y_wk_ar[wk_ar_size];

static float xp0;	/* internal xoffset in pixels */
static float yp0;	/* internal yoffset in pixels */

static int xoffset, yoffset;	/* external offsets in pixels */
static int xsize, ysize;	/* requested page size in pixels */

/* handle rotations, clipping, scaling here, use macros */
extern int uclip(); /* in utils.c */
#define UCLIP(xin, yin) \
  ((int) a3->clip_ind && !(dev_info->capability & can_clip)) ? \
   uclip(&xin, &yin, a3->clip_rect.i) : 0
#define NEWX(xin, yin) ((int) (sx * xp0 + (sx * cosr * xin - sy * sinr * yin)))
#define newx(xin, yin) (UCLIP(xin, yin), NEWX(xin, yin))

#define NEWY(xin, yin) ((int) (sy * yp0 + (sy * cosr * yin + sx * sinr * xin)))
#define newy(xin, yin) (UCLIP(xin, yin), NEWY(xin, yin))



/* the setup procedure */
cgm_setup(do_list, full_oname, this_version, new_info, new_opt,
gl1, gl2, df2, gl3, df3, gl5, df5,
pf0, pf1, pf2, pf3, pf4, pf5, pf6, pf7, pfct, random_file)
char *full_oname, *this_version;
int do_list;
struct info_struct *new_info;
struct one_opt *new_opt;
struct 	mf_d_struct 	*gl1;	/* the class 1 elements */
struct 	pic_d_struct 	*gl2, *df2;	/* the class 2 elements */
struct 	control_struct	*gl3, *df3;	/* the class 3 elements */
struct 	attrib_struct	*gl5, *df5;	/* the class 5 elements */
int (**pf0)(), (**pf1)(), (**pf2)(), (**pf3)(), (**pf4)(), (**pf5)(), 
(**pf6)(), (**pf7)(), (**pfct)();	/* the function pointer arrays */
int random_file;	/* can use random access */
{
	/* store globals */
	list_cgm = do_list;
	g_in_name = full_oname;
	version_str = this_version;
	dev_info = new_info;
	opt = new_opt;
	glbl1 = gl1;
	glbl2 = gl2;
	dflt2 = df2;
	glbl3 = gl3;
	dflt3 = df3;
	glbl5 = gl5;
	dflt5 = df5;

	delim 	= pf0;
	mfdesc 	= pf1;
	pdesc	= pf2;
	mfctrl	= pf3;
	gprim	= pf4;
	attr	= pf5;
	escfun	= pf6;
	extfun	= pf7;
	ctrl 	= pfct;
	random_input = random_file;
	/* make the active pointers refer to the globals */
	a2 = glbl2;	a3 = glbl3;	a5 = glbl5;
	return(1);
}

/* parcel out the commands here */
cgm_command(s_ptr, p_len, this_ad, out_class, out_element)
unsigned char *s_ptr;
int p_len;
struct ad_struct *this_ad;
int *out_class, *out_element;
{
unsigned int  byte1, byte2, class, element, p_flag;
/* are we replacing defaults ? */
#define no_defs(cl) if (a2 == dflt2) {(void) \
fprintf(stderr, "illegal class cl while replacing defaults !\n"); return(2);}

	/* mark our spot */
	last_ad.r_ad = this_ad->r_ad;
	last_ad.b_ad = this_ad->b_ad;
	last_ad.offset = this_ad->offset;


	byte1 = *s_ptr;
	byte2 = *(s_ptr + 1);
	*out_class = class = byte1 >> 4;
	*out_element = element = ((byte1 << 3) & 127) + (byte2 >> 5);
	/* are we starting a new page ? */
	if ( (class == 0) && ((enum cgmcls0) element == B_Pic) ) {
	    ++in_page_no;	 /* new page */
	    if (use_random && index_read) { /* all set up for random access */
		if (p_header.first) {	/* a real live page */
		    if ((in_page_no == 1) && 
			(opt[(int) pages].set||opt[(int) title_string].set)) {
			cgm_goto(&(p_header.first->ad));
			return(1);
		    }
		    if (get_all) burp(stderr, "page %d ", p_header.first->no);
		}
	    } else if (get_all) burp(stderr, "page %d ", in_page_no);


	    if (get_all) {
#ifdef VMS
		burp(stderr, "record %d, offset %d\n", last_ad.r_ad, 
		    last_ad.offset);
#else
		burp(stderr, "byte %d\n", last_ad.b_ad);
#endif
	    }
	    skipping = (!want_page(in_page_no, opt[(int) pages].val.str)) &&
		!(use_random && index_read);
	}
	/* now check for the end of the page */
	if ((class == 0) && ((enum cgmcls0) element == E_Pic)){
	    /* if skipping is on, turn it off at the end of the page */
	    if (skipping) {
		skipping = 0;
		return(1);
	    }
	}

	/* now if we don't want this page, don't do anything until ended */
	if (skipping) return(1);
	/* note trouble if we were to get a B_Pic inside a metafile def rep */ 

	p_flag 	= byte2  & 31;

	if (byte1 || byte2) may_list(stderr, "(%d, %d)(%d,%d,%d,%d)", 
	    byte1, byte2, class, element, p_flag, p_len);
	s_ptr += 2;

	switch (class) {
	case 0: no_defs(0); 	return(class0(element, p_len, s_ptr));
	case 1: no_defs(1); 	return(class1(element, p_len, s_ptr));
	case 2: 		return(class2(element, s_ptr));
	case 3: 		return(class3(element, s_ptr));
	case 4: no_defs(4); 	return(class4(element, p_len, s_ptr));
	case 5: 		return(class5(element, p_len, s_ptr));
	case 6: no_defs(6); 	return(class6(element, s_ptr));
	case 7: no_defs(7);	return(class7(element, s_ptr));
	default: burp(stderr, "illegal class, cmd = (%d,%d,%d)\n",
	    class, element, p_len);
	}

	return(get_all);	/* may or may not want to terminate on error */
}
/* class 0, the delimiter elements */
static class0(el, p_len, dat_ptr)
unsigned char *dat_ptr;
enum cgmcls0 el;
int p_len;
{
	switch (el) {
case No_Op: 	return(1);
case B_Mf:	return(f_b_mf	(dat_ptr, p_len, delim[(int) el], 
					ctrl[(int) el]));
case E_Mf:	return(f_e_mf	(delim[(int) el], ctrl[(int) el]));
case B_Pic:	return(f_b_p	(dat_ptr, p_len, delim[(int) el], 
					ctrl[(int) el]));
case B_Pic_Body: return(f_b_p_body	(delim[(int) el], ctrl[(int) el]));
case E_Pic:	return(f_e_pic	(delim[(int) el], ctrl[(int) el]));
default:	burp(stderr, "illegal class 0 command, %d\n", (int) el);


    }
	return(get_all);	/* may or may not want to terminate on error */
}
/* class 1, the metafile descriptor elements */
static class1(el, p_len, dat_ptr)
unsigned char *dat_ptr;
int p_len;
enum cgmcls1 el;
{

	switch (el) {
case  MfVersion:	return(rd_mf_version	(dat_ptr, p_len, 
							mfdesc[(int)el]));
case  MfDescrip:	return(rd_mf_descriptor	(dat_ptr, p_len, 
							mfdesc[(int) el]));
case  vdcType:		return(s_vdc_type	(dat_ptr, mfdesc[(int) el]));
case  IntPrec:		return(s_int_prec	(dat_ptr, mfdesc[(int) el]));
case  RealPrec:		return(s_real_prec	(dat_ptr, mfdesc[(int) el]));
case  IndexPrec:	return(s_index_prec	(dat_ptr, mfdesc[(int) el]));
case  ColPrec:		return(s_col_prec	(dat_ptr, mfdesc[(int) el]));
case  CIndPrec:		return(s_cind_prec	(dat_ptr, mfdesc[(int) el]));
case  MaxCInd:		return(s_mcind		(dat_ptr, mfdesc[(int) el]));
case  CVExtent:		return(s_cvextent	(dat_ptr, mfdesc[(int) el]));
case  MfElList:		return(rd_mf_list	(dat_ptr, mfdesc[(int) el]));
case  MfDefRep:		return(s_mf_defs	(dat_ptr, p_len, 
							mfdesc[(int) el]));
case  FontList:		return(do_font_list	(dat_ptr, p_len, 
							mfdesc[(int) el]));
case  CharList:		return(do_char_list	(dat_ptr, p_len, 
							mfdesc[(int) el]));
case  CharAnnounce:	return(do_cannounce	(dat_ptr, mfdesc[(int) el]));
default:		burp(stderr, "illegal class 1 command, %d\n", 
							(int) el);
	
	}
	return(get_all);	/* may or may not want to terminate on error */
}

/* class 2, the picture descriptor elements */
static class2(el, dat_ptr)
unsigned char *dat_ptr;
enum cgmcls2 el;
{
/* the device may not want calls during the metafile defaults replacement */

#define maybe2 (((a2 == dflt2) && (dev_info->capability & no_def_calls)) \
	? NULL : pdesc[(int) el])

	switch (el) {
case  ScalMode:			return(s_scalmode	(dat_ptr, maybe2));
case  ColSelMode:		return(s_c_s_mode	(dat_ptr, maybe2));
case  LWidSpecMode:		return(s_lws_mode	(dat_ptr, maybe2));
case  MarkSizSpecMode:		return(s_ms_mode	(dat_ptr, maybe2));
case  EdWidSpecMode:		return(s_ew_mode	(dat_ptr, maybe2));
case  vdcExtent:		return(s_vdc_extent	(dat_ptr, maybe2));
case  BackCol:			return(s_back_col	(dat_ptr, maybe2));
default:			burp(stderr, "illegal class 2 command, %d\n", el);
	}
    return(get_all);
#undef maybe2
}
/* class 3, the control elements */
static class3(el, dat_ptr)
unsigned char *dat_ptr;
enum cgmcls3 el;
{
/* the device may not want calls during the metafile defaults replacement */

#define maybe3 (((a3 == dflt3) && (dev_info->capability & no_def_calls)) \
	? NULL : mfctrl[(int) el])

	switch (el) {
case  vdcIntPrec:	return(s_vdc_i_p	(dat_ptr, maybe3));
case  vdcRPrec:		return(s_vdc_r_p	(dat_ptr, maybe3));
case  AuxCol:		return(s_aux_col	(dat_ptr, maybe3));
case  Transp:		return(s_transp		(dat_ptr, maybe3));
case  ClipRect:		return(s_clip_rec	(dat_ptr, maybe3));
case  ClipIndic:	return(s_clip_ind	(dat_ptr, maybe3));
default:		burp(stderr, "illegal class 3 command, %d\n", el);
	}
    return(get_all);
#undef maybe3
}
/* class 4, the graphical primitive elements */
static class4(el, p_len, dat_ptr)
unsigned char *dat_ptr;
int p_len;
enum cgmcls4 el;
{
/* now go do it */

	switch (el) {
case  PolyLine: 	return(do_polyline	(dat_ptr, p_len, 
							gprim[(int) el]));
case  Dis_Poly:		return(do_dis_polyline	(dat_ptr, p_len, 
							gprim[(int) el]));
case  PolyMarker:	return(do_polymarker	(dat_ptr, p_len, 
							gprim[(int) el]));
case  Text:		return(s_text		(dat_ptr, gprim[(int) el]));
case  Rex_Text:		return(s_rex_text	(dat_ptr, gprim[(int) el]));
case  App_Text:		return(s_app_text	(dat_ptr, gprim[(int) el]));
case  Polygon:		return(do_polygon	(dat_ptr, p_len, 
							gprim[(int) el]));
case  Poly_Set:		return(do_polyset	(dat_ptr, p_len, 
							gprim[(int) el]));
case  Cell_Array:	return(do_cell_array	(dat_ptr, p_len, 
							gprim[(int) el]));
case  Gen_D_Prim:	return(do_g_d_p		(dat_ptr, p_len, 
							gprim[(int) el]));
case  Rectangle:	return(do_rectangle	(dat_ptr, gprim[(int) el]));
case  Cgm_Circle:	return(do_circle	(dat_ptr, gprim[(int) el]));
case  Circ_3:		return(do_c3		(dat_ptr, gprim[(int) el]));
case  Circ_3_Close:	return(do_c3_close	(dat_ptr, gprim[(int) el]));
case  Circ_Centre:	return(do_c_centre	(dat_ptr, gprim[(int) el]));
case  Circ_C_Close:	return(do_c_c_close	(dat_ptr, gprim[(int) el]));
case  Ellipse:		return(do_ellipse	(dat_ptr, gprim[(int) el]));
case  Ellip_Arc:	return(do_ell_arc	(dat_ptr, gprim[(int) el]));
case  El_Arc_Close:	return(do_e_a_close	(dat_ptr, gprim[(int) el]));
default:		burp(stderr, "illegal class 4 command, %d\n",
							(int) el); 
	}
	return(get_all);
}
/* class 5, the attribute elements */
static class5(el, p_len, dat_ptr)
unsigned char *dat_ptr;
int p_len;
enum cgmcls5 el;
{
/* the device may not want calls during the metafile defaults replacement */

#define maybe5 (((a5 == dflt5) && (dev_info->capability & no_def_calls)) \
	? NULL : attr[(int) el])

	switch (el) {
case   LBIndex:		return(s_lbindex	(dat_ptr, maybe5));
case   LType:		return(s_l_type		(dat_ptr, maybe5));
case   LWidth:		return(s_l_width	(dat_ptr, maybe5));
case   LColour:		return(s_l_colour	(dat_ptr, maybe5));
case   MBIndex:		return(s_mbindex	(dat_ptr, maybe5));
case   MType:		return(s_mk_type	(dat_ptr, maybe5));
case   MSize:		return(s_mk_size	(dat_ptr, maybe5));
case   MColour:		return(s_mk_colour	(dat_ptr, maybe5));
case   TBIndex:		return(s_tbindex	(dat_ptr, maybe5));
case   TFIndex:		return(s_t_index	(dat_ptr, maybe5));
case   TPrec:		return(s_t_prec		(dat_ptr, maybe5));
case   CExpFac:		return(s_c_exp		(dat_ptr, maybe5));
case   CSpace:		return(s_c_space	(dat_ptr, maybe5));
case   TColour:		return(s_t_colour	(dat_ptr, maybe5));
case   CHeight:		return(s_c_height	(dat_ptr, maybe5));
case   COrient:		return(s_c_orient	(dat_ptr, maybe5));
case   TPath:		return(s_tpath		(dat_ptr, maybe5));
case   TAlign:		return(s_t_align	(dat_ptr, maybe5));
case   CSetIndex:	return(s_csindex	(dat_ptr, maybe5));
case   AltCSetIndex: 	return(s_acsindex	(dat_ptr, maybe5));
case   FillBIndex:	return(s_fbindex	(dat_ptr, maybe5));
case   IntStyle:	return(s_interior_style	(dat_ptr, maybe5));
case   FillColour:	return(s_fill_colour	(dat_ptr, maybe5));
case   HatchIndex:	return(s_hindex		(dat_ptr, maybe5));
case   PatIndex:	return(s_pindex		(dat_ptr, maybe5));
case   EdBIndex:	return(s_e_b_index	(dat_ptr, maybe5));
case   EType:		return(s_edge_t		(dat_ptr, maybe5));
case   EdWidth:		return(s_edge_w		(dat_ptr, maybe5));
case   EdColour:	return(s_edge_c		(dat_ptr, maybe5));
case   EdVis:		return(s_edge_v		(dat_ptr, maybe5));
case   FillRef:		return(s_fill_ref	(dat_ptr, maybe5));
case   PatTab:		return(p_tab_entry	(dat_ptr, p_len, maybe5));
case   PatSize:		return(s_pat_size	(dat_ptr, maybe5));
case   ColTab:		return(c_tab_entry	(dat_ptr, p_len, maybe5));
case   AspsFlags:	return(do_aspsflags	(dat_ptr, p_len, maybe5));
default:		burp(stderr, "illegal class 5 command, %d\n", el);
	}
	return(get_all);
#undef maybe5
}
/* class 6, the escape element */
static class6(el, dat_ptr)
unsigned char *dat_ptr;
enum cgmcls6 el;
{

	switch (el) {
case Escape:	return(do_escape	(dat_ptr, escfun[(int) el]));
default:	burp(stderr, "illegal class 6 command, %d\n", (int) el);
	}
	return(get_all);
}
/* class 7, the external elements */
/* not really implemented yet */
static class7(el, dat_ptr)
enum cgmcls7 el;
unsigned char *dat_ptr;
{

	switch (el) {
case Message:	return(do_message(dat_ptr, extfun[(int) el]));
case Ap_Data:	return(do_apdata(dat_ptr, extfun[(int) el]));
default:	burp(stderr, "illegal class 7 command, %d\n", (int) el);
	}
	return(get_all);
}
	/* now the routines that do the work, class by class */
/* class 0 routines */
/* function to start a metafile */
static f_b_mf(dat_ptr, p_len, dev_func, ctrl_func)
int (*ctrl_func)();
int (*dev_func)();
unsigned char *dat_ptr;
int p_len;
{
int ret = 1;
char buffer[2 * max_str + 1], prog_name[max_str], *name_ptr = NULL;
extern void rs_defaults(); /* in utils.c */

	/* indexing stuff */
	index_present = 0;
	index_read = 0;
	p_header.no_pages = 0;
	p_header.first = NULL;
	p_header.last = NULL;

	/* first put in the defaults */
	cosr = 1.0;
	sinr = 0.0;
	xp0 = 0;
	yp0 = 0;
	pxl_vdc = 1.0;
	sx = sy = 1.0;
	pages_done = 0;
	/* set the defaults in case anything happens before the first pic */
	rs_defaults(glbl1, glbl2, glbl3, glbl5, dflt2, dflt3, dflt5, pxl_vdc);

	(void) strcpy(prog_name, "GPLOT ");
	buffer[0] = '\0';
	if (p_len > 1) (void) strcpy(buffer, cgm_str(dat_ptr, NULL));
	if (get_all) {
	    burp(stderr, "Begin Metafile");
		burp(stderr, ", name: [%s]\n", buffer);
	}

	in_page_no = 0;
	if (dev_func) ret = (*dev_func) 
	    (buffer, g_in_name, strcat(prog_name, version_str));

	if (ctrl_func) (*ctrl_func) ();
	return(ret);
}
/* function to end a metafile */
static f_e_mf(dev_func, ctrl_func)
int (*ctrl_func)();
int (*dev_func)();
{
int ret = 1;

	if (get_all) burp(stderr, "End Metafile\n");
	if (dev_func) ret = (*dev_func) (pages_done);
	if (ctrl_func) (*ctrl_func) ();
	if (opt[(int) index_file].set && !index_read)
	    burp(stderr, "no index blocks\n");
	return(0);	/* time to stop */
}
/* function to reset all of the defaults before starting picture body */
static f_b_p(dat_ptr, p_len, dev_func, ctrl_func)
int (*ctrl_func)();
int (*dev_func)();
char *dat_ptr;
int p_len;
{
int ret = 1;
extern void rs_defaults(), get_smart(); /* in utils.c */

	if (p_len > 1)	{
	    strncpy(pic_name, cgm_str(dat_ptr, NULL), max_str);
	    pic_name[max_str-1] = '\0';
	} else 	pic_name[0] = '\0';

	if (get_all) {
	    burp(stderr, "Begin Picture");
	    if (p_len > 1) 
		burp(stderr, ", name: [%s]\n", pic_name);
	    else {
		burp(stderr, "\n");
	    }
	}
	if (dev_func) ret = (*dev_func) (pic_name);
	if (ctrl_func) (*ctrl_func) ();

	/* go back to the defaults for get_smart, use dummy pxl_vdc */
	pxl_vdc = 1;
	rs_defaults(glbl1, glbl2, glbl3, glbl5, dflt2, dflt3, dflt5, pxl_vdc);

	/* call get_smart and get real pxl_vdc */
	get_smart(0, glbl1, a2, opt, dev_info, &cosr, &sinr, &pxl_vdc,
	    &xoffset, &yoffset, &xp0, &yp0, &xsize, &ysize, &font_check,
	    &font_text, &sx, &sy);

	/* go back to the defaults for real */
	rs_defaults(glbl1, glbl2, glbl3, glbl5, dflt2, dflt3, dflt5, pxl_vdc);

	return(ret);
}
 /* function to get the printer ready to start drawing; implement defaults */
static f_b_p_body(dev_func, ctrl_func)
int (*ctrl_func)();
int (*dev_func)();
{
int ret = 1, i;
extern void get_smart();	/* utils.c */
	get_smart(1, glbl1, a2, opt, dev_info, &cosr, &sinr, &pxl_vdc,
	    &xoffset, &yoffset, &xp0, &yp0, &xsize, &ysize, &font_check,
	    &font_text, &sx, &sy);

	may_list(stderr, "Begin Picture Body\n");

	if (dev_func) ret = (*dev_func) (pic_name, xoffset, yoffset, 
		opt[(int) degrees].val.r,
		*(a5->ctab), *(a5->ctab + 1), *(a5->ctab + 2), 
		(use_random && index_read) ? p_header.first->no : in_page_no,
		xsize, ysize);

	if ((ctrl_func) && (!opt[(int) included].val.i)) (*ctrl_func) ();

	return(ret);
}
/* function to end a picture */
static f_e_pic(dev_func, ctrl_func)
int (*ctrl_func)();
int (*dev_func)();
{
int ret = 1;
	++pages_done;
	if (get_all) burp(stderr, "End Picture\n");
	if (dev_func) ret = (*dev_func) (opt[(int) copies].val.i);
	if ((ctrl_func) && (!opt[(int) included].val.i)) 
	    (*ctrl_func) (opt[(int) copies].val.i);

	    /* now handle random stuff */
	if (use_random && index_read) { 
	    if (p_header.first) p_header.first = p_header.first->next;
	    else return(f_e_mf	(delim[(int) E_Mf], ctrl[(int) E_Mf]));
	    if (p_header.first) cgm_goto(&(p_header.first->ad));
	    else return(f_e_mf	(delim[(int) E_Mf], ctrl[(int) E_Mf]));
	}
	return(ret);
}
/* now the class1 functions */
/* read the metafile version number (if present) */
static rd_mf_version(dat_ptr, p_len, dev_func)
unsigned char *dat_ptr;
int p_len;
int (*dev_func)();
{
int ret = 1, vers_no;
	if (p_len > 0) vers_no = cgm_sint(dat_ptr);
	if (get_all) { 
	    burp(stderr, "Metafile Version");
	    if (p_len > 0) 
		burp(stderr, ", number: [%d]\n", vers_no);
	    	else burp(stderr, "\n");
	    } 
	if ((p_len > 0) && dev_func) ret = (*dev_func) (vers_no);

	return(ret);
}

/* read the metafile descriptor */
static rd_mf_descriptor(dat_ptr, p_len, dev_func)
unsigned char *dat_ptr;
int p_len;
int (*dev_func)();
{
int ret = 1;
char *char_ptr;
	if (p_len > 1) char_ptr = cgm_str(dat_ptr, NULL);
	else char_ptr = NULL;
	/* send off to logging line */
	log_line(char_ptr, 0, NULL, 0, 0, 2);
	if (get_all) {
	    burp(stderr, "Metafile Description");
	    if (p_len > 1) 
		burp(stderr, ", name: [%s]\n", char_ptr);
	    else burp(stderr, "\n");
	} 
	if ((p_len > 1) && dev_func) ret = (*dev_func) (char_ptr);
	return(ret);
}

/* set the VDC type */
static s_vdc_type(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_type, ret = 1;

	new_type = cgm_eint(dat_ptr);
	may_list(stderr, "changing vdc type from %d to %d\n", 
	    (int) glbl1->vdc_type, new_type);
	glbl1->vdc_type = (enum vdc_enum) new_type;

	switch (glbl1->vdc_type) {
case vdc_int: cgm_vdc = cgm_vint; break;
case vdc_real: cgm_vdc = cgm_vireal; break;
default:    burp(stderr, "illegal vdc_type = %d\n", glbl1->vdc_type);
	}

	if (dev_func) ret = (*dev_func) (new_type);
	return(ret);
}
/* set the integer precision */
static s_int_prec(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_prec, ret = 1;

	new_prec = cgm_sint(dat_ptr);
	may_list(stderr, "changing integer precision from %d to %d\n",
	    glbl1->ind_prec, new_prec);
	glbl1->ind_prec = new_prec;

	if (dev_func) ret = (*dev_func) (new_prec);
	return(ret);
}
/* set the real  precision */
static s_real_prec(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int step_size, new_fixed, new_exp, new_fract, ret = 1;

	step_size = glbl1->int_prec / byte_size;
	new_fixed = cgm_sint(dat_ptr);
	dat_ptr += step_size;
	new_exp = cgm_sint(dat_ptr);
	dat_ptr += step_size;
	new_fract = cgm_sint(dat_ptr);

	may_list(stderr, 
	    "changing real precision from (%d, %d, %d) to (%d, %d, %d)\n",
	    glbl1->real_prec.fixed, glbl1->real_prec.exp, 
	    glbl1->real_prec.fract, new_fixed, new_exp, new_fract);

	glbl1->real_prec.fixed	= new_fixed;
	glbl1->real_prec.exp 	= new_exp;
	glbl1->real_prec.fract	= new_fract;

	if (dev_func) ret = (*dev_func) (new_fixed, new_exp, new_fract);
	return(ret);
}
/* set the index precision */
static s_index_prec(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_prec, ret = 1;

	new_prec = cgm_sint(dat_ptr);
	may_list(stderr, "changing index precision from %d to %d\n",
	    glbl1->ind_prec, new_prec);
	glbl1->ind_prec = new_prec;
	if (dev_func) ret = (*dev_func) (new_prec);
	return(ret);
}
/* set the colour precision */
static s_col_prec(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_prec, ret = 1;

	new_prec = cgm_sint(dat_ptr);
	may_list(stderr, "changing colour precision from %d to %d\n",
	    glbl1->col_prec, new_prec);
	glbl1->col_prec = new_prec;
	if (dev_func) ret = (*dev_func) (new_prec);
	return(ret);
}
/* set the colour index precision */
static s_cind_prec(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_prec, ret = 1;

	new_prec = cgm_sint(dat_ptr);
	may_list(stderr, "changing colour index precision from %d to %d\n",
	    glbl1->col_i_prec, new_prec);
	glbl1->col_i_prec = new_prec;
	if (dev_func) ret = (*dev_func) (new_prec);
	return(ret);
}
/* set the colour value extent */
static s_cvextent(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int cvmin[3], cvmax[3], ret = 1, step_size, i;

	step_size = (glbl1->col_prec / byte_size);

	for (i=0; i<3; ++i) {
	    cvmin[i] = cgm_dcint(dat_ptr);
	    dat_ptr += step_size;
	}
	for (i=0; i<3; ++i) {
	    cvmax[i] = cgm_dcint(dat_ptr);
	    dat_ptr += step_size;
	}

	may_list(stderr, 
"changing col value extent from (%d,%d,%d,%d,%d,%d) to (%d,%d,%d,%d,%d,%d)\n",
	    glbl1->c_v_extent.min[0], glbl1->c_v_extent.min[1], 
	    glbl1->c_v_extent.min[2], 
	    glbl1->c_v_extent.max[0], glbl1->c_v_extent.max[1], 
	    glbl1->c_v_extent.max[2], 
	    cvmin[0], cvmin[1], cvmin[1], cvmax[0], cvmax[1], cvmax[2]);

	for (i=0; i<3; ++i) {
	    glbl1->c_v_extent.min[i] = cvmin[i];
	    glbl1->c_v_extent.max[i] = cvmax[i];
	}

	if (dev_func) ret = (*dev_func) (cvmin, cvmax);

	return(ret);
}
/* set the maximum colour index */
/* split into two functions as we may have to handle illegal 
   metafiles (DI3000) that add c table entries before increasing
   glbl1->max_c_index */
static s_mcind(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_index, ret = 1;
int do_mcind();
	new_index = cgm_cxint(dat_ptr);
	if (!do_mcind(new_index)) burp(stderr, "trouble setting max colind\n");
	if (dev_func) ret = (*dev_func) (new_index);
	return(ret);
}
static do_mcind(new_index)
int new_index;
{
int i;
float *new_d_ptr, *new_g_ptr;

	may_list(stderr, "changing max colour index from %d to %d\n",
	    glbl1->max_c_index, new_index);
	if (new_index > glbl1->max_c_index) { /* need to make some new memory */
	    new_d_ptr = 
		(float *)allocate_mem(float_size * 3 * (new_index + 1), 1);
	    new_g_ptr = 
		(float *)allocate_mem(float_size * 3 * (new_index + 1), 1);
	    if ((!new_d_ptr) || (!new_g_ptr)) return(0);
	    /* move over the old data */
	    for (i=0; i<(glbl1->max_c_index + 1) * 3; ++i) {
		*(new_d_ptr + i) = *(a5->ctab + i);
		*(new_g_ptr + i) = *(a5->ctab + i);
	    }
	    /* free up memory */
	    free(dflt5->ctab);
	    free(glbl5->ctab);
	    /* and reassign the pointers */
	    dflt5->ctab = new_d_ptr;
	    glbl5->ctab = new_g_ptr;
	}
	glbl1->max_c_index = new_index;
	return(1);
}

/* read the metafile element list (may be useful) */
static rd_mf_list(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int no_pairs, i, class, el, ret = 1, *array_ptr = NULL, *cur_ptr;

	no_pairs = cgm_sint(dat_ptr);
	if (no_pairs > 0) 
	    cur_ptr = array_ptr = 
		(int *) allocate_mem(2 * no_pairs * sizeof(int), 0);

	dat_ptr += glbl1->int_prec / byte_size;
	for (i=0; i<no_pairs; ++i) {
	    *cur_ptr++ = class = cgm_xint(dat_ptr);
	    dat_ptr += glbl1->ind_prec / byte_size;
	    *cur_ptr++ = el = cgm_xint(dat_ptr);
	    dat_ptr += glbl1->ind_prec / byte_size;
	    if ( (class == -1) && (el == -1) ) {
		may_list(stderr, "drawing plus control set\n");
	    } else if ( (class == -1) && (el == 0) ) {
		may_list(stderr, "drawing set\n");
	    } else may_list(stderr, "may use (%d,%d)\n", class, el);
	}
	if (dev_func) ret = (*dev_func) (no_pairs, array_ptr);
	if (array_ptr) free(array_ptr);
	return(ret);
}
/* replace the metafile defaults */
#define round_up(a) (a = (a % 2) ? a + 1 : a)
static s_mf_defs(dat_ptr, tot_len, dev_func)
unsigned char *dat_ptr;
int tot_len;
int (*dev_func)();
{
unsigned char *my_ptr, *new_mem, *start_mem, *out_ptr;
int to_go, p_len, i, int_len, done, ret = 1, class, element;
unsigned char sec_word[2];

	may_list(stderr, "replacing metafile defaults[\n");
	if (tot_len <= 0) {
	    may_list(stderr, "no defaults]\n");
	    return(1);
	}
	if (dev_func) (*dev_func) (0);		/* start */
	a2 = dflt2;	a3 = dflt3;	a5 = dflt5;
	/* default values are active */
	/* first make some new memory */
	out_ptr = new_mem = allocate_mem(tot_len, 0);	/* ptrs to outgoing data */
	if (!out_ptr) return(2);
	my_ptr = dat_ptr;
	to_go = tot_len;
	/* now move args into it */
	while (to_go > 0) {
	    start_mem 		= my_ptr;	/* ptrs into the incoming data */
	    out_ptr		= new_mem;	/* ptrs into the outgoing data */
	    *out_ptr++ 		= *my_ptr++;
	    *out_ptr++		= *my_ptr++;
	    p_len = *(new_mem + 1)  & 31;

	    if (p_len < 31) {	/* short form command */
	        round_up(p_len);
		for (i=0; i<p_len; ++i) *out_ptr++ = *my_ptr++;
	    }
	    else {		/* long form */
		p_len = 0;
		done = 0;
		while (!done) {
		    for (i=0; i<2; ++i) sec_word[i] = *my_ptr++;
		    int_len = ( (sec_word[0] << 8) + (sec_word[1]) ) % (1 << 15) ;
		    round_up(int_len);
		    for (i=0; i<int_len; ++i)  *out_ptr++ = *my_ptr++;
		    p_len += int_len;
		    done = !(sec_word[0] & (1 << 7));
		}
	    }
	    to_go -= (my_ptr - start_mem);
	    if (!cgm_command(new_mem, p_len, &last_ad, &class, &element)) 
		burp(stderr, "trouble setting defaults\n");
	}
	a2 = glbl2;	a3 = glbl3;	a5 = glbl5;
	/* globals are now active again */
	may_list(stderr, "] finished replacing defaults\n");
	if (dev_func) ret = (*dev_func) (1);		/* end */
	return(ret);
}
#undef round_up
/* read the font list */
static do_font_list(dat_ptr, p_len, dev_func)
unsigned char *dat_ptr;
int p_len;
int (*dev_func)();
{
int ret = 1;
int no_strings;
char font_list[max_fonts][max_str+1];
unsigned char *new_ptr;

	new_ptr = dat_ptr;
	no_strings = 0;
	may_list(stderr, "recording font list\n");
	while ((no_strings < max_fonts) && (new_ptr < dat_ptr + p_len - 3)) {
	    strncpy(font_list[no_strings], cgm_str(new_ptr, &new_ptr), 
		max_str);
	    font_list[no_strings][max_str] = '\0';	/* for safety */
	    may_list(stderr, "[%s]", font_list[no_strings]);
	    ++no_strings;
	}
	may_list(stderr, "\n");
	if (no_strings > max_fonts) no_strings = max_fonts;

	if (dev_func) ret = (*dev_func) (no_strings, font_list);
	return(ret);
}
/* read the character list (implement fully later) */
static do_char_list(dat_ptr, p_len, dev_func)
unsigned char *dat_ptr;
int p_len;
int (*dev_func)();
{
int ret = 1;
int no_strings, type_array[max_fonts];
char font_list[max_fonts][max_str + 1];
unsigned char *new_ptr;

	new_ptr = dat_ptr;
	no_strings = 0;
	may_list(stderr, "recording character list\n");
	while ((no_strings < max_fonts) && (new_ptr < dat_ptr + p_len - 3)) {
	    type_array[no_strings] = cgm_eint(new_ptr);
	    new_ptr += e_size;
	    strncpy(font_list[no_strings], cgm_str(new_ptr, &new_ptr),
		max_str);
	    font_list[no_strings][max_str] = '\0';	/* for safety */
	    may_list(stderr, "[%s]", font_list[no_strings]);
	    ++no_strings;
	}
	may_list(stderr, "\n");

	if (dev_func) ret = (*dev_func) (no_strings, type_array, font_list);
	return(ret);
}
/* do the character announcer */
static do_cannounce(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int ret = 1;
int new_announce;
	
	new_announce = cgm_eint(dat_ptr);
	may_list(stderr, "changing character announcer from %d to %d\n",
	    glbl1->char_c_an, new_announce);
	glbl1->char_c_an = new_announce;
	if (dev_func) ret = (*dev_func) (new_announce);
	return(ret);
}
/* now the class2 functions */
/* set the scaling mode */
static s_scalmode(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int ret = 1, new_mode, step_size;
float my_scale = 1;
	
	step_size = e_size;
	new_mode = cgm_eint(dat_ptr);
	dat_ptr += e_size;
	if (new_mode) my_scale = cgm_float(dat_ptr, &glbl1->real_prec);

	may_list(stderr, "changing scaling mode from %d, %.4f to %d, %.4f\n",
	    a2->scale_mode.s_mode, a2->scale_mode.m_scaling,
	    new_mode, my_scale);

	a2->scale_mode.s_mode = new_mode;
	a2->scale_mode.m_scaling = my_scale;

	if (dev_func) ret = (*dev_func) (new_mode, my_scale);

	return(ret);
}

/* set the colour selection mode */
static s_c_s_mode(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_c_s_mode, ret = 1;
	new_c_s_mode = cgm_eint(dat_ptr);
	may_list(stderr, "changing colour selection mode from %d to %d\n",
		(int) a2->c_s_mode, new_c_s_mode);

	a2->c_s_mode = (enum cs_enum) new_c_s_mode;
	if (dev_func) ret = (*dev_func) ((enum cs_enum) new_c_s_mode);
	return(ret);
}
/* set the line width specification mode */
static s_lws_mode(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_lws_mode, ret = 1;
	new_lws_mode = cgm_eint(dat_ptr);
	may_list(stderr, "changing line width spec mode from %d to %d\n",
		(int) a2->l_w_s_mode, new_lws_mode);

	a2->l_w_s_mode = (enum spec_enum) new_lws_mode;
	if (dev_func) ret = (*dev_func) ((enum spec_enum) new_lws_mode);
	return(ret);
}
/* set the marker size specification mode */
static s_ms_mode(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_mode, ret = 1;
	new_mode = cgm_eint(dat_ptr);
	may_list(stderr, "changing marker size spec mode from %d to %d\n",
		(int) a2->m_s_s_mode, new_mode);

	a2->m_s_s_mode = (enum spec_enum) new_mode;
	if (dev_func) ret = (*dev_func) ((enum spec_enum) new_mode);
	return(ret);
}
/* set the edge width specification mode */
static s_ew_mode(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_mode, ret = 1;
	new_mode = cgm_eint(dat_ptr);
	may_list(stderr, "changing edge width spec mode from %d to %d\n",
		(int) a2->e_w_s_mode, new_mode);

	a2->e_w_s_mode = (enum spec_enum) new_mode;
	if (dev_func) ret = (*dev_func) ((enum spec_enum) new_mode);
	return(ret);
}
/* set the VDC extent */
static s_vdc_extent(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_coords[4], i, ret = 1; float new_real[4];

	switch (glbl1->vdc_type) {
case vdc_int:	for (i = 0; i<4; ++i) {
		    new_coords[i] = cgm_gint(dat_ptr, glbl3->vdc_i_prec);
		    new_real[i] = new_coords[i] / pxl_vdc;
		    dat_ptr += a3->vdc_i_prec / byte_size;
		}
		may_list(stderr, 
		   "changing VDC extent from (%d,%d,%d,%d) to (%d,%d,%d,%d)\n",
		    a2->vdc_extent.i[0], a2->vdc_extent.i[1], 
		    a2->vdc_extent.i[2], a2->vdc_extent.i[3], 	    	    
		    new_coords[0], new_coords[1], new_coords[2],new_coords[3]);
		break;
case vdc_real:	for (i = 0; i<4; ++i) {
		    new_real[i] = cgm_vreal(dat_ptr);
		    new_coords[i] = new_real[i] * pxl_vdc;
		    dat_ptr += (a3->vdc_r_prec.exp + a3->vdc_r_prec.fract)
			/ byte_size;
		}
		may_list(stderr, 
		   "changing VDC extent from (%f,%f,%f,%f) to (%f,%f,%f,%f)\n",
		    a2->vdc_extent.r[0], a2->vdc_extent.r[1], 
		    a2->vdc_extent.r[2], a2->vdc_extent.r[3],
		    new_real[0], new_real[1], new_real[2],new_real[3]);
		 break;
default:	burp(stderr, "illegal vdc_type = %d\n", glbl1->vdc_type);
	}
	for (i=0; i<4; ++i) {
	    a2->vdc_extent.i[i] = new_coords[i];
	    a2->vdc_extent.r[i] = new_real[i];
	}
	if (dev_func) ret = (*dev_func) (new_coords, new_real);
	return(ret);
}
/* set the background colour */
static s_back_col(dat_ptr,dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int step_size, ret = 1;
unsigned int ir, ig, ib;
float r, g, b;

	step_size = (glbl1->col_prec / byte_size);
	ir = cgm_dcint(dat_ptr);
	r = dcr(ir);
	dat_ptr += step_size;
	ig = cgm_dcint(dat_ptr);
	g = dcg(ig);
	dat_ptr += step_size;
	ib = cgm_dcint(dat_ptr);
	b = dcb(ib);
	may_list(stderr, 
		"changing background colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
		a2->back_col.red, a2->back_col.green, a2->back_col.blue,
		r, g, b);
	a2->back_col.red = r;
	a2->back_col.green = g;
	a2->back_col.blue = b;

	/* we have to override the index 0 colour by the background 
	    colour (sounds screwy, but I think correct) */

	*(a5->ctab)	 	= a2->back_col.red;
	*(a5->ctab + 1) 	= a2->back_col.green;
	*(a5->ctab + 2) 	= a2->back_col.blue;

	if (dev_func) ret = (*dev_func)(ir, ig, ib);
	return(ret);
}
/* now the class3 functions */
/* set the vdc integer precision */
static s_vdc_i_p(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int precision, ret = 1;
	precision = cgm_sint(dat_ptr);
	may_list(stderr, "changing vdc int precision from %d to %d",
	    a3->vdc_i_prec, precision);
	a3->vdc_i_prec = precision;
	if (dev_func) ret = (*dev_func) (precision);
	return(ret);
}
/* set the vdc real precision */
static s_vdc_r_p(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int step_size, new_fixed, new_exp, new_fract, ret = 1;

	step_size = glbl1->int_prec / byte_size;
	new_fixed = cgm_sint(dat_ptr);
	dat_ptr += step_size;
	new_exp = cgm_sint(dat_ptr);
	dat_ptr += step_size;
	new_fract = cgm_sint(dat_ptr);

	may_list(stderr, 
	    "changing vdc real precision from (%d, %d, %d) to (%d, %d, %d)\n",
	    a3->vdc_r_prec.fixed, a3->vdc_r_prec.exp, 
	    a3->vdc_r_prec.fract, new_fixed, new_exp, new_fract);

	a3->vdc_r_prec.fixed	= new_fixed;
	a3->vdc_r_prec.exp 	= new_exp;
	a3->vdc_r_prec.fract	= new_fract;

	if (dev_func) ret = (*dev_func) (new_fixed, new_exp, new_fract);
	return(ret);
}
/* set the auxiliary colour */
static s_aux_col(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int ret = 1;
int new_index = 0, step_size, ir, ig, ib;
float r, g, b, *rptr;

	switch ((int) a2->c_s_mode) {
case (int) i_c_mode: /* indexed mode */
	    new_index = cgm_cxint(dat_ptr);
	    if (list_cgm) 
		burp(stderr, "changing auxiliary index from %d to %d\n",
		    a3->aux_col.ind, new_index);
	    a3->aux_col.ind = new_index;
	    rptr = a5->ctab + a3->aux_col.ind * 3;
	    r = *rptr;
	    g = *++rptr;
	    b = *++rptr;
	    break;
case (int) d_c_mode: /* direct mode */
	    step_size = (glbl1->col_prec / byte_size);
	    ir = cgm_dcint(dat_ptr);
	    r = dcr(ir);
	    dat_ptr += step_size;
	    ig = cgm_dcint(dat_ptr);
	    g = dcg(ig);
	    dat_ptr += step_size;
	    ib = cgm_dcint(dat_ptr);
	    b = dcb(ib);
	    may_list(stderr, 
"changing auxiliary colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
		a3->aux_col.red, a3->aux_col.green, 
		a3->aux_col.blue, r, g, b);
	    a3->aux_col.red = r;
	    a3->aux_col.green = g;
	    a3->aux_col.blue = b;
	    break;
	default:
	    burp(stderr, "illegal colour mode = %d\n", (int) a2->c_s_mode);
	}
	if (dev_func) ret = (*dev_func) (r, g, b, new_index);
	return(ret);
}
/* set the transparency */
static s_transp(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_trans, ret = 1;
	new_trans = cgm_eint(dat_ptr);
	may_list(stderr, "changing transparency from %d to %d\n",
	    (int) a3->transparency, new_trans);
	a3->transparency = (enum boolean) new_trans;
	if (dev_func) ret = (*dev_func) (a3->transparency);
	return(ret);
}


/* set the clipping rectangle */
static s_clip_rec(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_coords[4], i; float new_real[4], ret = 1;


	switch (glbl1->vdc_type) {
case vdc_int:	for (i = 0; i<4; ++i) {
		    new_coords[i] = (*cgm_vdc)(dat_ptr);
		    dat_ptr += a3->vdc_i_prec / byte_size;
		}
		may_list(stderr, 
	"changing clipping rectangle from (%d,%d,%d,%d)to (%d,%d,%d,%d)\n",
		    a3->clip_rect.i[0], a3->clip_rect.i[1], a3->clip_rect.i[2],
		    a3->clip_rect.i[3],
		    new_coords[0], new_coords[1], new_coords[2],new_coords[3]);
		for (i=0; i<4; ++i) a3->clip_rect.i[i] = new_coords[i];
		break;
case vdc_real:	for (i = 0; i<4; ++i) {
		    new_real[i] = cgm_vreal(dat_ptr);
		    a3->clip_rect.i[i] = (*cgm_vdc)(dat_ptr);
		    dat_ptr += (a3->vdc_r_prec.exp + a3->vdc_r_prec.fract)
		        / byte_size;
		}
		may_list(stderr, 
	"changing clipping rectangle from (%d,%d,%d,%d)to (%d,%d,%d,%d)\n",
		    a3->clip_rect.r[0], a3->clip_rect.r[1], a3->clip_rect.r[2],
		    a3->clip_rect.r[3], 
		    new_real[0], new_real[1], new_real[2],new_real[3]);
		for (i=0; i<4; ++i) a3->clip_rect.r[i] = new_real[i];
		break;
default:	burp(stderr, "illegal vdc_type = %d\n", glbl1->vdc_type);

	}
/* get into right format for device */
	for (i=0; i<2; ++i) {
	  new_coords[2 * i] = NEWX(a3->clip_rect.i[2*i],
				   a3->clip_rect.i[2*i + 1]);
	  new_coords[2 * i + 1] = NEWY(a3->clip_rect.i[2*i],
				       a3->clip_rect.i[2*i + 1]);
	}

	if (dev_func) ret = (*dev_func) (new_coords, new_real);
	return(ret);
}
/* set the clipping indicator */
static s_clip_ind(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int new_clip, ret = 1;
	new_clip = cgm_eint(dat_ptr);
	may_list(stderr, "changing clipping indicator from %d to %d\n",
	    (int) a3->clip_ind, new_clip);
	a3->clip_ind = (enum boolean) new_clip;
	if (dev_func) ret = (*dev_func) (a3->clip_ind);
	return(ret);
}
/* now the class 4 functions */
/* take care of a series of points that need a line between them */
static do_polyline(dat_ptr, p_len, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
int p_len;
{
int ret = 1;
int no_pairs, step_size, x, y, i, *x_ptr, *y_ptr, *x1_ptr, *y1_ptr;

	step_size = (vdc_step / byte_size);
	no_pairs = p_len / (step_size * 2);

	/* first get the memory */
	if (no_pairs > wk_ar_size) {
	    x1_ptr = intalloc(no_pairs * bytes_per_word);
	    y1_ptr = intalloc(no_pairs * bytes_per_word);
	}
	else {
	    x1_ptr = x_wk_ar;
	    y1_ptr = y_wk_ar;
	}
	if ( (!x1_ptr) || (!y1_ptr) ) 
	    burp(stderr, "trouble with polyline memory !\n");
	x_ptr = x1_ptr;
	y_ptr = y1_ptr;
	    
	/* now grab the data */
	for (i=0;i<no_pairs;++i) {
	    x = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	    y = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	    *x_ptr++ = newx(x, y);
	    *y_ptr++ = newy(x, y);
	}

	/* and take care of the output */
	if (list_cgm) {
	    (void) fprintf(stderr, "polyline = [");
	    for (i=0;i<no_pairs;++i) 
	         fprintf(stderr, "\n(%d,%d)", x1_ptr[i], y1_ptr[i]);
	    (void) fprintf(stderr, "]\n");
	}
	
	/* NO emulation function, can't do anything without a polyline */
	if (dev_func) ret = (*dev_func) (no_pairs, x1_ptr, y1_ptr);

	if (x1_ptr != x_wk_ar) free(x1_ptr);
	if (y1_ptr != y_wk_ar) free(y1_ptr);

	return(ret);
}
	
/* take care of a series of points that need a line between alternate points*/
static do_dis_polyline(dat_ptr, p_len, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
int p_len;
{
int ret = 1;
int no_pairs, step_size, x, y, i, *x_ptr, *y_ptr, *x1_ptr, *y1_ptr;
extern em_dpline();	/* in emul.c */

	step_size = (vdc_step / byte_size);
	no_pairs = p_len / (step_size * 2);

	/* some cell arrays have odd number of points ! */
	if (no_pairs % 2) --no_pairs;

	/* first get the memory */
	if (no_pairs > wk_ar_size) {
	    x1_ptr = intalloc(no_pairs * bytes_per_word);
	    y1_ptr = intalloc(no_pairs * bytes_per_word);
	}
	else {
	    x1_ptr = x_wk_ar;
	    y1_ptr = y_wk_ar;
	}
	if ( (!x1_ptr) || (!y1_ptr) ) 
	    burp(stderr, "trouble with polyline memory !\n");
	x_ptr = x1_ptr;
	y_ptr = y1_ptr;
	    
	/* now grab the data */
	for (i=0;i<no_pairs;++i) {
	    x = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	    y = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	    *x_ptr++ = newx(x, y);
	    *y_ptr++ = newy(x, y);
	}

	/* and take care of the output */
	if (list_cgm) {
	    (void) fprintf(stderr, "disjoint polyline = [");
	    for (i=0;i<no_pairs;++i) 
	         fprintf(stderr, "\n(%d,%d)", x1_ptr[i], y1_ptr[i]);
	    (void) fprintf(stderr, "]\n");
	}
	
	/* might emulate */
	ret = (dev_func) ? (*dev_func) (no_pairs, x1_ptr, y1_ptr):
	    em_dpline(no_pairs, x1_ptr, y1_ptr);

	if (x1_ptr != x_wk_ar) free(x1_ptr);
	if (y1_ptr != y_wk_ar) free(y1_ptr);

	return(ret);
}
/* do a series of markers at the specified points */
static do_polymarker(dat_ptr, p_len, dev_func)
unsigned char *dat_ptr;
int p_len;
int (*dev_func)();
{
int ret = 1;
int no_pairs, step_size, x, y, i, *x_ptr, *y_ptr, *x1_ptr, *y1_ptr, memsize;
extern int em_pmarker();	/* emul.c */

	step_size = (vdc_step / byte_size);
	no_pairs = p_len / (step_size * 2);

	/* first get the memory */
	if (no_pairs > wk_ar_size) {
	    x1_ptr = intalloc(no_pairs * bytes_per_word);
	    y1_ptr = intalloc(no_pairs * bytes_per_word);
	}
	else {
	    x1_ptr = x_wk_ar;
	    y1_ptr = y_wk_ar;
	}
	if ( (!x1_ptr) || (!y1_ptr) ) 
	    burp(stderr, "trouble with polymarker memory !\n");
	x_ptr = x1_ptr;
	y_ptr = y1_ptr;
	    
	/* now grab the data */
	for (i=0;i<no_pairs;++i) {
	    x = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	    y = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	    *x_ptr++ = newx(x, y);
	    *y_ptr++ = newy(x, y);
	}
/* and take care of the output */
	if (list_cgm) {
	    (void) fprintf(stderr, "polymarker = [");
	    for (i=0;i<no_pairs;++i) 
	         fprintf(stderr, "\n(%d,%d)", x1_ptr[i], y1_ptr[i]);
	    (void) fprintf(stderr, "]\n");
	}
	memsize = a5->mk_size.i;
	/* might emulate */
	ret = (dev_func) ? (*dev_func) (no_pairs, x1_ptr, y1_ptr) :
	    em_pmarker(no_pairs, x1_ptr, y1_ptr);

	if (x1_ptr != x_wk_ar) free(x1_ptr);
	if (y1_ptr != y_wk_ar) free(y1_ptr);

	return(ret);
}
/* set actual text, also take care of raster character descriptions */
static s_text(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int step_size, x, y, final, xin, yin, no_chars, device_ok, i, ind, ret=1,
xdelta, ydelta, old_c_index, old_l_width = 1, new_l_width;
float r0, g0, b0, l0;
double mag;
char buffer[2 * max_str + 1];
static int em_type = 0, font_size;
enum line_enum old_l_type = solid_l;

	/* first see if the device can handle it */
	device_ok = 
	    ((a5->t_prec == string) &&(dev_info->capability & string_text)) ||
	    ((a5->t_prec == character)&&(dev_info->capability & char_text)) ||
	    ((a5->t_prec == stroke) &&(dev_info->capability & stroke_text)) ||
	    (opt[(int) font_type].set && !opt[(int) font_type].val.i) ||
	    (font_check == NULL) || (font_text == NULL);

	/* may be overrriding */
	if (opt[(int) font_type].set && opt[(int) font_type].val.i &&
	    font_check && font_text) device_ok = 0;

	/* if don't have device_ok, but can't emulate, we'll force device_ok */
	if (!device_ok) {
	    if (!em_type) em_type = (*font_check)(&font_size);
	    switch (em_type) {
case CA_EMULATION:	/* emulation via Cell Arrays */
	    	if (!gprim[(int) Cell_Array]) device_ok = 1;
		break;
case PL_EMULATION:	/* emulation via Polyline */
	    	if (!gprim[(int) PolyLine]) device_ok = 1;
		break;
	    }
	}

	/* now ready to procede */
	step_size = (vdc_step / byte_size);

	xin = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	yin = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	x = newx(xin, yin);
	y = newy(xin, yin);

	step_size = e_size;
	final = cgm_eint(dat_ptr);
	dat_ptr += step_size;

	(void) strcpy(buffer, cgm_str(dat_ptr, NULL));
	no_chars = strlen(buffer);

	if ( (final != 0) && (final != 1) ) {
	    burp(stderr, "illegal (%d)  final in TEXT\n", final);
	    return(0);
	}

	may_list(stderr, "text[%d] at (%d,%d) = [%s]\n",
	    *(dat_ptr + 1), x, y, buffer);

	/* may have to do the positioning adjustment */
	xdelta = ydelta = 0;
	if (!(dev_info->capability & v_center) && (device_ok)) {
	switch (a5->text_align.ver) {
case top_v:		ydelta = - 1.1 * dev_info->c_height; break;
case cap_v:		ydelta = - dev_info->c_height; break;
case half_v:		ydelta = - 0.5 * dev_info->c_height; break;
case bottom_v:		ydelta = - 0.1 * dev_info->c_height; break;
default:		ydelta = 0;
}
	}

	if (!(dev_info->capability & h_center) && (device_ok)) { 
	    switch (a5->text_align.hor) {	/* fix later */
    case center_h:	xdelta = - 0.5 * no_chars * dev_info->c_width; break;
    case right_h:	xdelta = - no_chars * dev_info->c_width; break;
    default:		xdelta = 0;
	    }
	}

	/* adjust the position */
	if (xdelta || ydelta) {
	    x += xdelta * cosr - ydelta * sinr;
	    y += ydelta * cosr + xdelta * sinr;
	}

/* now do the actual text */
	if (device_ok) { 
	    ret = (dev_func) ? (*dev_func) (x, y, final, buffer) : 2;
	} else {	/* may do it with emulation */
	    /* now decide the relevant font magnification */
	    mag = (double) a5->c_height / font_size;

	    switch (em_type) {
case CA_EMULATION:	/* emulation via Cell Arrays */
	    	if (gprim[(int) Cell_Array])
		    ret = (*font_text)(x, y, buffer, gprim[(int) Cell_Array], 
		        a2, a5, no_chars, dev_info->capability, NULL, mag);
		break;
case PL_EMULATION:	/* emulation via Polylines */
	    	if (gprim[(int) PolyLine]) {
		    /* need to set the colour, line size/type */
		    /* first the colour */
		    old_c_index = a5->line_colour.ind;
		    r0 = a5->line_colour.red;
		    g0 = a5->line_colour.green;
		    b0 = a5->line_colour.blue;
		    a5->line_colour.ind = a5->text_colour.ind;
		    a5->line_colour.red = a5->text_colour.red;
		    a5->line_colour.green = a5->text_colour.green;
		    a5->line_colour.blue = a5->text_colour.blue;
		    if (attr[(int) LColour]) (*attr[(int) LColour])
			(a5->text_colour.red, a5->text_colour.green, 
			 a5->text_colour.blue, a5->text_colour.ind);
		    /* now the line type */
		    if (a5->line_type != solid_l) {
			old_l_type = a5->line_type;
			a5->line_type = solid_l;
			if (attr[(int) LType]) (*attr[(int) LType])
			    (a5->line_type);
		    }
		    /* now the line width */
		    new_l_width = 0.5 + mag;
		    if (new_l_width < 1) new_l_width = 1;
		    if (new_l_width != a5->line_width.i) {
			old_l_width = a5->line_width.i;
			l0 = a5->line_width.r;
			a5->line_width.i = new_l_width;
			if (!dev_info->d_l_width) dev_info->d_l_width = 1;
			a5->line_width.r = (float) a5->line_width.i /
			    dev_info->d_l_width;
			if (attr[(int) LWidth]) (*attr[(int) LWidth])
			    (a5->line_width.i, a5->line_width.r);
		    }
		    /* actually set the text */
	  	    ret = (*font_text)(x, y, buffer, gprim[(int) PolyLine], 
		        a2, a5, no_chars, dev_info->capability, NULL, mag);

		    /* reset attributes */
		    a5->line_colour.ind = old_c_index;
		    a5->line_colour.red = r0;
		    a5->line_colour.green = g0;
		    a5->line_colour.blue = b0;
		    if (attr[(int) LColour]) (*attr[(int) LColour])
			(a5->line_colour.red, a5->line_colour.green, 
			 a5->line_colour.blue, a5->line_colour.ind);
		    if (old_l_type != solid_l) {
			a5->line_type = old_l_type;
			if (attr[(int) LType]) (*attr[(int) LType])
			    (a5->line_type);
		    }
		    if (old_l_width != 1) {
			a5->line_width.i = old_l_width;
			a5->line_width.r = l0;
			if (attr[(int) LWidth]) (*attr[(int) LWidth])
			    (a5->line_width.i, a5->line_width.r);
		    }
		    /* all done */

		}
		break;
default:	ret = 0;
		break;
	    }
	}
	return(ret);
}
/* restricted text */
static s_rex_text(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int ret = 1;
int step_size, width, height, xin, yin, x, y;
enum boolean final;
char buffer[2 * max_str + 1];

	step_size = (vdc_step / byte_size);

	width = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	height = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	xin = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	yin = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	x = newx(xin, yin);
	y = newy(xin, yin);

	step_size = e_size;
	final = (enum boolean) cgm_eint(dat_ptr);
	dat_ptr += step_size;

	(void) strcpy(buffer, cgm_str(dat_ptr, NULL));
	
	if (dev_func) ret = (*dev_func) (width, height, x, y, final, buffer);

	return(ret);
}
/* appended text */
static s_app_text(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int ret = 1;
int step_size;
enum boolean final;
char buffer[2 * max_str + 1];
	step_size = e_size;
	final = (enum boolean) cgm_eint(dat_ptr);
	dat_ptr += step_size;

	(void) strcpy(buffer, cgm_str(dat_ptr, NULL));
	
	if (dev_func) ret = (*dev_func) (final, buffer);
	return(ret);
}

/* handle a polygon */
static do_polygon(dat_ptr, p_len, dev_func)
unsigned char *dat_ptr;
int p_len;
int (*dev_func)();
{
int ret = 1;
int no_pairs, step_size, x, y, i, *x_ptr, *y_ptr, *x1_ptr, *y1_ptr;
extern em_pgon();	/* in emul.c */

	step_size = (vdc_step / byte_size);
	no_pairs = p_len / (step_size * 2);

	/* first get the memory */
	if (no_pairs > wk_ar_size) {
	    x1_ptr = intalloc(no_pairs * bytes_per_word);
	    y1_ptr = intalloc(no_pairs * bytes_per_word);
	}
	else {
	    x1_ptr = x_wk_ar;
	    y1_ptr = y_wk_ar;
	}
	if ( (!x1_ptr) || (!y1_ptr) ) 
	    burp(stderr, "trouble with polygon memory !\n");
	x_ptr = x1_ptr;
	y_ptr = y1_ptr;
	    
	/* now grab the data */
	for (i=0;i<no_pairs;++i) {
	    x = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	    y = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	    *x_ptr++ = newx(x, y);
	    *y_ptr++ = newy(x, y);
	}

	/* and take care of the output */
	if (list_cgm) {
	    (void) fprintf(stderr, "polygon = [");
	    for (i=0;i<no_pairs;++i) 
	         fprintf(stderr, "\n(%d,%d)", x1_ptr[i], y1_ptr[i]);
	    (void) fprintf(stderr, "]\n");
	}
	/* might emulate */
	ret = (dev_func) ? (*dev_func) (no_pairs, x1_ptr, y1_ptr) :
	    em_pgon(no_pairs, x1_ptr, y1_ptr);

	if (x1_ptr != x_wk_ar) free(x1_ptr);
	if (y1_ptr != y_wk_ar) free(y1_ptr);

	return(ret);
}
/* do a polyset */
static do_polyset(dat_ptr, p_len, dev_func)
unsigned char *dat_ptr;
int p_len;
int (*dev_func)();
{
int ret = 1;
int no_pairs, step_size, x, y, i, *x_ptr, *y_ptr, *x1_ptr, *y1_ptr;
unsigned char *edge_ptr; 
extern em_polyset();	/* in emul.c */

	/* figure out how many pairs */
	no_pairs = p_len * byte_size / (vdc_step * 2 + e_size * byte_size);

	/* first get the memory */
	if (no_pairs > wk_ar_size) {
	    x1_ptr = intalloc(no_pairs * bytes_per_word);
	    y1_ptr = intalloc(no_pairs * bytes_per_word);
	}
	else {
	    x1_ptr = x_wk_ar;
	    y1_ptr = y_wk_ar;
	}
	edge_ptr = allocate_mem(no_pairs, 0);
	if (!edge_ptr) return(2);
	if ( (!x1_ptr) || (!y1_ptr) ) 
	    burp(stderr, "trouble with polyset memory !\n");
	x_ptr = x1_ptr;
	y_ptr = y1_ptr;
	    
	/* now grab the data */
	step_size = (vdc_step / byte_size);
	for (i=0;i<no_pairs;++i) {
	    x = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	    y = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	    *x_ptr++ = newx(x, y);
	    *y_ptr++ = newy(x, y);
	    edge_ptr[i] = cgm_eint(dat_ptr);
	    dat_ptr += e_size;
	}

	/* and take care of the output */
	if (list_cgm) {
	    (void) fprintf(stderr, "polyset = [");
	    for (i=0;i<no_pairs;++i) 
	        fprintf(stderr, "\n(%d,%d, %d)", x1_ptr[i], y1_ptr[i], 
		edge_ptr[i]);
	    (void) fprintf(stderr, "]\n");
	}
	/* might emulate */
	ret = (dev_func) ? (*dev_func) (no_pairs, x1_ptr, y1_ptr, edge_ptr) :
	    em_polyset(no_pairs, x1_ptr, y1_ptr, edge_ptr);

	free(edge_ptr);
	if (x1_ptr != x_wk_ar) free(x1_ptr);
	if (y1_ptr != y_wk_ar) free(y1_ptr);
	return(ret);
}
/* now try to take care of a general cell array */
static do_cell_array(dat_ptr, plen, dev_func)
unsigned char *dat_ptr;
int plen, (*dev_func)();
{
int ret = 1;
int p0[2], cp[2];	/* corner p */
int q0[2], cq[2];	/* corner q */
int r0[2], cr[2];	/* corner r */
/* this is a parallelogram, diagonal between p and q, first row is p-r */
unsigned int nx;		/* columns of data */
unsigned int ny;		/* rows of data */
int l_col_prec;	/* local colour precision */
int rep_mode;	/* cell representation mode */
int step_size, i, c_size, row_size;
unsigned char *orig_ptr;		/* original pointer */
long int no_bytes;	/* number of bytes of data */
extern em_cell_array();	/* in emul.c */

	orig_ptr = dat_ptr;
	step_size = (vdc_step / byte_size);
	for (i=0; i<2; ++i) {
	    p0[i] = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	}
	for (i=0; i<2; ++i) {
	    q0[i] = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	}
	for (i=0; i<2; ++i) {
	    r0[i] = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	}

	step_size = (glbl1->int_prec / byte_size);
	
	nx = cgm_sint(dat_ptr);
	dat_ptr += step_size;

	ny = cgm_sint(dat_ptr);
	dat_ptr += step_size;

	l_col_prec = cgm_sint(dat_ptr);
	dat_ptr += step_size;

	step_size = e_size;
	rep_mode = cgm_eint(dat_ptr);
	dat_ptr += step_size;
/* get the precion right */
	if (a2->c_s_mode) { 	/* direct */
	    l_col_prec = (l_col_prec) ? l_col_prec : glbl1->col_prec;
	} else {		/* indexed */
	    l_col_prec = (l_col_prec) ? l_col_prec : glbl1->col_i_prec;
	}
	cp[0] = newx(p0[0], p0[1]);
	cp[1] = newy(p0[0], p0[1]);
	cq[0] = newx(q0[0], q0[1]);
	cq[1] = newy(q0[0], q0[1]);
	cr[0] = newx(r0[0], r0[1]);
	cr[1] = newy(r0[0], r0[1]);

	if ( (rep_mode > 1) || (rep_mode < 0) ) {
	    burp(stderr, "illegal rep_mode %d\n", rep_mode); 
	    return(2);
	}


/* just pass them the whole shebang, let them use cell array library */

	may_list(stderr, 
	"cell array (%d,%d,%d,%d,%d,%d), size %d,%d, prec %d, mode %d\n",
	    cp[0], cp[1], cq[0], cq[1], cr[0], cr[1], nx, ny, l_col_prec,
	    rep_mode);

	no_bytes = plen - (dat_ptr - orig_ptr);
	/* might emulate */
	ret = (dev_func) ? (*dev_func) (cp, cq, cr, nx, ny, l_col_prec, 
	    dat_ptr, rep_mode, no_bytes) : em_cell_array(cp, cq, cr, nx, ny, 
	    l_col_prec, dat_ptr, rep_mode, no_bytes);
	return(ret);
}
/* generalised drawing primitive */
/* format is identifier, no_pairs, list of points, set of strings */
static do_g_d_p(dat_ptr, plen, dev_func)
unsigned char *dat_ptr;
int plen, (*dev_func)();
{
int ret = 1;
int gdp_id, no_pairs, *x_ptr, *y_ptr, step_size, i, x, y;
char *data_record;
extern em_g_d_p();	/* in emul.c */

	step_size = (glbl1->int_prec / byte_size);
	
	gdp_id = cgm_sint(dat_ptr);
	dat_ptr += step_size;

	no_pairs = cgm_sint(dat_ptr);
	dat_ptr += step_size;
	
	/* first get the memory */
	if (no_pairs > wk_ar_size) {
	    x_ptr = intalloc(no_pairs * bytes_per_word);
	    y_ptr = intalloc(no_pairs * bytes_per_word);
	}
	else {
	    x_ptr = x_wk_ar;
	    y_ptr = y_wk_ar;
	}
	/* now grab the data */
	step_size = (vdc_step / byte_size);
	for (i=0;i<no_pairs;++i) {
	    x = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	    y = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	    x_ptr[i] = newx(x, y);
	    y_ptr[i] = newy(x, y);
	}
	data_record = cgm_str(dat_ptr, NULL);

	if (list_cgm) {
	    (void) fprintf(stderr, 
	"generalised drawing primitive %d, %d points, data record = [%s]\n", 
		gdp_id, no_pairs, data_record);
	    for (i=0; i<no_pairs; ++i) 
		(void) fprintf(stderr, "(%d, %d)\n", x_ptr[i], y_ptr[i]);
	}
	/* might emulate */
	ret = (dev_func) ? (*dev_func) (gdp_id, no_pairs, x_ptr, y_ptr, 
	    data_record) : em_g_d_p(gdp_id, no_pairs, x_ptr, y_ptr, 
	    data_record);

	if (x_ptr != x_wk_ar) free(x_ptr);
	if (y_ptr != y_wk_ar) free(y_ptr);

	return(ret);
}
/* do a rectangle */
static do_rectangle(dat_ptr, dev_func)
unsigned char *dat_ptr;
int(*dev_func)();
{
int ret = 1;
int x1, y1, x2, y2, step_size, x_1, y_1, x_2, y_2;
extern int em_rectangle(); /* in emul.c */
	step_size = (vdc_step / byte_size);
	x1 = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	y1 = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	x2 = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	y2 = (*cgm_vdc)(dat_ptr);
	x_1 = newx(x1, y1);
	y_1 = newy(x1, y1);
	x_2 = newx(x2, y2);
	y_2 = newy(x2, y2);


	may_list(stderr, "rectangle = (%d, %d, %d, %d)\n",
	     x_1, y_1, x_2, y_2);
	/* might emulate */
	ret = (dev_func)? (*dev_func) (x_1, y_1, x_2, y_2) :
	    em_rectangle(x_1, y_1, x_2, y_2);
	return(ret);
}
/* set a circle */
static do_circle(dat_ptr, dev_func)
unsigned char *dat_ptr;
int(*dev_func)();
{
int ret = 1, step_size, x, y, r, x1, y1;
extern int em_circle();		/* emul.c */

	step_size = (vdc_step / byte_size);
	x1 = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	y1 = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	r = (*cgm_vdc)(dat_ptr);
	x = newx(x1, y1);
	y = newy(x1, y1);
	may_list(stderr, "circle, center (%d, %d), radius %d\n", x, y, r);

	/* might emulate */
	ret = (dev_func) ? (*dev_func) (x, y, r) : em_circle(x, y, r);

	return(ret);
}
/* set an arc, get the positions of 1st pt, intermdiate pt, end pt */
static do_c3(dat_ptr, dev_func)
unsigned char *dat_ptr;
int(*dev_func)();
{
int ret = 1, step_size, x_array[6], i, first_array[6];
extern em_c3();	/* in emul.c */

	step_size = (vdc_step / byte_size);
	for (i=0; i<6; ++i) {
	    first_array[i] = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	}
	for (i=0; i<3; ++i) {
	    x_array[2 * i] = newx(first_array[2 * i], first_array[2*i+1]);
	    x_array[2 * i + 1] = newy(first_array[2 * i], first_array[2*i+1]);
	}
	may_list(stderr, "arc thru (%d, %d) (%d, %d) (%d, %d)\n",
	    x_array[0], x_array[1], x_array[2], x_array[3], 
	    x_array[4], x_array[5]);
	ret = (dev_func) ? (*dev_func) (x_array) : em_c3(x_array);
	return(ret);
}
/* set a closed arc, get the positions of 1st pt, intermdiate pt, end pt */
static do_c3_close(dat_ptr, dev_func)
unsigned char *dat_ptr;
int(*dev_func)();
{
int ret = 1, step_size, x_array[6], i, first_array[6];
enum boolean chord;
extern em_c3_close();	/* in emul.c */
	step_size = (vdc_step / byte_size);
	for (i=0; i<6; ++i) {
	    first_array[i] = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	}
	for (i=0; i<3; ++i) {
	    x_array[2 * i] = newx(first_array[2 * i], first_array[2*i+1]);
	    x_array[2 * i + 1] = newy(first_array[2 * i], first_array[2*i+1]);
	}
	chord = (enum boolean) cgm_eint(dat_ptr);
	may_list(stderr, "closed arc thru (%d, %d) (%d, %d) (%d, %d), %d\n",
	    x_array[0], x_array[1], x_array[2], x_array[3], 
	    x_array[4], x_array[5], (int) chord);
	ret = (dev_func) ? (*dev_func) (x_array, chord) : 
	    em_c3_close(x_array, chord);
	return(ret);
}
/* set an arc, ends specified by vectors */
static do_c_centre(dat_ptr, dev_func)
unsigned char *dat_ptr;
int(*dev_func)();
{
int ret = 1, step_size, vec_array[4], i, x, y, r, x1, y1;
extern em_c_centre();	/* in emul.c */

	step_size = (vdc_step / byte_size);
	x1 = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	y1 = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	for (i=0; i<4; ++i) {
	    vec_array[i] = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	}
	r = (*cgm_vdc)(dat_ptr);
	x = newx(x1, y1);
	y = newy(x1, y1);

	may_list(stderr, "circle centre (%d, %d) (%d, %d) (%d, %d), %d\n",
	    x, y, vec_array[0], vec_array[1], 
	    vec_array[2], vec_array[3], r);

	/* might emulate */
	ret = (dev_func) ? (*dev_func) (x, y, vec_array, r) : 
	    em_c_centre(x, y, vec_array, r);
	return(ret);
}
/* set an arc, ends specified by vectors, close it */
static do_c_c_close(dat_ptr, dev_func)
unsigned char *dat_ptr;
int(*dev_func)();
{
int ret = 1, step_size, vec_array[4], i, x, y, r, x1, y1;
enum boolean chord;
extern em_c_c_close();	/* in emul.c */
	step_size = (vdc_step / byte_size);
	x1 = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	y1 = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	for (i=0; i<4; ++i) {
	    vec_array[i] = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	}
	r = (*cgm_vdc)(dat_ptr);
	x = newx(x1, y1);
	y = newy(x1, y1);

	dat_ptr += step_size;
	chord = (enum boolean) cgm_eint(dat_ptr);
	may_list(stderr, 
	    "closed circle centre (%d, %d) (%d, %d) (%d, %d), %d, %d\n",
	    x, y, vec_array[0], vec_array[1], 
	    vec_array[2], vec_array[3], r, (int) chord);
	/* might emulate */
	ret = (dev_func) ? (*dev_func) (x, y, vec_array, r, chord) : 
	    em_c_c_close(x, y, vec_array, r, chord);
	return(ret);
}
/* set an ellipse, specify centre, two conjugate diameters (see the book !) */
static do_ellipse(dat_ptr, dev_func)
unsigned char *dat_ptr;
int(*dev_func)();
{
int ret = 1, step_size, pt_array[6], i;
extern em_ellipse(); /* in emul.c */

	step_size = (vdc_step / byte_size);
	for (i=0; i<6; ++i) {
	    pt_array[i] = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	}
	may_list(stderr, "ellipse (%d, %d), (%d, %d), (%d, %d)\n",
	    pt_array[0], pt_array[1], pt_array[2], 
	    pt_array[3], pt_array[4], pt_array[5]);

	/* might emulate */
	ret = (dev_func) ? (*dev_func) (pt_array) : 
	    em_ellipse(pt_array);
	return(ret);
}
/* set an elliptical arc, specify centre two conjugate diameters end vectors */
static do_ell_arc(dat_ptr, dev_func)
unsigned char *dat_ptr;
int(*dev_func)();
{
int ret = 1, step_size, pt_array[6], vec_array[4], i;
extern em_all_arc();	/* in emul.c */

	step_size = (vdc_step / byte_size);
	for (i=0; i<6; ++i) {
	    pt_array[i] = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	}
	for (i=0; i<4; ++i) {
	    vec_array[i] = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	}
	may_list(stderr,
	    "elliptical arc (%d, %d), (%d, %d), (%d, %d), (%d,%d), (%d, %d)\n",
	    pt_array[0], pt_array[1], pt_array[2], 
	    pt_array[3], pt_array[4], pt_array[5],
	    vec_array[0], vec_array[1], vec_array[2], vec_array[3]);
	/* might emulate */
	ret = (dev_func) ? (*dev_func) (pt_array, vec_array) : 
	    em_ell_arc(pt_array, vec_array);

	return(ret);
}
/* set an elliptical arc, close it */
static do_e_a_close(dat_ptr, dev_func)
unsigned char *dat_ptr;
int(*dev_func)();
{
int ret = 1, step_size, pt_array[6], vec_array[4], i;
enum boolean chord;
extern em_e_a_close();	/* in emul.c */

	step_size = (vdc_step / byte_size);
	for (i=0; i<6; ++i) {
	    pt_array[i] = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	}
	for (i=0; i<4; ++i) {
	    vec_array[i] = (*cgm_vdc)(dat_ptr);
	    dat_ptr += step_size;
	}
	chord = (enum boolean) cgm_eint(dat_ptr);

	may_list(stderr,
"elliptical arc close (%d, %d), (%d, %d), (%d, %d), (%d,%d), (%d, %d), %d\n",
	    pt_array[0], pt_array[1], pt_array[2], 
	    pt_array[3], pt_array[4], pt_array[5],
	    vec_array[0], vec_array[1], vec_array[2], vec_array[3], 
	    (int) chord);
	/* might emulate */
	ret = (dev_func) ? (*dev_func) (pt_array, vec_array, chord) : 
	    em_e_a_close(pt_array, vec_array, chord);

	return(ret);
}

/* now the class5 functions */
/* set the line bundle index */
static s_lbindex(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_lbindex;
	new_lbindex = cgm_xint(dat_ptr);
	may_list(stderr, "changing line bundle index from %d to %d\n",
	    (int) a5->l_b_index, new_lbindex);
	a5->l_b_index = new_lbindex;

	if (dev_func) ret = (*dev_func) (a5->l_b_index);

	return(ret);
}
/* set the line type */
static s_l_type(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_l_type;
	new_l_type = cgm_xint(dat_ptr);
	may_list(stderr, "changing line type from %d to %d\n",
	    (int) a5->line_type, new_l_type);
	a5->line_type = (enum line_enum) new_l_type;

	if (dev_func) ret = (*dev_func) (a5->line_type);

	return(ret);
}
/* set the line width */
static s_l_width(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
float rmul;
int new_l_width;
	
	if (!dev_info->d_l_width) dev_info->d_l_width = 1;	/* safety */

	if (a2->l_w_s_mode == absolute) {
	    new_l_width =  (*cgm_vdc)(dat_ptr) + 0.5;
	    may_list(stderr, "changing abs line width from %d to %d\n",
	    	a5->line_width.i, new_l_width);
	    a5->line_width.i = new_l_width;
	} else if (a2->l_w_s_mode == scaled) {
	    rmul = cgm_real(dat_ptr);
	    may_list(stderr, 
		"changing rel line width from %.4f to %.4f\n",
	    	a5->line_width.r, rmul);
	    a5->line_width.r = rmul;
	    a5->line_width.i = rmul * dev_info->d_l_width;
	} else burp(stderr, "illegal line spec mode = %d\n", a2->l_w_s_mode);

	if (dev_func) ret = (*dev_func) 
	    (a5->line_width.i, (float) a5->line_width.i  / dev_info->d_l_width);

	return(ret);
}
/* set the line colour */
static s_l_colour(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_index = -1, step_size, ir, ig, ib;
float r, g, b, *rptr;
	switch ((int) a2->c_s_mode) {
case (int) i_c_mode: /* indexed mode */
	    new_index = cgm_cxint(dat_ptr);
	    if (list_cgm) 
		burp(stderr, "changing line index from %d to %d\n",
		    a5->line_colour.ind, new_index);
	    a5->line_colour.ind = new_index;
	    rptr = a5->ctab + a5->line_colour.ind * 3;
	    r = *rptr;
	    g = *++rptr;
	    b = *++rptr;
	    break;
case (int) d_c_mode: /* direct mode */
	    step_size = (glbl1->col_prec / byte_size);
	    ir = cgm_dcint(dat_ptr);
	    r = dcr(ir);
	    dat_ptr += step_size;
	    ig = cgm_dcint(dat_ptr);
	    g = dcg(ig);
	    dat_ptr += step_size;
	    ib = cgm_dcint(dat_ptr);
	    b = dcb(ib);
	    may_list(stderr, 
		"changing line colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
		a5->line_colour.red, a5->line_colour.green, a5->line_colour.blue,
		r, g, b);
	    a5->line_colour.red = r;
	    a5->line_colour.green = g;
	    a5->line_colour.blue = b;
	    break;
	default:
	    burp(stderr, "illegal colour mode = %d\n", a2->c_s_mode);
	}
	if (dev_func) ret = (*dev_func) (r, g, b, new_index);
	return(ret);
}
/* set the marker bundle index */
static s_mbindex(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_index;
	new_index = cgm_xint(dat_ptr);
	may_list(stderr, "changing marker bundle index from %d to %d\n",
	    (int) a5->mk_b_index, new_index);
	a5->mk_b_index = new_index;

	if (dev_func) ret = (*dev_func) (a5->mk_b_index);

	return(ret);
}
/* set the marker type */
static s_mk_type(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_mk_type;
	new_mk_type = cgm_xint(dat_ptr);
	may_list(stderr, "changing marker type from %d to %d\n",
	    	 a5->mk_type, new_mk_type);
	a5->mk_type = new_mk_type;
	if (dev_func) ret = (*dev_func) (a5->mk_type);
	return(ret);
}
/* set the marker size */
static s_mk_size(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
float rmul;
int new_mk_size;

	if (!dev_info->d_m_size) dev_info->d_m_size = 1;	/* safety */

	if (a2->m_s_s_mode == absolute) {
	    new_mk_size = (*cgm_vdc)(dat_ptr);
	    may_list(stderr, "changing marker size from %d to %d\n",
	    	a5->mk_size.i, new_mk_size);
	    a5->mk_size.i = new_mk_size;
	} else if (a2->m_s_s_mode == scaled) {
	    rmul = cgm_real(dat_ptr);
	    may_list(stderr, 
		"changing rel mkr size from %.4f to %.4f\n",
	    	a5->mk_size.r, rmul);
	    a5->mk_size.r = rmul;
	    a5->mk_size.i = rmul * dev_info->d_m_size;
	} else burp(stderr, "illegal marker size mode = %d\n", a2->m_s_s_mode);

	if (dev_func) ret = (*dev_func) 
	    (a5->mk_size.i, (float) a5->mk_size.i / dev_info->d_m_size);

	return(ret);
}
/* set the marker colour */
static s_mk_colour(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_index = -1, step_size, ir, ig, ib;
float r, g, b, *rptr;

	
	switch ((int) a2->c_s_mode) {
case (int) i_c_mode : /* indexed mode */
	    new_index = cgm_cxint(dat_ptr);
	    if (list_cgm) 
		burp(stderr, "changing marker index from %d to %d\n",
		    new_index, a5->mk_colour.ind);
	    a5->mk_colour.ind = new_index;
	    rptr = a5->ctab + a5->mk_colour.ind * 3;
	    r = *rptr;
	    g = *++rptr;
	    b = *++rptr;
	    break;
case (int) d_c_mode : /* direct mode */
	    step_size = (glbl1->col_prec / byte_size);
	    ir = cgm_dcint(dat_ptr);
	    r = dcr(ir);
	    dat_ptr += step_size;
	    ig = cgm_dcint(dat_ptr);
	    g = dcg(ig);
	    dat_ptr += step_size;
	    ib = cgm_dcint(dat_ptr);
	    b = dcb(ib);
	    may_list(stderr, 
		"changing mk colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
		a5->mk_colour.red, a5->mk_colour.green, a5->mk_colour.blue,
		r, g, b);
	    a5->mk_colour.red = r;
	    a5->mk_colour.green = g;
	    a5->mk_colour.blue = b;
	    break;
	default:
	    burp(stderr, "illegal colour mode = %d\n", a2->c_s_mode);
	}
	if (dev_func) ret = (*dev_func) (r, g, b, new_index);
	return(ret);
}
/* set the text bundle index */
static s_tbindex(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_index;
	new_index = cgm_xint(dat_ptr);
	may_list(stderr, "changing text bundle index from %d to %d\n",
	    (int) a5->t_b_index, new_index);
	a5->t_b_index = new_index;

	if (dev_func) ret = (*dev_func) (a5->t_b_index);

	return(ret);
}
/* set the text font index */
static s_t_index(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_font_index;
	new_font_index = cgm_xint(dat_ptr);
	may_list(stderr, "changing font index from %d to %d\n",
	    a5->t_f_index, new_font_index);
	a5->t_f_index = new_font_index;
	if (dev_func) ret = (*dev_func) (new_font_index);	
	return(ret);
}
/* set the text precision */
static s_t_prec(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_t_prec;
	new_t_prec = cgm_eint(dat_ptr);	
	may_list(stderr, "changing text precision from %d to %d\n",
	    (int) a5->t_prec, new_t_prec);
	a5->t_prec = (enum txt_enum) new_t_prec;
	if (dev_func) ret = (*dev_func) (a5->t_prec);
	return(ret);
}
/* set the character expansion factor */
static s_c_exp(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
float new_c_exp_fac;
	new_c_exp_fac = cgm_real(dat_ptr);
	may_list(stderr, 
	    "changing character expansion factor from %.4f to %.4f\n",
	    a5->c_exp_fac, new_c_exp_fac);
	a5->c_exp_fac = new_c_exp_fac;
	if (dev_func) ret = (*dev_func) (new_c_exp_fac);
	return(ret);
}
/* set the character space */
static s_c_space(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
float new_c_space;
	new_c_space = cgm_real(dat_ptr);
	may_list(stderr, "changing character space %f to %f\n",
	    a5->c_space, new_c_space);
	a5->c_space = new_c_space;
	if (dev_func) ret = (*dev_func) (a5->c_space);
	return(ret);
}
/* set the text colour */
static s_t_colour(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_index = -1, step_size, ir, ig, ib;
float r, g, b, *rptr;

	switch ((int) a2->c_s_mode) {
	case 0: /* indexed mode */
	    new_index = cgm_cxint(dat_ptr);
	    if (list_cgm) burp(stderr, "changing text colour index from %d to %d\n",
		    a5->text_colour.ind, new_index);
	    a5->text_colour.ind = new_index;
	    rptr = a5->ctab + a5->text_colour.ind * 3;
	    r = *rptr;
	    g = *++rptr;
	    b = *++rptr;
	    break;
	case 1: /* direct mode */
	    step_size = (glbl1->col_prec / byte_size);
	    ir = cgm_dcint(dat_ptr);
	    r = dcr(ir);
	    dat_ptr += step_size;
	    ig = cgm_dcint(dat_ptr);
	    g = dcg(ig);
	    dat_ptr += step_size;
	    ib = cgm_dcint(dat_ptr);
	    b = dcb(ib);
	    may_list(stderr, 
		"changing text colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
		a5->text_colour.red, a5->text_colour.green, a5->text_colour.blue,
		r, g, b);
	    a5->text_colour.red = r;
	    a5->text_colour.green = g;
	    a5->text_colour.blue = b;
	    break;
	default:
	    burp(stderr, "illegal colour mode = %d\n", a2->c_s_mode);
	}
	
	if (dev_func) ret = (*dev_func) (r, g, b, new_index);

	return(ret);
}
/* set character height */
static s_c_height(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_height, dev_height;

	new_height =  (*cgm_vdc)(dat_ptr) + 0.5;
	if (opt[(int) text_mag].set) new_height *= opt[(int) text_mag].val.r;
	may_list(stderr, "changing character height from %d to %d\n",
	    a5->c_height, new_height);
	a5->c_height = new_height;
	if (dev_func) ret = (*dev_func) (a5->c_height);
	return(ret);
}
/* set the character orientation structure */
static s_c_orient(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
struct orient_struct new_orient;
int step_size;

	step_size = (vdc_step / byte_size);

	new_orient.x_up = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	new_orient.y_up = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	new_orient.x_base = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	new_orient.y_base = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;

	may_list(stderr, 
	    "changing orientation from (%d,%d,%d,%d) to (%d,%d,%d,%d)\n",
	    a5->c_orient.x_up,a5->c_orient.y_up,a5->c_orient.x_base,
	    a5->c_orient.y_base,
	    new_orient.x_up,new_orient.y_up,new_orient.x_base,new_orient.y_base);

	a5->c_orient.x_up = new_orient.x_up * sy;
	a5->c_orient.y_up = new_orient.y_up * sy;
	a5->c_orient.x_base = new_orient.x_base * sx;
	a5->c_orient.y_base = new_orient.y_base * sx;
	if (dev_func) ret = (*dev_func) (new_orient.x_up, 
	    new_orient.y_up, new_orient.x_base, new_orient.y_base);
	return(ret);
}
/* set the text path */
static s_tpath(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
enum path_enum new_path;

	new_path = (enum path_enum) cgm_eint(dat_ptr);
	may_list(stderr, "changing text path from %d to %d\n",
	    (int) a5->text_path, (int) new_path);
	a5->text_path = new_path;
	if (dev_func) ret = (*dev_func) (new_path);
	return(ret);
}
/* set the text alignment */
static s_t_align(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int step_size;
struct align_struct new_align;

	step_size = e_size;
	new_align.hor = (enum hor_align) cgm_eint(dat_ptr);
	dat_ptr += step_size;

	new_align.ver = (enum ver_align) cgm_eint(dat_ptr);
	dat_ptr += step_size;

	step_size = (glbl1->real_prec.exp + glbl1->real_prec.fract) 
	    / byte_size;
	new_align.cont_hor = cgm_real(dat_ptr);
	dat_ptr += step_size;
	new_align.cont_ver = cgm_real(dat_ptr);
	dat_ptr += step_size;

	may_list(stderr, 
	    "changing text alignment from (%d,%d,%f,%f) to (%d,%d,%f,%f)\n",
	    a5->text_align.hor, a5->text_align.ver, a5->text_align.cont_hor,
	    a5->text_align.cont_ver, new_align.hor, new_align.ver,
	    new_align.cont_hor, new_align.cont_ver);
	a5->text_align.hor = new_align.hor;
	a5->text_align.ver = new_align.ver;
	a5->text_align.cont_hor = new_align.cont_hor;
	a5->text_align.cont_ver = new_align.cont_ver;

	if (dev_func) ret = (*dev_func) (a5->text_align.hor, 
	    a5->text_align.ver, a5->text_align.cont_hor, 
	    a5->text_align.cont_ver);

	return(ret);
}
/* set the character set index */
static s_csindex(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1, new_index;

	new_index = cgm_xint(dat_ptr);
	may_list(stderr, "changing character set index from %d to %d\n",
	    a5->c_set_index, new_index);
	a5->c_set_index = new_index;
	if (dev_func) ret = (*dev_func) (new_index);
	return(ret);
}
/* set the alternate character set index */
static s_acsindex(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1, new_index;

	new_index = cgm_xint(dat_ptr);
	may_list(stderr, 
	    "changing alternate character set index from %d to %d\n",
	    a5->a_c_set_index, new_index);
	a5->a_c_set_index = new_index;
	if (dev_func) ret = (*dev_func) (new_index);
	return(ret);
}
/* set the fill bundle index */
static s_fbindex(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1, new_index;

	new_index = cgm_xint(dat_ptr);
	may_list(stderr, "changing fill bundle index from %d to %d\n",
	    a5->f_b_index, new_index);
	a5->f_b_index = new_index;
	if (dev_func) ret = (*dev_func) (new_index);
	return(ret);
}

/* set the interior style */
static s_interior_style(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_style;
	new_style = cgm_eint(dat_ptr);
	may_list(stderr, "changing interior style from %d to %d\n",
	    	(int) a5->int_style, new_style);
	a5->int_style = (enum is_enum) new_style;
	if (dev_func) ret = (*dev_func)(a5->int_style);
	return(ret);
}
/* set the fill colour */
static s_fill_colour(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_index = -1, step_size, ir, ig, ib;
float r, g, b, *rptr;


	switch ((int) a2->c_s_mode) {
case (int) i_c_mode: /* indexed mode */
	    new_index = cgm_cxint(dat_ptr);
	    may_list(stderr, "changing fill index from %d to %d\n",
		    a5->fill_colour.ind, new_index);
	    a5->fill_colour.ind = new_index;
	    rptr = a5->ctab + a5->fill_colour.ind * 3;
	    r = *rptr;
	    g = *++rptr;
	    b = *++rptr;
	    break;
case (int) d_c_mode: /* direct mode */
	    step_size = (glbl1->col_prec / byte_size);
	    ir = cgm_dcint(dat_ptr);
	    r = dcr(ir);
	    dat_ptr += step_size;
	    ig = cgm_dcint(dat_ptr);
	    g = dcg(ig);
	    dat_ptr += step_size;
	    ib = cgm_dcint(dat_ptr);
	    b = dcb(ib);
	    may_list(stderr, 
		"changing fill colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
		a5->fill_colour.red, a5->fill_colour.green, a5->fill_colour.blue,
		r, g, b);
	    a5->fill_colour.red = r;
	    a5->fill_colour.green = g;
	    a5->fill_colour.blue = b;
	    break;
	default:
	    burp(stderr, "illegal colour mode = %d\n", a2->c_s_mode);
	}
	
	if (dev_func) ret = (*dev_func) (r, g, b, new_index);
	return(ret);
}
/* set the hatch index */
static s_hindex(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1, new_index;

	new_index = cgm_xint(dat_ptr);
	may_list(stderr, "changing hatch index from %d to %d\n",
	    a5->hatch_index, new_index);
	a5->hatch_index = new_index;
	if (dev_func) ret = (*dev_func) (new_index);
	return(ret);
}
/* set the pattern index */
static s_pindex(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1, new_index;

	new_index = cgm_xint(dat_ptr);
	may_list(stderr, "changing pattern index from %d to %d\n",
	    a5->pat_index, new_index);
	a5->pat_index = new_index;
	if (dev_func) ret = (*dev_func) (new_index);
	return(ret);
}
/* set the edge bundle index */
static s_e_b_index(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1, new_index;

	new_index = cgm_xint(dat_ptr);
	may_list(stderr, "changing edge bundle index from %d to %d\n",
	    a5->e_b_index, new_index);
	a5->e_b_index = new_index;
	if (dev_func) ret = (*dev_func) (new_index);
	return(ret);
}

/* set the edge type */
static s_edge_t(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_l_type;
	new_l_type = cgm_sint(dat_ptr);
	may_list(stderr, "changing edge type from %d to %d\n",
	    (int) a5->edge_type, new_l_type);
	a5->edge_type = (enum line_enum) new_l_type;

	if (dev_func) ret = (*dev_func) (a5->edge_type);

	return(ret);
}

/* set the edge width */
static s_edge_w(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_flag;
float rmul;
int new_e_width;
	
	if (!dev_info->d_e_width) dev_info->d_e_width = 1;	/* safety */

	if (a2->e_w_s_mode == absolute) {
	    new_e_width = (*cgm_vdc)(dat_ptr) + 0.5;
	    may_list(stderr, "changing abs edge width from %d to %d\n",
	    	a5->edge_width.i, new_e_width);
	    a5->edge_width.i = new_e_width;
	} else if (a2->e_w_s_mode == scaled) {
	    rmul = cgm_real(dat_ptr);
	    may_list(stderr, 
		"changing rel edge width from %.4f to %.4f\n",
	    	a5->edge_width.r, rmul);
	    a5->edge_width.r = rmul;
	    a5->edge_width.i = rmul * dev_info->d_e_width;
	} else burp(stderr, "illegal edge spec mode = %d\n", a2->e_w_s_mode);

	if (dev_func) ret = (*dev_func) 
	    (a5->edge_width.i, (float) a5->edge_width.i / dev_info->d_e_width);

	return(ret);
}
/* set the edge colour */
static s_edge_c(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_index = -1, step_size, ir, ig, ib;
float r, g, b, *rptr;
	switch ((int) a2->c_s_mode) {
case (int) i_c_mode : /* indexed mode */
	    new_index = cgm_cxint(dat_ptr);
	    if (list_cgm) 
		burp(stderr, "changing edge index from %d to %d\n",
		    a5->edge_colour.ind, new_index);
	    a5->edge_colour.ind = new_index;
	    rptr = a5->ctab + a5->line_colour.ind * 3;
	    r = *rptr;
	    g = *++rptr;
	    b = *++rptr;
	    break;
case (int) d_c_mode : /* direct mode */
	    step_size = (glbl1->col_prec / byte_size);
	    ir = cgm_dcint(dat_ptr);
	    ir = dcr(ir);
	    dat_ptr += step_size;
	    ig = cgm_dcint(dat_ptr);
	    g = dcg(ig);
	    dat_ptr += step_size;
	    ib = cgm_dcint(dat_ptr);
	    b = dcb(ib);
	    may_list(stderr, 
		"changing edge colour from (%.4f, %.4f, %.4f) to (%.4f, %.4f, %.4f)\n",
		a5->edge_colour.red, a5->edge_colour.green, a5->edge_colour.blue,
		r, g, b);
	    a5->edge_colour.red = r;
	    a5->edge_colour.green = g;
	    a5->edge_colour.blue = b;
	    break;
	default:
	    burp(stderr, "illegal colour mode = %d\n", a2->c_s_mode);
	}
	if (dev_func) ret = (*dev_func) (r, g, b, new_index);
	return(ret);
}
/* set the edge visibility */
static s_edge_v(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_flag;
	new_flag = cgm_eint(dat_ptr);
	may_list(stderr, "changing edge visibility from %d to %d\n",
	    (int) a5->edge_vis, new_flag);
	a5->edge_vis = (enum boolean) new_flag;
	if (dev_func) ret = (*dev_func)(a5->edge_vis);
	return(ret);
}

/* set the fill reference point */
static s_fill_ref(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_fill[2], i, step_size, x, y;

	step_size = (vdc_step / byte_size);
	x = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;
	y = (*cgm_vdc)(dat_ptr);
	dat_ptr += step_size;

	new_fill[0] = newx(x, y);
	new_fill[1] = newy(x, y);

	may_list(stderr, "setting fill ref pt to (%d,%d)\n",
	    	new_fill[0], new_fill[1]);
	for (i=0;i<2;++i) a5->fill_ref.i[i] = new_fill[i];
	if (dev_func) ret = (*dev_func)(a5->fill_ref.i[0], a5->fill_ref.i[1]);
	return(ret);
}
/* make a pattern table entry */
/* not really implemented yet */
static p_tab_entry(dat_ptr, p_len, dev_func)
int p_len;
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int step_size, index, nx, ny, col_prec, no_bytes, l_col_prec;
	
	step_size = glbl1->ind_prec / byte_size;
	index = cgm_xint(dat_ptr);
	dat_ptr += step_size;
	step_size = glbl1->int_prec / byte_size;
	nx = cgm_sint(dat_ptr);
	dat_ptr += step_size;
	ny = cgm_sint(dat_ptr);
	dat_ptr += step_size;
	col_prec = cgm_sint(dat_ptr);
	dat_ptr += step_size;
	if (a2->c_s_mode == d_c_mode) { 	/* direct */
	    l_col_prec = (col_prec) ? col_prec : glbl1->col_prec;
	} else {		/* indexed */
	    l_col_prec = (col_prec) ? col_prec : glbl1->col_i_prec;
	}

	no_bytes = nx * ny * l_col_prec / byte_size;
	may_list(stderr, "pattern table entry (%d), (%d, %d), %d, %d\n",
	    index, nx, ny, col_prec, no_bytes);

	if (dev_func) ret = (*dev_func) (index, nx, ny, col_prec, dat_ptr,
	    no_bytes);
	return(ret);
}

/* set the pattern size */
static s_pat_size(dat_ptr, dev_func)
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int new_size[4], i, step_size;

	step_size = (vdc_step / byte_size);
	for (i=0;i<4;++i) {
	    new_size[i] = (*cgm_vdc)(dat_ptr) + 0.5;
	    dat_ptr += step_size;
	}
	may_list(stderr, 
	    "changing pattern size from (%d,%d,%d,%d) to (%d,%d,%d,%d)\n",
	    a5->pat_size.i[0], a5->pat_size.i[1], a5->pat_size.i[2], 
	    a5->pat_size.i[3],
	    new_size[0], new_size[1], new_size[2], new_size[3]);
	for (i=0;i<4;++i) a5->pat_size.i[i] = new_size[i];
	if (dev_func) ret = (*dev_func)(a5->pat_size.i);
	return(ret);
}
/* make a colour table entry */
static c_tab_entry(dat_ptr, p_len, dev_func)
int p_len;
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
int step1_size, step2_size, beg_index, i, no_entries, col[3], j,
max_index, do_mcind();
float rcol[3];

	may_list(stderr, "colour table entry\n");
	step1_size = (glbl1->col_i_prec / byte_size);	
	step2_size = (glbl1->col_prec / byte_size);	
	beg_index = cgm_cxint(dat_ptr);
	dat_ptr += step1_size;
	no_entries = (p_len - step1_size) / (3 * step2_size);
	max_index = beg_index + no_entries - 1;
	if (max_index > glbl1->max_c_index) {
	    if (get_all) {
		burp(stderr, 
		"illegally many entries, %d > max_col_index = %d,",
	    	max_index, glbl1->max_c_index);
		burp(stderr, "will change max_col_index\n");
	    }
	    if (!do_mcind(max_index)) burp(stderr, "trouble setting mcind\n");
	}
	may_list(stderr, "beginning index = %d, %d entries\n", 
	    beg_index, no_entries);
	
	for (i=0; i<no_entries; ++i) {
	    for (j=0; j<3; ++j) {
		col[j] = cgm_dcint(dat_ptr);
		dat_ptr += step2_size;
		rcol[j] = dcind(j, col[j], glbl1->c_v_extent);
	    }
	    may_list(stderr, "(%d, %d, %d) = (%.4f, %.4f, %.4f)\n", 
		col[0], col[1], col[2], rcol[0], rcol[1], rcol[2]);
	    for (j=0; j<3; ++j) 
		*(a5->ctab + 3 * (beg_index + i) + j) = rcol[j];
	}
	if (dev_func) ret = (*dev_func) (beg_index, no_entries, a5->ctab);
	return(ret);
}
/* take care of the asps flags */
/* not really implemented yet */
static do_aspsflags(dat_ptr, p_len, dev_func)
int p_len;
int (*dev_func)();
unsigned char *dat_ptr;
{
int ret = 1;
#define max_pairs 18	/* maximum no of pairs */
int no_pairs, flags[max_pairs * 2], i;

	no_pairs = p_len / (2 * e_size);
	if (no_pairs > max_pairs) {
	    (void) fprintf(stderr, "illegal no_pairs in aspsflags %d > %d\n",
		no_pairs, max_pairs);
	    no_pairs = max_pairs;
	}
	may_list(stderr, "setting asps flags\n");
	for (i=0; i<(no_pairs * 2); ++i) {
	    flags[i] = cgm_eint(dat_ptr);
	    dat_ptr += e_size;
	    may_list(stderr, ", %d", flags[i]);
	}
	if (dev_func) ret = (*dev_func) (no_pairs, flags);
	return(ret);
}
/* now the class6 functions */
/* do the special command */
static do_escape(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int ret = 1;
int escape_id, step_size;
char *str_ptr;

	escape_id = cgm_sint(dat_ptr);
	step_size = (glbl1->int_prec / byte_size);		
	dat_ptr += step_size;
	str_ptr = cgm_str(dat_ptr, NULL);
	if ((escape_id == INDEX_FLAG) ||
	    (escape_id == -1)){	/* our index format */
	    if (!index_read) {
		if (!opt[(int) nindex].set) read_index(str_ptr);
	    }
	} else {
	    may_list(stderr, "escape id = %d, string = [%s]\n",
	    	escape_id, str_ptr);
	    if (dev_func) ret = (*dev_func) (escape_id, str_ptr);
	}

	return(ret);
}
/* process an index block here */
static read_index(str_ptr)
unsigned char *str_ptr;
{
int i, j, no_pages, page_no;
long int byte_ad, next_index;
struct p_struct *this_page, *temp_page = NULL, *last_page;
struct ad_struct next_ad;
#define have_number(c) (((c)=='-')||((c)=='+')||(('0'<=(c))&&((c)<='9')))

	if (index_present == 0) {
	    index_present = 1;
	    if (get_all) burp(stderr, "indexed file\n");
	    if (!random_input) 
		burp(stderr, "indexed file, but no random access\n");
	    /* and mark where we are */
	    first_index.r_ad = last_ad.r_ad;
	    first_index.b_ad = last_ad.b_ad;
	    first_index.offset = last_ad.offset;
	}
	may_list(stderr, "index block:\n%s\n", str_ptr);
	while (*str_ptr == ' ') ++str_ptr;
	sscanf(str_ptr, "%d", &no_pages);
	while have_number(*str_ptr) ++str_ptr;
	for (i=0; i < no_pages; ++i) {
	    if (this_page =  (struct p_struct *) allocate_mem(p_s_size, 0)){
		if (i==0) temp_page = this_page;
		if (p_header.first == NULL){ 	/* first time */
		    p_header.first = this_page;
		    this_page->no = 1;
		} else {
		    this_page->no = p_header.last->no + 1;
		    p_header.last->next = this_page;
		}
		p_header.last = this_page;
		this_page->next = NULL;
		while (*str_ptr == ' ') ++str_ptr;
		sscanf(str_ptr, "%ld", &byte_ad);
		if (!fill_ad(&(this_page->ad), &last_ad, byte_ad)) {
		    (void) fprintf(stderr, "illegal offset %ld\n", byte_ad);
		}
		while have_number(*str_ptr) ++str_ptr;
		while (*str_ptr == ' ') ++str_ptr;
		sscanf(str_ptr, "%d", &this_page->len);
		while have_number(*str_ptr) ++str_ptr;

/* now take care of the string */
		++str_ptr;		/* one-char gap */
		if ((this_page->len > 0) && 
			(this_page->str = (char *) 
			    allocate_mem(this_page->len, 0)))
		    for (j=0; j<this_page->len; ++j)
			this_page->str[j] = *str_ptr++;
	    } else fprintf(stderr, "couldn't allocate this_page !\n");
	}
/* read the indices */
	if (1 != sscanf(str_ptr, " %ld", &next_index))
	    fprintf(stderr, "couldn't read next index\n");
	if (get_all) {
	    this_page = temp_page;
	    while (this_page != NULL) {
	    	fprintf(stderr, "%d %d %ld %d ", this_page->ad.r_ad, 
		    this_page->ad.offset, this_page->ad.b_ad, this_page->len);
	    	for (i=0; i<this_page->len; ++i) fputc(this_page->str[i], 
		    stderr);
	    	fputc('\n', stderr);
	    	this_page = this_page->next;
	    }
	    fprintf(stderr, "next_index = %ld\n", next_index);
	}
 	if (use_random && next_index) {
	    /* want to jump to next block */
	    /* last_ad holds the current command address */
	    if (fill_ad(&next_ad, &last_ad, next_index)) {
		if (use_random) cgm_goto(&next_ad);
	    } else burp(stderr, "trouble with fill_ad\n");
	}
	if (next_index == 0) /* all done */ index_read = 1;
 	if (next_index) return(1); 
/* have read last index so may show the data */
	/* may trim the list of pages */
	/* first trim by page numbers */
	if (opt[(int) pages].set) {
	    page_no = 1;
	    this_page = p_header.first;
	    /* get first wanted page */
	    while ((this_page) && 
		(!want_page(page_no, opt[(int) pages].val.str))) {
		++page_no;
		this_page = this_page->next;
	    }
	    last_page = p_header.first = this_page;
	    /* now trim off unwanted pages */
	    while (this_page) {
		this_page = this_page->next;
		++page_no;
		if (this_page && want_page(page_no, opt[(int) pages].val.str)){
		    last_page->next = this_page;
		    last_page = this_page;
		}
	    }
	    if (last_page) last_page->next = NULL;
	}
	/* now trim by page title */
	if (opt[(int) title_string].set) {
	    this_page = p_header.first;
	    /* get first wanted page */
	    while ((this_page) && 
		(!want_title(opt[(int) title_string].val.str, 
		    this_page->str, this_page->len))) {
		this_page = this_page->next;
	    }
	    last_page = p_header.first = this_page;
	    /* now trim off unwanted pages */
	    while (this_page) {
		this_page = this_page->next;
		if (this_page && want_title(opt[(int) title_string].val.str, 
		    this_page->str, this_page->len)){
		    last_page->next = this_page;
		    last_page = this_page;
		}
	    }
	    if (last_page) last_page->next = NULL;
	}
	/* now show the pages */
	if (opt[(int) index_file].set) {
	    this_page = p_header.first;
	    while (this_page != NULL) {
	    	fprintf(stderr, "page %d ", this_page->no);
#ifdef VMS
	    	fprintf(stderr, "record %d, offset %d", this_page->ad.r_ad,
		    this_page->ad.offset);
#else
	        fprintf(stderr, "byte address %ld", this_page->ad.b_ad);
#endif
	        if (this_page->len > 0) {
		    fputc(' ', stderr);
		    fputc('[', stderr);
		    for (j=0; j<this_page->len; ++j) 
		    	fputc(this_page->str[j], stderr);
		    fputc(']', stderr);
	        }
	        fputc('\n', stderr);
	        this_page = this_page->next;
	    }
	}
 	if ((!opt[(int) device].set) && (!opt[(int) screen].set) &&
	    (!opt[(int) list].set)) /*  no output */ 
		return(f_e_mf	(delim[(int) E_Mf], ctrl[(int) E_Mf])); 
	/* else go back to where we started */
	if (use_random) cgm_goto(&first_index);
	return(1);
}
/* figure out if the input string is included in the title */
want_title(in_str, long_str, str_len)
char *in_str, *long_str; 	/* input and title string */
int str_len;			/* length of title string */
{
int in_len, first_char, match = -1, i;
	
	if (str_len <= 0) return(0);	/* nothing to do */

	in_len = strlen(in_str);	/* input length */
	if (in_len <= 0) return(0);	/* nothing to do */
	if (in_len > str_len) return(0);	/* doesn't fit */

	first_char = 0;			/* start at beginning */

	while (match < 0) {
	    /* find next possibility */
	    while ((first_char + in_len <= str_len) && 
		(long_str[first_char] != *in_str)) ++first_char;
	    if (first_char + in_len > str_len) match = 0;
	    else {
		for (i=0; (i<in_len) && (in_str[i]==long_str[first_char + i]);
		    ++i);
	        if (i>=in_len) match = 1;
	        else ++first_char;
	    }
	}
	return(match);
}
/* now the class 7 functions */
/* message command */
static do_message(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int ret = 1;
char *str_ptr;
enum boolean action;

	action = (enum boolean) cgm_eint(dat_ptr);
	dat_ptr += e_size;
	str_ptr = cgm_str(dat_ptr, NULL);
	may_list(stderr, "message, action = %d, string = [%s]\n",
	    (int) action, str_ptr);

	if (dev_func) ret = (*dev_func) (action, str_ptr);

	return(ret);
}
/* application data */
static do_apdata(dat_ptr, dev_func)
unsigned char *dat_ptr;
int (*dev_func)();
{
int ret = 1;
char *str_ptr;
int id, step_size, i, j;

	id = cgm_sint(dat_ptr);
	step_size = (glbl1->int_prec / byte_size);		
	dat_ptr += step_size;
	str_ptr = cgm_str(dat_ptr, NULL);
	may_list(stderr, "apdata, id = %d, string = [%s]\n",
	    id, str_ptr);
	if (dev_func) ret = (*dev_func) (id, str_ptr);

	return(ret);
}
