/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997	Tim Janik	<timj@psynet.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"rcs.h"
RCS_ID("$Id: gsi.c,v 1.2 1997/08/19 01:34:12 timj Exp $")


#define		__gsi_c__

#include	"gsi.h"
#include	<stdio.h>

#ifdef	GUBI
	#include	"defines.h"
	#include	"widdata.h"
#endif	/*GUBI*/


/* --- defines --- */
#define	GSI_FIELD_VALUE_U(struct_begin,offset_union)	\
	((gsi_value_U*)((guchar*)struct_begin+((offset_union).pos)))



/* --- variables --- */
static	GHashTable	*gsi_struct_info_hash_table=NULL;
static	GHashTable	*gsi_struct_info_by_name_hash_table=NULL;
static	GsiReferenceFunc *gsi_reference_func_p=NULL;



/* --- prototypes --- */
static	void		gsi_struct_info_list_foreach	(gpointer	key,
							 gpointer	value,
							 gpointer	user_data);
static	int		gsi_compare_struct_types	(const void	*p1,
							 const void	*p2);
static	int		gsi_compare_struct_names	(const void	*p1,
							 const void	*p2);
static	guint		gsi_hash_func		(gpointer		v);
static	gint		gsi_compare_func	(gpointer		v,
						 gpointer		v2);
static	gsi_value_U	gsi_field_get_default	(gsi_base_S		*struct_p,
						 const gsi_field_info_S	*field_info);
static	void		gsi_struct_zero_fields	(const gsi_struct_info_S *struct_info,
						 gsi_base_S		*struct_p);



/* --- functions --- */
guint
gsi_hash_func	(gpointer	v)
{
	return (guint) v;
}


gint
gsi_compare_func	(gpointer	v,
			 gpointer	v2)
{
	return v==v2;
}


void
gsi_initialize	(const gsi_struct_info_S	**info_array)
{
	if (gsi_struct_info_hash_table) {
		g_hash_table_destroy(gsi_struct_info_hash_table);
		g_hash_table_destroy(gsi_struct_info_by_name_hash_table);
	}

	gsi_struct_info_hash_table=g_hash_table_new(gsi_hash_func, gsi_compare_func);
	gsi_struct_info_by_name_hash_table=g_hash_table_new(g_string_hash, g_string_equal);
	
	if (info_array) {
		register guint	i;
		
		for (i=0; info_array[i]; i++)
			gsi_struct_info_add(info_array[i]);
	}
}


void
gsi_struct_info_add	(const gsi_struct_info_S	*struct_info)
{
	g_assert(struct_info);
	
	g_hash_table_insert	(gsi_struct_info_hash_table,
				 (gpointer)struct_info->struct_type,
				 (gpointer)struct_info);
	g_hash_table_insert	(gsi_struct_info_by_name_hash_table,
				 (gpointer)struct_info->struct_name,
				 (gpointer)struct_info);
}


void
gsi_struct_info_remove	(const gsi_struct_info_S	*struct_info)
{
	g_assert(struct_info);
	
	g_hash_table_remove	(gsi_struct_info_hash_table,
				 (gpointer)struct_info->struct_type);
	g_hash_table_remove	(gsi_struct_info_by_name_hash_table,
				 (gpointer)struct_info->struct_name);
}


void
gsi_set_reference_func	(GsiReferenceFunc	func_p)
{
	gsi_reference_func_p=func_p;
}


const gsi_struct_info_S*
gsi_struct_info		(const gsi_struct_type_E	struct_type)
{
	register const gsi_struct_info_S	*struct_info;
	
	struct_info=(const gsi_struct_info_S*) g_hash_table_lookup(
							gsi_struct_info_hash_table,
							(gpointer)struct_type);
	
	return struct_info;
}


void
gsi_struct_info_list_foreach	(gpointer	key,
				 gpointer	value,
				 gpointer	user_data)
{
	register GList	**list_p;
	
	list_p=user_data;
	
	*list_p=g_list_append(*list_p, value);
}


int	gsi_compare_struct_types(const void	*p1,
				 const void	*p2)
{
	register const gsi_struct_info_S	*const*struct_info_p1;
	register const gsi_struct_info_S	*const*struct_info_p2;
	
	struct_info_p1=p1;
	struct_info_p2=p2;
	
	if ((*struct_info_p1)->struct_type < (*struct_info_p2)->struct_type)
		return -1;
	return ((*struct_info_p1)->struct_type - (*struct_info_p2)->struct_type);
}


int	gsi_compare_struct_names(const void	*p1,
				 const void	*p2)
{
	register const gsi_struct_info_S	*const*struct_info_p1;
	register const gsi_struct_info_S	*const*struct_info_p2;
	
	struct_info_p1=p1;
	struct_info_p2=p2;
	return strcmp( (*struct_info_p1)->struct_name, (*struct_info_p2)->struct_name);
}


GList*
gsi_struct_info_list	(const gsi_sort_type_E	sort_type)
{
	static	 GList		*struct_list;
	
	struct_list=NULL;
	g_hash_table_foreach(gsi_struct_info_hash_table, gsi_struct_info_list_foreach, &struct_list);
	
	/* hmm, because the unique struct_type is used as hash value for
	 * gsi_struct_info_hash_table there seems to be no real need to
	 * actualy sort with gsi_compare_struct_types().
	 * but leave it like this, because i'm not too sure about the
	 * outcome of a g_hash_table_foreach().
	 * this way qsort() just sorts an already sorted array (hope
	 * qsort() is not implemented as brain damaged plain recursion,
	 * wich would cause Cn=n^2/2 in this case).
	*/
	
	if (sort_type!=GSI_SORT_NONE) {
		register guint		i;
		register gpointer	*array;
		register GList		*list;
		
		i=g_list_length(struct_list);
		array=g_new(gpointer, i);
		
		list=struct_list;
		i=0;
		while (list) {
			array[i++]=list->data;
			list=list->next;
		}
		
		switch (sort_type) {
		
		case	GSI_SORT_BY_NAME:
			qsort(array, i, sizeof(gpointer), gsi_compare_struct_names);
			break;
		
		case	GSI_SORT_BY_TYPE:
		default:
			qsort(array, i, sizeof(gpointer), gsi_compare_struct_types);
			break;
		}
		
		list=struct_list;
		i=0;
		while (list) {
			list->data=array[i++];
			list=list->next;
		}
		
		g_free(array);
	}
	
	return struct_list;
}


gboolean
gsi_struct_info_is_a	(const gsi_struct_info_S	*struct_info,
			 const gsi_struct_type_E	struct_type)
{
	while (struct_info) {
		if (struct_info->struct_type==struct_type)
			return TRUE;
		struct_info=struct_info->parent;
	}
	return FALSE;
}


guint
gsi_struct_info_n_parents	(const gsi_struct_info_S	*struct_info)
{
	register guint	i;
	
	i=0;
	while (struct_info) {
		struct_info=struct_info->parent;
		if (struct_info)
			i++;
	};
	
	return i;
}


const gsi_struct_info_S*
gsi_struct_info_by_name	(const gchar	*struct_name)
{
	register const gsi_struct_info_S	*struct_info;
	
	g_assert(struct_name);
	
	struct_info=(const gsi_struct_info_S*) g_hash_table_lookup(
							gsi_struct_info_by_name_hash_table,
							(gpointer)struct_name);
	
	return struct_info;
}


GList*
gsi_struct_field_list_from_to	(const gsi_struct_info_S	*struct_info,
				 const gsi_struct_type_E	from_struct_type,
				 const gsi_struct_type_E	to_struct_type)
{
	register GList		*list;
	register GList		*offset_list;
	register gboolean	add_fields;
	
	add_fields=FALSE;
	list=NULL;
	offset_list=NULL;
	while (struct_info) {
		register gint	n;
		
		if (!add_fields && struct_info->struct_type == from_struct_type)
			add_fields=TRUE;
		
		if (add_fields) {
			for (n = struct_info->n_fields - 1; n >= 0; n--) {
				if ( ! g_list_find(offset_list,
						   (gpointer) struct_info->fields[n].field_offset.pos) ) {
					list=g_list_prepend(list, (gpointer)&struct_info->fields[n]);
					offset_list=g_list_prepend(offset_list,
								   (gpointer)struct_info->fields[n].field_offset.pos);
				}
			}
		}
	
		if (struct_info->struct_type==to_struct_type)
			break;
		
		struct_info=struct_info->parent;
	}
	
	g_list_free(offset_list);
	
	return list;
}


GList*
gsi_struct_field_list	(const gsi_struct_info_S	*struct_info)
{
	return gsi_struct_field_list_from_to(struct_info, struct_info->struct_type, GB_STRUCT_NONE);
}


const guint
gsi_struct_n_fields	(const gsi_struct_info_S	*struct_info)
{
	register guint	num;
	
	num=0;
	while (struct_info) {
		num+=struct_info->n_fields;
		struct_info=struct_info->parent;
	}
	
	return num;
}


const gsi_field_info_S*
gsi_field_info	(const gsi_struct_info_S	*struct_info,
		 const guint			field_indx)
{
	register guint			max_fields;
	
	g_assert(struct_info);
	
	max_fields=gsi_struct_n_fields(struct_info);
	
	g_assert(field_indx < max_fields);
	
	while (struct_info) {
		if (max_fields-(field_indx+1) < struct_info->n_fields)
			return &struct_info->fields[struct_info->n_fields-(max_fields-field_indx)];
		
		max_fields=max_fields-struct_info->n_fields;
		struct_info=struct_info->parent;
	}
	
	g_assert_not_reached();
	
	return NULL;
}


const gsi_field_info_S*
gsi_field_info_by_name	(const gsi_struct_info_S	*struct_info,
			 const gchar			*field_name)
{
	g_assert(field_name);
	
	while (struct_info) {
		register guint i;
	
		for (i=0; i < struct_info->n_fields; i++)
			if (strcmp(field_name, struct_info->fields[i].field_name)==0)
				return &struct_info->fields[i];
		
		struct_info=struct_info->parent;
	}
	
	return NULL;
}


const gsi_field_info_S*
gsi_field_info_by_name2	(const gchar			*struct_name,
			 const gchar			*field_name)
{
	register const gsi_struct_info_S	*struct_info;
	
	
	struct_info=gsi_struct_info_by_name(struct_name);
	
	if (struct_info)
		return gsi_field_info_by_name(struct_info, field_name);
	
	return NULL;
}


gsi_value_U
gsi_field_get_default	(gsi_base_S		*struct_p,
			 const gsi_field_info_S	*field_info)
{
	register gsi_value_U	value;
	
	g_assert(field_info);
	
	switch (field_info->field_type) {
	case	GSI_FIELD_DOUBLE:
		value.v_double=field_info->v_default.v_double;
		break;
	
	case	GSI_FIELD_FLOAT:
		value.v_float=field_info->v_default.v_float;
		break;
	
	case	GSI_FIELD_LONG:
		value.v_long=field_info->v_default.v_long;
		break;
	
	case	GSI_FIELD_INT:
		value.v_int=field_info->v_default.v_int;
		break;
	
	case	GSI_FIELD_ENUM:
		value.v_enum=field_info->v_default.v_enum;
		break;
	
	case	GSI_FIELD_BITS:
		value.v_bits=field_info->v_default.v_bits;
		break;
	
	case	GSI_FIELD_BOOLEAN:
		value.v_boolean=field_info->v_default.v_boolean;
		break;
	
	case	GSI_FIELD_STRING:
		value.v_string=field_info->v_default.v_string;
		if (GSI_FIELD_OPT(field_info, GSI_OPT_STRING_FUNCTION) &&
		    field_info->field_func) {
			register GsiStringFieldFunc	*string_func;
			
			string_func=(GsiStringFieldFunc*)field_info->field_func;
			value.v_string=(*string_func)	(struct_p,
							 &(GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_string));
		}
		if (!value.v_string && GSI_FIELD_OPT(field_info, GSI_OPT_STRING_NONNULL))
			value.v_string="";
		break;
	
	case	GSI_FIELD_POINTER:
		value.v_pointer=field_info->v_default.v_pointer;
		break;
	
	case	GSI_FIELD_STRUCT_LINK:
		value.v_struct_link=NULL;
		break;
	
	default:
		g_assert_not_reached();
		break;
	}
	
	return value;
}


void
gsi_field_set_default	(gsi_base_S		*struct_p,
			 const gsi_field_info_S	*field_info)
{
	gsi_field_set_value(struct_p, field_info, gsi_field_get_default(struct_p, field_info));
}


gboolean
gsi_field_has_default	(gsi_base_S		*struct_p,
			 const gsi_field_info_S	*field_info)
{
	register gsi_value_U	value;
	register gsi_value_U	default_value;
	
	default_value=gsi_field_get_default(struct_p, field_info);
	value=gsi_field_get_value(struct_p, field_info);
	
	return gsi_values_equal	(field_info->field_type,
				 GSI_FIELD_OPT(field_info, GSI_OPT_MASK),
				 value,
				 default_value);
}


void
gsi_struct_zero_fields	(const gsi_struct_info_S	*struct_info,
			 gsi_base_S			*struct_p)
{
	register GList	*list, *free_list;
	
	g_assert(struct_p);
	
	free_list=list=gsi_struct_field_list(struct_info);
	
	while (list) {
		register const gsi_field_info_S	*field_info;
		
		field_info=(const gsi_field_info_S*) list->data;
		
		switch (field_info->field_type) {
		
		case	GSI_FIELD_DOUBLE:
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_double=0;
			break;
		
		case	GSI_FIELD_FLOAT:
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_float=0;
			break;
		
		case	GSI_FIELD_LONG:
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_long=0;
			break;
		
		case	GSI_FIELD_INT:
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_int=0;
			break;
		
		case	GSI_FIELD_ENUM:
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_enum=0;
			break;
		
		case	GSI_FIELD_BITS:
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_bits=0;
			break;
		
		case	GSI_FIELD_BOOLEAN:
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_boolean=FALSE;
			break;
		
		case	GSI_FIELD_STRING:
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_string=NULL;
			break;
		
		case	GSI_FIELD_POINTER:
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_pointer=NULL;
			break;
		
		case	GSI_FIELD_STRUCT_LINK:
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_struct_link=NULL;
			break;
		
		default:
			g_assert_not_reached();
			break;
		}
		
		list=list->next;
	}
	
	g_list_free(free_list);
}


void
gsi_struct_init_fields	(const gsi_struct_info_S	*struct_info,
			 gsi_base_S			*struct_p)
{
	register GList	*list, *free_list;
	
	g_assert(struct_p);
	
	gsi_struct_zero_fields(struct_info, struct_p);
	
	free_list=list=gsi_struct_field_list(struct_info);
	
	while (list) {
		register const gsi_field_info_S	*field_info;
		
		field_info=(const gsi_field_info_S*) list->data;
		
		gsi_field_set_default(struct_p, field_info);
		
		list=list->next;
	}
	
	g_list_free(free_list);
}


gboolean
gsi_values_equal	(const gsi_field_type_E		field_type,
			 const gsi_field_options_E	field_options,
			 gsi_value_U			value1,
			 gsi_value_U			value2)
{
	switch (field_type) {
	
	case	GSI_FIELD_DOUBLE:
		return value1.v_double==value2.v_double;
	
	case	GSI_FIELD_FLOAT:
		return value1.v_float==value2.v_float;
	
	case	GSI_FIELD_LONG:
		return value1.v_long==value2.v_long;
	
	case	GSI_FIELD_INT:
		return value1.v_int==value2.v_int;
	
	case	GSI_FIELD_ENUM:
		return value1.v_enum==value2.v_enum;
	
	case	GSI_FIELD_BITS:
		return value1.v_bits==value2.v_bits;
	
	case	GSI_FIELD_BOOLEAN:
		return value1.v_boolean==value2.v_boolean;
	
	case	GSI_FIELD_STRING:
		if (field_options&GSI_OPT_STRING_NONNULL &&
		    (!value2.v_string || value2.v_string[0]==0) &&
		    (!value1.v_string || value1.v_string[0]==0))
			return TRUE;
		else if (!value2.v_string || !value1.v_string)
			return value2.v_string==value1.v_string;
		else
			return strcmp(value2.v_string, value1.v_string)==0;
	
	case	GSI_FIELD_POINTER:
		return value1.v_pointer==value2.v_pointer;
	
	case	GSI_FIELD_STRUCT_LINK:
		return value1.v_struct_link==value2.v_struct_link;
	
	default:
		g_assert_not_reached();
		return TRUE;
	}
}


gboolean
gsi_field_set_value	(gsi_base_S		*struct_p,
			 const gsi_field_info_S	*field_info,
			 gsi_value_U		value)
{
	register gsi_value_U	old_value;
	register gboolean	string_race;
	register gboolean	ergo;
	
	g_assert(struct_p);
	g_assert(field_info);
	
	old_value=gsi_field_get_value(struct_p, field_info);
	if (field_info->field_type==GSI_FIELD_STRING) {
		string_race=TRUE;
		old_value.v_string=g_strdup(old_value.v_string);
	} else
		string_race=FALSE;
	
	switch (field_info->field_type) {
	
	case	GSI_FIELD_DOUBLE:
		value.v_double=CLAMP	(value.v_double,
					 field_info->minimum.v_double,
					 field_info->maximum.v_double);
		GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_double=value.v_double;
		break;
	
	case	GSI_FIELD_FLOAT:
		value.v_float=CLAMP	(value.v_float,
					 field_info->minimum.v_float,
					 field_info->maximum.v_float);
		GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_float=value.v_float;
		break;
	
	case	GSI_FIELD_LONG:
		value.v_long=CLAMP	(value.v_long,
					 field_info->minimum.v_long,
					 field_info->maximum.v_long);
		GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_long=value.v_long;
		break;
	
	case	GSI_FIELD_INT:
		value.v_int=CLAMP	(value.v_int,
					 field_info->minimum.v_int,
					 field_info->maximum.v_int);
		GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_int=value.v_int;
		break;
	
	case	GSI_FIELD_ENUM:
		value.v_enum=CLAMP	(value.v_enum,
					 field_info->minimum.v_enum,
					 field_info->maximum.v_enum);
		GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_enum=value.v_enum;
		break;
	
	case	GSI_FIELD_BITS:
		value.v_bits &=	field_info->maximum.v_bits;
		value.v_bits |=	(gsi_field_get_default(struct_p, field_info).v_bits &
				 ~field_info->maximum.v_bits);
		GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_bits=value.v_bits;
		break;
	
	case	GSI_FIELD_BOOLEAN:
		GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_boolean=value.v_boolean!=0;
		break;
	
	case	GSI_FIELD_STRING:
		g_free(GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_string);
		GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_string=NULL;
		
		if ( (!value.v_string || (value.v_string && value.v_string[0]==0)) &&
		     GSI_FIELD_OPT(field_info, GSI_OPT_STRING_DEFAULTS) )
			value.v_string=gsi_field_get_default(struct_p, field_info).v_string;
		
		
#ifdef	GUBI
		if (GSI_FIELD_OPT(field_info, GB_OPT_STRING_IS_CHILD) && GB_IS_WIDDAT(struct_p)) {
			register gb_wdat_base_S	*widget_data=GB_wCAST(base, struct_p);
			
			g_assert(!GSI_FIELD_OPT(field_info, GSI_OPT_STRING_NONNULL));
			
			if (value.v_string && value.v_string[0]==0)
				value.v_string=NULL;
			
			if (value.v_string) {
				UPDATE_CHILD_COUNT(widget_data);
				if (widget_data_check_add_type(widget_data, GB_WIDGET_LABEL, 0))
					value.v_string=NULL;
			}
		}
#endif	/*GUBI*/
		
		if (GSI_FIELD_OPT(field_info, GSI_OPT_STRING_NONNULL) && !value.v_string)
			value.v_string="";
		else if (!GSI_FIELD_OPT(field_info, GSI_OPT_STRING_NONNULL) &&
		         value.v_string && value.v_string[0]==0)
			value.v_string=NULL;
		if (value.v_string)
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_string=g_strdup(value.v_string);

#ifdef	GUBI
		if (GSI_FIELD_OPT(field_info, GB_OPT_STRING_IS_CHILD) && GB_IS_WIDDAT(struct_p))
			UPDATE_CHILD_COUNT(GB_wCAST(base, struct_p));
#endif	/*GUBI*/
		
		break;
	
	case	GSI_FIELD_POINTER:
		GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_pointer=value.v_pointer;
		break;
	
	case	GSI_FIELD_STRUCT_LINK:
		if (GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_struct_link &&
		    gsi_reference_func_p)
			(*gsi_reference_func_p)	(struct_p,
						 GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_struct_link,
						 FALSE);
		if (value.v_struct_link &&
		    gsi_reference_func_p) {
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_struct_link=NULL;
			(*gsi_reference_func_p)(struct_p, value.v_struct_link, TRUE);
		}
		GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_struct_link=value.v_struct_link;
		break;
	
	default:
		g_assert_not_reached();
		break;
	}

	ergo=!gsi_values_equal(field_info->field_type,
				GSI_FIELD_OPT(field_info, GSI_OPT_MASK),
				old_value,
				gsi_field_get_value(struct_p, field_info));
	if (string_race)
		g_free(old_value.v_string);
	
	return ergo;
}


gboolean
gsi_field_set_value_from_string	(gsi_base_S		*struct_p,
				 const gsi_field_info_S	*field_info,
				 const gchar		*value_as_string)
{
	static	gsi_value_U	value;
	
	g_assert(struct_p);
	g_assert(field_info);
	
	switch (field_info->field_type) {
	
	case	GSI_FIELD_DOUBLE:
		value.v_double=0;
		sscanf(value_as_string, " %lf", &value.v_double);
		break;
	
	case	GSI_FIELD_FLOAT:
		value.v_float=0;
		sscanf(value_as_string, " %f", &value.v_float);
		break;
	
	case	GSI_FIELD_LONG:
		value.v_long=0;
		sscanf(value_as_string, " %li", &value.v_long);
		break;
	
	case	GSI_FIELD_INT:
		value.v_int=0;
		sscanf(value_as_string, " %i", &value.v_int);
		break;
	
	case	GSI_FIELD_ENUM:
		value.v_long=0;
		sscanf(value_as_string, " %li", &value.v_long);
		value.v_enum=value.v_long;
		break;
	
	case	GSI_FIELD_BITS:
		value.v_long=0;
		sscanf(value_as_string, " %li", &value.v_long);
		value.v_bits=value.v_long;
		break;
	
	case	GSI_FIELD_BOOLEAN:
		value._char=0;
		sscanf(value_as_string, " %c", &value._char);
		switch (value._char) {
		
		case 1 ... 9:
		case 'T':
		case 't':
		case 'Y':
		case 'y':
			value.v_boolean=TRUE;
			break;
		
		case 0:
		case 'F':
		case 'f':
		case 'N':
		case 'n':
		default:
			value.v_boolean=FALSE;
			break;
		}
		break;
	
	case	GSI_FIELD_STRING:
		value.v_string=(gchar*)value_as_string;
		break;
	
	case	GSI_FIELD_POINTER:
		return FALSE;
	
	case	GSI_FIELD_STRUCT_LINK:
		return FALSE;
	
	default:
		g_assert_not_reached();
		break;
	}
	
	return gsi_field_set_value(struct_p, field_info, value);
}


gsi_value_U
gsi_field_get_value	(gsi_base_S		*struct_p,
			 const gsi_field_info_S	*field_info)
{
	register gsi_value_U	value;
	
	g_assert(struct_p);
	g_assert(field_info);
	
	switch (field_info->field_type) {
	
	case	GSI_FIELD_DOUBLE:
		value.v_double=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_double;
		break;
	
	case	GSI_FIELD_FLOAT:
		value.v_float=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_float;
		break;
	
	case	GSI_FIELD_LONG:
		value.v_long=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_long;
		break;
	
	case	GSI_FIELD_INT:
		value.v_int=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_int;
		break;
	
	case	GSI_FIELD_ENUM:
		value.v_enum=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_enum;
		break;
	
	case	GSI_FIELD_BITS:
		value.v_bits=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_bits;
		break;
	
	case	GSI_FIELD_BOOLEAN:
		value.v_boolean=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_boolean;
		break;
	
	case	GSI_FIELD_STRING:
		value.v_string=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_string;
		break;
	
	case	GSI_FIELD_POINTER:
		value.v_pointer=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_pointer;
		break;
	
	case	GSI_FIELD_STRUCT_LINK:
		value.v_struct_link=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_struct_link;
		break;
	
	default:
		g_assert_not_reached();
		break;
	}
	
	return value;
}


const gchar*
gsi_field_get_value_as_string	(gsi_base_S		*struct_p,
				 const gsi_field_info_S	*field_info)
{
	static	 gchar		buffer[64+1];
	register gsi_value_U	value;
	register guint		i;
	
	value=gsi_field_get_value(struct_p, field_info);
	
	switch (field_info->field_type) {
	register guint		j;
	
	case	GSI_FIELD_DOUBLE:
		value.v_double=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_double;
		i=snprintf(buffer, 64, "%.9f", value.v_double);
		j=i-1;
		while (buffer[j]=='0')
			buffer[j--]=0;
		break;
	
	case	GSI_FIELD_FLOAT:
		value.v_float=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_float;
		i=snprintf(buffer, 64, "%.6f", value.v_float);
		j=i-1;
		while (buffer[j]=='0')
			buffer[j--]=0;
		break;
	
	case	GSI_FIELD_LONG:
		value.v_long=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_long;
		i=snprintf(buffer, 64, "%ld", value.v_long);
		break;
	
	case	GSI_FIELD_INT:
		value.v_int=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_int;
		i=snprintf(buffer, 64, "%d", value.v_int);
		break;
	
	case	GSI_FIELD_ENUM:
		value.v_enum=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_enum;
		if (!field_info->field_func)
			i=snprintf(buffer, 64, "%ld", (gulong)value.v_enum);
		else {
			register GsiUNum2StringFunc	*unum2string_func;
			
			unum2string_func=(GsiUNum2StringFunc*)field_info->field_func;
			i=snprintf	(buffer, 64, "%s",
					 (*unum2string_func)(value.v_enum, FALSE));
		}
		break;
	
	case	GSI_FIELD_BITS:
		value.v_bits=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_bits;
		i=snprintf(buffer, 64, "0x%x", value.v_bits);
		break;
	
	case	GSI_FIELD_BOOLEAN:
		value.v_boolean=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_boolean;
		if (!field_info->field_func)
			i=snprintf(buffer, 64, "%s", value.v_boolean ? "TRUE" : "FALSE");
		else {
			register GsiUNum2StringFunc	*unum2string_func;
			
			unum2string_func=(GsiUNum2StringFunc*)field_info->field_func;
			i=snprintf	(buffer, 64, "%s",
					 (*unum2string_func)(value.v_boolean, FALSE));
		}
		break;
	
	case	GSI_FIELD_STRING:
		value.v_string=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_string;
		if (value.v_string)
			i=snprintf(buffer, 64, "%s%s%s",
				GSI_FIELD_OPT(field_info, GSI_OPT_STRING_UNQUOTED) ? "" : "\"",
				value.v_string,
				GSI_FIELD_OPT(field_info, GSI_OPT_STRING_UNQUOTED) ? "" : "\"");
		else
			i=snprintf(buffer, 64, "");
		break;
	
	case	GSI_FIELD_POINTER:
		value.v_pointer=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_pointer;
		i=snprintf	(buffer, 64,
				 GSI_FIELD_OPT(field_info, GSI_OPT_POINTER_UPCASED) ?
				    "0X%lX" : "0x%lx",
				 (gulong) value.v_pointer);
		break;
	
	case	GSI_FIELD_STRUCT_LINK:
		value.v_struct_link=GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_struct_link;
		i=snprintf(buffer, 64, "");
		break;
	
	default:
		g_assert_not_reached();
		i=0;
		break;
	}
	
	if (i>=(64-1)) {
		buffer[64]=0;
		g_warning("possible buffer overflow in sprintf()");
	}
	
	return buffer;
}


void
gsi_struct_free_fields	(const gsi_struct_info_S	*struct_info,
			 gsi_base_S			*struct_p)
{
	register GList	*list, *free_list;
	
	g_assert(struct_p);
	
	
	/* NOTE: better make successive calls to this function permittable!
	*/
	
	free_list=list=gsi_struct_field_list(struct_info);
	
	while (list) {
		register const gsi_field_info_S	*field_info;
		
		field_info=(const gsi_field_info_S*) list->data;
		
		switch (field_info->field_type) {
		register gsi_value_U	value;
		
		case	GSI_FIELD_DOUBLE:
		case	GSI_FIELD_FLOAT:
		case	GSI_FIELD_LONG:
		case	GSI_FIELD_INT:
		case	GSI_FIELD_ENUM:
		case	GSI_FIELD_BITS:
		case	GSI_FIELD_BOOLEAN:
			break;
		
		case	GSI_FIELD_STRING:
			g_free(GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_string);
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_string=NULL;
			break;
		
		case	GSI_FIELD_POINTER:
			GSI_FIELD_VALUE_U(struct_p, field_info->field_offset)->v_pointer=NULL;
			break;
		
		case	GSI_FIELD_STRUCT_LINK:
			value.v_struct_link=NULL;
			gsi_field_set_value(struct_p, field_info, value);
			break;
		
		default:
			g_assert_not_reached();
			break;
		}
		
		list=list->next;
	}
	
	g_list_free(free_list);
}


gsi_base_S*
gsi_struct_new0	(const gsi_struct_info_S	*struct_info)
{
	register gsi_base_S	*struct_p;
	
	g_assert(struct_info);
	
	struct_p=g_malloc0(struct_info->struct_size);
	struct_p->type=struct_info->struct_type;
	
	/* gsi_struct_zero_fields(struct_info, struct_p); */
	
	gsi_struct_init_fields(struct_info, struct_p);
	
	return struct_p;
}


void
gsi_struct_free	(gsi_base_S	*struct_p)
{
	register const gsi_struct_info_S	*struct_info;
	
	if (struct_p) {
		struct_info=gsi_struct_info(struct_p->type);
		
		if (struct_info) {
			gsi_struct_free_fields(struct_info, struct_p);
			
			g_free(struct_p);
		}
	}
}


guint
gsi_field_n_bits	(const gsi_field_info_S	*field_info)
{
	register guint	count, i;
	
	g_assert(field_info);
	g_assert(field_info->field_type==GSI_FIELD_BITS);
	
	count=0;
	
	for (i=0; i<sizeof(field_info->maximum.v_bits)*8; i++)
		if (field_info->maximum.v_bits & (1<<i))
			count++;
	
	return count;
}


guint32
gsi_field_bit_value	(const gsi_field_info_S	*field_info,
			 const guint		nth_bit)
{
	register guint	count, i;
	
	g_assert(field_info);
	g_assert(field_info->field_type==GSI_FIELD_BITS);
	
	count=0;
	
	for (i=0; i<sizeof(field_info->maximum.v_bits)*8; i++) {
		if (field_info->maximum.v_bits & (1<<i))
			count++;
		if (count==nth_bit)
			break;
	}
	
	return field_info->maximum.v_bits & (1<<i);
}


void
gsi_field_set_bit	(gsi_base_S		*struct_p,
			 const gsi_field_info_S	*field_info,
			 const guint		nth_bit,
			 gboolean		state)
{
	if (state)
		gsi_field_set_value	(struct_p,
					 field_info,
					 gsi_field_bit_value(field_info, nth_bit) |
					   gsi_field_get_value	(struct_p,
					 			 field_info).v_bits);
	else
		gsi_field_set_value	(struct_p,
					 field_info,
					 ~gsi_field_bit_value(field_info, nth_bit) &
					   gsi_field_get_value	(struct_p,
					 			 field_info).v_bits);
}


void
gsi_struct_unlink	(gsi_base_S	*struct_p,
			 gsi_base_S	*linking_struct_p)
{
	register const gsi_struct_info_S	*struct_info;
	register GList				*list, *free_list;
	
	g_assert(struct_p);
	g_assert(linking_struct_p);
	
	struct_info=gsi_struct_info(linking_struct_p->type);
	g_assert(struct_info);
	
	free_list=list=gsi_struct_field_list(struct_info);
	
	while (list) {
		register const gsi_field_info_S	*field_info;
		
		field_info=(const gsi_field_info_S*) list->data;
		if (field_info->field_type==GSI_FIELD_STRUCT_LINK &&
		    gsi_field_get_value(linking_struct_p, field_info).v_struct_link==struct_p)
			gsi_field_set_default(linking_struct_p, field_info);
		
		list=list->next;
	}
	g_list_free(free_list);
}
