/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997	Tim Janik	<timj@psynet.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"rcs.h"
RCS_ID("$Id: widdata.c,v 1.10 1997/08/18 03:38:12 timj Exp $")


#define		__widdata_c__

#include	"gsi.h"
#include	"widdata.h"
#include	"misc.h"
#include	"editor.h"
#include	"defines.h"
#include	"structures.h"


/* --- external global variables --- */
GList		*Gubi_auxillary_list=NULL;



/* --- variables --- */
static	GHashTable	*symbol_name_hash_table=NULL;



/* --- prototypes --- */
static	void	symbol_name_hash_table_init	(void);
static	void	widget_data_delete_foreach	(gpointer	data,
						 gpointer	user_data);
static	void	SigH_widget_data_item_destroy	(GtkWidget	*item,
						 gpointer	func_data);



/* --- functions --- */
void
symbol_name_hash_table_init	(void)
{
	if (!symbol_name_hash_table) {
		symbol_name_hash_table=g_hash_table_new((GHashFunc)g_string_hash,
							(GCompareFunc)g_string_equal);
	}
	
	g_assert(symbol_name_hash_table);
}


void
widget_data_initialize	(void)
{
	symbol_name_hash_table_init();
	
	gsi_initialize(gsi_struc_info_list);
	gsi_set_reference_func(widget_data_reference_func);
}


gb_wdat_base_S*
widget_data_new	(gb_struct_type_E	struct_type,
		 const gchar		*symbol_name,
		 gboolean		auxillary)
{
	register gb_wdat_base_S			*WidDat;
	register const gsi_struct_info_S	*struct_info;
	
	struct_info=gsi_struct_info(struct_type);
	g_assert(struct_info);
	
	WidDat=(gb_wdat_base_S*)g_malloc0(struct_info->struct_size);
	
	
	/* initialize private fields of base widget data
	*/
	WidDat->type=struct_type;
	WidDat->widget=NULL;
	WidDat->gfree_on_destroy=FALSE;
	WidDat->clone=NULL;
	WidDat->next=NULL;
	WidDat->keyed_data_stack=NULL;
	WidDat->signal_handler_stack=NULL;
	WidDat->user_data=NULL;
	WidDat->parent=NULL;
	WidDat->linkage=NULL;
	
	
	/* allocate and pre-initialize gubi_data_S
	*/
	WidDat->GUBI_DATA_FIELD=(gpointer)g_malloc0(sizeof(gubi_data_S));
	GUBI_DATA(WidDat)->type=GB_STRUCT_NONE;
	GUBI_DATA(WidDat)->widget_data=WidDat;
	GUBI_DATA(WidDat)->symbol_name=NULL;
	GUBI_DATA(WidDat)->struct_info=struct_info;
	GUBI_DATA(WidDat)->category=WID_CAT_NORMAL | (auxillary ? WID_CAT_AUXILLARY : 0);
	GUBI_DATA(WidDat)->children=NULL;
	GUBI_DATA(WidDat)->max_child_count=struct_info->gb_widget_child_count;
	GUBI_DATA(WidDat)->child_count=0;
	GUBI_DATA(WidDat)->tree=NULL;
	GUBI_DATA(WidDat)->editor=NULL;
	GUBI_DATA(WidDat)->browser=NULL;
	GUBI_DATA(WidDat)->item_refs=NULL;
	GUBI_DATA(WidDat)->link_refs=NULL;
	
	
	/* set symbol name,
	 * right now we need an unregistered symbol name!
	*/
	widget_data_symbol_name_set(WidDat, symbol_name);
	widget_data_symbol_name_unregister(WidDat);
	
	
	/* initialize widget specific fields
	*/
	gsi_struct_init_fields(GUBI_DATA(WidDat)->struct_info, GB_CAST(any, WidDat));
	
	
	/* reset the child count
	*/
	UPDATE_CHILD_COUNT(WidDat);
	
	return WidDat;
}


void
widget_data_delete_foreach	(gpointer	data,
				 gpointer	user_data)
{
	register gb_wdat_base_S	*WidDat;
	
	WidDat=data;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	widget_data_delete_R(WidDat);
}


void
widget_data_delete_R	(gb_wdat_base_S	*WidDat)
{
	register const gsi_struct_info_S	*struct_info;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(GUBI_DATA(WidDat));
	g_assert(!WidDat->widget);
	g_assert(!GUBI_DATA(WidDat)->editor);
	g_assert(!GUBI_DATA(WidDat)->browser);
	g_assert(GUBI_DATA(WidDat)->tree==NULL);
	g_assert(!widget_data_lookup(widget_data_symbol_name_get(WidDat)));
	if (GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY) {
		g_assert(WidDat->parent=NULL);
		g_assert(WidDat->next=NULL);
        }
        
	if (GB_TYPE_IS_WIDDAT_WINDOW(WidDat->type))
		g_assert(g_list_length(GUBI_DATA(WidDat)->children)==0);
	
	
	/* delete all list items that might refer to us
	*/
	while (GUBI_DATA(WidDat)->item_refs)
		widget_data_item_destroy(WidDat, GUBI_DATA(WidDat)->item_refs->data);
	
	
	/* delete references from other widgets in widget data
	*/
	while (GUBI_DATA(WidDat)->link_refs) {
		register gb_wdat_base_S	*RefDat;
		
		RefDat=GUBI_DATA(WidDat)->link_refs->data;
		g_assert(GB_IS_WIDDAT(RefDat));
		
		gsi_struct_unlink(GB_CAST(any, WidDat), GB_CAST(any, RefDat));
		
		if (GUBI_DATA(RefDat)->editor)
			Editor_refresh(RefDat);
	}
	
	
	/* delete children and children list
	*/
	g_list_foreach(GUBI_DATA(WidDat)->children,
			widget_data_delete_foreach,
			NULL);
	g_list_free(GUBI_DATA(WidDat)->children);
	
	
	/* free signal_handler_stack (used as string fields)
	*/
	widget_data_handler_stack_remove(WidDat);
	
	
	/* free gubi_data_S
	*/
	struct_info=GUBI_DATA(WidDat)->struct_info;
	g_free(GUBI_DATA(WidDat)->symbol_name);
	g_free(GUBI_DATA(WidDat));
	
	
	/* free other widget field specific memory allocations
	*/
	gsi_struct_free_fields(struct_info, GB_CAST(any, WidDat));
	
	/* free gb_wdat_base_S
	 * this also frees the linkage (and signal_handler_stack)
	*/
	gb_sigh_structure_free(NULL, GB_CAST(any, WidDat));
}


const	gchar*
widget_data_symbol_name_get	(gb_wdat_base_S	*WidDat)
{
	g_assert(GB_IS_WIDDAT(WidDat));
	
	return GUBI_DATA(WidDat)->symbol_name ? GUBI_DATA(WidDat)->symbol_name : "";
}


void
widget_data_symbol_name_set	(gb_wdat_base_S	*WidDat,
				 const gchar	*name)
{
	register gchar		*buffer;
	register guint		buffer_len;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	buffer_len=DFL_BUFFER_SIZE;
	buffer=g_new(gchar, buffer_len+1);
	
	if (GUBI_DATA(WidDat)->symbol_name) {
		if (widget_data_lookup(GUBI_DATA(WidDat)->symbol_name))
			widget_data_symbol_name_unregister(WidDat);
		g_free(GUBI_DATA(WidDat)->symbol_name);
	}
	
	
	/* if the given name already exists or is invalid
	 * we make it generic
	*/
	if (!name || strlen(name)==0 || widget_data_lookup(name))
		name=NULL;
	
	
	/* if no name is given, we genrate one
	*/
	if (!name) {
		GUBI_DATA(WidDat)->category|=WID_CAT_GENERIC;
		snprintf(buffer, buffer_len, "%s_%%d",
			to_UpCase(GUBI_DATA(WidDat)->struct_info->struct_name));
	} else {
		GUBI_DATA(WidDat)->category&=~WID_CAT_GENERIC;
		snprintf(buffer, buffer_len, "%s", name);
	}
	
	
	/* if name is generic, check for existence
	*/
	if (GUBI_DATA(WidDat)->category & WID_CAT_GENERIC) {
		register guint	i;
		
		GUBI_DATA(WidDat)->symbol_name=g_strdup(buffer);
		i=1;
		do {
			snprintf(buffer, buffer_len, GUBI_DATA(WidDat)->symbol_name, i);
			i++;
		} while (widget_data_lookup(buffer));
		g_free(GUBI_DATA(WidDat)->symbol_name);
	}
	GUBI_DATA(WidDat)->symbol_name=g_strdup(buffer);
	g_free(buffer);
	
	
	/* we don't allow generic names in auxillary pool
	*/
	if (GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY &&
	    GUBI_DATA(WidDat)->category & WID_CAT_GENERIC) {
		GUBI_DATA(WidDat)->category&=~WID_CAT_GENERIC;
		g_warning("%s\ngeneric name generation forced for auxillary widget: `%s'",
			FUNCNAME,
			GUBI_DATA(WidDat)->symbol_name);
	}
	
	widget_data_symbol_name_register(WidDat);
}


gb_wdat_base_S*
widget_data_lookup	(const gchar	*name)
{
	register gpointer	Ret;
	
	g_assert(name);
	
	Ret=g_hash_table_lookup(symbol_name_hash_table, (gpointer)name);
	
	return GB_wCAST(base, Ret);
}


void
widget_data_symbol_name_register	(gb_wdat_base_S	*WidDat)
{
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(!widget_data_lookup(widget_data_symbol_name_get(WidDat)));
	if (GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY)
		g_assert(!GUBI_DATA(WidDat)->category&WID_CAT_GENERIC);
	
	g_hash_table_insert(symbol_name_hash_table,
			    (gpointer)widget_data_symbol_name_get(WidDat),
			    WidDat);
}


void
widget_data_symbol_name_unregister	(gb_wdat_base_S	*WidDat)
{
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(widget_data_lookup(widget_data_symbol_name_get(WidDat)));
	
	g_hash_table_remove(symbol_name_hash_table,
			    (gpointer)widget_data_symbol_name_get(WidDat));
}


GtkWidget*
widget_data_item_new		(gb_wdat_base_S	*WidDat,
				 guint		indent_level,
				 GtkList	*List,
				 gint		position)
{
	register GtkWidget	*item;
	register gchar		*name;
	register const gchar	*symbol_name;
	register guint		i;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	symbol_name=widget_data_symbol_name_get(WidDat);
	name=g_new(gchar, strlen(symbol_name)+indent_level+1);
	for (i=0; i<indent_level; i++)
		name[i]=ITEM_INDENT_CHAR;
	strcpy(&name[i], symbol_name);
	
	item=gtk_list_item_new_with_label(name);
	g_free(name);
	
	gtk_object_set_user_data(GTK_OBJECT(item), (gpointer)WidDat);
	
	GUBI_DATA(WidDat)->item_refs=g_list_append(GUBI_DATA(WidDat)->item_refs, item);
	
	gtk_signal_connect(GTK_OBJECT(item),
			   "destroy",
			   GTK_SIGNAL_FUNC(SigH_widget_data_item_destroy),
			   NULL);
	
	if (List) {
		register GList	*item_list;
	
		item_list=g_list_alloc();
		item_list->data=item;
		gtk_list_insert_items(List, item_list, position);
	}
	
	gtk_widget_show(item);
	
	return item;
}


void
SigH_widget_data_item_destroy	(GtkWidget	*item,
				 gpointer	func_data)
{
	register gb_wdat_base_S	*WidDat;
	
	WidDat=gtk_object_get_user_data(GTK_OBJECT(item));
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	GUBI_DATA(WidDat)->item_refs=g_list_remove(GUBI_DATA(WidDat)->item_refs, item);
}


void
widget_data_item_destroy	(gb_wdat_base_S	*WidDat,
				 GtkWidget	*item)
{
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(g_list_find(GUBI_DATA(WidDat)->item_refs, item)); /* FIXME */
	
	if (item->parent) {
		register GList		*list;
	
		g_assert(GTK_IS_LIST(item->parent));
		
		list=g_list_append(NULL, item);
		
		gtk_list_remove_items(GTK_LIST(item->parent), list);
	}
	
	g_assert(GTK_IS_ITEM(item));
	gtk_widget_destroy(item);
}


void
widget_data_items_update	(gb_wdat_base_S	*WidDat)
{
	register const gchar	*symbol_name;
	register GtkWidget	*Item;
	register GList		*list;
	register GList		*children;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	symbol_name=widget_data_symbol_name_get(WidDat);
	
	list=GUBI_DATA(WidDat)->item_refs;
	
	while (list) {
		g_assert((Item=list->data));
		
		children=gtk_container_children(GTK_CONTAINER(Item));
		while (children) {
			register GtkWidget	*Child;
			
			g_assert((Child=children->data));
			
			if (GTK_IS_LABEL(Child)) {
				register guint	indent;
				gchar		*label;
				register guchar	i;
				
				gtk_label_get(GTK_LABEL(Child), &label);
				indent=0;
				if (label)
					while (label[indent]==ITEM_INDENT_CHAR)
						indent++;
				label=g_new(gchar, indent + strlen(symbol_name) +1);
				for (i=0; i<indent; i++)
					label[i]=ITEM_INDENT_CHAR;
				strcpy(&label[i], symbol_name);
				gtk_label_set(GTK_LABEL(Child), label);
				g_free(label);
			}
			children=children->next;
		}
		
		g_list_free(children);
		gtk_widget_draw(Item, NULL);
		
		list=list->next;
	}
}


guint
widget_data_parent_level	(gb_wdat_base_S	*WidDat)
{
	register guint	level;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	level=0;
	WidDat=WidDat->parent;
	while (WidDat) {
		level++;
		WidDat=WidDat->parent;
	}
	
	return level;
}


void
widget_data_linkage_new	(gb_wdat_base_S	*WidDat)
{
	register const gsi_struct_info_S	*struct_info;
	
	struct_info=NULL;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(!WidDat->linkage);
	g_assert(!(GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY));
	
	if (!WidDat->parent)
		return;
	
	
	/* we only need the linkage information on
	 * special parent widgets
	*/
	switch (WidDat->parent->type) {
	
	case	GB_WIDGET_H_BOX:
	case	GB_WIDGET_V_BOX:
	case	GB_WIDGET_FILE_SELECTION:
		struct_info=gsi_struct_info(GB_LINKAGE_BOX);
		break;
	
	case	GB_WIDGET_TABLE:
		struct_info=gsi_struct_info(GB_LINKAGE_TABLE);
		break;
	
	case	GB_WIDGET_NOTEBOOK:
		struct_info=gsi_struct_info(GB_LINKAGE_NOTEBOOK);
		break;
	
	default:
		break;
	}
	
	if (struct_info)
		WidDat->linkage=GB_CAST(any, gsi_struct_new0(struct_info));
	widget_data_linkage_check(WidDat);
}


gboolean
widget_data_linkage_check	(gb_wdat_base_S	*WidDat)
{
	register GList				*list, *free_list;
	register gboolean			has_default;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(!(GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY));
	
	
	/* do we have a linkage to check?
	*/
	if (!WidDat->linkage)
		return TRUE;
	
	
	/* we only need to check the linkage validity on
	 * special parent widgets
	*/
	switch (WidDat->parent->type) {
		register gb_linkage_table_S	*l_table;
		register gb_wdat_table_S	*Table;
		
	case	GB_WIDGET_H_BOX:
	case	GB_WIDGET_V_BOX:
	case	GB_WIDGET_FILE_SELECTION:
		g_assert(WidDat->linkage->type==GB_LINKAGE_BOX);
		break;
		
	case	GB_WIDGET_NOTEBOOK:
		g_assert(WidDat->linkage->type==GB_LINKAGE_NOTEBOOK);
		break;
		
	case	GB_WIDGET_TABLE:
		g_assert(WidDat->linkage->type==GB_LINKAGE_TABLE);
		l_table=(gb_linkage_table_S*)WidDat->linkage;
		Table=GB_wCAST(table, WidDat->parent);
		
		g_assert(Table->columns>=1 && Table->rows>=1);
		l_table->left_attach=CLAMP(l_table->left_attach, 0, Table->columns-1);
		l_table->right_attach=CLAMP(l_table->right_attach, 1, Table->columns);
		l_table->top_attach=CLAMP(l_table->top_attach, 0, Table->rows-1);
		l_table->bottom_attach=CLAMP(l_table->bottom_attach, 1, Table->rows);
		l_table->left_attach=MIN(l_table->left_attach, l_table->right_attach-1);
		l_table->top_attach=MIN(l_table->top_attach, l_table->bottom_attach-1);
		break;
	
	default:
		g_assert_not_reached();
		break;
	}
	
	
	/* let's determine wether we have default values
	*/
	free_list=list=gsi_struct_field_list(gsi_struct_info(WidDat->linkage->type));
	has_default=TRUE;
	while (list) {
		has_default&=gsi_field_has_default(GB_CAST(any, WidDat->linkage), list->data);
		list=list->next;
	}
	g_list_free(free_list);
	
	return has_default;
}


gboolean
widget_data_children_linkage_check	(gb_wdat_base_S	*WidDat)
{
	register gboolean	something_changed=FALSE;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	
	/* if we are a table we need to check our childrens
	 * linkage
	*/
	if (WidDat->type==GB_WIDGET_TABLE) {
		register GList	*list;
		
		list=GUBI_DATA(WidDat)->children;
		if (list)
			something_changed=TRUE;
		while (list) {
			register gb_wdat_base_S	*ChildDat=list->data;
			
			g_assert(GB_IS_WIDDAT(ChildDat));
			
			widget_data_linkage_check(ChildDat);
			
			list=list->next;
		}
	}
	
	return something_changed;
}


void
widget_data_handler_stack_remove	(gb_wdat_base_S	*WidDat)
{
	g_assert(GB_IS_WIDDAT(WidDat));
	
	
	/* free signal_handler_stack string fields
	*/
	if (WidDat->signal_handler_stack) {
		register gb_signal_handler_S	*handler;
		
		handler=WidDat->signal_handler_stack;
		
		while (handler->signal_name) {
			g_free(handler->signal_name);
			g_free((void*)handler->handler_func);
			g_free(handler->data.func_data);
			handler++;
		}
		
		g_free(WidDat->signal_handler_stack);
		WidDat->signal_handler_stack=NULL;
	}
}


guint
widget_data_handler_add(gb_wdat_base_S	*WidDat,
			const gchar	*signal_name,
			const gchar	*func_name,
			const gchar	*data_name,
			const guint	connect_options)
{
	register gb_signal_handler_S	*handler;
	register guint			count;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	g_assert(signal_name);
	g_assert(func_name);
	g_assert(data_name);
	
	count=0;
	if (!WidDat->signal_handler_stack)
		WidDat->signal_handler_stack=g_new(gb_signal_handler_S, 2);
	else {
		register gb_signal_handler_S	*old_list;
		
		old_list=WidDat->signal_handler_stack;
		
		for (count=0; old_list[count].signal_name; count++);
		
		WidDat->signal_handler_stack=g_new(gb_signal_handler_S, count+1+1);
		memcpy(WidDat->signal_handler_stack, old_list, count*sizeof(gb_signal_handler_S));
		g_free(old_list);
	}

	handler=&WidDat->signal_handler_stack[count];
	
	handler->connect_options=connect_options;
	handler->signal_name=g_strdup(signal_name);
	handler->handler_func=(GtkSignalFunc)g_strdup(func_name);
	handler->data.func_data=g_strdup(data_name);
	
	handler++;
	
	handler->connect_options=0;
	handler->handler_id=0;
	handler->signal_name=NULL;
	handler->handler_func=NULL;
	handler->data.func_data=NULL;
	
	return count;
}

void
widget_data_reference_func	(gb_any_S	*_WidDat,
				 gb_any_S	*_RefDat,
				 gboolean	add_link)
{
	register gb_wdat_base_S	*WidDat;
	register gb_wdat_base_S	*RefDat;
	
	g_assert(GB_IS_WIDDAT(_WidDat));
	g_assert(GB_IS_WIDDAT(_RefDat));
	
	WidDat=(gb_wdat_base_S*)_WidDat;
	RefDat=(gb_wdat_base_S*)_RefDat;
	
#ifdef	PARANOID
	if (add_link) {
		if (g_list_find(GUBI_DATA(RefDat)->link_refs, WidDat))
			g_warning("%s: link to `%s' in `%s' already there?",
				__PRETTY_FUNCTION__,
				widget_data_symbol_name_get(WidDat),
				widget_data_symbol_name_get(RefDat));
	} else {
		if (!g_list_find(GUBI_DATA(RefDat)->link_refs, WidDat))
			g_warning("%s: link to `%s' in `%s' missing!",
				__PRETTY_FUNCTION__,
				widget_data_symbol_name_get(WidDat),
				widget_data_symbol_name_get(RefDat));
	}
#endif	/*PARANOID*/
	
	if (add_link)
		GUBI_DATA(RefDat)->link_refs=g_list_prepend(GUBI_DATA(RefDat)->link_refs, WidDat);
	else
		GUBI_DATA(RefDat)->link_refs=g_list_remove(GUBI_DATA(RefDat)->link_refs, WidDat);
}


guint
widget_data_base_child_count	(gb_wdat_base_S	*WidDat)
{
	register GList			*list, *free_list;
	register guint			child_count;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	child_count=0;
	free_list=list=gsi_struct_field_list(GUBI_DATA(WidDat)->struct_info);
	while (list) {
		register const gsi_field_info_S	*field_info;
		
		field_info=list->data;
		
		if (field_info->field_type==GSI_FIELD_STRING &&
		    GSI_FIELD_OPT(field_info, GB_OPT_STRING_IS_CHILD) &&
		    gsi_field_get_value(GB_CAST(any, WidDat), field_info).v_string )
			child_count++;
		
		list=list->next;
	}
	g_list_free(free_list);
	
	if (child_count > GUBI_DATA(WidDat)->struct_info->gb_widget_child_count)
		g_error("%s\nencountered internal child_count oddity: %s: %d>%d",
			FUNCNAME,
			to_UpCase(GUBI_DATA(WidDat)->struct_info->struct_name),
			child_count,
			GUBI_DATA(WidDat)->struct_info->gb_widget_child_count);
	
	return child_count;
}
