/*  GUBI - Gtk+ User Interface Builder
 *  Copyright (C) 1997  Tim Janik
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#include	"RCS.h"
RCS_ID("$Id: gbcio.c,v 1.9 1997/05/20 23:50:59 tim Exp $")


#define		__gbcio_c__

#include	"gbcio.h"
#include	"widdata.h"
#include	"structures.h"
#include	"misc.h"
#include	"defines.h"
#include	<errno.h>



/* --- defines --- */
#define	STRUCT_HAS_FIELD_HASH_TABLE(struct_type)	\
	(						\
	  GB_TYPE_IS_OBJDAT(struct_type) ||		\
	  GB_TYPE_IS_LINKAGE(struct_type)		\
	)

/*	FIXME: remove
	  struct_type==GB_OBJECT ||			\
	  struct_type==GB_WIDGET_BASE ||		\
	  struct_type==GB_WIDGET_MISC ||		\
	  struct_type==GB_WIDGET_CONTAINER ||		\
	  struct_type==GB_WIDGET_BIN			\
	)
*/

/* macros for scanning/parsing errors
 */
#define	FILE_WARNING	(stderr)
#define	FILE_ERROR	(stderr)
#define	YY_ABORT()		(!yyin || yyerror>=yymaxerror)
#define	PARSER_UNEXP_WID(widget_name, reason)	{			\
	yyerror++;							\
	file_printf_set(FILE_ERROR);					\
	file_printf("%s:%d: error: unexpected widget `%s'%s%s%s\n",	\
		yyname,							\
		yyline,							\
		widget_name,						\
		reason ? " (" : "",					\
		reason ? reason : "",					\
		reason ? ")" : "");					\
}
#define	PARSER_ERROR(printf_args)	{				\
	yyerror++;							\
	file_printf_set(FILE_ERROR);					\
	file_printf("%s:%d: error: ", yyname, yyline);			\
	file_printf_set(FILE_ERROR);					\
	file_printf printf_args ;					\
	file_printf_set(FILE_ERROR);					\
	file_printf("\n");						\
}
#define	PARSER_WARNING(printf_args)	{				\
	file_printf_set(FILE_WARNING);					\
	file_printf("%s:%d: warning: ", yyname, yyline);		\
	file_printf_set(FILE_WARNING);					\
	file_printf printf_args;					\
	file_printf_set(FILE_WARNING);					\
	file_printf("\n");						\
}


#define	HANDLE_DUPS(org_name, widget_data)	{			\
	if (strcmp((org_name),						\
	           widget_data_symbol_name_get((widget_data)))!=0) {	\
		g_hash_table_insert(dups_ref_hash_table,		\
				    g_strdup((org_name)),		\
				    (widget_data));			\
		PARSER_WARNING((					\
			"duplicate widget name `%s': renamed to: `%s'",	\
			(org_name),					\
			widget_data_symbol_name_get((widget_data))));	\
	}								\
}



/* --- variables --- */
extern	FILE		*yyin;	/* from gbclex.l */
	guint		yyline=0;
	token_U		yylval;
static	gchar		*yyname;
static	guint		yyerror=0;
static	guint		yymaxerror=0;
static	char		write_buffer[DFL_BUFFER_SIZE];
static	guint		indent_level=0;
static	GHashTable	*struct_type_hash_table=NULL;
static	GHashTable	**struct_field_hash_table_array=NULL;
static	GHashTable	*dups_ref_hash_table=NULL;


/* --- static prototypes --- */


/* scan expected_token from yyin, if something else is found return
 * TOKEN_ERROR.
 * if expected_token is TOKEN_ANY any tokens but TOKEN_EOF will be accepted.
 * if expected_token is TOKEN_EOF any tokens including TOKEN_EOF will be
 * accepted.
 * if TOKEN_ERROR is returned and EOF is not reached,
 * a warning will be issued.
 * this is unflexible, but no one should play around with the config file!
*/
static	gint	GetToken		(gint		expected_token);
static	void	UnexpectedToken		(gint		token,
					 gint		expected_token);


/* write (String,) into (Dest)
*/
static	void	SaveString		(const	gchar		*string,
					 FILE			*f_out);


/* widget data io functions
*/
static	void	hash_tables_init	(void);
static	void	hash_tables_destroy	(void);
static	void	hash_table_free_key	(gpointer	key,
					 gpointer	value,
					 gpointer	user_data);
static	void	Save_widget_data_R	(gb_wdat_base_S		*WidDat,
					 gboolean		skip_defaults,
					 FILE			*f_out);
static	void	Save_generic_fields	(gb_any_S		*Struct,
					 const struct_info_S	*StrInf,
					 gboolean		skip_defaults,
					 FILE			*f_out);
static	void	Save_linkage		(gb_wdat_base_S		*WidDat,
					 gboolean		skip_defaults,
					 FILE			*f_out);
static	void	Save_handler_stack	(gb_wdat_base_S		*WidDat,
					 FILE			*f_out);
static	void	Scan_structure_R	(tree_S			*tree,
					 gb_any_S		*Struct);
static	void	Scan_generic_field	(gb_any_S		*Struct,
					 gchar			*field_name);
static	void	Scan_handler_old		(gb_wdat_base_S		*WidDat);



/* --- functions --- */
void
hash_tables_init	(void)
{
	register guint	i;
	
	if (!struct_type_hash_table) {
		g_assert(!struct_field_hash_table_array);
		struct_type_hash_table=g_hash_table_new((GHashFunc)g_string_hash,
							(GCompareFunc)g_string_equal);
		struct_field_hash_table_array=g_new0(GHashTable*, GB_STRUCT_LAST);
	}
	
	g_assert(struct_type_hash_table);
	g_assert(struct_field_hash_table_array);
	
	for (i=0; i<GB_STRUCT_LAST; i++) {
		if (STRUCT_HAS_FIELD_HASH_TABLE(i))
		    	g_hash_table_insert(struct_type_hash_table,
				    (gpointer)structure_info(i)->struct_name,
				    (gpointer)i);
	}
	
	for (i=0; i<GB_STRUCT_LAST; i++) {
		register guint			j;
		register const struct_info_S	*StrInf;
		
		if (!STRUCT_HAS_FIELD_HASH_TABLE(i)) {
			struct_field_hash_table_array[i]=NULL;
			continue;
		}
		struct_field_hash_table_array[i]=g_hash_table_new((GHashFunc)g_string_hash,
								  (GCompareFunc)g_string_equal);
		g_assert(struct_field_hash_table_array[i]);
		
		StrInf=structure_info(i);
		
		for (j=0; j<StrInf->field_count; j++) {
			register const field_info_S	*FldInf;
			
			FldInf=field_info(StrInf, j);
			g_hash_table_insert(struct_field_hash_table_array[i],
					    (gpointer)FldInf->field_name,
					    (gpointer)FldInf);
		}
	}
	
	g_assert(dups_ref_hash_table==NULL);
	dups_ref_hash_table=g_hash_table_new((GHashFunc)g_string_hash,
					     (GCompareFunc)g_string_equal);
}


void
hash_table_free_key	(gpointer	key,
			 gpointer	value,
			 gpointer	user_data)
{
	g_free(key);
}


void
hash_tables_destroy	(void)
{
	g_hash_table_foreach(dups_ref_hash_table, hash_table_free_key, NULL);
	g_hash_table_destroy(dups_ref_hash_table);
	dups_ref_hash_table=NULL;
}


gint
GetToken	(gint	expected_token)
{
	register int	yytoken;
	
	g_assert(expected_token!=TOKEN_ERROR);
	
	if (!yyin)
		if (expected_token==TOKEN_EOF)
			return TOKEN_EOF;
		else
			return TOKEN_ERROR;
	
	yytoken=yylex();
	
	if (yytoken<=0) {
		yyin=NULL;
		if (expected_token!=TOKEN_EOF) {
			UnexpectedToken(TOKEN_EOF, expected_token);
			return TOKEN_ERROR;
		} else
			return TOKEN_EOF;
	}
	
	if (yytoken!=expected_token && expected_token!=TOKEN_ANY && expected_token!=TOKEN_EOF) {
		UnexpectedToken(yytoken, expected_token);
		if (yytoken==TOKEN_STRING)
			g_free(yylval.String);
		return TOKEN_ERROR;
	}
	
	return yytoken;
}


void
UnexpectedToken	(gint	token,
		 gint	expected_token)
{
	register gchar	*token_string;
	register gchar	*expected_string;
	
	token_string=g_new(gchar, 64);
	expected_string=g_new(gchar, 64);
	
	switch (token) {
	
	case	' ' ... '~':
		sprintf(token_string, "character `%c'", token);
		break;
	
	case	TOKEN_NULL:
		sprintf(token_string, "identifier `NULL'");
		break;
	
	case	TOKEN_LONG:
		sprintf(token_string, "number `%ld'", yylval.Long);
		break;
	
	case	TOKEN_DOUBLE:
		sprintf(token_string, "number `%.1f'", yylval.Double);
		break;
	
	case	TOKEN_STRING:
		sprintf(token_string, "%sstring \"%s\"",
			strlen(yylval.String)>0 ? "" : "empty ",
			yylval.String);
		break;
	
	case	TOKEN_TREE:
		sprintf(token_string, "tree identifier `%s'", yylval.String);
		break;
	
	case	TOKEN_STRUCT:
		sprintf(token_string, "structure identifier `%s'", yylval.String);
		break;
	
	case	TOKEN_WIDGET:
		sprintf(token_string, "widget identifier `%s'", yylval.String);
		break;
	
	
	case	TOKEN_FIELD:
		sprintf(token_string, "field identifier `%s'", yylval.String);
		break;
	
	case	TOKEN_ERROR:
		g_assert_not_reached();
		break;
	
	default:
		sprintf(token_string, "token <%d>", token);
		break;
	}

	switch (expected_token) {
	
	case	TOKEN_ANY:
	case	TOKEN_EOF:
		g_free(expected_string);
		expected_string=NULL;
		break;
	
	case	' ' ... '~':
		sprintf(expected_string, "character `%c'", expected_token);
		break;
	
	case	TOKEN_NULL:
		sprintf(expected_string, "`NULL'");
		break;
	
	case	TOKEN_LONG:
		sprintf(expected_string, "number (integer)");
		break;
	
	case	TOKEN_DOUBLE:
		sprintf(expected_string, "number (float)");
		break;
	
	case	TOKEN_STRING:
		sprintf(expected_string, "string constant");
		break;
	
	case	TOKEN_TREE:
		sprintf(expected_string, "tree identifier");
		break;
	
	case	TOKEN_STRUCT:
	case	TOKEN_WIDGET:
		sprintf(expected_string, "structure identifier");
		break;
	
	case	TOKEN_FIELD:
		sprintf(expected_string, "field identifier");
		break;
	
	default:
		sprintf(expected_string, "token <%d>", expected_token);
		break;
	}


	if (token==TOKEN_EOF) {
		g_assert(!yyin);
		PARSER_ERROR(("unexpected end of file"));
	} else if (expected_string) {
		PARSER_ERROR(("unexpected %s, expected %s", token_string, expected_string));
	} else
		PARSER_ERROR(("unexpected %s", token_string));
	
	g_free(token_string);
	g_free(expected_string);
}


void
SaveString	(const	gchar	*string,
		 FILE		*f_out)
{
	register int	i;
	register int	Ret=~EOF;
	
	if (string) {
		for (i=0; Ret!=EOF && i<indent_level; i++)
			Ret=fputs(GBC_INDENT_STRING, f_out);
		if (Ret!=EOF)
			Ret=fputs(string, f_out);
	}

	if (Ret==EOF)
		g_error("fputs(): `%s'", g_strerror(errno));
}


void
Save_widget_data_R	(gb_wdat_base_S	*WidDat,
			 gboolean	skip_defaults,
			 FILE		*f_out)
{
	register GList	*list;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	if (GUBI_DATA(WidDat)->category & WID_CAT_AUXILLARY) {
		g_assert(WidDat->parent=NULL);
		g_assert(WidDat->next=NULL);
	}
		
	
	list=GUBI_DATA(WidDat)->children;
	
	
	/* write widget header
	*/
	if ((!GUBI_DATA(WidDat)->category&WID_CAT_GENERIC) ||
	    GUBI_DATA(WidDat)->link_refs)
		sprintf(write_buffer, "GB_WIDGET_%s  \"%s\" {\n",
			to_UPCASE(structure_info(WidDat->type)->struct_name),
			widget_data_symbol_name_get(WidDat));
	else
		sprintf(write_buffer, "GB_WIDGET_%s  \"\" {\n",
			to_UPCASE(structure_info(WidDat->type)->struct_name));
	SaveString(write_buffer, f_out);
	indent_level++;
	
	
	/* save object fields
	*/
	Save_generic_fields(GB_CAST(any, WidDat),
			    structure_info(GB_OBJECT),
			    skip_defaults,
			    f_out);
	
	
	/* save base widget fields
	*/
	Save_generic_fields(GB_CAST(any, WidDat),
			    structure_info(GB_WIDGET_BASE),
			    skip_defaults,
			    f_out);
	
	
	/* save misc widget fields
	*/
	if (GB_TYPE_IS_WIDDAT_MISC(WidDat->type)) {
		Save_generic_fields(GB_CAST(any, WidDat),
				    structure_info(GB_WIDGET_MISC),
				    skip_defaults,
				    f_out);
	}
	
	
	/* save container widget fields
	*/
	if (GB_TYPE_IS_WIDDAT_CONTAINER(WidDat->type)) {
		Save_generic_fields(GB_CAST(any, WidDat),
				    structure_info(GB_WIDGET_CONTAINER),
				    skip_defaults,
				    f_out);
	}
	
	
	/* save bin widget fields
	*/
	if (GB_TYPE_IS_WIDDAT_BIN(WidDat->type)) {
		Save_generic_fields(GB_CAST(any, WidDat),
				    structure_info(GB_WIDGET_BIN),
				    skip_defaults,
				    f_out);
	}
	
	
	/* save specific widget fields
	*/
	Save_generic_fields(GB_CAST(any, WidDat),
			    structure_info(WidDat->type),
			    skip_defaults,
			    f_out);
	
	
	/* save linkage information
	*/
	Save_linkage(WidDat, skip_defaults, f_out);
	
	
	/* save handler stack
	*/
	if (WidDat->handler_stack)
		Save_handler_stack(WidDat, f_out);
	
	
	/* save children
	*/
	while (list) {
		register gb_wdat_base_S	*ChildDat=list->data;
		
		g_assert(GB_IS_WIDDAT(ChildDat));
		
		Save_widget_data_R(ChildDat, skip_defaults, f_out);
		
		list=list->next;
	}
	
	
	/* write widget footer
	*/
	indent_level--;
	sprintf(write_buffer, "}\n");
	SaveString(write_buffer, f_out);
}


void
Save_linkage		(gb_wdat_base_S	*WidDat,
			 gboolean	skip_defaults,
			 FILE		*f_out)
{
	g_assert(GB_IS_WIDDAT(WidDat));
	
	if (widget_data_linkage_check(WidDat))
		return;
	
	
	/* write linkage header
	*/
	sprintf(write_buffer, "GB_%s {\n",
		to_UPCASE(structure_info(WidDat->linkage->type)->struct_name));
	SaveString(write_buffer, f_out);
	indent_level++;
	
	
	/* save linkage fields
	*/
	Save_generic_fields(WidDat->linkage,
			    structure_info(WidDat->linkage->type),
			    skip_defaults,
			    f_out);
	
	
	/* write linkage footer
	*/
	indent_level--;
	sprintf(write_buffer, "}\n");
	SaveString(write_buffer, f_out);
}


void
Save_handler_stack	(gb_wdat_base_S	*WidDat,
			 FILE		*f_out)
{
	register gb_handler_S	*h_next;
	register guint		count;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	h_next=WidDat->handler_stack;
	count=0;
	
	if (h_next) while (h_next->handler_func) {
		sprintf(write_buffer, "HANDLER_spec {\n");
		SaveString(write_buffer, f_out);
		indent_level++;
		sprintf(write_buffer, "HANDLER_connect_options %d\n",
			h_next->connect_options);
		SaveString(write_buffer, f_out);
		sprintf(write_buffer, "HANDLER_handler_func    \"%s\"\n",
			(char*)h_next->handler_func);
		SaveString(write_buffer, f_out);
		sprintf(write_buffer, "HANDLER_signal_name     \"%s\"\n",
			h_next->signal_name);
		SaveString(write_buffer, f_out);
		sprintf(write_buffer, "HANDLER_data            \"%s\"\n",
			(char*)h_next->data.func_data);
		SaveString(write_buffer, f_out);
		indent_level--;
		sprintf(write_buffer, "}\n");
		SaveString(write_buffer, f_out);
		count++;
		h_next++;
	}
}


void
Save_generic_fields	(gb_any_S		*Struct,
			 const struct_info_S	*StrInf,
			 gboolean		skip_defaults,
			 FILE			*f_out)
{
	register guint			i;
	
	g_assert(GB_IS_STRUCT(Struct));
	g_assert(StrInf);
	
	
	for (i=0; i<StrInf->field_count; i++) {
		register const	field_info_S	*FldInf;
		
		FldInf=field_info(StrInf, i);
		
		
		if (!(structure_field_has_default(Struct, FldInf)&&skip_defaults)) {
			register const	gchar		*string;
			
			switch (FldInf->field_type) {
			case	FIELD_BOOLEAN:
				sprintf(write_buffer, "%s        %u\n",
					FldInf->field_name,
					structure_field_get_value(Struct, FldInf).bool_val);
				SaveString(write_buffer, f_out);
				break;
			
			case	FIELD_BITS:
			case	FIELD_ENUM:
				sprintf(write_buffer, "%s        %u\n",
					FldInf->field_name,
					structure_field_get_value(Struct, FldInf).bit_val);
				SaveString(write_buffer, f_out);
				break;
			
			case	FIELD_STRING:
			case	FIELD_POINTER:
				string=structure_field_get_value_str(Struct, FldInf);
				sprintf(write_buffer, "%s        \"%s\"\n",
					FldInf->field_name,
					string ? string : "");
				SaveString(write_buffer, f_out);
				break;
			
			case	FIELD_WD_LINK:
				string=structure_field_get_value_str(Struct, FldInf);
				sprintf(write_buffer, "%s        \"%s\"\n",
					FldInf->field_name,
					string);
				SaveString(write_buffer, f_out);
				break;
			
			default:
				string=structure_field_get_value_str(Struct, FldInf);
				sprintf(write_buffer, "%s        %s\n",
					FldInf->field_name,
					string);
				SaveString(write_buffer, f_out);
				break;
			}
		}
	}
}


void
Save_window	(FILE			*f_out,
		 gchar			*filename,
		 gb_wdat_window_S	*WinDat,
		 gboolean		skip_defaults)
{
	g_assert(f_out);
	g_assert(filename);
	g_assert(GB_IS_WIDDAT_WINDOW(WinDat));
	
	
	
	/* prepare global vars
	*/
	yyname=filename;
	yyline=1;
	indent_level=0;
	
	
	/* save window's widget tree
	*/
	sprintf(write_buffer, "\n\n/* widget tree for %s\n*/\n", PRGNAME_Short);
	SaveString(write_buffer, f_out);
	Save_widget_data_R(GB_wCAST(base, WinDat), skip_defaults, f_out);
}


guint
Read_trees	(FILE			*f_in,
		 gchar			*filename,
		 guint			max_errors)
{
	register int		yytoken;
	register tree_S		*tree;
	register guint		i;
	
	
	/* prepare global variables
	*/
	for (i=strlen(filename); i>0; i--)
		if (filename[i-1]=='/')
			break;
	if (i>0 && i<strlen(filename))
		yyname=&filename[i];
	else
		yyname=filename;
	yyin=0;
	yyrestart(f_in);
	yyline=1;
	yyerror=0;
	yymaxerror=max_errors;
	
	hash_tables_init();
	
	/* scan for new trees
	*/
	yytoken=TOKEN_EOF;
	while (!YY_ABORT()) {
		
		yytoken=GetToken(TOKEN_EOF);

		switch (yytoken) {
		register char			*string;
		register gb_struct_type_E	struct_type;
		register gb_wdat_window_S	*WinDat;

		case	TOKEN_WIDGET:
			struct_type=(gb_struct_type_E)g_hash_table_lookup(struct_type_hash_table,
							(gpointer)to_downcase(&yylval.String[10]));
			if (!GB_TYPE_IS_WIDDAT_WINDOW(struct_type)) {
				PARSER_UNEXP_WID(&yylval.String[10], NULL);
				g_free(yylval.String);
				break;
			}
			g_free(yylval.String);
			
			if (GetToken(TOKEN_STRING)==TOKEN_ERROR)
				break;
			string=yylval.String;
			
			if (GetToken('{')==TOKEN_ERROR) {
				g_free(string);
				break;
			}
			tree=tree_new(TRUE, FALSE);
			
			WinDat=GB_wCAST(window, widget_data_new(struct_type, string, FALSE));
			
			
			/* we need to check for duplicate identifiers
			*/
			if (string && string[0]!=0)
				HANDLE_DUPS(string, GB_wCAST(base, WinDat));
			
			g_free(string);
			tree_widget_data_insert_R(tree, NULL, GB_wCAST(base, WinDat), 0);
			tree_set_current_widget_data(tree, GB_wCAST(base, WinDat));
			
			Scan_structure_R(tree, GB_CAST(any, WinDat));
			
			tree_show(tree);
			break;
			
		default:
			UnexpectedToken(yytoken, TOKEN_TREE);
		case	TOKEN_ERROR:
		case	TOKEN_EOF:
			break;
		}
	}
	
	hash_tables_destroy();
	
	return yyerror;
}


void
Scan_structure_R	(tree_S		*tree,
			 gb_any_S	*Struct)
{
	register int		yytoken;
	register gboolean	scan_children_only;
	
	scan_children_only=FALSE;
	while ((yytoken=GetToken(TOKEN_ANY))!='}') {
		
		if (YY_ABORT())
			break;
		
		
		/* if we have reached the end of a widget
		 * definition, we only accept new widgets
		 * (yytoken==TOKEN_WIDGET) or
		 * the footer (yytoken=='}')
		*/
		if (scan_children_only &&
		    yytoken!='}' &&
		    yytoken!=TOKEN_WIDGET &&
		    yytoken!=TOKEN_ERROR) {
			UnexpectedToken(yytoken, TOKEN_ANY);
			continue;
		}
		
		switch (yytoken) {
		register gb_struct_type_E	struct_type;
		register gchar			*string;
		register gb_wdat_base_S		*WidDat;
		register gb_wdat_base_S		*ChildDat;
		
		case	TOKEN_FIELD:
			string=yylval.String;
			Scan_generic_field(Struct, string);
			g_free(string);
			break;

		case	TOKEN_HANDLER:
			/* FIXME: remove this case */
			if (GB_TYPE_IS_WIDDAT(Struct->type))
				Scan_handler_old(GB_wCAST(base, Struct));
			else
				UnexpectedToken(TOKEN_HANDLER, TOKEN_FIELD);
			break;
		
		case	TOKEN_STRUCT:
			if (!GB_TYPE_IS_WIDDAT(Struct->type)) {
				UnexpectedToken(TOKEN_STRUCT, TOKEN_FIELD);
				break;
			}
			
			WidDat=GB_wCAST(base, Struct);
				
			struct_type=(gb_struct_type_E)
				g_hash_table_lookup(struct_type_hash_table,
						(gpointer)to_downcase(&yylval.String[3]));
			
			switch (struct_type) {
			
			case	GB_LINKAGE_BOX:
			case	GB_LINKAGE_TABLE:
				if (WidDat->linkage) {
					UnexpectedToken(TOKEN_STRUCT, TOKEN_FIELD);
					break;
				}
				widget_data_linkage_new(WidDat);
				if (!WidDat->linkage) {
					UnexpectedToken(TOKEN_STRUCT, TOKEN_FIELD);
					break;
				}
				
				if (GetToken('{')==TOKEN_ERROR) {
					g_free(WidDat->linkage);
					WidDat->linkage=NULL;
					break;
				}
				Scan_structure_R(tree, WidDat->linkage);
				break;
			
			default:
				UnexpectedToken(yytoken, TOKEN_FIELD);
			case	TOKEN_ERROR:
			case	TOKEN_EOF:
				break;
			}
			break;

		case	TOKEN_WIDGET:
			if (!GB_TYPE_IS_WIDDAT(Struct->type)) {
				UnexpectedToken(TOKEN_WIDGET, TOKEN_FIELD);
				break;
			}
			
			struct_type=(gb_struct_type_E)g_hash_table_lookup(struct_type_hash_table,
							(gpointer)to_downcase(&yylval.String[10]));
			
			if (!GB_TYPE_IS_WIDDAT(struct_type)) {
				PARSER_ERROR(("structure is no widget: <%s>", yylval.String));
				g_free(yylval.String);
				break;
			}
			
			
			/* we don't accept windows as children
			*/
			if (GB_TYPE_IS_WIDDAT_WINDOW(struct_type)) {
				PARSER_UNEXP_WID(yylval.String, "window widget");
				g_free(yylval.String);
				break;
			}
			
			
			/* we need to be a container if we get a widget
			*/
			if (!GB_TYPE_IS_WIDDAT_CONTAINER(Struct->type)) {
				PARSER_UNEXP_WID(yylval.String, "parent is no container");
				g_free(yylval.String);
				break;
			}
			
			
			/* we need to have space for our widget
			*/
			if (HAS_MAX_CHILD_COUNT(GB_wCAST(base, Struct))) {
				PARSER_UNEXP_WID(yylval.String, "maximum child count reached in parent");
				g_free(yylval.String);
				break;
			}
			
			g_free(yylval.String);
			
			
			if (GetToken(TOKEN_STRING)==TOKEN_ERROR)
				break;
			string=yylval.String;
			if (GetToken('{')==TOKEN_ERROR) {
				g_free(string);
				break;
			}
			ChildDat=widget_data_new(struct_type, string, FALSE);
			
			
			/* we need to check for duplicate identifiers
			*/
			if (string && string[0]!=0)
				HANDLE_DUPS(string, ChildDat);
			
			g_free(string);
			
			tree_widget_data_insert_R(tree, GB_wCAST(base, Struct), ChildDat, ~0);
			
			Scan_structure_R(tree, GB_CAST(any, ChildDat));
			
			scan_children_only=TRUE;
			break;

		default:
			UnexpectedToken(yytoken, TOKEN_FIELD);
		case	TOKEN_ERROR:
		case	TOKEN_EOF:
			break;
		}
	}
	
	
	/* check for linkage specific stuff
	*/
	if (GB_TYPE_IS_WIDDAT(Struct->type)) {
		register gb_wdat_base_S	*WidDat=GB_wCAST(base, Struct);
		
		if (widget_data_linkage_check(WidDat)) {
			g_free(WidDat->linkage);
			WidDat->linkage=NULL;
		}
		
		widget_data_children_linkage_check(WidDat);
 	}
}


void
Scan_generic_field	(gb_any_S	*Struct,
			 gchar		*field_name)
{
	register const field_info_S	*FldInf;
	register field_value_U		value;
	register gboolean		set_value;
	
	g_assert(GB_IS_STRUCT(Struct));
	g_assert(field_name);
	
	FldInf=NULL;
	
	
	/* lookup the field type
	*/
	FldInf=g_hash_table_lookup(struct_field_hash_table_array[Struct->type], field_name);
	
	
	/* if we are a widget we accept other structure fields as well
	*/
	if (!FldInf && GB_TYPE_IS_OBJDAT(Struct->type))
		FldInf=g_hash_table_lookup(struct_field_hash_table_array[GB_OBJECT],
					   field_name);
	if (!FldInf && GB_TYPE_IS_WIDDAT(Struct->type))
		FldInf=g_hash_table_lookup(struct_field_hash_table_array[GB_WIDGET_BASE],
					   field_name);
	if (!FldInf && GB_TYPE_IS_WIDDAT_MISC(Struct->type))
		FldInf=g_hash_table_lookup(struct_field_hash_table_array[GB_WIDGET_MISC],
					   field_name);
	if (!FldInf && GB_TYPE_IS_WIDDAT_CONTAINER(Struct->type))
		FldInf=g_hash_table_lookup(struct_field_hash_table_array[GB_WIDGET_CONTAINER],
					   field_name);
	if (!FldInf && GB_TYPE_IS_WIDDAT_BIN(Struct->type))
		FldInf=g_hash_table_lookup(struct_field_hash_table_array[GB_WIDGET_BIN],
					   field_name);
	
	
	/* warn if we don't know the field type
	*/
	if (!FldInf) {
		PARSER_WARNING(("invalid field identifier (foreign?): `%s'", field_name));
		GetToken(TOKEN_EOF);
		return;
	}
	
	
	/* scan the field's value
	*/
	set_value=FALSE;
	switch (FldInf->field_type) {
	register guint	yytoken;
	register gb_wdat_base_S	*WidDat;
	register gb_wdat_base_S	*LnkDat;
	
	case	FIELD_FLOAT:
		yytoken=GetToken(TOKEN_ANY);
		switch (yytoken) {
		
		case	TOKEN_LONG:
			yylval.Double=yylval.Long;
		case	TOKEN_DOUBLE:
			value.float_val=yylval.Double;
			set_value=TRUE;
			break;
		
		case	TOKEN_ERROR:
			break;
			
		default:
			UnexpectedToken(yytoken, TOKEN_DOUBLE);
			break;
		}
		yylval.String=NULL;
		break;
	
	case	FIELD_BITS:
		if (GetToken(TOKEN_LONG)==TOKEN_ERROR) {
			yylval.String=NULL;
			break;
		}
		value.bit_val=yylval.Long;
		set_value=TRUE;
		yylval.String=NULL;
		break;
		
	case	FIELD_INT:
		if (GetToken(TOKEN_LONG)==TOKEN_ERROR) {
			yylval.String=NULL;
			break;
		}
		value.int_val=yylval.Long;
		set_value=TRUE;
		yylval.String=NULL;
		break;
		
	case	FIELD_ENUM:
		if (GetToken(TOKEN_LONG)==TOKEN_ERROR) {
			yylval.String=NULL;
			break;
		}
		value.enum_val=yylval.Long;
		set_value=TRUE;
		yylval.String=NULL;
		break;
	
	case	FIELD_BOOLEAN:
		if (GetToken(TOKEN_LONG)==TOKEN_ERROR) {
			yylval.String=NULL;
			break;
		}
		value.bool_val=yylval.Long;
		set_value=TRUE;
		yylval.String=NULL;
		break;
	
	case	FIELD_STRING:
	case	FIELD_POINTER:
		if (GetToken(TOKEN_STRING)==TOKEN_ERROR) {
			yylval.String=NULL;
			break;
		}
		value.string=yylval.String;
		set_value=TRUE;
		break;
	
	case	FIELD_WD_LINK:
		if (GetToken(TOKEN_STRING)==TOKEN_ERROR) {
			yylval.String=NULL;
			break;
		}
		
		value=structure_field_get_value(Struct, FldInf);
		
		WidDat=GB_wCAST(base, Struct);
		
		g_assert(GB_IS_WIDDAT(WidDat));
		
		LnkDat=g_hash_table_lookup(dups_ref_hash_table, yylval.String);
		if (!LnkDat)
			LnkDat=widget_data_lookup(yylval.String);
		
		if (LnkDat) {
			if (GUBI_DATA(WidDat)->tree!=GUBI_DATA(LnkDat)->tree) {
				PARSER_WARNING(("out of tree reference `%s' for `%s'",
						yylval.String,
						widget_data_symbol_name_get(WidDat)));
			} else if (LnkDat->type!=WidDat->type) {
				PARSER_WARNING(("invalid type in reference `%s' for `%s'",
						yylval.String,
						widget_data_symbol_name_get(WidDat)));
			} else {
				value.link=LnkDat;
				set_value=TRUE;
			}
		} else {
			if (yylval.String[0]!=0) {
				PARSER_WARNING(("unknown reference `%s' for `%s'",
						yylval.String,
						widget_data_symbol_name_get(WidDat)));
			}
		}
		break;
		
	default:
		yylval.String=NULL;
		g_assert_not_reached();
		break;
	}
	
	
	/* assign the field's value
	*/
	if (set_value)
		structure_field_set_value(Struct, value, FldInf);
	
	g_free(yylval.String);
}


void
Scan_handler_old	(gb_wdat_base_S	*WidDat)
{
	register int		yytoken;
	register int		connect_options;
	register gchar		*func_name;
	register gchar		*signal_name;
	register gchar		*data_name;
	
	g_assert(GB_IS_WIDDAT(WidDat));
	
	if (GetToken('{')==TOKEN_ERROR)
		return;
	
	connect_options=0;
	func_name=g_strdup("SigH_widget_dump");
	signal_name=g_strdup("destroy");
	data_name=g_strdup("NULL");
	
	while ((yytoken=GetToken(TOKEN_ANY))!='}') {
		
		if (YY_ABORT())
			break;
		
		switch (yytoken) {
		
		case	TOKEN_HANDLER_connect_options:
			if (GetToken(TOKEN_LONG)==TOKEN_ERROR)
				break;
			connect_options=yylval.Long;
			break;
		
		case	TOKEN_HANDLER_handler_func:
			if (GetToken(TOKEN_STRING)==TOKEN_ERROR)
				break;
			g_free(func_name);
			func_name=yylval.String;
			break;
		
		case	TOKEN_HANDLER_signal_name:
			if (GetToken(TOKEN_STRING)==TOKEN_ERROR)
				break;
			g_free(signal_name);
			signal_name=yylval.String;
			break;
		
		case	TOKEN_HANDLER_data:
			if (GetToken(TOKEN_STRING)==TOKEN_ERROR)
				break;
			g_free(data_name);
			data_name=yylval.String;
			break;
		
		default:
			UnexpectedToken(yytoken, TOKEN_FIELD);
		case	TOKEN_ERROR:
		case	TOKEN_EOF:
			break;
		}
	}
	
	widget_data_handler_add(WidDat,
				connect_options,
				func_name,
				signal_name,
				data_name);
}
