/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998, 1999 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 */
#include	<gle/gleconfig.h>

#include	"glegwidget.h"
#include	"gleprivate.h"
#include	"glegcontainer.h"



/* --- prototypes --- */
static void  gle_gwidget_class_init		(GleGWidgetClass*class);
static void  gle_gwidget_init			(GleGWidget	*gwidget);
static void  gle_gwidget_do_initialize		(GleGObject	*gobject);
static void  gle_gwidget_do_pre_destroy		(GleGObject	*gobject);
static void  gle_gwidget_do_instantiate		(GleGObject	*gobject);
static void  gle_gwidget_do_associate		(GleGObject	*gobject,
						 GtkObject	*object);
static void  gle_gwidget_do_foreign_associated	(GleGObject	*gobject);
static void  gle_gwidget_do_disassociate	(GleGObject	*gobject);
static void  gle_gwidget_do_check_args		(GleGObject	*gobject);
static void  gle_gwidget_do_apply_all_args	(GleGObject	*gobject);
static void  gle_gwidget_do_reset_all_args	(GleGObject	*gobject);
static void  gle_gwidget_do_restore_all_args	(GleGObject	*gobject);
static void  gle_gwidget_do_save_all_args	(GleGObject	*gobject);
static void  gle_gwidget_do_update_all_args	(GleGObject	*gobject);
static void  gle_gwidget_do_set_parent		(GleGWidget	*gwidget,
						 GleGWidget	*gcontainer);
static void  gle_gwidget_handle_parent_set	(GleGWidget	*gwidget,
						 GtkWidget	*previous_parent);


/* --- variables --- */
static GQuark		 quark_child_args = 0;
static GQuark		 quark_style_args = 0;
static GQuark		 quark_composite_name = 0;
static GleGObjectClass	*parent_class = NULL;
/* from glegobject.c */
extern GHashTable	*gle_gname_hash_table;


/* --- functions --- */
GtkType
gle_gwidget_get_type (void)
{
  static GtkType gwidget_type = 0;
  
  if (!gwidget_type)
    {
      GtkTypeInfo gwidget_info =
      {
	"GleGWidget",
	sizeof (GleGWidget),
	sizeof (GleGWidgetClass),
	(GtkClassInitFunc) gle_gwidget_class_init,
	(GtkObjectInitFunc) gle_gwidget_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
	(GtkClassInitFunc) NULL,
      };
      
      gwidget_type = gtk_type_unique (GLE_TYPE_GOBJECT, &gwidget_info);
      gtk_type_set_chunk_alloc (gwidget_type, GLE_GWIDGETS_PREALLOC);
    }
  
  return gwidget_type;
}

static void
gle_gwidget_class_init (GleGWidgetClass *class)
{
  GleGObjectClass *gobject_class;
  GleArgInfo *garg_info;
  
  parent_class = gtk_type_class (GLE_TYPE_GOBJECT);
  gobject_class = (GleGObjectClass*) class;
  
  quark_child_args = g_quark_from_static_string ("Child-Args");
  quark_style_args = g_quark_from_static_string ("Style-Args");
  quark_composite_name = g_quark_from_static_string ("gle-composite-name");
  
  gobject_class->initialize = gle_gwidget_do_initialize;
  gobject_class->pre_destroy = gle_gwidget_do_pre_destroy;
  
  gobject_class->instantiate = gle_gwidget_do_instantiate;
  gobject_class->associate = gle_gwidget_do_associate;
  gobject_class->foreign_associated = gle_gwidget_do_foreign_associated;
  gobject_class->disassociate = gle_gwidget_do_disassociate;
  
  gobject_class->check_args = gle_gwidget_do_check_args;
  gobject_class->apply_all_args = gle_gwidget_do_apply_all_args;
  gobject_class->reset_all_args = gle_gwidget_do_reset_all_args;
  gobject_class->restore_all_args = gle_gwidget_do_restore_all_args;
  gobject_class->save_all_args = gle_gwidget_do_save_all_args;
  gobject_class->update_all_args = gle_gwidget_do_update_all_args;

  class->set_parent = gle_gwidget_do_set_parent;
  GLE_CLASS_ADD_NOTIFY_METHOD (class, GleGWidgetClass, set_parent);

  /* bann arguments that are implicitely handled by GleGWidgetClass */
  garg_info = gle_arg_get_info ("GtkWidget::style", FALSE);
  gobject_class->banned_args = g_slist_prepend (gobject_class->banned_args, garg_info);
  garg_info = gle_arg_get_info ("GtkWidget::parent", FALSE);
  gobject_class->banned_args = g_slist_prepend (gobject_class->banned_args, garg_info);
}

static void
gle_gwidget_init (GleGWidget	 *gwidget)
{
  gwidget->parent = NULL;
}

GleGWidget*
gle_gwidget_new (GtkType	 type)
{
  GleGWidget *gwidget;
  
  g_return_val_if_fail (gtk_type_is_a (type, GTK_TYPE_WIDGET), NULL);
  
  gwidget = (GleGWidget*) gle_gobject_new (type);
  
  return gwidget;
}

static void
gle_gwidget_do_initialize (GleGObject *gobject)
{
  GleGWidget *gwidget;
  
  gwidget = GLE_GWIDGET (gobject);
  
  parent_class->initialize (gobject);
  
  //  gle_gobject_add_garg_group (gobject, quark_style_args, GLE_TYPE_GWIDGET, NULL);
  gle_gobject_add_garg_group (gobject, quark_child_args, GLE_TYPE_GWIDGET, NULL);
}

void
gle_gwidget_ref (GleGWidget *gwidget)
{
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  
  gle_gobject_ref (GLE_GOBJECT (gwidget));
}

void
gle_gwidget_unref (GleGWidget *gwidget)
{
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  
  gle_gobject_unref (GLE_GOBJECT (gwidget));
}

void
gle_gwidget_destroy (GleGWidget	    *gwidget)
{
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  
  gle_gobject_destroy (GLE_GOBJECT (gwidget));
}

void
gle_gwidget_do_pre_destroy (GleGObject *gobject)
{
  GleGWidget *gwidget;
  
  gwidget = GLE_GWIDGET (gobject);
  
  /* unlink from parent
   */
  if (gwidget->parent)
    gle_gcontainer_remove_child (GLE_GCONTAINER (gwidget->parent), gwidget);
  
  /* chain parent class' handler
   */
  parent_class->pre_destroy (gobject);
}

static void
gle_gwidget_do_instantiate (GleGObject *gobject)
{
  GleGWidget *gwidget;
  
  gwidget = GLE_GWIDGET (gobject);

  if (GLE_GWIDGET_COMPOSITE (gwidget))
    {
      gchar *name;

      name = gle_gwidget_get_composite_name (gwidget);
      if (name)
	{
	  if (!gwidget->parent)
	    g_warning ("cannot instantiate unparented composite `%s' proxy for `%s'",
		       gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gwidget)),
		       gtk_type_name (GLE_GOBJECT_OTYPE (gwidget)));
	  else
	    {
	      GtkWidget *widget;

	      widget = gle_gcontainer_find_composite (GLE_GCONTAINER (gwidget->parent), name);
	      if (!widget)
		g_message ("cannot instantiate empty composite `%s' proxy for `%s'",
			   gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gwidget)),
			   gtk_type_name (GLE_GOBJECT_OTYPE (gwidget)));
	      else
		GLE_GOBJECT (gwidget)->object = GTK_OBJECT (widget);
	    }
	}
      else
	g_message ("cannot instantiate unnamed composite `%s' proxy for `%s'",
		   gtk_type_name (GLE_GOBJECT_PROXY_TYPE (gwidget)),
		   gtk_type_name (GLE_GOBJECT_OTYPE (gwidget)));
    }
  else if (GLE_GWIDGET_INTERNAL (gwidget))
    g_message ("instantiation of internal `%s' requested - unhandled",
	       gtk_type_name (GLE_GOBJECT_OTYPE (gwidget)));
  else
    {
      GtkWidget *widget;

      parent_class->instantiate (gobject);
      
      widget = GLE_GWIDGET_WIDGET (gwidget);
      
      if (gwidget->parent && GLE_GOBJECT_IS_INSTANTIATED (gwidget->parent))
	gtk_container_add (GLE_GCONTAINER_CONTAINER (gwidget->parent), widget);
    }
}

static void
gle_gwidget_do_associate (GleGObject *gobject,
			  GtkObject  *object)
{
  GleGWidget *gwidget;
  
  gwidget = GLE_GWIDGET (gobject);
  
  parent_class->associate (gobject, object);
  
  gtk_signal_connect_object_after (gobject->object,
				   "parent_set",
				   GTK_SIGNAL_FUNC (gle_gwidget_handle_parent_set),
				   (gpointer) gwidget);
}

static void
gle_gwidget_do_foreign_associated (GleGObject *gobject)
{
  GleGWidget *gwidget;
  GtkWidget *widget;
  
  gwidget = GLE_GWIDGET (gobject);
  
  /* chain parent class' handler and check for correct association
   */
  parent_class->foreign_associated (gobject);
  
  widget = GLE_GWIDGET_WIDGET (gwidget);
  
  /* FIXME: rebuilding of child args can be optimized (reduced) */
  gle_gwidget_rebuild_child_args (gwidget);
  
  if (widget->parent)
    {
      GleGWidget *gparent;
      
      gparent = gle_widget_get_gwidget (widget->parent);
      if (gparent != gwidget->parent)
	{
	  if (gwidget->parent)
	    gle_gcontainer_remove_child (GLE_GCONTAINER (gwidget->parent), gwidget);
	  gle_gcontainer_add_child (GLE_GCONTAINER (gparent), gwidget);
	}
    }
  else if (gwidget->parent)
    gle_gcontainer_remove_child (GLE_GCONTAINER (gwidget->parent), gwidget);
  
  if (GTK_WIDGET_COMPOSITE_CHILD (widget))
    {
      GLE_GOBJECT_SET_FLAG (gwidget, GLE_GWIDGET_COMPOSITE);
      gle_gobject_set_qdata_full (GLE_GOBJECT (gwidget),
				 quark_composite_name,
				 gtk_widget_get_composite_name (widget),
				 g_free);
    }
  else
    {
      GLE_GOBJECT_UNSET_FLAG (gwidget, GLE_GWIDGET_COMPOSITE);
      gle_gobject_set_qdata (GLE_GOBJECT (gwidget), quark_composite_name, NULL);
    }
}

static void
gle_gwidget_handle_parent_set (GleGWidget *gwidget,
			       GtkWidget  *previous_parent)
{
  GtkWidget *widget;
  
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  
  widget = GLE_GWIDGET_WIDGET (gwidget);
  
  if (widget->parent)
    {
      GleGWidget *gparent;
      gboolean rebuild_args;
      
      gparent = gle_widget_get_gwidget (widget->parent);
      rebuild_args = TRUE;
      
      if (gwidget->parent)
	{
	  gle_gcontainer_remove_child (GLE_GCONTAINER (gwidget->parent), gwidget);
	  rebuild_args = FALSE;
	}
      
      if (gparent)
	{
	  gle_gcontainer_add_child (GLE_GCONTAINER (gparent), gwidget);
	  rebuild_args = FALSE;
	}
      
      if (rebuild_args)
	{
	  gle_gwidget_rebuild_child_args (gwidget);
	  gle_gwidget_rebuild_style_args (gwidget);
	}
    }
  else
    {
      /* we just got removed from our parent, in this case we keep
       * the child argument values
       */
    }
}

static void
gle_gwidget_do_disassociate (GleGObject	    *gobject)
{
  GleGWidget *gwidget;
  
  gwidget = GLE_GWIDGET (gobject);
  
  if (!GTK_OBJECT_DESTROYED (gobject->object))
    gtk_signal_disconnect_by_func (gobject->object,
				   GTK_SIGNAL_FUNC (gle_gwidget_handle_parent_set),
				   gwidget);
  
  parent_class->disassociate (gobject);
}

static void
gle_gwidget_do_check_args (GleGObject *gobject)
{
  GleGArgGroup *group;
  GleGWidget *gwidget;
  GSList *slist;
  
  gwidget = GLE_GWIDGET (gobject);
  
  parent_class->check_args (gobject);
  
  /* we override the checking of the parent class for those
   * arguments that are ours (child arguments are always
   * considered dirty for instantiation).
   */
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    {
      if (group->proxy_type == GLE_TYPE_GWIDGET)
	{
	  for (slist = group->gargs; slist; slist = slist->next)
	    {
	      GleGArg *garg;
	      
	      garg = slist->data;
	      if (GLE_GARG_IS_WRITABLE (garg))
		garg->needs_set = TRUE;
	    }
	}
    }
}

void
gle_gwidget_apply_child_args (GleGWidget *gwidget)
{
  GleGObject *gobject;
  GtkWidget *widget;
  
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gwidget));
  
  gobject = GLE_GOBJECT (gwidget);
  
  widget = GLE_GWIDGET_WIDGET (gwidget);
  
  if (widget->parent)
    {
      GleGArgGroup *group;
      GSList *slist;
      
      for (group = gobject->garg_groups;
	   group < gobject->garg_groups + gobject->n_garg_groups;
	   group++)
	if (group->garg_group == quark_child_args)
	  break;
      g_return_if_fail (group->garg_group == quark_child_args &&
			group->proxy_type == GLE_TYPE_GWIDGET);
      
      for (slist = group->gargs; slist; slist = slist->next)
	{
	  GleGArg *garg;
	  
	  garg = slist->data;
	  if (GLE_GARG_IS_WRITABLE (garg) &&
	      GLE_GARG_NEEDS_SET (garg))
	    gle_garg_set (garg, GTK_OBJECT (widget->parent), widget);
	}
    }
}

void
gle_gwidget_reset_child_args (GleGWidget *gwidget)
{
  GleGObject *gobject;
  GleGArgGroup *group;
  GSList *slist;
  
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  
  gobject = GLE_GOBJECT (gwidget);
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    if (group->garg_group == quark_child_args)
      break;
  g_return_if_fail (group->garg_group == quark_child_args &&
		    group->proxy_type == GLE_TYPE_GWIDGET);
  
  for (slist = group->gargs; slist; slist = slist->next)
    {
      GleGArg *garg;
      
      garg = slist->data;
      gle_garg_reset (garg);
    }
}

void
gle_gwidget_restore_child_args (GleGWidget *gwidget)
{
  GleGObject *gobject;
  GleGArgGroup *group;
  GSList *slist;
  
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  
  gobject = GLE_GOBJECT (gwidget);
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    if (group->garg_group == quark_child_args)
      break;
  g_return_if_fail (group->garg_group == quark_child_args &&
		    group->proxy_type == GLE_TYPE_GWIDGET);
  
  for (slist = group->gargs; slist; slist = slist->next)
    {
      GleGArg *garg;
      
      garg = slist->data;
      gle_garg_restore (garg);
    }
}

void
gle_gwidget_save_child_args (GleGWidget *gwidget)
{
  GleGObject *gobject;
  GleGArgGroup *group;
  GSList *slist;
  
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  
  gobject = GLE_GOBJECT (gwidget);
  
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    if (group->garg_group == quark_child_args)
      break;
  g_return_if_fail (group->garg_group == quark_child_args &&
		    group->proxy_type == GLE_TYPE_GWIDGET);
  
  for (slist = group->gargs; slist; slist = slist->next)
    {
      GleGArg *garg;
      
      garg = slist->data;
      gle_garg_save (garg);
    }
}

void
gle_gwidget_update_child_args (GleGWidget *gwidget)
{
  GleGObject *gobject;
  
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  
  gobject = GLE_GOBJECT (gwidget);
  
  if (GLE_GOBJECT_IS_INSTANTIATED (gwidget) &&
      !GTK_OBJECT_DESTROYED (gobject->object))
    {
      GtkWidget *widget;
      
      widget = GLE_GWIDGET_WIDGET (gwidget);
      
      if (widget->parent)
	{
	  GleGArgGroup *group;
	  GSList *slist;
	  
	  for (group = gobject->garg_groups;
	       group < gobject->garg_groups + gobject->n_garg_groups;
	       group++)
	    if (group->garg_group == quark_child_args)
	      break;
	  g_return_if_fail (group->garg_group == quark_child_args &&
			    group->proxy_type == GLE_TYPE_GWIDGET);
	  
	  for (slist = group->gargs; slist; slist = slist->next)
	    {
	      GleGArg *garg;
	      
	      garg = slist->data;
	      if (GLE_GARG_IS_READABLE (garg))
		{
		  GLE_GARG_NEEDS_SET (garg) = FALSE;
		  
		  gle_garg_get (garg, GTK_OBJECT (widget->parent), widget);
		}
	    }
	}
    }
}

void
gle_gwidget_rebuild_child_args (GleGWidget *gwidget)
{
  GleGObject *gobject;
  GleGArgGroup *group;
  GtkWidget *widget;
  GSList *slist;
  GtkType type;
  
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  
  gobject = GLE_GOBJECT (gwidget);
  
  /* figure our argument group (child args)
   */
  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    if (group->garg_group == quark_child_args)
      break;
  g_return_if_fail (group->garg_group == quark_child_args &&
		    group->proxy_type == GLE_TYPE_GWIDGET);
  
  /* remove existing args
   */
  slist = NULL;
  while (group->gargs)
    {
      GSList *tmp;
      GleGArg *garg;
      
      tmp = group->gargs;
      group->gargs = tmp->next;
      tmp->next = slist;
      slist = tmp;
      
      garg = slist->data;
      
      if (garg->widget)
	{
	  gtk_widget_destroy (garg->widget);
	  if (garg->widget)
	    {
	      g_warning ("auto destruction of widget for arg \"%s\" failed",
			 garg->arg_name);
	      garg->widget = NULL;
	    }
	}
      gle_garg_free (garg);
    }
  g_slist_free (slist);
  
  /* figure parent type to query the args from
   */
  widget = GLE_GWIDGET_WIDGET (gwidget);
  if (widget && widget->parent)
    type = GTK_OBJECT_TYPE (widget->parent);
  else if (gwidget->parent)
    type = GLE_GOBJECT_OTYPE (gwidget->parent);
  else
    type = 0;
  
  /* setup the child args group
   */
  while (type && gtk_type_is_a (type, GTK_TYPE_CONTAINER))
    {
      GtkArg  *args;
      GleGArg *garg;
      guint    nargs;
      guint    i;
      
      args = gtk_container_query_child_args (type, NULL, &nargs);
      
      for (i = 0; i < nargs; i++)
	{
	  const GleArgInfo *garg_info;
	  
	  garg_info = gle_arg_get_info (args[i].name, TRUE);
	  if (!garg_info)
	    {
	      g_warning ("unresolved child arg: \"%s\"", args[i].name);
	      continue;
	    }
          if (gle_class_arg_info_banned (GLE_GOBJECT_GET_CLASS (gwidget), garg_info))
	    continue;
	  
	  /* child args should actually never be flagged as GTK_ARG_CONSTRUCT_ONLY
	   */
	  if (!(garg_info->arg_flags & GTK_ARG_CONSTRUCT_ONLY))
	    {
	      garg = gle_garg_new (garg_info);
	      
	      g_assert (GTK_FUNDAMENTAL_TYPE (GLE_GARG_ARG_TYPE (garg)) < GTK_TYPE_OBJECT);
	      
	      group->gargs = g_slist_prepend (group->gargs, garg);
	    }
	}
      
      g_free (args);
      
      type = gtk_type_parent (type);
    }
  group->gargs = g_slist_reverse (group->gargs);
  
  /* notify */
  gle_gobject_rebuilt_args (gobject);
  
  gle_gwidget_update_child_args (gwidget);
  gle_gwidget_save_child_args (gwidget);
}

void
gle_gwidget_apply_style_args (GleGWidget *gwidget)
{
}

void
gle_gwidget_reset_style_args (GleGWidget *gwidget)
{
}

void
gle_gwidget_restore_style_args (GleGWidget *gwidget)
{
}

void
gle_gwidget_save_style_args (GleGWidget *gwidget)
{
}

void
gle_gwidget_update_style_args (GleGWidget *gwidget)
{
}

void
gle_gwidget_rebuild_style_args (GleGWidget *gwidget)
{
}

static void
gle_gwidget_do_apply_all_args (GleGObject *gobject)
{
  GleGWidget *gwidget;
  
  gwidget = GLE_GWIDGET (gobject);
  
  parent_class->apply_all_args (gobject);
  
  gle_gwidget_apply_style_args (gwidget);
  gle_gwidget_apply_child_args (gwidget);
}

static void
gle_gwidget_do_reset_all_args (GleGObject *gobject)
{
  GleGWidget *gwidget;
  
  gwidget = GLE_GWIDGET (gobject);
  
  parent_class->reset_all_args (gobject);
  
  gle_gwidget_reset_style_args (gwidget);
  gle_gwidget_reset_child_args (gwidget);
}

static void
gle_gwidget_do_restore_all_args (GleGObject *gobject)
{
  GleGWidget *gwidget;
  
  gwidget = GLE_GWIDGET (gobject);
  
  parent_class->restore_all_args (gobject);
  
  gle_gwidget_restore_style_args (gwidget);
  gle_gwidget_restore_child_args (gwidget);
}

static void
gle_gwidget_do_save_all_args (GleGObject *gobject)
{
  GleGWidget *gwidget;
  
  gwidget = GLE_GWIDGET (gobject);
  
  parent_class->save_all_args (gobject);
  
  gle_gwidget_save_style_args (gwidget);
  gle_gwidget_save_child_args (gwidget);
}

static void
gle_gwidget_do_update_all_args (GleGObject *gobject)
{
  GleGWidget *gwidget;
  
  gwidget = GLE_GWIDGET (gobject);
  
  parent_class->update_all_args (gobject);
  
  gle_gwidget_update_style_args (gwidget);
  gle_gwidget_update_child_args (gwidget);
}

void
gle_gwidget_set_parent (GleGWidget     *gwidget,
			GleGWidget     *gcontainer)
{
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  if (gcontainer)
    {
      g_return_if_fail (GLE_IS_GCONTAINER (gcontainer));
      g_return_if_fail (gwidget->parent == NULL);
    }
  else
    g_return_if_fail (gwidget->parent != NULL);

  gle_gwidget_ref (gwidget);
  if (gcontainer)
    gle_gwidget_ref (gcontainer);
  
  GLE_GWIDGET_GET_CLASS (gwidget)->set_parent (gwidget, gcontainer);

  GLE_GOBJECT_NOTIFY (gwidget, set_parent, gcontainer, NULL);

  if (gcontainer)
    gle_gwidget_unref (gcontainer);
  gle_gwidget_unref (gwidget);
}

static void
gle_gwidget_do_set_parent (GleGWidget *gwidget,
			   GleGWidget *gcontainer)
{
  gwidget->parent = gcontainer;
  gle_gwidget_rebuild_child_args (gwidget);
}

gboolean
gle_gwidget_is_shell_owned (GleGWidget *gwidget)
{
  g_return_val_if_fail (gwidget != NULL, FALSE);
  g_return_val_if_fail (GLE_IS_GWIDGET (gwidget), FALSE);

  while (gwidget->parent)
    {
      if (GLE_GOBJECT_SHELL_OWNED (gwidget))
	return TRUE;

      gwidget = gwidget->parent;
    }

  return GLE_GOBJECT_SHELL_OWNED (gwidget);
}

gchar*
gle_gwidget_get_composite_name (GleGWidget *gwidget)
{
  g_return_val_if_fail (gwidget != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GWIDGET (gwidget), NULL);

  if (GLE_GWIDGET_COMPOSITE (gwidget))
    {
      if (GLE_GOBJECT_IS_INSTANTIATED (gwidget) && GLE_GWIDGET_WIDGET (gwidget)->parent)
	gle_gobject_set_qdata_full (GLE_GOBJECT (gwidget),
				   quark_composite_name,
				   gtk_widget_get_composite_name (GLE_GWIDGET_WIDGET (gwidget)),
				   g_free);

      return gle_gobject_get_qdata (GLE_GOBJECT (gwidget), quark_composite_name);
    }

  return NULL;
}

gboolean
gle_gwidget_set_visible (GleGWidget *gwidget,
			 gboolean    visible)
{
  GleGArg *garg;
  gboolean was_visible;
  GtkWidget *widget;
  
  g_return_val_if_fail (gwidget != NULL, FALSE);
  g_return_val_if_fail (GLE_IS_GWIDGET (gwidget), FALSE);

  garg = gle_gobject_get_object_garg (GLE_GOBJECT (gwidget), "GtkWidget::visible");
  was_visible = GTK_VALUE_BOOL (garg->object_arg);
  GTK_VALUE_BOOL (garg->object_arg) = visible != FALSE;

  widget = GLE_GWIDGET_WIDGET (gwidget);
  if (widget)
    {
      if (GTK_VALUE_BOOL (garg->object_arg))
	gtk_widget_show (widget);
      else
	gtk_widget_hide (widget);
    }

  return was_visible;
}

GleGArg*
gle_gwidget_get_child_garg (GleGWidget  *gwidget,
			    const gchar *arg_name)
{
  GleGObject *gobject;
  GleGArgGroup *group;
  const GleArgInfo* arg_info;

  g_return_val_if_fail (gwidget != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GWIDGET (gwidget), NULL);
  g_return_val_if_fail (arg_name != NULL, NULL);

  gobject = GLE_GOBJECT (gwidget);

  arg_info = gle_arg_get_info (arg_name, TRUE);
  if (!arg_info)
    return NULL;

  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    {
      GSList *slist;

      if (group->garg_group != quark_child_args)
	continue;

      g_return_val_if_fail (group->garg_group == quark_child_args &&
			    group->proxy_type == GLE_TYPE_GWIDGET, NULL);

      for (slist = group->gargs; slist; slist = slist->next)
	{
	  GleGArg *garg;

	  garg = slist->data;
	  if (garg->info == arg_info)
	    return garg;
	}
    }

  return NULL;
}

GleGArg*
gle_gwidget_get_style_garg (GleGWidget  *gwidget,
			    const gchar *arg_name)
{
  GleGObject *gobject;
  GleGArgGroup *group;
  const GleArgInfo* arg_info;

  g_return_val_if_fail (gwidget != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GWIDGET (gwidget), NULL);
  g_return_val_if_fail (arg_name != NULL, NULL);

  gobject = GLE_GOBJECT (gwidget);

  arg_info = gle_arg_get_info (arg_name, TRUE);
  if (!arg_info)
    return NULL;

  for (group = gobject->garg_groups;
       group < gobject->garg_groups + gobject->n_garg_groups;
       group++)
    {
      GSList *slist;

      if (group->garg_group != quark_style_args)
	continue;

      g_return_val_if_fail (group->garg_group == quark_style_args &&
			    group->proxy_type == GLE_TYPE_GWIDGET, NULL);

      for (slist = group->gargs; slist; slist = slist->next)
	{
	  GleGArg *garg;

	  garg = slist->data;
	  if (garg->info == arg_info)
	    return garg;
	}
    }

  return NULL;
}

GtkWidget*
gle_widget_from_gname (const gchar *gname)
{
  GtkObject *object;

  g_return_val_if_fail (gname != NULL, NULL);

  object = gle_object_from_gname (gname);
  if (object && GTK_IS_WIDGET (object))
    return (GtkWidget*) object;
  else
    return NULL;
}

GleGWidget*
gle_widget_get_gwidget (GtkWidget      *widget)
{
  g_return_val_if_fail (widget != NULL, NULL);
  g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);
  
  return (GleGWidget*) gle_object_get_gobject (GTK_OBJECT (widget));
}

GleGWidget*
gle_widget_force_gwidget (GtkWidget	 *widget)
{
  g_return_val_if_fail (widget != NULL, NULL);
  g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);
  g_return_val_if_fail (!GLE_HAS_TAG (widget), NULL);
  
  return (GleGWidget*) gle_object_force_gobject (GTK_OBJECT (widget));
}

static void
gwidgets_collect (gpointer	 key,
		  gpointer	 value,
		  gpointer	 user_data)
{
  GSList **slist_p = user_data;
  GleGObject *gobject = value;
  
  if (gtk_type_is_a (GLE_GOBJECT_PROXY_TYPE (gobject), GLE_TYPE_GWIDGET))
    *slist_p = g_slist_prepend (*slist_p, gobject);
}

GSList*
gle_gwidgets_list (void)
{
  GSList *slist;
  
  slist = NULL;
  if (gle_gname_hash_table)
    g_hash_table_foreach (gle_gname_hash_table, gwidgets_collect, &slist);
  
  return slist;
}
