/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"gleselector.h"


/* --- defines ---*/
#define	GLE_SELECTOR_WARNING_TIMEOUT	(3000)


/* --- signals --- */
enum
{
  SIGNAL_CANDIDATE_CHECK,
  SIGNAL_CANDIDATE_SELECTED,
  SIGNAL_LAST
};
typedef	void	(*SignalCandidateCheck)		(GtkObject	*object,
						 GtkWidget	*new_candidate,
						 gint		*candidate_ok,
						 gpointer	 func_data);
typedef	void	(*SignalCandidateSelected)	(GtkObject	*object,
						 GtkWidget	*candidate,
						 gpointer	 func_data);


/* --- prototypes --- */
static void		gle_selector_class_init		(GleSelectorClass *class);
static void		gle_selector_init		(GleSelector	*selector);
static void		gle_selector_destroy		(GtkObject	*object);
static gint		gle_selector_event		(GtkWidget	*widget,
							 GdkEvent	*event);
static void		gle_selector_cleanup		(GleSelector	*selector);
static void		gle_selector_candidate_label_set(GleSelector	*selector,
							 GtkWidget	*candidate);
static void		gle_selector_warning_draw	(GleSelector	*selector);
     

/* --- variables --- */
static GtkWindowClass	*parent_class = NULL;
static gint		 selector_signals[SIGNAL_LAST] = { 0 };
static const gchar	*warning_rc_string =
"style'GleSelectorWarning'"
"{"
"font='-adobe-helvetica-medium-r-normal--*-120-*-*-*-*-*-*'"
"bg[NORMAL]={0.5,0.5,0.5}"
"base[NORMAL]={1.,0,0}"
"fg[NORMAL]={1.,1.,0}"
"}"
"widget'GleSelectorWarning*'style'GleSelectorWarning'"
;


/* --- functions --- */
GtkType
gle_selector_get_type (void)
{
  static GtkType selector_type = 0;

  if (!selector_type)
    {
      GtkTypeInfo selector_info =
      {
	"GleSelector",
	sizeof (GleSelector),
	sizeof (GleSelectorClass),
	(GtkClassInitFunc) gle_selector_class_init,
	(GtkObjectInitFunc) gle_selector_init,
	(GtkArgSetFunc) NULL,
	(GtkArgGetFunc) NULL,
      };

      selector_type = gtk_type_unique (gtk_window_get_type (), &selector_info);
    }

  return selector_type;
}

static void
gle_selector_marshal_candidate_check (GtkObject      *object,
				      GtkSignalFunc  func,
				      gpointer       func_data,
				      GtkArg         *args)
{
  SignalCandidateCheck sfunc = (SignalCandidateCheck) func;

  (* sfunc) (object, GTK_VALUE_WIDGET (args[0]), GTK_VALUE_POINTER (args[1]), func_data);
}

static void
gle_selector_marshal_candidate_selected (GtkObject      *object,
					 GtkSignalFunc  func,
					 gpointer       func_data,
					 GtkArg         *args)
{
  SignalCandidateSelected sfunc = (SignalCandidateSelected) func;

  (* sfunc) (object, GTK_VALUE_WIDGET (args[0]), func_data);
}

static void
gle_selector_class_init (GleSelectorClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;
  GtkWindowClass *window_class;

  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  container_class = (GtkContainerClass*) class;
  window_class = (GtkWindowClass*) class;

  parent_class = gtk_type_class (gtk_window_get_type ());

  selector_signals[SIGNAL_CANDIDATE_CHECK] =
    gtk_signal_new ("candidate_check",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleSelectorClass, candidate_check),
		    gle_selector_marshal_candidate_check,
		    GTK_TYPE_NONE, 2,
		    GTK_TYPE_WIDGET,
		    GTK_TYPE_POINTER);
  selector_signals[SIGNAL_CANDIDATE_SELECTED] =
    gtk_signal_new ("candidate_selected",
		    GTK_RUN_LAST,
		    object_class->type,
		    GTK_SIGNAL_OFFSET (GleSelectorClass, candidate_selected),
		    gle_selector_marshal_candidate_selected,
		    GTK_TYPE_NONE, 1,
		    GTK_TYPE_WIDGET);
  gtk_object_class_add_signals (object_class, selector_signals, SIGNAL_LAST);

  object_class->destroy = gle_selector_destroy;
  widget_class->event = gle_selector_event;
  widget_class->event = gle_selector_event;

  class->candidate_check = NULL;
  class->candidate_selected = NULL;
}

static void
gle_selector_init (GleSelector *selector)
{
  GtkWidget *main_vbox;
  GtkWidget *label;
  GtkWidget *any;
  GtkWidget *button;

  selector->in_selection = FALSE;
  selector->warning_visible = FALSE;
  selector->candidate = NULL;
  selector->query_cursor = NULL;
  selector->warning_timer = 0;
  
  gtk_widget_set (GTK_WIDGET (selector),
		  "GtkWindow::type", GTK_WINDOW_TOPLEVEL,
		  "GtkWindow::title", "GLE-Selector",
		  "GtkWindow::window_position", GTK_WIN_POS_CENTER,
		  "GtkWindow::allow_shrink", FALSE,
		  "GtkWindow::allow_grow", FALSE,
		  "GtkWindow::auto_shrink", FALSE,
		  NULL);
  main_vbox =
    gtk_widget_new (gtk_vbox_get_type (),
		    "GtkBox::homogeneous", FALSE,
		    "GtkBox::spacing", 5,
		    "GtkContainer::border_width", 10,
		    "GtkWidget::parent", selector,
		    "GtkWidget::visible", TRUE,
		    NULL);
  label	= gtk_widget_new (gtk_label_get_type (),
			  "GtkLabel::label", "Select a window by clicking",
			  "GtkWidget::visible", TRUE,
			  NULL);
  gtk_box_pack_start (GTK_BOX (main_vbox), label, FALSE, FALSE, 10);
  any =
    gtk_widget_new (gtk_frame_get_type (),
		    "GtkFrame::shadow", GTK_SHADOW_ETCHED_OUT,
		    "GtkFrame::label_xalign", 0.5,
		    "GtkFrame::label", "Candidate",
		    "GtkWidget::parent", main_vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  selector->candidate_label =
    gtk_widget_new (gtk_label_get_type (),
		    "GtkLabel::label", "<None>",
		    "GtkWidget::sensitive", FALSE,
		    "GtkWidget::parent", any,
		    "GtkWidget::visible", TRUE,
		    NULL);
  any =
    gtk_widget_new (gtk_hseparator_get_type (),
		    "GtkWidget::parent", main_vbox,
		    "GtkWidget::visible", TRUE,
		    NULL);
  button =
    gtk_widget_new (gtk_button_get_type (),
		    "GtkButton::label", "Abort",
		    "GtkContainer::border_width", 5,
		    "GtkObject::object_signal::clicked", gle_selector_abort, selector,
		    "GtkWidget::can_default", TRUE,
		    "GtkWidget::parent", main_vbox,
		    "GtkWidget::visible", TRUE,
		    "GtkWidget::has_default", TRUE,
		    NULL);
  
  selector->warning_window =
    gtk_widget_new (gtk_window_get_type (),
		    "GtkWindow::type", GTK_WINDOW_POPUP,
		    "GtkWindow::title", "GLE-Warning",
		    "GtkWindow::window_position", GTK_WIN_POS_CENTER,
		    "GtkWindow::allow_shrink", FALSE,
		    "GtkWindow::allow_grow", FALSE,
		    "GtkWindow::auto_shrink", FALSE,
		    "GtkWidget::name", "GleSelectorWarning",
		    "GtkContainer::border_width", 10,
		    "GtkObject::object_signal::expose_event", gle_selector_warning_draw, selector,
		    "GtkObject::signal::delete_event", gtk_true, NULL,
		    NULL);
  label	= gtk_widget_new (gtk_label_get_type (),
			  "GtkWidget::visible", TRUE,
			  "GtkWidget::parent", selector->warning_window,
			  "GtkLabel::label",
			  "GLE-Selector Warning:\n"
			  "There already is a server grab in effect.\n"
			  "Overriding this might lead to strange results.\n"
			  "Reinvoke to override.\n",
			  NULL);
}

static void
gle_selector_destroy (GtkObject	*object)
{
  GleSelector *selector;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GLE_IS_SELECTOR (object));

  selector = GLE_SELECTOR (object);

  if (selector->in_selection)
    gle_selector_abort (selector);

  gle_selector_reset (selector);
  gtk_widget_destroy (selector->warning_window);
  selector->warning_window = NULL;

  GTK_OBJECT_CLASS (parent_class)->destroy (object);
}

void
gle_selector_abort (GleSelector	   *selector)
{
  g_return_if_fail (selector != NULL);
  g_return_if_fail (GLE_IS_SELECTOR (selector));

  if (selector->in_selection)
    gle_selector_cleanup (selector);
  gle_selector_reset (selector);
}

static void
gle_selector_warning_draw (GleSelector *selector)
{
  GdkGC *frame_gc;
  GdkDrawable *drawable;
  guint max_width;
  guint max_height;

  g_return_if_fail (selector != NULL);
  g_return_if_fail (GLE_IS_SELECTOR (selector));

  if (!selector->warning_visible)
    return;

  drawable = selector->warning_window->window;
  frame_gc = selector->warning_window->style->base_gc[GTK_STATE_NORMAL];
  max_width = selector->warning_window->allocation.width;
  max_height = selector->warning_window->allocation.height;

  gdk_draw_rectangle (drawable, frame_gc,
		      FALSE,
		      0,
		      0,
		      max_width,
		      max_height);
  gdk_draw_rectangle (drawable, frame_gc,
		      FALSE,
		      1,
		      1,
		      max_width - 2,
		      max_height - 2);
  gdk_draw_rectangle (drawable, frame_gc,
		      FALSE,
		      2,
		      2,
		      max_width - 4,
		      max_height - 4);
}

static gint
gle_selector_warning_timeout (GleSelector *selector)
{
  g_return_val_if_fail (selector != NULL, FALSE);
  g_return_val_if_fail (GLE_IS_SELECTOR (selector), FALSE);

  selector->warning_timer = 0;
  selector->warning_visible = FALSE;
  gtk_widget_hide (selector->warning_window);

  return FALSE;
}

void
gle_selector_reset (GleSelector	   *selector)
{
  g_return_if_fail (selector != NULL);
  g_return_if_fail (GLE_IS_SELECTOR (selector));
  g_return_if_fail (selector->in_selection == FALSE);

  if (selector->warning_visible)
    {
      gtk_timeout_remove (selector->warning_timer);
      gle_selector_warning_timeout (selector);
    }

  if (selector->candidate)
    {
      gtk_widget_unref (selector->candidate);
      selector->candidate = NULL;
      gle_selector_candidate_label_set (selector, NULL);
    }
}

void
gle_selector_popup_warning (GleSelector    *selector)
{
  g_return_if_fail (selector != NULL);
  g_return_if_fail (GLE_IS_SELECTOR (selector));
  g_return_if_fail (selector->in_selection == FALSE);
  g_return_if_fail (selector->warning_visible == FALSE);
  g_return_if_fail (gdk_pointer_is_grabbed () == TRUE);

  selector->warning_visible = TRUE;
  if (warning_rc_string)
    {
      gtk_rc_parse_string (warning_rc_string);
      gtk_widget_set_rc_style (selector->warning_window);
      warning_rc_string = NULL;
    }
  gtk_widget_show (selector->warning_window);
  selector->warning_timer = gtk_timeout_add (GLE_SELECTOR_WARNING_TIMEOUT,
					     (GtkFunction) gle_selector_warning_timeout,
					     selector);
}

GtkWidget*
gle_selector_make_selection (GleSelector    *selector)
{
  GtkWidget *result;
  
  g_return_val_if_fail (selector != NULL, NULL);
  g_return_val_if_fail (GLE_IS_SELECTOR (selector), NULL);
  g_return_val_if_fail (selector->in_selection == FALSE, NULL);

  gtk_widget_ref (GTK_WIDGET (selector));

  gle_selector_reset (selector);

  gtk_container_need_resize (GTK_CONTAINER (selector));
  gtk_widget_show (GTK_WIDGET (selector));
  gdk_window_raise (GTK_WIDGET (selector)->window);

  gtk_grab_add (GTK_WIDGET (selector));
  selector->in_selection = TRUE;

  result = NULL;
  selector->query_cursor = gdk_cursor_new (GDK_DOT);
  while (gdk_pointer_grab (GTK_WIDGET (selector)->window,
			   TRUE,
			   GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK |
			   GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK,
			   NULL,
			   selector->query_cursor,
			   GDK_CURRENT_TIME));
  
  while (!gtk_main_iteration ());

  gtk_widget_hide (GTK_WIDGET (selector));
  
  if (selector->in_selection)
    {
      g_warning ("GleSelector: Gtk-mainloop quitted for unknown reason, aborting...");
      gle_selector_abort (selector);
      
      /* we will leave gtk's quit value set, since it wasn't
       * set by the selector
       */
    }
  else
    {
      /* reset Gtk's quit value,
       * give it something to process
       */
      gtk_widget_hide (GTK_WIDGET (selector));
      gdk_flush ();
      
      gtk_main_iteration ();
    }
  
  result = selector->candidate;
  if (!GTK_OBJECT_DESTROYED (selector) && result)
    gtk_signal_emit (GTK_OBJECT (selector), selector_signals[SIGNAL_CANDIDATE_SELECTED], result);
  
  gdk_cursor_destroy (selector->query_cursor);
  selector->query_cursor = NULL;
  
  gtk_widget_unref (GTK_WIDGET (selector));
  
  return result;
}

static void
gle_selector_cleanup (GleSelector *selector)
{
  g_return_if_fail (selector != NULL);
  g_return_if_fail (GLE_IS_SELECTOR (selector));
  g_return_if_fail (selector->in_selection == TRUE);
  
  gtk_grab_remove (GTK_WIDGET (selector));
  gdk_pointer_ungrab (GDK_CURRENT_TIME);
  selector->in_selection = FALSE;
  gtk_main_quit ();
}

static void
gle_selector_candidate_label_set (GleSelector	*selector,
				  GtkWidget	*candidate)
{
  GtkLabel *label;
  gchar *string;

  g_return_if_fail (selector != NULL);
  g_return_if_fail (GLE_IS_SELECTOR (selector));

  label = GTK_LABEL (selector->candidate_label);

  if (!candidate)
    string = "<None>";
  else
    {
      if (GTK_CHECK_TYPE (candidate, gtk_window_get_type ()))
	{
	  if (GTK_WINDOW (candidate)->title)
	    string = GTK_WINDOW (candidate)->title;
	  else
	    string = "<NULL> Title";
	}
      else if (GTK_CHECK_TYPE (candidate, gtk_menu_get_type ()))
	string = "GtkMenu";
      else
	string = "<Unknown Candidate>";
    }
  if (!g_str_equal (label->label, string))
    gtk_label_set (label, string);
  gtk_widget_set_sensitive (GTK_WIDGET (label), candidate && candidate == selector->candidate);
}

static gint
gle_selector_check_candidate (GleSelector    *selector,
			      GtkWidget	     *new_candidate)
{
  gint candidate_ok;

  candidate_ok = new_candidate && new_candidate != GTK_WIDGET (selector);
  if (candidate_ok)
    gtk_signal_emit (GTK_OBJECT (selector),
		     selector_signals[SIGNAL_CANDIDATE_CHECK],
		     new_candidate,
		     &candidate_ok);
  if (selector->candidate)
    gtk_widget_unref (selector->candidate);
  if (candidate_ok)
    {
      selector->candidate = new_candidate;
      gtk_widget_ref (selector->candidate);
    }
  else
    selector->candidate = NULL;

  gle_selector_candidate_label_set (selector, new_candidate);
  
  return candidate_ok;
}

static gint
gle_selector_event (GtkWidget	       *widget,
		    GdkEvent	       *event)
{
  GtkWidget *event_window;
  GleSelector *selector;
  gboolean event_handled;
  
  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GLE_IS_SELECTOR (widget), FALSE);

  selector = GLE_SELECTOR (widget);
  if (!selector->in_selection)
    {
      if (GTK_WIDGET_CLASS (parent_class)->event)
	return GTK_WIDGET_CLASS (parent_class)->event (widget, event);
      else
	return FALSE;
    }
  
  event_window = gtk_get_event_widget (event);

  if (event_window)
    while (event_window->parent)
      event_window = event_window->parent;
  
  event_handled = FALSE;
  switch (event->type)
    {
    case  GDK_LEAVE_NOTIFY:
    case  GDK_ENTER_NOTIFY:
      gle_selector_check_candidate (selector, event_window);
      break;
      
    case  GDK_BUTTON_PRESS:
    case  GDK_BUTTON_RELEASE:
      event_window = selector->candidate;
      if (gle_selector_check_candidate (selector, event_window))
	{
	  gle_selector_cleanup (selector);
	  gtk_signal_emit (GTK_OBJECT (selector),
			   selector_signals[SIGNAL_CANDIDATE_SELECTED],
			   selector->candidate);
	}
      event_handled = TRUE;
      break;
      
    case  GDK_DELETE:
      gle_selector_abort (selector);
      event_handled = TRUE;
      break;
      
    default:
      break;
    }
  
  return event_handled;
}
