/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"glegwidget.h"
#include	"gleprivate.h"



/* --- prototypes --- */
extern gpointer		_gle_gobject_new (GtkType        type,
					  gchar         *gle_name,
					  gboolean       generic_name,
					  guint          alloc_type);


/* --- functions --- */
GleGWidget*
gle_gwidget_new (GtkType         type,
		 const gchar    *gle_name,
		 gboolean        generic_name)
{
  GleGWidget *gwidget;
  gchar *name;

  g_return_val_if_fail (gtk_type_is_a (type, GTK_TYPE_WIDGET), NULL);
  g_return_val_if_fail (gle_name != NULL, NULL);
  g_return_val_if_fail (gle_name[0] != 0, NULL);

  name = gle_canonicalize_gle_name (g_strdup (gle_name));

  if (gle_gobject_lookup (name))
    {
      g_warning ("gle_gwidget_new(): GleGWidget \"%s\" already exists", name);
      g_free (name);
      return NULL;
    }

  gwidget = _gle_gobject_new (type, name, generic_name, 2);

  gwidget->parent = NULL;
  gwidget->children = NULL;
  gwidget->destroyed_handler = 0;
  gwidget->parent_set_handler = 0;

  return gwidget;
}


static void
gle_gwidget_parent_set (GleGWidget *gwidget,
			GtkWidget  *previous_parent)
{
  GtkWidget *widget;

  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));

  widget = GLE_GWIDGET_WIDGET (gwidget);
}

static void
gle_gwidget_widget_destroyed (GleGWidget     *gwidget)
{
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));

  gwidget->destroyed_handler = 0;
  gwidget->parent_set_handler = 0;
}

void
gle_gwidget_associate (GleGWidget     *gwidget,
		       GtkWidget      *widget)
{
  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_WIDGET (widget));
  g_return_if_fail (!GTK_OBJECT_DESTROYED (widget));
  g_return_if_fail (!GLE_HAS_TAG (widget));
  g_return_if_fail (!GLE_GOBJECT_IS_INSTANTIATED (gwidget));
  
  gle_gobject_associate (GLE_GOBJECT (gwidget), GTK_OBJECT (widget));
  
  gwidget->destroyed_handler = gtk_signal_connect_object (GTK_OBJECT (widget),
							  "destroy",
							  GTK_SIGNAL_FUNC (gle_gwidget_widget_destroyed),
							  (GtkObject*) gwidget);
  gwidget->parent_set_handler = gtk_signal_connect_object (GTK_OBJECT (widget),
							   "parent_set",
							   GTK_SIGNAL_FUNC (gle_gwidget_parent_set),
							   (GtkObject*) gwidget);
}

GtkWidget*
gle_gwidget_instantiate (GleGWidget     *gwidget)
{
  g_return_val_if_fail (gwidget != NULL, NULL);
  g_return_val_if_fail (GLE_IS_GWIDGET (gwidget), NULL);
  g_return_val_if_fail (!GLE_GOBJECT_IS_INSTANTIATED (gwidget), NULL);

  return NULL;
}

void
gle_gwidget_disassociate (GleGWidget     *gwidget)
{
  GtkWidget *widget;

  g_return_if_fail (gwidget != NULL);
  g_return_if_fail (GLE_IS_GWIDGET (gwidget));
  g_return_if_fail (GLE_GOBJECT_IS_INSTANTIATED (gwidget));

  widget = GLE_GWIDGET_WIDGET (gwidget);

  gtk_signal_disconnect (GTK_OBJECT (widget), gwidget->parent_set_handler);
  gwidget->parent_set_handler = 0;
  gtk_signal_disconnect (GTK_OBJECT (widget), gwidget->destroyed_handler);
  gwidget->destroyed_handler = 0;

  gle_gobject_disassociate (GLE_GOBJECT (gwidget));
}

gchar*
gle_widget_create_gle_name (GtkWidget *widget)
{
  gchar *gle_name;
  gchar *name_base;
  guint counter;
  GtkWidget *parent;

  g_return_val_if_fail (widget != NULL, NULL);
  g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);

  gle_name = gle_canonicalize_object_name (g_strdup (gtk_widget_get_name (widget)));
  parent = widget->parent;
  while (parent)
    {
      gchar *old_name;
      const gchar *parent_name;

      parent_name = gle_object_get_gle_name (GTK_OBJECT (parent));
      if (!parent_name)
	parent_name = gle_canonicalize_object_name_static (gtk_widget_get_name (parent));

      if ((parent_name[0] == 'G' || parent_name[0] == 'g') &&
	  (parent_name[1] == 'T' || parent_name[1] == 't') &&
	  (parent_name[2] == 'K' || parent_name[2] == 'k') &&
	  parent_name[4])
	parent_name += 3;

      old_name = gle_name;
      if ((gle_name[0] == 'G' || gle_name[0] == 'g') &&
	  (gle_name[1] == 'T' || gle_name[1] == 't') &&
	  (gle_name[2] == 'K' || gle_name[2] == 'k') &&
	  gle_name[4])
	gle_name += 3;
      gle_name = g_strconcat (parent_name,
			      ".",
			      gle_name,
			      NULL);
      g_free (old_name);

      parent = parent->parent;
    }

  gle_name = gle_canonicalize_gle_name (gle_name);

  name_base = gle_name;
  gle_name = g_strdup (name_base);
  counter = 1;
  while (gle_gobject_lookup (gle_name))
    {
      gchar buffer[64];
      
      sprintf (buffer, "-%u", counter++);
      g_free (gle_name);
      gle_name = g_strconcat (name_base, buffer, NULL);
    }
  g_free (name_base);
  
  return gle_name;
}

GleGWidget*
gle_widget_get_gwidget (GtkWidget      *widget)
{
  g_return_val_if_fail (widget != NULL, NULL);
  g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);

  return (GleGWidget*) gle_object_get_gobject (GTK_OBJECT (widget));
}

GleGWidget*
gle_widget_force_gwidget (GtkWidget      *widget)
{
  GleGWidget *gwidget;

  g_return_val_if_fail (widget != NULL, NULL);
  g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);
  g_return_val_if_fail (!GLE_HAS_TAG (widget), NULL);

  gwidget = gle_widget_get_gwidget (widget);
  if (!gwidget)
    {
      gwidget = gle_gwidget_new (GTK_OBJECT_TYPE (widget),
				 gle_widget_create_gle_name (widget),
				 TRUE);
      gle_gwidget_associate (gwidget, widget);
    }

  return gwidget;
}
