/* GLE - The GTK+ Layout Engine
 * Copyright (C) 1998 Tim Janik
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include	"config.h"

#include	"glemisc.h"
#include	"glepopups.h"
#include	<gdk/gdk.h>
#include	<string.h>
#include	<ctype.h>
#include	<gtk/gtkprivate.h>




/* --- variables --- */
static GtkWidget *gle_flash_widget = NULL;


/* --- functions --- */
void
gle_widget_make_sensitive (GtkWidget	*widget)
{
  if (!GTK_WIDGET_SENSITIVE (widget))
    gtk_widget_set_sensitive (widget, TRUE);
}

void
gle_widget_make_insensitive (GtkWidget	*widget)
{
  if (GTK_WIDGET_SENSITIVE (widget))
    gtk_widget_set_sensitive (widget, FALSE);
}

GtkWidget*
gle_class_show_signals (GtkObjectClass *klass,
			const gchar	*selected)
{
  GtkWidget	*window;
  GList		*item_list;
  GtkWidget	*item;
  guint		 i;
  gint		 pos;
  gchar		*message;
  GtkType	ctype;
  guint		tmp_pos;

  g_return_val_if_fail (klass != NULL, NULL);
  g_return_val_if_fail (GTK_OBJECT_CLASS (klass) != NULL, NULL);
  if (!selected)
    selected = "";

  /* create list items
   */
  pos = -1;
  item_list = NULL;
  ctype = klass->type;
  tmp_pos = 0;
  while (ctype)
    {
      GtkObjectClass *class;

      class = gtk_type_class (ctype);
	
      for (i = 0; i < class->nsignals; i++)
	{
	  GtkSignalQuery *query;
	  gchar *label;

	  query = gtk_signal_query (class->signals[i]);

	  if (!query)
	    {
	      if (i == 0)
		g_warning ("%s: signal slot 0 empty", gtk_type_name (ctype));
	      else
		g_warning ("%s: signal slot %d (after \"%s\") empty",
			   gtk_type_name (ctype),
			   i,
			   gtk_signal_name (class->signals[i - 1]));
	      continue;
	    }

	  if (g_str_equal ((gchar*) selected, (gchar*) query->signal_name) == 0)
	    pos = tmp_pos + i + 1;


	  label = g_strconcat (gtk_type_name (query->object_type),
			       "::",
			       query->signal_name,
			       "  (",
			       query->is_user_signal ? "U" : "-",
			       query->run_type & GTK_RUN_FIRST ? "F" : "-",
			       query->run_type == GTK_RUN_LAST ? "L" : "-",
			       ")  returns: ",
			       gtk_type_name (query->return_val),
			       NULL);
	  item = gtk_list_item_new_with_label (label);
	  g_free (label);
	  g_free (query);

	  gtk_label_get (GTK_LABEL (GTK_BIN (item)->child), &label);
	  gtk_object_set_user_data (GTK_OBJECT(item), label);
	  gtk_widget_show (GTK_WIDGET (item));
	  
	  item_list = g_list_prepend (item_list, item);
	}

      tmp_pos += i;
      ctype = gtk_type_parent (ctype);
    }
  item_list = g_list_reverse (item_list);
  if (pos != -1)
    pos = tmp_pos - pos;

  /* pop up the signal_name list
   */
  message = g_strconcat (gtk_type_name (klass->type), " signals:", NULL);
  window = gle_popup_simple_list (NULL,
				  "GLE Signal List",
				  NULL,
				  NULL,
				  NULL,
				  FALSE,
				  item_list,
				  pos,
				  NULL,
				  TRUE,
				  message);
  g_free (message);

  return window;
}

static struct
{
  gchar *name;
  guint32 value;
} gle_widget_flags[] =
{
  /* flag section 0 */
  { "destroyed",	GTK_DESTROYED },
  { "floating",		GTK_FLOATING },
  { "reserved_1",	GTK_RESERVED_1 },
  { "reserved_2",	GTK_RESERVED_2 },
  /* flag section 1 */
  { "----------------",	0 },
  { "toplevel",		GTK_TOPLEVEL },
  { "no_window",	GTK_NO_WINDOW },
  { "realized",		GTK_REALIZED },
  { "mapped",		GTK_MAPPED },
  { "visible",		GTK_VISIBLE },
  { "sensitive",	GTK_SENSITIVE },
  { "parent_sensitive",	GTK_PARENT_SENSITIVE },
  { "can_focus",	GTK_CAN_FOCUS },
  { "has_focus",	GTK_HAS_FOCUS },
  { "can_default",	GTK_CAN_DEFAULT },
  { "has_default",	GTK_HAS_DEFAULT },
  { "has_grab",		GTK_HAS_GRAB },
  { "basic",		GTK_BASIC },
  /* flag section 2 */
  { "----------------",	0 },
  { "user_style",	PRIVATE_GTK_USER_STYLE },
  { "redraw_pending",	PRIVATE_GTK_REDRAW_PENDING },
  { "resize_pending",	PRIVATE_GTK_RESIZE_PENDING },
  { "resize_needed",	PRIVATE_GTK_RESIZE_NEEDED },
  { "leave_pending",	PRIVATE_GTK_LEAVE_PENDING },
  { "has_shape_mask",	PRIVATE_GTK_HAS_SHAPE_MASK },
  { "gtk_in_reparent",	PRIVATE_GTK_IN_REPARENT },
};
static guint	gle_n_widget_flag_names = sizeof (gle_widget_flags) / sizeof (gle_widget_flags[0]);

static void
gle_item_toggle_x_flag (GtkItem   *item,
			GdkEventButton *event,
			gpointer   func_data,
			gboolean   priv)
{
  guint f;
  GtkObject *object;
  GtkLabel *label;
  gchar *string;

  g_return_if_fail (item != NULL);
  g_return_if_fail (GTK_IS_ITEM (item));

  if (event->type != GDK_2BUTTON_PRESS)
    return;

  f = (guint) func_data;
  object = gtk_object_get_data (GTK_OBJECT (item), "gle-toggle-object");
  g_return_if_fail (object != NULL);

  label = GTK_LABEL (GTK_BIN (item)->child);

  if (priv)
    GTK_PRIVATE_FLAGS (object) ^= f;
  else
    GTK_OBJECT_FLAGS (object) ^= f;

  string = g_strdup (label->label);
  if (priv)
    string[0] = GTK_PRIVATE_FLAGS (object) & f ? '+' : '-';
  else
    string[0] = GTK_OBJECT_FLAGS (object) & f ? '+' : '-';
  gtk_label_set (label, string);
  g_free (string);
}

static void
gle_item_toggle_object_flag (GtkItem   *item,
			     GdkEventButton *event,
			     gpointer   func_data)
{
  gle_item_toggle_x_flag (item, event, func_data, 0);
}

static void
gle_item_toggle_private_flag (GtkItem   *item,
			      GdkEventButton *event,
			      gpointer   func_data)
{
  gle_item_toggle_x_flag (item, event, func_data, 1);
}

GtkWidget*
gle_widget_show_flags (GtkWidget        *widget)
{
  GtkWidget	*window;
  GList		*item_list;
  GtkWidget	*item;
  guint		 i;
  gint		 pos;
  gint		 flag_sect;
  gchar		*message;

  g_return_val_if_fail (widget != NULL, NULL);
  g_return_val_if_fail (GTK_IS_WIDGET (widget), NULL);

  /* create list items
   */
  pos = -1;
  item_list = NULL;
  flag_sect = 0;
  for (i = 0; i < gle_n_widget_flag_names; i++)
    {
      gchar  *string;

      if (!gle_widget_flags[i].value)
	flag_sect++;
      
      string = g_strconcat (" ", gle_widget_flags[i].name, NULL);
      if (flag_sect > 1)
	string[0] = GTK_PRIVATE_FLAGS (widget) & gle_widget_flags[i].value ? '+' : '-';
      else
	string[0] = GTK_OBJECT_FLAGS (widget) & gle_widget_flags[i].value ? '+' : '-';
      item = gtk_list_item_new_with_label (string);
      g_free (string);
      /* gtk_label_get (GTK_LABEL (GTK_BIN (item)->child), &string);
       * gtk_object_set_user_data (GTK_OBJECT (item), string);
       */
      gtk_object_set_data (GTK_OBJECT (item), "gle-toggle-object", widget);
      gtk_widget_show (GTK_WIDGET (item));
      if (gle_widget_flags[i].value)
	gtk_signal_connect (GTK_OBJECT (item),
			    "button_press_event",
			    flag_sect > 1 ?
			    GTK_SIGNAL_FUNC (gle_item_toggle_private_flag) :
			    GTK_SIGNAL_FUNC (gle_item_toggle_object_flag),
			    (gpointer) gle_widget_flags[i].value);
      else
	gtk_widget_set_sensitive (item, FALSE);
      item_list = g_list_prepend (item_list, item);
    }
  item_list = g_list_reverse (item_list);
  if (pos != -1)
    pos = i - pos;

  /* pop up the flag list
   */
  message = g_strconcat (gtk_type_name (GTK_OBJECT (widget)->klass->type),
			 "\"",
			 widget->name ? widget->name : "",
			 "\"",
			 " flags:",
			 NULL);
  window = gle_popup_simple_list (NULL,
				  "GLE Flag List",
				  NULL,
				  NULL,
				  NULL,
				  FALSE,
				  item_list,
				  pos,
				  NULL,
				  FALSE,
				  message);
  gtk_signal_connect_object_while_alive (GTK_OBJECT (widget),
					 "destroy",
					 GTK_SIGNAL_FUNC (gtk_widget_destroy),
					 GTK_OBJECT (window));
  g_free (message);

  return window;
}

static gboolean	gtk_types_hard_initialized = FALSE;

static void
gle_hard_initialize_gtk_types (void)
{
  gtk_types_hard_initialized = TRUE;
#define GLE_TYPE_CALL(type_func)	type_func ();
#  include "gletypecalls.c"
#undef	GLE_TYPE_CALL
}

GtkType
gle_type_from_name (const gchar      *arg_name,
		    const gchar	**class_name_p)
{
  static gchar	*buffer = NULL;
  static guint	buffer_len = 0;
  GtkType	widget_type;
  const gchar	*p;
  guint		class_length;

  g_return_val_if_fail (arg_name != NULL, 0);
  
  p = strchr (arg_name, ':');
  if (!p)
    p = arg_name + strlen (arg_name);
  
  class_length = p - arg_name;

  g_return_val_if_fail (class_length > 0, 0);
  
  if (class_length < buffer_len)
    {
      g_free (buffer);
      buffer_len = class_length + 1;
      buffer = g_new (gchar, buffer_len);
    }
  g_memmove (buffer, arg_name, class_length);
  buffer[class_length + 1] = 0;
  
  widget_type = gtk_type_from_name (buffer);
  if (!widget_type && !gtk_types_hard_initialized)
    {
      gle_hard_initialize_gtk_types ();
      widget_type = gtk_type_from_name (buffer);
    }

  if (widget_type && class_name_p)
    *class_name_p = buffer;

  return widget_type;
}

static gint
gle_flash_timer (gpointer data)
{
  static gboolean toggle = FALSE;
  GtkWidget *widget;
  GdkGC *gc;

  toggle = !toggle;

  widget = gle_flash_widget;
  if (!widget || !widget->window)
    return TRUE;

  if (toggle)
    gc = gtk_widget_get_default_style ()->white_gc;
  else
    gc = gtk_widget_get_default_style ()->black_gc;

  gdk_draw_rectangle	  (widget->window,
			   gc,
			   FALSE,
			   widget->allocation.x,
			   widget->allocation.y,
			   widget->allocation.width-1,
			   widget->allocation.height-1);
  gdk_draw_rectangle	  (widget->window,
			   gc,
			   FALSE,
			   widget->allocation.x,
			   widget->allocation.y,
			   widget->allocation.width,
			   widget->allocation.height);
  gdk_draw_line	  (widget->window,
		   gc,
		   widget->allocation.x,
		   widget->allocation.y,
		   widget->allocation.x +
		   widget->allocation.width,
		   widget->allocation.y +
		   widget->allocation.height);
  gdk_draw_line	  (widget->window,
		   gc,
		   widget->allocation.x +
		   widget->allocation.width,
		   widget->allocation.y,
		   widget->allocation.x,
		   widget->allocation.y +
		   widget->allocation.height);
  

  return TRUE;
}

static void
gle_expose_widget (GtkWidget *widget)
{
  if (widget->window)
    gdk_window_clear_area_e (widget->window, 0, 0, 0, 0);
}

void
gle_set_flash_widget (GtkWidget *widget)
{
  static gint flash_timer = 0;
  static gint flash_widget_handler1;
  static gint flash_widget_handler2;

  if (widget)
    g_return_if_fail (GTK_IS_WIDGET (widget));

  if (!flash_timer && widget)
    flash_timer = gtk_timeout_add (GLE_FLASH_MSEC, gle_flash_timer, NULL);

  if (flash_timer && !widget)
    {
      gtk_timeout_remove (flash_timer);
      flash_timer = 0;
    }

  if (gle_flash_widget)
    {
      gtk_signal_disconnect (GTK_OBJECT (gle_flash_widget), flash_widget_handler1);
      gtk_signal_disconnect (GTK_OBJECT (gle_flash_widget), flash_widget_handler2);
      gle_expose_widget (gle_flash_widget);
    }

  gle_flash_widget = widget;
  if (gle_flash_widget)
    {
      flash_widget_handler1 = gtk_signal_connect_object (GTK_OBJECT (gle_flash_widget),
							 "destroy",
							 GTK_SIGNAL_FUNC (gle_set_flash_widget),
							 NULL);
      flash_widget_handler2 = gtk_signal_connect_object (GTK_OBJECT (gle_flash_widget),
							 "size_allocate",
							 GTK_SIGNAL_FUNC (gle_expose_widget),
							 GTK_OBJECT (gle_flash_widget));
    }
}

GtkWidget*
gle_get_flash_widget (void)
{
  return gle_flash_widget;
}

gint
gtk_clist_find_row (GtkCList *clist,
		    gpointer  data)
{
  GList *list;
  gint n;

  g_return_val_if_fail (clist != NULL, -1);
  g_return_val_if_fail (GTK_IS_CLIST (clist), -1);

  if (clist->rows < 1)
    return -1;

  n = 0;
  list = clist->row_list;
  while (list)
    {
      GtkCListRow *clist_row;

      clist_row = list->data;
      if (clist_row->data == data)
	break;
      n++;
      list = list->next;
    }

  if (list)
    return n;

  return -1;
}

static	gint	destroy_handler = 0;
static GList	*destroy_queue = NULL;

static gint
gtk_delayed_destroy_handler (gpointer data)
{
  GList *list;

  list = destroy_queue;
  while (list)
    {
      GtkWidget *widget;

      widget = list->data;
      list = list->next;

      gtk_widget_destroy (widget);
      destroy_queue = g_list_remove (destroy_queue, widget);
    }

  destroy_handler = 0;
  return FALSE;
}

void
gtk_widget_queue_destroy (GtkWidget      *widget)
{
  if (!g_list_find (destroy_queue, widget))
    {
      if (!destroy_handler)
	destroy_handler = gtk_idle_add (gtk_delayed_destroy_handler, NULL);
      destroy_queue = g_list_prepend (destroy_queue, widget);
    }
}

GList*
gtk_widget_query_arg_list (GtkWidget *widget,
			   GtkType    class_type)
{
  GtkArg* args;
  gint nargs;
  GList *list;
  guint i;

  nargs = 0;
  args = gtk_object_query_args	 (class_type, &nargs);
  gtk_widget_set (widget, "GtkObject::signal::destroy", g_free, args, NULL);

  list = NULL;
  for (i = nargs; i > 0; i--)
    {
      list = g_list_prepend (list, &args[i - 1]);
    }

  return list;
}

gchar*
g_strdown (gchar  *string)
{
  register gchar *s;

  g_return_val_if_fail (string, NULL);

  s = string;

  while (*s)
    {
      *s = tolower (*s);
      s++;
    }

  return string;
}

gchar*
g_strup (gchar	*string)
{
  register gchar *s;

  g_return_val_if_fail (string, NULL);

  s = string;

  while (*s)
    {
      *s = toupper (*s);
      s++;
    }

  return string;
}

gchar*
g_basename (const gchar	   *file_name)
{
  register gchar *base;

  g_return_val_if_fail (file_name != NULL, NULL);

  base = strrchr (file_name, '/');
  if (base)
    base++;
  else
    base = (gchar*) file_name;

  return base;
}

gchar*
g_strconvert (gchar	     *mod_string,
	      gchar	     new_delim,
	      gboolean	     downcase,
	      gboolean	     upcase)
{
  register gchar *delimiters = "_-|> <.";
  register gchar *string;

  g_return_val_if_fail (mod_string != NULL, NULL);

  string = mod_string;
  while (*string)
    {
      if (new_delim && strchr (delimiters, *string))
	*string = new_delim;
      else if (upcase)
	*string = toupper (*string);
      else if (downcase)
	*string = tolower (*string);

      string++;
    }

  return mod_string;
}
