/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include "buffer_tile.h"


#define DATA   (&tile_buffer->tiles[i].data)
#define VALID  (tile_buffer->tiles[i].valid)
#define VALID2 (tile_buffer->tiles[i].validating)


static void
gimp_tile_buffer_delete (GimpBuffer * buffer)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint i;

  for (i = 0; i < tile_buffer->ntiles; i++)
    {
      d_uninit (DATA);
    }

  g_free (tile_buffer->tiles);
  gimp_buffer_uninit (buffer);
  g_free (tile_buffer);
}


static gboolean
gimp_tile_buffer_alloc (GimpBuffer   *buffer,
                        GimpArea     *area,
                        Alloc         how)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  static GimpMemPtr junk[4];
  static int need_init = 1;
  gint fails = 0;
  GimpPoint p;
  
  /* for now, assume all tiles are 64 x 64 */
  if (need_init)
    {
      gint i, n;
      
      for (i = 0; i < 4; i++)
        {
          n = (buffer->_x.period * buffer->_y.period * (i+1));
          d_init (&junk[i]);
          d_alloc (&junk[i], n);
          d_update (&junk[i]);
          memset (guchar_d (&junk[i]), 128, n);
          d_release (&junk[i]);
        }
      
      need_init = 0;
    }

  for (p.y = area->a.y;
       p.y < area->b.y;
       p.y += buffer->_y.period)
    {
      for (p.x = area->a.x;
           p.x < area->b.x;
           p.x += buffer->_x.period)
        {
          gint i = gimp_buffer_tilenum (buffer, &p);
          
          switch (how)
            {
            case ALLOC_ALLOC:
              if (! d_is_alloced (DATA))
                {
                  gint n = gimp_buffer_depth(buffer);
                  if (d_join (&junk[n-1], DATA) != TRUE)
                    {
                      fails++;
                    }
                }
              break;
              
            case ALLOC_UNALLOC:
              if (d_is_alloced (DATA))
                {
                  if (d_unalloc (DATA) == TRUE)
                    {
                      VALID = FALSE;
                    }
                  else
                    {
                      fails++;
                    }
                }
              break;
              
            case ALLOC_NONE:
              g_warning ("bad value");
              return FALSE;
            }
        }
    }
  
  if (fails != 0)
    g_warning ("%d tiles failed alloc", fails);

  return TRUE;
}


static gboolean
gimp_tile_buffer_map (GimpBuffer    *dst,
                      GimpArea      *darea,
                      GimpBuffer    *src,
                      GimpArea      *sarea)
{
  GimpTileBuffer * db = GIMP_TILE_BUFFER (dst);
  GimpTileBuffer * sb = GIMP_TILE_BUFFER (src);
  GimpPoint sp, dp;
  GimpTile *s, *d;
  gint fails = 0;
  

  /* for each dest row */
  for (dp.y = darea->a.y, sp.y = sarea->a.y;
       dp.y < darea->b.y;
       dp.y += dst->_y.period, sp.y += src->_y.period)
    {
      if (sp.y == sarea->b.y)
        sp.y = sarea->a.y;

      /* for each dest col */
      for (dp.x = darea->a.x, sp.x = sarea->a.x;
           dp.x < darea->b.x;
           dp.x += dst->_x.period, sp.x += src->_x.period)
        {
          if (sp.x == sarea->b.x)
            sp.x = sarea->a.x;

          /* get the src tile */
          s = &sb->tiles[gimp_buffer_tilenum (src, &sp)];
          if (s->validating == TRUE)
            {
              fails++;
              continue;
            }

          /* get the dest tile */
          d = &db->tiles[gimp_buffer_tilenum (dst, &dp)];
          if (d->validating == TRUE)
            {
              fails++;
              continue;
            }

          /* temp, is this a good idea? */
          if (&s->data == &d->data)
            {
              continue;
            }
          
          /* copy the data */
          d_uninit (&d->data);
          if (d_join (&s->data, &d->data) != TRUE)
            {
              fails++;
              continue;
            }

          /* copy the valid state */
          d->valid = s->valid;
        }
    }

  if (fails != 0)
    g_warning ("%d tiles failed map", fails);
  
  return TRUE;
}


static gboolean
gimp_tile_buffer_validate (GimpBuffer  *buffer,
                           GimpArea    *area, 
                           Validate     how)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint fails = 0;
  GimpArea a;

  /* for each row */
  for (a.a.y = area->a.y;
       a.a.y < area->b.y;
       a.a.y += buffer->_y.period)
    {
      /* for each column */
      for (a.a.x = area->a.x;
           a.a.x < area->b.x;
           a.a.x += buffer->_x.period)
        {
          gint i = gimp_buffer_tilenum (buffer, &a.a);

          switch (how)
            {
            case VALIDATE_VALIDATE:
              if (VALID == TRUE)
                break;

              a.b.x = a.a.x + buffer->_x.period;
              a.b.y = a.a.y + buffer->_y.period;
          
              if (buffer->vfunc == NULL)
                {
                  gimp_tile_buffer_alloc (buffer, &a, ALLOC_UNALLOC);
                  gimp_tile_buffer_alloc (buffer, &a, ALLOC_ALLOC);
                  VALID = TRUE;
                }
              else
                {
                  if (VALID2 == TRUE)
                    {
                      g_warning ("recursive validate");
                      fails++;
                      break;
                    }

                  if (! d_is_alloced (DATA))
                    {
                      gimp_tile_buffer_alloc (buffer, &a, ALLOC_ALLOC);
                    }
                  
                  if (d_write (DATA) != TRUE)
                    {
                      fails++;
                      break;
                    }

                  VALID2 = TRUE;
                  buffer->vfunc (buffer, &a, guchar_d (DATA));
                  VALID2 = FALSE;

                  d_release (DATA);
                  VALID = TRUE;
                }
              break;
              

            case VALIDATE_INVALIDATE:
              if (d_usecount (DATA) == 0)
                {
                  VALID = FALSE;
                }
              else
                {
                  g_warning ("in use");
                  fails++;
                }
              break;

            case VALIDATE_NONE:
              g_warning ("bad value");
              return FALSE;
            }
        }
    }
  
  if (fails != 0)
    g_warning ("%d tiles failed valid", fails);

  return TRUE;
}



static gboolean
gimp_tile_buffer_use (GimpBuffer  *buffer,
                      GimpPortion *portion, 
                      Use          how)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint i = gimp_buffer_tilenum (buffer, &portion->bounds.a);

  if ((VALID != TRUE) && ((how == USE_READ) ||
                          (how == USE_UPDATE) ||
                          (how == USE_WRITE)))
    {
      if (gimp_tile_buffer_validate (buffer, &portion->bounds,
                                     VALIDATE_VALIDATE) != TRUE)
        {
          g_warning ("tilebuffer autovalidate failed");
          return FALSE;
        }
    }

  switch (how)
    {
    case USE_READ:
      g_return_val_if_fail (d_read (DATA) == TRUE, FALSE);
      return TRUE;

    case USE_UPDATE:
      g_return_val_if_fail (d_update (DATA) == TRUE, FALSE);
      return TRUE;

    case USE_WRITE:
      g_return_val_if_fail (d_write (DATA) == TRUE, FALSE);
      return TRUE;

    case USE_RELEASE:
      g_return_val_if_fail (d_release (DATA) == TRUE, FALSE);
      return TRUE;

    case USE_NONE:
      g_warning ("bad value");
      return FALSE;
    }

  return FALSE;
}


static gboolean
gimp_tile_buffer_query (GimpBuffer    *buffer,
                        GimpPortion   *portion,
                        GimpMemStatus *status)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint i = gimp_buffer_tilenum (buffer, &portion->bounds.a);

  status->alloced   = (d_is_alloced (DATA) ? TRUE : FALSE);
  status->valid     = VALID;
  status->usecount  = (d_usecount (DATA));

  return TRUE;
}


static gboolean
gimp_tile_buffer_data (GimpBuffer      *buffer,
                       GimpPortion     *portion,
                       GimpPixelArray  *array)
{
  GimpTileBuffer * tile_buffer = GIMP_TILE_BUFFER (buffer);
  gint i = gimp_buffer_tilenum (buffer, &portion->bounds.a);

  /* describe the memory layout */
  array->tag       = buffer->_z.tag;
  array->width     = portion->focus.b.x - portion->focus.a.x;
  array->height    = portion->focus.b.y - portion->focus.a.y;
  array->pixstride = buffer->_z.len;
  array->rowstride = buffer->_z.len * buffer->_x.period;

  /* get a pointer to the memory */
  array->data = DATA;

  /* and save an offset to the first pixel */
  array->offset =
    ((portion->focus.a.y - portion->bounds.a.y) * array->rowstride) +
    ((portion->focus.a.x - portion->bounds.a.x) * array->pixstride);

  return TRUE;
}



static GimpTileBufferClass my_class =
{
  {
    BUFFER_TILED,
    gimp_tile_buffer_delete,
    gimp_tile_buffer_alloc,
    gimp_tile_buffer_map,
    gimp_tile_buffer_validate,
    gimp_tile_buffer_use,
    gimp_tile_buffer_query,
    gimp_tile_buffer_data
  }
};






GimpTileBuffer *
gimp_tile_buffer_new (Tag   tag,
                      gint  width,
                      gint  height,
                      gint  tile_width,
                      gint  tile_height)
{
  GimpTileBuffer * tile_buffer;
  gint i;

  if (tile_width == 0)
    tile_width = 64;

  if (tile_height == 0)
    tile_height = 64;

  tile_buffer = g_new (GimpTileBuffer, 1);

  gimp_buffer_init (GIMP_BUFFER (tile_buffer), tag,
                    width, tile_width, 0,
                    height, tile_height, 0);

  tile_buffer->ntiles =
    ((width + tile_width - 1)   / tile_width) *
    ((height + tile_height - 1) / tile_height);
  
  tile_buffer->tiles = g_new (GimpTile, tile_buffer->ntiles);

  for (i = 0; i < tile_buffer->ntiles; i++)
    {
      d_init (DATA);
      VALID = FALSE;
      VALID2 = FALSE;
    }

  GIMP_BUFFER (tile_buffer)->klass = (void*) &my_class;

  return tile_buffer;
}


