/* The GIMP -- an image manipulation program
 * Copyright (C) 1995 Spencer Kimball and Peter Mattis
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <sys/errno.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>

#include "memory.h"
#include "debug.h"


/* cache */
static GimpMemList active   = {NULL, NULL, 0, 0};
static GimpMemList inactive = {NULL, NULL, 0, 0};
static GimpMemList swapped  = {NULL, NULL, 0, 0};

/* swap */
static SwapFile * swapfile = NULL;
static gint mem_high = 8 * 1024 * 1024;
static gint mem_low  = 1 * 1024 * 1024;



void
gimp_memory_init (gchar * filename)
{
  static SwapFile file;
  if (swap_new (&file, filename) != TRUE)
    g_warning ("swap failed to init");
  else
    swapfile = &file;
}



#define gimp_mem_is_joined(mem) \
          ((mem) && ((mem)->ptrcount>1))

#define gimp_mem_dirty(mem) \
          ((mem) && ((mem)->swap.dirty = TRUE))

#define gimp_mem_unshare(mem) \
          ((mem) && ((mem)->sharable = FALSE))

#define gimp_mem_ptr(mem) \
          (((mem) && (mem)->usecount) ? (mem)->data : NULL)


static void
gimp_mem_link (GimpMem *     mem,
               GimpMemList * list)
{
  if (list->head)
    list->head->cache.prev = mem;
  
  mem->cache.list = list;
  mem->cache.next = list->head;
  mem->cache.prev = NULL;

  list->head = mem;
  if (list->tail == NULL)
    list->tail = mem;
  list->len++;
  list->size += mem->size;
}


static void
gimp_mem_unlink (GimpMem * mem)
{
  GimpMemList * list = mem->cache.list;

  if (list == NULL)
    return;
  
  if (mem->cache.prev)
    mem->cache.prev->cache.next = mem->cache.next;
  if (mem->cache.next)
    mem->cache.next->cache.prev = mem->cache.prev;

  if (list->head == mem)
    list->head = mem->cache.next;
  if (list->tail == mem)
    list->tail = mem->cache.prev;
  list->len--;
  list->size -= mem->size;
  
  mem->cache.list = NULL;
  mem->cache.next = NULL;
  mem->cache.prev = NULL;      
}


static void
gimp_mem_move (GimpMem *     mem,
               GimpMemList * list)
{
  g_return_if_fail (mem != NULL);
  g_return_if_fail (list != NULL);

  if (mem->cache.list == list)
    return;

  gimp_mem_unlink (mem);
  gimp_mem_link (mem, list);
      
  if (active.size + inactive.size < mem_high)
    return;
      
  while (((mem = inactive.tail) != NULL) &&
         (active.size + inactive.size > mem_low))
    {
      if (mem->swap.dirty == TRUE)
        {
          if (mem->swap.tag == -1)
            mem->swap.tag = swap_alloc (mem->swap.file,
                                        mem->size);
          
          if (swap_write (mem->swap.file,
                          mem->swap.tag,
                          mem->data,
                          mem->size) != TRUE)
            {
              g_warning ("ouch, dropping memory..");
              mem->swap.tag = -1;
            }
          
          mem->swap.dirty = FALSE;
        }
      
      g_free (mem->data);
      mem->data = NULL;
      
      gimp_mem_unlink (mem);
      gimp_mem_link (mem, &swapped);
    }
}


static GimpMem *
gimp_mem_new (gint size)
{
  GimpMem * mem;

  mem = g_new (GimpMem, 1);

  mem->data = NULL;
  mem->size = size;
  mem->ptrcount = 0;
  mem->usecount = 0;
  mem->sharable = TRUE;
  mem->cache.list = NULL;
  mem->cache.next = NULL;
  mem->cache.prev = NULL;
  mem->swap.file = swapfile;
  mem->swap.tag = -1;
  mem->swap.dirty = FALSE;

  gimp_mem_move (mem, &inactive);

  return mem;
}


static gboolean
gimp_mem_use (GimpMem * mem)
{
  g_return_val_if_fail (mem != NULL, FALSE);

  if (mem->usecount == 0)
    {
      if (mem->data == NULL)
        {
          mem->data = g_malloc (mem->size);

          if (mem->swap.tag == -1)
            {
              mem->swap.dirty = TRUE;
            }
          else
            {
              if (swap_read (mem->swap.file, mem->swap.tag,
                             mem->data, mem->size) != TRUE)
                {
                  g_free (mem->data);
                  mem->data = NULL;
                  return FALSE;
                }
              mem->swap.dirty = FALSE;
            }
        }

      gimp_mem_move (mem, &active);
    }
  
  mem->usecount++;

  return TRUE;
}


static gboolean
gimp_mem_unuse (GimpMem * mem)
{
  g_return_val_if_fail (mem != NULL, FALSE);

  mem->usecount--;
  
  if (mem->usecount == 0)
    {
      mem->sharable = TRUE;
      gimp_mem_move (mem, &inactive);
    }
  
  return TRUE;
}


static GimpMem *
gimp_mem_dup (GimpMem * mem)
{
  GimpMem * newmem;

  g_return_val_if_fail (mem != NULL, NULL);

  newmem = gimp_mem_new (mem->size);
  g_return_val_if_fail (newmem != NULL, NULL);

  if (gimp_mem_use (mem) == TRUE)
    {
      if (gimp_mem_use (newmem) == TRUE)
        {
          memcpy (newmem->data, mem->data, mem->size);
          gimp_mem_unuse (newmem);
        }
      gimp_mem_unuse (mem);
    }
  
  return newmem;
}


static GimpMem *
gimp_mem_attach (GimpMem * mem,
                 gboolean  use)
{
  g_return_val_if_fail (mem != NULL, NULL);

  if (mem->sharable != TRUE)
    {
      mem = gimp_mem_dup (mem);
      g_return_val_if_fail (mem != NULL, NULL);
    }

  mem->ptrcount++;
  
  if (use == TRUE)
    if (gimp_mem_use (mem) != TRUE)
      g_warning ("gimp_mem_attach use failed");
  
  return mem;
}


static gboolean
gimp_mem_detach (GimpMem * mem,
                 gboolean  unuse)
{
  g_return_val_if_fail (mem != NULL, FALSE);

  if (unuse == TRUE)
    if (gimp_mem_unuse (mem) != TRUE)
      g_warning ("gimp_mem_detach unuse failed");

  mem->ptrcount--;

  if (mem->ptrcount == 0)
    {
      if ((mem->swap.tag != -1) &&
          (swap_free (mem->swap.file,
                      mem->swap.tag) != TRUE))
        {
          g_warning ("swap unreserve failed");
        }
      gimp_mem_unlink (mem);
      g_free (mem->data);
      g_free (mem);
    }
  
  return TRUE;
}



static void
gimp_memptr_drop (GimpMemPtr  *ptr)
{
  g_return_if_fail (ptr != NULL);

  if (d_is_alloced (ptr))
    {
      gboolean unuse = (d_usecount (ptr) != 0 ? TRUE : FALSE);

      if (gimp_mem_detach (ptr->mem, unuse) != TRUE)
        g_warning ("gimp_memptr_drop detach failed");
    }

  d_init (ptr);
}


void
gimp_memptr_init (GimpMemPtr *ptr)
{
  g_return_if_fail (ptr != NULL);
  
  ptr->mem = NULL;
  ptr->usecount = 0;
  ptr->joined = FALSE;
  ptr->size = 0;
  ptr->data = NULL;
}


void
gimp_memptr_uninit (GimpMemPtr *ptr)
{
  g_return_if_fail (ptr != NULL);
  
  if (d_usecount (ptr) != 0)
    g_warning ("uniniting an in-use gmp");
  
  gimp_memptr_drop (ptr);  
}


gboolean
gimp_memptr_alloc (GimpMemPtr  *ptr,
                   gint         size)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (size > 0, FALSE);
  g_return_val_if_fail (! d_is_alloced (ptr), FALSE);
  
  {
    GimpMem * mem = gimp_mem_new (size);
    g_return_val_if_fail (mem != NULL, FALSE);
 
    mem = gimp_mem_attach (mem, FALSE);
    g_return_val_if_fail (mem != NULL, FALSE);

    ptr->mem = mem;
    ptr->size = size;
  }

  return TRUE;
}


gboolean 
gimp_memptr_unalloc (GimpMemPtr *ptr)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (d_usecount (ptr) == 0, FALSE);

  gimp_memptr_drop (ptr);  

  return TRUE;
}


gboolean
gimp_memptr_use (GimpMemPtr *ptr,
                 gboolean    dirty,
                 gboolean    cow)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (d_is_alloced (ptr), FALSE);

  if ((dirty == TRUE) && (d_is_joined (ptr)) && (cow == TRUE))
    {
      if (d_split (ptr) != TRUE)
        {
          g_warning ("gimp_memptr_use split failed");
          return FALSE;
        }
      gimp_mem_unshare (ptr->mem);
    }

  if (ptr->usecount == 0)
    {
      if (gimp_mem_use (ptr->mem) != TRUE)
        {
          g_warning ("gimp_memptr_use use failed");
          return FALSE;
        }
      ptr->data = gimp_mem_ptr (ptr->mem);
    }
  
  ptr->usecount++;

  if (dirty == TRUE)
    gimp_mem_dirty (ptr->mem);

  return TRUE;
}


gboolean
gimp_memptr_unuse (GimpMemPtr *ptr)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (d_usecount (ptr) != 0, FALSE);

  if (ptr->usecount == 1)
    {
      if (gimp_mem_unuse (ptr->mem) != TRUE)
        {
          g_warning ("gimp_memptr_unuse unuse failed");
          return FALSE;
        }
      ptr->data = NULL;
    }
  
  ptr->usecount--;
  
  return TRUE;
}


gboolean
gimp_memptr_join (GimpMemPtr *ptr,
                  GimpMemPtr *newptr)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (newptr != NULL, FALSE);
  g_return_val_if_fail (d_is_alloced (ptr), FALSE);
  g_return_val_if_fail (! d_is_alloced (newptr), FALSE);

  newptr->mem = gimp_mem_attach (ptr->mem, FALSE);
  g_return_val_if_fail (newptr->mem != NULL, FALSE);

  if (newptr->mem == ptr->mem)
    {
      ptr->joined = TRUE;
      newptr->joined = TRUE;
    }

  newptr->size = d_size (ptr);
  
  return TRUE;
}


gboolean
gimp_memptr_split (GimpMemPtr *ptr)
{
  g_return_val_if_fail (ptr != NULL, FALSE);
  g_return_val_if_fail (d_is_alloced (ptr), FALSE);

  if (d_is_joined (ptr))
    {
      if (gimp_mem_is_joined (ptr->mem))
        {
          GimpMem * mem = ptr->mem;
          gboolean use  = (ptr->usecount ? TRUE : FALSE);
          gboolean rc   = gimp_mem_detach (mem, use);

          g_return_val_if_fail (rc == TRUE, FALSE);

          mem = gimp_mem_dup (mem);
          g_return_val_if_fail (mem != NULL, FALSE);

          mem = gimp_mem_attach (mem, use);
          g_return_val_if_fail (mem != NULL, FALSE);

          ptr->mem = mem;
          if (ptr->usecount != 0)
            ptr->data = gimp_mem_ptr (mem);
        }

      ptr->joined = FALSE;
    }
  
  return TRUE;
}

