
/* BCG 2.4 -- image filter plug-in for The Gimp image manipulation program
 * Copyright (C) 1996 Adam D. Moss  -  aspirin@tigerden.com
 * Copyright (C) 1996 Ian Tester - 94024831@athene.mit.csu.edu.au
 * Copyright (C) 1996 Federico Mena Quintero - quartic@polloux.fciencias.unam.mx
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * This filter allows the image gamma, brightness, and contrast to
 * be adjusted.  The gamma of each channel may be set independantly.
 *
 *      I have added the feature of an "RGB lock" when used on RGB images.
 *      When set, any change to either of the R, G, or B gamma values
 *      causes the other two gamma values to change also.
 *      - Ian Tester
 *
 * NOTE: This is written for The GIMP version 0.54 - it might not
 *   compile or run on other versions.
 *
 */

/* Some days after this plug-in's first version was finished, I wrote
 * another brightness and contrast plug-in, which used a (traditional)
 * additive brightness and stretch contrast algorithm.  This is different
 * with what this plug-in originally had; a very neat multiplicative
 * brightness and biased contrast algorithm.  My version of the plug-in
 * was never released to the public.
 * 
 * So a few weeks later (i.e. now) I decided to integrate them together.
 * The dialog box is a bit ugly, IMO, but the plug-in now provides a lot
 * of functionality.  I don't know what Adam will say, as I have made
 * *my* algorithms the default... :) This is because it's what most
 * people are accustomed to.
 * 
 * Adam's algorithms frequently give more pleasing results when used
 * against photographs, however, since they tend not to discard image
 * information --- values do not get clamped.  Choose the algorithms
 * which serve your purposes best.
 *
 * The code could use some cleaning up.  It may have bugs, although I
 * have not detected any.
 *
 * - Federico Mena Quintero (quartic@polloux.fciencias.unam.mx)
 */

/* Changes in version 2.1, by Quartic
 *
 * Arghh... fixed stupid bugs with the Auto Apply function, which I
 * introduced in version 2.0.  Sorry, guys :)
 */

/* Changes in version 2.3, by Adam
 *
 * Rearranged the dialog box a little, Fixed overflow and underflow in
 * Quartic's additive/subtractive brightness adjustment... and made my
 * algorithms the default again, teehee.  =)
 * Added 'Reset All' button.
 * Fixed a lot of auto-apply bugs.
 */

/* Changes in version 2.4, by Quartic
 *
 * Hehehe... Adam sent his own 2.2 to the mailing list, so we had two
 * versions... his 2.2 and my 2.3 :) So I integrated them.  The dialog
 * is Adam's redesigned one (much better than the original!).  I
 * cleaned up the algorithms, and hopefully they are now Bug-Free (tm).
 *
 * I cleaned up all the source code, since it was getting kinda messy.
 * Adam did some minor bug-fixes to the auto-apply code.  Let's hope
 * we all got it right this time :)
 */
 

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include "gimp.h"


/* Some useful macros */

#define CLAMP(a, x, b) (((x) <= (a)) ? (a) : (((x) <= (b)) ? (x) : (b)))


/***** Prototypes *****/

static void saveimage(void);
static void freshen(void);

static void scale_callback(int item_ID, void *client_data, void *call_data);
static void RGB_scale_callback(int item_ID, void *client_data, void *call_data);

static void ok_callback(int item_ID, void *client_data, void *call_data);
static void cancel_callback(int item_ID, void *client_data, void *call_data);

static void reset_callback(int item_ID, void *client_data, void *call_data);
static void toggle_callback(int item_ID, void *client_data, void *call_data);
static void lock_toggle_callback(int item_ID, void *client_data, void *call_data);
static void radio_callback(int dialog_ID, void *client_data, void *call_data);

static int do_gamma(int tmp, long howmuch);
static int brightness(int tmp, long howmuch);
static int contrast(int tmp, long howmuch);

static void bcg(Image linput, Image loutput);


/***** Global variables *****/

static char *prog_name;

static long aapply, lockRGB;

static long bamount = 0, camount = 0;
static long b_additive, b_multiplicative;
static long c_stretch, c_bias;

static Image input, output;
static unsigned char *saved;

static long gamount[3] = {10, 10, 10};
static long last_changed = 10;

static int aapply_ID, lockRGB_ID;
static int dialog_ID;
static int group_ID;
static int scale_R_ID, scale_G_ID, scale_B_ID, scale_ID, scale2_ID;


/***** Functions *****/

/*****/

int
main(int argc, char **argv)
{
	int frame_ID, cols_ID;
	int row_ID, radio_ID;
	int b_additive_ID, b_multiplicative_ID;
	int c_stretch_ID, c_bias_ID;
	int reset_ID;

	/* Save the program name so we can use it later in reporting errors */

	prog_name = argv[0];

	/* Call 'gimp_init' to initialize this filter.
	 * 'gimp_init' makes sure that the filter was properly called and
	 * it opens pipes for reading and writing.
	 */
	
	if (!gimp_init(argc, argv))
		return 0;

	input  = gimp_get_input_image(0);
	output = gimp_get_output_image(0);

	/* If input image is available, then do some work.
	 * Then update the output image.
	 */
	
	if (input && output)
		if ((gimp_image_type(input) == RGB_IMAGE) ||
		    (gimp_image_type(input) == GRAY_IMAGE)) {
			saveimage();
					
			dialog_ID = gimp_new_dialog("Brightness/Contrast/Gamma");
			cols_ID   = gimp_new_column_group(dialog_ID, DEFAULT, NORMAL, "");
			group_ID  = gimp_new_row_group(dialog_ID, cols_ID, NORMAL, "");
					
			aapply_ID = gimp_new_check_button(dialog_ID, group_ID, "Auto Apply");
			aapply = 1;
			gimp_change_item(dialog_ID, aapply_ID, sizeof(aapply), &aapply);
			gimp_add_callback(dialog_ID, aapply_ID, toggle_callback, &aapply);
					
			reset_ID = gimp_new_push_button(dialog_ID, group_ID, "Reset All");
			gimp_add_callback(dialog_ID, reset_ID, reset_callback, NULL);
					
			frame_ID = gimp_new_frame(dialog_ID, group_ID, "");
					
			if (gimp_image_type(input) == GRAY_IMAGE) {
				row_ID = gimp_new_row_group(dialog_ID, frame_ID, DEFAULT, "");
						
				scale_R_ID = gimp_new_scale(dialog_ID, row_ID, 1, 100, gamount[0], 1);
				gimp_new_label(dialog_ID, scale_R_ID, "Gamma:");
				gimp_add_callback(dialog_ID, scale_R_ID, scale_callback, &gamount[0]);
			} else if (gimp_image_type(input) == RGB_IMAGE) {
				row_ID = gimp_new_row_group(dialog_ID, frame_ID, DEFAULT, "");
						
				lockRGB_ID = gimp_new_check_button(dialog_ID, row_ID, "Lock RGB");
				lockRGB = 0;
				gimp_change_item(dialog_ID, lockRGB_ID, sizeof(lockRGB), &lockRGB);
				gimp_add_callback(dialog_ID, lockRGB_ID, lock_toggle_callback, &lockRGB);
						
				scale_R_ID = gimp_new_scale(dialog_ID, row_ID, 1, 100, gamount[0], 1);
				gimp_new_label(dialog_ID, scale_R_ID, "Red gamma:");
				gimp_add_callback(dialog_ID, scale_R_ID, RGB_scale_callback, &gamount[0]);
						
				scale_G_ID = gimp_new_scale(dialog_ID, row_ID, 1, 100, gamount[1], 1);
				gimp_new_label(dialog_ID, scale_G_ID, "Green gamma:");
				gimp_add_callback(dialog_ID, scale_G_ID, RGB_scale_callback, &gamount[1]);
						
				scale_B_ID = gimp_new_scale(dialog_ID, row_ID, 1, 100, gamount[2], 1);
				gimp_new_label(dialog_ID, scale_B_ID, "Blue gamma:");
				gimp_add_callback(dialog_ID, scale_B_ID, RGB_scale_callback, &gamount[2]);
			} /* else */
			
			group_ID = gimp_new_row_group(dialog_ID, cols_ID, NORMAL, "");
					
			/* Initialize radio buttons */
					
			b_additive       = 0;
			b_multiplicative = 1;
			c_stretch        = 0;
			c_bias           = 1;
					
			/* Additive brightness */
			
			frame_ID = gimp_new_frame(dialog_ID, group_ID, "");
					
			row_ID = gimp_new_row_group(dialog_ID, frame_ID, DEFAULT, "");
			radio_ID = gimp_new_row_group(dialog_ID, row_ID, RADIO, "");
					
			b_additive_ID = gimp_new_radio_button(dialog_ID, radio_ID, "Additive brightness");
			gimp_change_item(dialog_ID, b_additive_ID, sizeof(b_additive), &b_additive);
			gimp_add_callback(dialog_ID, b_additive_ID, radio_callback, &b_additive);
					
			/* Multiplicative brightness */
					
			b_multiplicative_ID = gimp_new_radio_button(dialog_ID, radio_ID,
								    "Multiplicative brightness");
			gimp_change_item(dialog_ID, b_multiplicative_ID, sizeof(b_multiplicative),
					 &b_multiplicative);
			gimp_add_callback(dialog_ID, b_multiplicative_ID, radio_callback,
					  &b_multiplicative);
					
			/* Brightness */
					
			scale_ID = gimp_new_scale(dialog_ID, row_ID, -255, 255, bamount, 0);
			gimp_new_label(dialog_ID, scale_ID, "Brightness adjustment:");
			gimp_add_callback(dialog_ID, scale_ID, scale_callback, &bamount);
					
			/* Stretch contrast */
					
			frame_ID = gimp_new_frame(dialog_ID, group_ID, "");
					
			row_ID = gimp_new_row_group(dialog_ID, frame_ID, DEFAULT, "");
			radio_ID = gimp_new_row_group(dialog_ID, row_ID, RADIO, "");
					
			c_stretch_ID = gimp_new_radio_button(dialog_ID, radio_ID, "Stretch contrast");
			gimp_change_item(dialog_ID, c_stretch_ID, sizeof(c_stretch), &c_stretch);
			gimp_add_callback(dialog_ID, c_stretch_ID, radio_callback, &c_stretch);
					
			/* Bias contrast */
					
			c_bias_ID = gimp_new_radio_button(dialog_ID, radio_ID, "Bias contrast");
			gimp_change_item(dialog_ID, c_bias_ID, sizeof(c_bias), &c_bias);
			gimp_add_callback(dialog_ID, c_bias_ID, radio_callback, &c_bias);
					
			/* Contrast */
					
			scale2_ID = gimp_new_scale(dialog_ID, row_ID, -127, 127, camount, 0);
			gimp_new_label(dialog_ID, scale2_ID, "Contrast adjustment:");
			gimp_add_callback(dialog_ID, scale2_ID, scale_callback, &camount);
					
			/* Buttons */
					
			gimp_add_callback(dialog_ID, gimp_ok_item_id(dialog_ID), ok_callback, 0);
			gimp_add_callback(dialog_ID, gimp_cancel_item_id(dialog_ID), cancel_callback, 0);
					
			if (!gimp_show_dialog(dialog_ID)) {
				if ((bamount != 0 || camount != 0 || gamount[0] != 10
				     || gamount[1] != 10 || gamount[2] != 10)
				    && (aapply)) {
					freshen();
					gimp_update_image(output);
				} /* if */
			} else 
				if ((bamount != 0 || camount != 0 || gamount[0] != 10
				     || gamount[1] != 10 || gamount[2] != 10)
				    && (!aapply)) {
					bcg(input, output);
					gimp_update_image(output);
				} /* if */ 
					
			free(saved);
		} else
			gimp_message("bcg: can only operate on 24-bit or grayscale images");
	
	/* Free both images. */
		
	if (input)
		gimp_free_image(input);
		
	if (output)
		gimp_free_image(output);
		
	/* Quit */
		
	gimp_quit();
	
	return 0;
} /* main */


/*****/

/* save a copy of the original image to work on.... */

static void
saveimage(void)
{
	saved = malloc(gimp_image_width(input) * gimp_image_height(input) * gimp_image_channels(input));

	memcpy(saved, gimp_image_data(input),
	       gimp_image_width(input) * gimp_image_height(input) * gimp_image_channels(input));
} /* saveimage */


/*****/

/* copy the original image to the working buffer... */

static void
freshen(void)
{
	memcpy(gimp_image_data(output), saved,
	       gimp_image_width(input) * gimp_image_height(input) * gimp_image_channels(input));
} /* freshen */


/*****/

static void
scale_callback(int item_ID, void *client_data, void *call_data)
{
	if (aapply && (*((long *) client_data) != *((long *) call_data))) {
		*((long *) client_data) = *((long *) call_data);

		bcg(input, output);
		gimp_update_image(output);
	} else
		*((long *) client_data) = *((long *) call_data);
} /* scale_callback */


/*****/

static void
RGB_scale_callback(int item_ID, void *client_data, void *call_data)
{
	if (lockRGB) {
		gamount[0] = gamount[1] = gamount[2] = *((long *) call_data);

		gimp_change_item(dialog_ID, scale_R_ID, sizeof(gamount[0]), &gamount[0]);
		gimp_change_item(dialog_ID, scale_G_ID, sizeof(gamount[1]), &gamount[1]);
		gimp_change_item(dialog_ID, scale_B_ID, sizeof(gamount[2]), &gamount[2]);

		if (aapply) {
			bcg(input, output);
			gimp_update_image(output);
		} /* if */
	} else {
		last_changed = *((long *) client_data) = *((long *) call_data);

		if (aapply) {
			bcg(input, output);
			gimp_update_image(output);
		} /* if */
	} /* else */
} /* RGB_scale_callback */


/*****/

static void
ok_callback(int item_ID, void *client_data, void *call_data)
{
	gimp_close_dialog(dialog_ID, 1);
} /* ok_callback */


/*****/

static void
cancel_callback(int item_ID, void *client_data, void *call_data)
{
	gimp_close_dialog(dialog_ID, 0);
} /* cancel_callback */


/*****/

static void
reset_callback(int item_ID, void *client_data, void *call_data)
{
	freshen();

	bamount = 0;
	camount = 0;
	gamount[0] = 10;

	gimp_change_item(dialog_ID, scale_ID, sizeof(bamount), &bamount);
	gimp_change_item(dialog_ID, scale2_ID, sizeof(camount), &camount);
	gimp_change_item(dialog_ID, scale_R_ID, sizeof(gamount[0]), &gamount[0]);

	if (gimp_image_type(input) == RGB_IMAGE) {
		gamount[1] = 10;
		gamount[2] = 10;
		last_changed = 10;

		gimp_change_item(dialog_ID, scale_G_ID, sizeof(gamount[1]), &gamount[1]);
		gimp_change_item(dialog_ID, scale_B_ID, sizeof(gamount[2]), &gamount[2]);
	} /* if */

	if (aapply)
		gimp_update_image(output);
} /* reset_callback */


/*****/

static void
toggle_callback(int item_ID, void *client_data, void *call_data)
{
	*((long *) client_data) = *((long *) call_data);

	if (aapply) {
		bcg(input, output);
		gimp_update_image(output);
	} else {
		freshen();
		gimp_update_image(output);
	} /* else */
} /* toggle_callback */


/*****/

static void
lock_toggle_callback(int item_ID, void *client_data, void *call_data)
{
	*((long *) client_data) = *((long *) call_data);

	if (lockRGB) {
		gamount[0] = gamount[1] = gamount[2] = last_changed;

		gimp_change_item(dialog_ID, scale_R_ID, sizeof(gamount[0]), &gamount[0]);
		gimp_change_item(dialog_ID, scale_G_ID, sizeof(gamount[1]), &gamount[1]);
		gimp_change_item(dialog_ID, scale_B_ID, sizeof(gamount[2]), &gamount[2]);

		if (aapply) {
			bcg(input, output);
			gimp_update_image(output);
		} /* if */
	} /* if */
} /* lock_toggle_callback */


/*****/

static void
radio_callback(int dialog_ID, void *client_data, void *call_data)
{
	if (aapply && (*((long *) client_data) != *((long *) call_data))) {
		*((long *) client_data) = *((long *) call_data);

		bcg(input, output);
		gimp_update_image(output);
	} else
		*((long *) client_data) = *((long *) call_data);
} /* radio_callback */


/*****/

static int
do_gamma(int tmp, long howmuch)
{
	/* This piece is basically from Spencer & Peter's gamma.c, but
	 * without the overflow bug.  ;) :P =)
	 */

	double one_over_gamma, ind, v;

	one_over_gamma = 10.0 / howmuch;

	ind = (double) tmp / 255.0;
	v = 255.0 * pow(ind, one_over_gamma);

	return (int) (v + 0.5);
} /* do_gamma */


/*****/

static int
brightness(int tmp, long howmuch)
{
	if (b_additive)
		return (tmp + howmuch);
	else {
		if (howmuch < 0)
			return (tmp * 256) / (65536 / (256 + howmuch));
		else if (howmuch > 0)
			return tmp + ((255 - tmp) * 256 / (65536 / howmuch));
		else
			return tmp;
	} /* else */
} /* brightness */


/*****/

static int
contrast(int tmp, long howmuch)
{
	double ii;
	int    i0, i1, o0, o1;
	int    result;

	if (c_stretch) {
		if (howmuch >= 0) {
			i0 = howmuch;
			i1 = 255 - howmuch;
			o0 = 0;
			o1 = 255;
		} else {
			i0 = 0;
			i1 = 255;
			o0 = -howmuch;
			o1 = 255 + howmuch;
		} /* else */

		tmp = (o1 - o0) * (tmp - i0) / (i1 - i0) + o0;

		return tmp;
	} else {
		howmuch = -howmuch;

		tmp = CLAMP(0, tmp, 255);

		if (howmuch > 0) {
			if (tmp > 127)
				result = 255 - tmp;
			else
				result = tmp;
			
			ii = 127.0 * pow((double) result / 127.5, (double) (127 - howmuch) / 127.0);
			
			result = ii;

			if (tmp > 127)
				result = 255 - result;
		} else if (howmuch < 0) {
			if (tmp > 127)
				result = 255 - tmp;
			else
				result = tmp;
			
			if (howmuch == -127)
				ii = (result < 127) ? 0.0 : 127.0;
			else
				ii = 127.0 * pow((double) result / 127.5, 127.0 / (double) (howmuch + 127));

			result = ii;

			if (tmp > 127)
				result = 255 - result;
		} else
			result = tmp;

		return result;
	} /* else */
} /* contrast */


/*****/

static void
bcg(Image linput, Image loutput)
{
	long width, height;
	long channels, rowstride;
	unsigned char *src_row, *dest_row;
	unsigned char *src, *dest;
	short row, col;
	int x1, y1, x2, y2;
	int tmp;
	int table[256][3];

	/* Get the input area. This is the bounding box of the
	 * selection in the image (or the entire image if there is no
	 * selection). Only operating on the input area is simply an
	 * optimization. It doesn't need to be done for correct
	 * operation. (It simply makes it go faster, since fewer
	 * pixels need to be operated on).
	 */

	gimp_image_area(linput, &x1, &y1, &x2, &y2);

	/* Get the size of the input image. (This will/must be the
	 * same as the size of the output image.
	 */

	width     = gimp_image_width(linput);
	height    = gimp_image_height(linput);
	channels  = gimp_image_channels(linput);
	rowstride = width * channels;

	src_row   = saved;
	dest_row  = gimp_image_data(loutput);

	x1 *= channels;
	x2 *= channels;

	/* Advance the source and destination pointers */
	
	src_row  += rowstride * y1 + x1;
	dest_row += rowstride * y1 + x1;

	/* tables */

	for (tmp = 0; tmp < 256; tmp++) {
		table[tmp][0] = contrast(brightness(do_gamma(tmp, gamount[0]), bamount), camount);
		table[tmp][0] = CLAMP(0, table[tmp][0], 255);
	} /* for */

	if (gimp_image_type(input) == RGB_IMAGE) {
		for (tmp = 0; tmp < 256; tmp++) {
			table[tmp][1] = contrast(brightness(do_gamma(tmp, gamount[1]), bamount), camount);
			table[tmp][1] = CLAMP(0, table[tmp][1], 255);
		} /* for */

		for (tmp = 0; tmp < 256; tmp++) {
			table[tmp][2] = contrast(brightness(do_gamma(tmp, gamount[2]), bamount), camount);
			table[tmp][2] = CLAMP(0, table[tmp][2], 255);
		} /* for */
	} /* if */
	
	for (row = y1; row < y2; row++) {
		src = src_row;
		dest = dest_row;

		/*  Calculate across the scanline  */
		for (col = x1; col < x2;) {
			int ch;

			for (ch = 0; ch < channels; ch++) {
				col++;
				*dest++ = (unsigned char) (table[(unsigned char) (*src++)][ch]);
			} /* for */
		} /* for */

		src_row += rowstride;
		dest_row += rowstride;
	} /* for */ 
} /* bcg */

