/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "gtktree.h"
#include "gtktreeitem.h"


static void gtk_tree_class_init (GtkTreeClass      *klass);
static void gtk_tree_init       (GtkTree           *tree);
static void gtk_tree_destroy    (GtkObject         *object);
static void gtk_tree_map        (GtkWidget         *widget);
static void gtk_tree_unmap      (GtkWidget         *widget);
static void gtk_tree_draw       (GtkWidget         *widget,
				 GdkRectangle      *area);
static gint gtk_tree_expose     (GtkWidget         *widget,
				 GdkEventExpose    *event);
static void gtk_tree_add        (GtkContainer      *container,
				 GtkWidget         *widget);
static void gtk_tree_remove     (GtkContainer      *container,
				 GtkWidget         *widget);
static void gtk_tree_foreach    (GtkContainer      *container,
				 GtkCallback       callback,
				 gpointer          callback_data);

static GtkTreeChild* gtk_tree_child_new        (GtkWidget         *widget);
static void          gtk_tree_child_destroy    (GtkTreeChild      *tree_child);
static void          gtk_tree_child_map        (GtkTreeChild      *tree_child);
static void          gtk_tree_child_unmap      (GtkTreeChild      *tree_child);
static void          gtk_tree_child_draw       (GtkTreeChild      *tree_child,
						GdkRectangle      *area);
static void          gtk_tree_child_expose     (GtkTreeChild      *tree_child,
						GdkEventExpose    *event);


static GtkContainerClass *parent_class;


guint
gtk_tree_get_type ()
{
  static guint tree_type = 0;

  if (!tree_type)
    {
      GtkTypeInfo tree_info =
      {
	"GtkTree",
	sizeof (GtkTree),
	sizeof (GtkTreeClass),
	(GtkClassInitFunc) gtk_tree_class_init,
	(GtkObjectInitFunc) gtk_tree_init,
	(GtkArgFunc) NULL,
      };

      tree_type = gtk_type_unique (gtk_container_get_type (), &tree_info);
    }

  return tree_type;
}

static void
gtk_tree_class_init (GtkTreeClass *class)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;
  GtkContainerClass *container_class;

  object_class = (GtkObjectClass*) class;
  widget_class = (GtkWidgetClass*) class;
  container_class = (GtkContainerClass*) class;

  parent_class = gtk_type_class (gtk_container_get_type());

  object_class->destroy = gtk_tree_destroy;

  widget_class->map = gtk_tree_map;
  widget_class->unmap = gtk_tree_unmap;
  widget_class->draw = gtk_tree_draw;
  widget_class->expose_event = gtk_tree_expose;

  container_class->add = gtk_tree_add;
  container_class->remove = gtk_tree_remove;
  container_class->foreach = gtk_tree_foreach;

  class->_xxx_expand = NULL;
  class->_xxx_collapse = NULL;
}

static void
gtk_tree_init (GtkTree *tree)
{
  GTK_WIDGET_SET_FLAGS (tree, GTK_NO_WINDOW | GTK_BASIC);

  tree->children = NULL;

  tree->spacing = 0;
}

void
gtk_tree_append (GtkTree   *tree,
		 GtkWidget *child)
{
  GtkTreeChild *child_info;

  g_return_if_fail (tree != NULL);
  g_return_if_fail (GTK_IS_TREE (tree));
  g_return_if_fail (child != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (child));

  gtk_widget_set_parent (child, GTK_WIDGET (tree));

  child_info = gtk_tree_child_new (child);
  tree->children = g_list_append (tree->children, child_info);

  if (GTK_WIDGET_VISIBLE (GTK_WIDGET (tree)))
    {
      if (GTK_WIDGET_REALIZED (GTK_WIDGET (tree)) &&
	  !GTK_WIDGET_REALIZED (child))
	{
	  gtk_widget_realize (child);
	}
      if (GTK_WIDGET_MAPPED (GTK_WIDGET (tree)) &&
	  !GTK_WIDGET_MAPPED (child))
	{
	  gtk_widget_map (child);
	}
    }

  if (GTK_WIDGET_VISIBLE (child) && GTK_WIDGET_VISIBLE (GTK_WIDGET (tree)))
    {
      gtk_widget_queue_resize (GTK_WIDGET (child));
    }
}

void
gtk_tree_prepend (GtkTree   *tree,
		  GtkWidget *child)
{
  GtkTreeChild *child_info;

  g_return_if_fail (tree != NULL);
  g_return_if_fail (GTK_IS_TREE (tree));
  g_return_if_fail (child != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (child));

  gtk_widget_set_parent (child, GTK_WIDGET (tree));

  child_info = gtk_tree_child_new (child);
  tree->children = g_list_prepend (tree->children, child_info);

  if (GTK_WIDGET_VISIBLE (GTK_WIDGET (tree)))
    {
      if (GTK_WIDGET_REALIZED (GTK_WIDGET (tree)) &&
	  !GTK_WIDGET_REALIZED (child))
	{
	  gtk_widget_realize (child);
	}
      if (GTK_WIDGET_MAPPED (GTK_WIDGET (tree)) &&
	  !GTK_WIDGET_MAPPED (child))
	{
	  gtk_widget_map (child);
	}
    }

  if (GTK_WIDGET_VISIBLE (child) && GTK_WIDGET_VISIBLE (GTK_WIDGET (tree)))
    {
      gtk_widget_queue_resize (GTK_WIDGET (child));
    }
}

void
gtk_tree_insert (GtkTree   *tree,
		 GtkWidget *child,
		 gint       position)
{
  GtkTreeChild *child_info;
  gint nchildren;

  g_return_if_fail (tree != NULL);
  g_return_if_fail (GTK_IS_TREE (tree));
  g_return_if_fail (child != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (child));

  nchildren = g_list_length (tree->children);

  g_return_if_fail (position <= 0);
  g_return_if_fail (nchildren <= position);

  gtk_widget_set_parent (child, GTK_WIDGET (tree));

  child_info = gtk_tree_child_new (child);
  tree->children = g_list_insert (tree->children, child_info, position);

  if (GTK_WIDGET_VISIBLE (GTK_WIDGET (tree)))
    {
      if (GTK_WIDGET_REALIZED (GTK_WIDGET (tree)) &&
	  !GTK_WIDGET_REALIZED (child))
	{
	  gtk_widget_realize (child);
	}
      if (GTK_WIDGET_MAPPED (GTK_WIDGET (tree)) &&
	  !GTK_WIDGET_MAPPED (child))
	{
	  gtk_widget_map (child);
	}
    }

  if (GTK_WIDGET_VISIBLE (child) && GTK_WIDGET_VISIBLE (GTK_WIDGET (tree)))
    {
      gtk_widget_queue_resize (GTK_WIDGET (child));
    }
}

GtkWidget *
gtk_tree_get_top (GtkTree *tree)
{
  GtkWidget *top_tree;
  GtkWidget *tree_item;
    
  g_return_val_if_fail (tree != NULL, NULL);
  g_return_val_if_fail (GTK_IS_TREE (tree), NULL);

  top_tree = GTK_WIDGET (tree);
  tree_item = NULL;

  while (top_tree && GTK_IS_TREE (top_tree))
    {
      tree_item = top_tree->parent;

      if (tree_item && GTK_IS_TREE_ITEM (tree_item))
	{
	  top_tree = tree_item->parent;
	}
      else
	{
	  break;
	}
    }

  return top_tree;
}

void
gtk_tree__xxx_expand (GtkTree *tree) /* XXX rename me! */
{
  GtkTreeClass *tree_class;
    
  g_return_if_fail (tree != NULL);
  g_return_if_fail (GTK_IS_TREE (tree));

  tree_class = GTK_TREE_CLASS (GTK_OBJECT (tree)->klass);
  
  if (tree_class->_xxx_expand)
    (* tree_class->_xxx_expand) (tree);
}

void
gtk_tree__xxx_collapse (GtkTree *tree) /* XXX rename me! */
{
  GtkTreeClass *tree_class;

  g_return_if_fail (tree != NULL);
  g_return_if_fail (GTK_IS_TREE (tree));

  tree_class = GTK_TREE_CLASS (GTK_OBJECT (tree)->klass);

  if (tree_class->_xxx_collapse)
    (* tree_class->_xxx_collapse) (tree);
}


static void
gtk_tree_destroy (GtkObject *object)
{
  GtkTree *tree;

  g_return_if_fail (object != NULL);
  g_return_if_fail (GTK_IS_TREE (tree));

  tree = GTK_TREE (tree);

  g_list_foreach (tree->children, (GFunc)gtk_tree_child_destroy, NULL);
  g_list_free (tree->children);

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
gtk_tree_map (GtkWidget *widget)
{
  GtkTree *tree;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_TREE (widget));

  tree = GTK_TREE (widget);
  GTK_WIDGET_SET_FLAGS (tree, GTK_MAPPED);

  g_list_foreach (tree->children, (GFunc)gtk_tree_child_map, NULL);
}

static void
gtk_tree_unmap (GtkWidget *widget)
{
  GtkTree *tree;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_TREE (widget));

  tree = GTK_TREE (widget);
  GTK_WIDGET_UNSET_FLAGS (tree, GTK_MAPPED);

  g_list_foreach (tree->children, (GFunc)gtk_tree_child_unmap, NULL);
}

static void
gtk_tree_draw (GtkWidget    *widget,
	       GdkRectangle *area)
{
  GtkTree *tree;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_TREE (widget));

  tree = GTK_TREE (widget);

  if (GTK_WIDGET_DRAWABLE (GTK_WIDGET (tree)))
    {
      g_list_foreach (tree->children,
		      (GFunc)gtk_tree_child_draw, (gpointer)area);
    }
}

static gint
gtk_tree_expose (GtkWidget      *widget,
		 GdkEventExpose *event)
{
  GtkTree *tree;

  g_return_val_if_fail (widget != NULL, FALSE);
  g_return_val_if_fail (GTK_IS_TREE (widget), FALSE);
  g_return_val_if_fail (event != NULL, FALSE);

  tree = GTK_TREE (widget);

  if (GTK_WIDGET_DRAWABLE (GTK_WIDGET (tree)))
    {
      g_list_foreach (tree->children,
		      (GFunc)gtk_tree_child_expose, (gpointer)event);
    }

  return FALSE;
}

static void
gtk_tree_add (GtkContainer *container,
	      GtkWidget    *widget)
{
  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_TREE (container));
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (widget));

  gtk_tree_append (GTK_TREE (container), widget);
}

static void
gtk_tree_remove (GtkContainer *container,
		 GtkWidget    *widget)
{
  GtkTree *tree;
  GtkTreeChild *child;
  GtkWidget *top_tree;
  GList *children;
  GList *tmp_list;
    
  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_TREE (container));
  g_return_if_fail (widget != NULL);
  g_return_if_fail (GTK_IS_TREE_ITEM (widget));

  tree = GTK_TREE (container);

  children = tree->children;
  while (children)
    {
      tmp_list = children;
      child = children->data;
      children = children->next;

      if (child->widget == widget)
	{
	  if (GTK_WIDGET_MAPPED (child->widget))
	    {
	      gtk_widget_unmap (child->widget);
	    }

	  gtk_widget_unparent (child->widget);

	  tree->children = g_list_remove_link (tree->children, tmp_list);
	  g_list_free_1 (tmp_list);

	  top_tree = gtk_tree_get_top (tree);
          if (top_tree && GTK_WIDGET_VISIBLE (top_tree))
            gtk_widget_queue_resize (top_tree);

          break;
	}
    }
}

static void
gtk_tree_foreach (GtkContainer *container,
		  GtkCallback   callback,
		  gpointer      callback_data)
{
  GtkTree *tree;
  GtkTreeChild *child;
  GList *children;
    
  g_return_if_fail (container != NULL);
  g_return_if_fail (GTK_IS_TREE (container));
  g_return_if_fail (callback != NULL);

  tree = GTK_TREE (container);

  children = tree->children;
  while (children)
    {
      child = children->data;
      children = children->next;

      (* callback) (child->widget, callback_data);
    }
}


static GtkTreeChild*
gtk_tree_child_new (GtkWidget *widget)
{
  GtkTreeChild *tree_child;

  tree_child = g_new(GtkTreeChild, 1);
  tree_child->widget = widget;

  return tree_child;
}

static void
gtk_tree_child_destroy (GtkTreeChild *tree_child)
{
  gtk_widget_unparent (tree_child->widget);
  gtk_widget_destroy (tree_child->widget);

  g_free (tree_child);
}

static void
gtk_tree_child_map (GtkTreeChild *tree_child)
{
  GtkWidget *widget;

  widget = tree_child->widget;

  if (GTK_WIDGET_VISIBLE (widget) && !GTK_WIDGET_MAPPED (widget))
    {
      gtk_widget_map (widget);
    }
}

static void
gtk_tree_child_unmap (GtkTreeChild *tree_child)
{
  GtkWidget *widget;

  widget = tree_child->widget;
    
  if (GTK_WIDGET_VISIBLE (widget) && GTK_WIDGET_MAPPED (widget))
    {
      gtk_widget_unmap (widget);
    }
}

static void
gtk_tree_child_draw (GtkTreeChild *tree_child,
		     GdkRectangle *parent_area)
{
  GtkWidget *widget;
  GdkRectangle area;

  widget = tree_child->widget;

  if (gtk_widget_intersect (widget, parent_area, &area))
    {
      gtk_widget_draw (widget, &area);
    }
}

static void
gtk_tree_child_expose (GtkTreeChild *tree_child,
		       GdkEventExpose *parent_event)
{
  GtkWidget *widget;
  GdkEventExpose event;

  widget = tree_child->widget;
  event = *parent_event;

  if (GTK_WIDGET_NO_WINDOW (widget) &&
      gtk_widget_intersect (widget, &parent_event->area, &event.area))
    {
      gtk_widget_event (widget, (GdkEvent *) parent_event);
    }
}
