/* $Id: dpu.c,v 1.7 89/09/20 17:59:58 mbp Exp $
 *
 * dpu.c: Display Processing Unit for SunView
 */

/************************************************************************
 *		Copyright (C) 1989 by Mark B. Phillips                  *
 * 									*
 * Permission to use, copy, modify, and distribute this software and    *
 * its documentation for any purpose and without fee is hereby granted, *
 * provided that the above copyright notice appear in all copies and    *
 * that both that copyright notice and this permission notice appear in *
 * supporting documentation, and that the name of Mark B. Phillips or   *
 * the University of Maryland not be used in advertising or publicity   *
 * pertaining to distribution of the software without specific, written *
 * prior permission.  This software is provided "as is" without express *
 * or implied warranty.                                                 *
 ************************************************************************/

#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include "dpu.h"
#include "dev.h"
#include "../lgd.h"
#include "../internal.h"
#include "../mf.h"
#include "../vector.h"
#include GR_HEADER

/************************************************************************
 *			  PUBLIC DEFINITIONS				*
 ************************************************************************/

char *dpu_error = NULL;
char *dpu_internal_error=NULL;

/************************************************************************
 *			 PRIVATE DEFINITIONS				*
 ************************************************************************/

#define		UP	0
#define		DOWN	1
#define		NO	0
#define		YES	1
#define		SQR(x)  ((x)*(x))
#define		MAX(x,y)	( (x) > (y) ? (x) : (y) )
#define		STATIC	/* make this static after debugging */

typedef double Matrix4x4[4][4];
typedef double R2vect[2];

/* Address of entry in MF which was most recently executed:	*/
static mfa last_executed;

/* Specification of current viewing transformation: */
static lgd_View3 current_view;

/* The viewing transformation matrix: */
static double
  	tmat_00,tmat_01,tmat_02,tmat_03,
	tmat_10,tmat_11,tmat_12,tmat_13,
	tmat_20,tmat_21,tmat_22,tmat_23,
	tmat_30,tmat_31,tmat_32,tmat_33;

/* Flag telling whether MF needs to be redrawn when updating display: */
static int need_to_redraw=NO;	

static int	dpu_pen_status	=	UP;
static double canvas_width, canvas_height;

static int	batching = NO;

STATIC double edist_r2vect(), point_segment_dist(), point_ray_dist(),
  point_line_dist(), dotprod_r2vect(), enorm_r2vect();

/***********************************************************************
 *			  PUBLIC PROCEDURES                            *
 ***********************************************************************/

/*-----------------------------------------------------------------------
 * Function:     dpu_initialize
 * Description:  initialize the DPU
 * Arguments:    (none)
 * Returns:      nothing
 * Notes:        This procedure may be called more than once.  It
 *               should always have the effect of leaving the DPU
 *               (and the graphics device) in a cleared state, ready
 *               to start displaying a new picture.
 */
dpu_initialize()
{
  last_executed = mf_first_addr() - 1;
  dpu_pen_status = UP;
  dev_initialize();
  canvas_width = gr_canvas_width();
  canvas_height = gr_canvas_height();
}

/*-----------------------------------------------------------------------
 * Function:     dpu_delete_segment
 * Description:  delete (erase) a segment
 * Arguments IN: n: the number of the segment to be deleted
 * Returns:      nothing
 * Notes:        n had better not be the current segment!
 */
dpu_delete_segment( n )
mfe n;
{
  /* This version just sets a flag which indicates that the current
   * display is no longer current.  That way we know to redraw the
   * entire MF on the next call to "dpu_display_current". */
  need_to_redraw = YES;
}

/*-----------------------------------------------------------------------
 * Function:     dpu_set_segment_visibility
 * Description:  Set the visibility of a segment
 * Arguments IN: n: the number of the segment
 *               s: visibility status to set to: 1=visible, 0=invisible
 * Returns:      nothing
 * Notes:        
 */
dpu_set_segment_visibility( n, s )
mfe n; 
int s;
{
  /* This version just sets a flag which indicates that the current
   * display is no longer current.  That way we know to redraw the
   * entire MF on the next call to "dpu_display_current". */
  need_to_redraw = YES;
}

/*-----------------------------------------------------------------------
 * Function:     dpu_update_display
 * Description:  update the display -- make sure it shows the current MF
 * Arguments:    (none)
 * Returns:      nothing
 */
dpu_update_display()
{
  if (need_to_redraw) {
    if (batching) gr_batch_on();
    redraw_mf();
    if (batching) gr_batch_off();
    }
  else {
    if (batching) gr_batch_on();
    flush_mf();
    if (batching) gr_batch_off();
  }
  if (dpu_internal_error!=NULL) {
    dpu_error = dpu_internal_error;
    dpu_internal_error = NULL;
    return;
  }
  need_to_redraw = NO;
  dev_update_display();
}

/*-----------------------------------------------------------------------
 * Function:     dpu_save
 * Description:  Write current dpu state to a file
 * Arguments IN: fp: file to write to
 * Returns:      nothing
 * Notes:        fp should point to a file which has been opened
 *               for binary writing.
 */
dpu_save( fp )
     FILE *fp;
{
  int items_written;

#define	SAVEIT(x,type)	{items_written = fwrite( \
		(char*)x,sizeof(type), 1, fp );\
		if (items_written!=1) \
		{dpu_error=E_svd_bad_write; return; } }
  /*
   *********************
   * THIS IS FORMAT 3  *
   *********************
   */

  SAVEIT( &last_executed, mfa );
  SAVEIT( &current_view, lgd_View3 );
  SAVEIT( &dpu_pen_status, int );

#undef	SAVEIT
}


/*-----------------------------------------------------------------------
 * Function:     dpu_load
 * Description:  Read a dpu state from a file
 * Arguments IN: fp: file to read from
 * Returns:      nothing
 * Notes:        fp should point to a file which has been opened
 *               for binary reading.  The current dpu is re-initialized
 *               first.
 */
dpu_load( fp )
     FILE *fp;
{
  int items_read;

#define	LOADIT(x,type)	{items_read = fread( \
		(char*)x,sizeof(type), 1, fp );\
		if (items_read!=1) \
		{dpu_error=E_svd_bad_read; return; } }
  /*
   *********************
   * THIS IS FORMAT 3  *
   *********************
   */
  dpu_initialize();
  LOADIT( &last_executed, mfa );
  LOADIT( &current_view, lgd_View3 );
  LOADIT( &dpu_pen_status, int );
  need_to_redraw = YES;
  dpu_update_display();
  if (dpu_internal_error!=NULL) {
    dpu_error = dpu_internal_error;
    dpu_internal_error = NULL;
    return;
  }

#undef	LOADIT
}

/*-----------------------------------------------------------------------
 * Function:     dpu_set_view
 * Description:  set the current viewwing transformation
 * Arguments IN: *view: the viewwing specs
 * Returns:      nothing
 * Notes:        This may not actually cause the display to change;
 *               call dpu_update_display to make sure the display
 *               is current.
 */
dpu_set_view( view )
lgd_View3 *view;
{
  LGD_copy_vec( current_view.eye, view->eye );
  LGD_copy_vec( current_view.focus,view->focus );
  LGD_copy_vec( current_view.up,view->up );
  current_view.u1 = view->u1;
  current_view.u2 = view->u2;
  current_view.v1 = view->v1;
  current_view.v2 = view->v2;

  /* Note: The Sun version currently does not use h1 or h2; so we
   *   give them fixed values here */
  current_view.h1 = -1;
  current_view.h2 = 1;

  compute_tmat();
  need_to_redraw = YES;
  if (dpu_internal_error!=NULL) {
    dpu_error = dpu_internal_error;
    dpu_internal_error = NULL;
  }
}

/*-----------------------------------------------------------------------
 * Function:     dpu_inquire_view
 * Description:  Get the current viewing transformation
 * Arguments OUT: *view: the viewwing transformation specs
 * Returns:      nothing
 */
dpu_inquire_view( view )
lgd_View3 *view;
{
  LGD_copy_vec( view->eye, current_view.eye );
  LGD_copy_vec( view->focus, current_view.focus );
  LGD_copy_vec( view->up, current_view.up );
  view->u1 = current_view.u1;
  view->u2 = current_view.u2;
  view->v1 = current_view.v1;
  view->v2 = current_view.v2;
}

/*-----------------------------------------------------------------------
 * Function:     dpu_get_default_view
 * Description:  Compute a standard default viewwing transformation
 * Arguments:    (none)
 * Returns:      nothing
 */
dpu_get_default_view( view )
lgd_View3 *view;
{
  int dimension, i;
  double wbox_low[MAXDIM], wbox_high[MAXDIM], d;
#define	WFACT	2.0

  lgd_inquire_world( &dimension, wbox_low, wbox_high );
  if (dimension==3) {
    /* Set FOCUS to center of world		*/
    for (i=0; i<3; ++i)
      view->focus[i] =
	wbox_low[i] + (wbox_high[i]-wbox_low[i]) / 2;
    /* Set EYE on line parallel to z axis	*/
    view->eye[0] = view->focus[0];
    view->eye[1] = view->focus[1];
    view->eye[2] = view->focus[2] + 2*(wbox_high[2]-wbox_low[2]);
    /* Set UP on line parallel to y axis	*/
    view->up[0] = view->focus[0];
    view->up[1] = view->focus[1] + wbox_high[1];
    view->up[2] = view->focus[2];
    /* Make square window to just include	*/
    /* all of world box.			*/
    d = MAX( wbox_high[0]-wbox_low[0],
	    wbox_high[1]-wbox_low[1] ) / 2;
    view->u1 = view->v1 =
      -(view->u2 = view->v2 = (WFACT*4*d/3) );
  }
#undef	WFACT
}

/*-----------------------------------------------------------------------
 * Function:     dpu_pick_segment
 * Description:  get a segment picked by user
 * Arguments OUT: *s: points to an array of segment numbers 
 *               *n: number of segment numbers in array s
 * Returns:      nothing
 * Notes:     1. The space to which *s points is allocated dynamically.
 *               LGD keeps track of this location, and on the next call
 *               to dpu_pick_segment this array is freed before the new
 *               array is allocated.  This means that the array returned
 *               by a call to dpu_pick_segment is valid only until the
 *               next call.
 *            2. If the user picks no segments, *s is set to NULL
 *               and *n to 0. 
 */
dpu_pick_segment( s, n )
int **s, *n;
{
  double d;
  int screenvec[2];
  double dscreenvec[2];
  R2vect p_cur, p_pick;
  register mfa addr;
  int i, segnum, segaddr, seglen, pick_pen_status;
  mfa last;
  mfe ivec[MAXDIM];
  register mfe entry;
  static int *last_pick_array=NULL;
  double pick_tolerance;

  pick_tolerance = canvas_width * lgd_pick_tolerance;
  
  /* Begin by freeing up the array returned on the last call, if there
     is one */
  if (last_pick_array!=NULL) free(last_pick_array);

  /* Get point from user */
  dev_get_ndc_point(&p_pick[0], &p_pick[1]);

  /* Then traverse MF looking for visible segments which contain
     a point within pick_tolerance of the picked point */
  *s = NULL;
  *n = 0;
  p_cur[0] = p_cur[1] = 0;
  pick_pen_status = UP;
  last = mf_last_addr();
  addr = mf_first_addr();
  while (addr<=last) {
    entry=mf_peek(addr);
    if (mf_is_command(entry)) {
      if (entry==MF_begin_segment) {
	/* Check visibility:	*/
	if (mf_peek(addr+VIS_OS)){	/* (visible)	 */
          segaddr = addr;
          seglen = (mfa) mf_peek(addr+LEN_OS);
          segnum = (int) mf_peek(addr+NAME_OS);
	  addr += lgd_header_length;
	}
	else			        /* (invisible) */
	  addr += mf_peek(addr+LEN_OS);
	continue;
      }
      if (entry==MF_end_segment) {	/* skip trailer	*/
	addr += lgd_trailer_length;
	continue;
      }
      if (entry==MF_pen_up) {
	pick_pen_status = UP;
	++addr;
	continue;
      }
      if (entry==MF_pen_down) {
	pick_pen_status = DOWN;
	++addr;
	continue;
      }
      if (entry==MF_point) {	
	/* Check distance of this (the current) point to pick point;
	   if dist is within tolerance, add this segment to list and
           skip to next segment, otherwise continue with this segment */
	d = edist_r2vect(p_cur, p_pick);
	if (d<=pick_tolerance) {
	  add_to_pick_array(s, n, segnum);
	  if (seglen==0)		/* length=0 --> this is the current */
	    addr = last+1;		/* segment. So go to end of MF */
	  else			/* Otherwise skip to next segment */
	    addr = segaddr + seglen;
	}
	else
	  ++addr;
	continue;
      }
      if (entry==MF_line_style) {
	addr += 2;
	continue;
      }
      if (entry==MF_color) {
	addr += 2;
	continue;
      }
      /* Invalid entry	*/
      dpu_internal_error = E_bad_mf_entry;
      return;
    } /* End of "if (mf_is_command..."		*/
    else {	/* We have a vector		*/
      /*   Get vector from MF:	*/
      for (i=0; i<lgd_dim; ++i)
	ivec[i] = mf_peek(addr+i);
      /*   convert to screen coords */
      screen_coord( screenvec, screenvec+1, ivec );

      /* If the pen is up, simply update current point and continue */
      if (pick_pen_status==UP) {
	p_cur[0] = screenvec[0];
	p_cur[1] = screenvec[1];
        addr += lgd_dim;
      }
      /* If pen is down, compute distance from pick point to segment
         between current point and new point, update current point to
         new point, and check to see if distance is within tolerance */
      else {
	dscreenvec[0] = screenvec[0];
	dscreenvec[1] = screenvec[1];
	d = point_segment_dist(p_pick, p_cur, dscreenvec);
	p_cur[0] = screenvec[0];
	p_cur[1] = screenvec[1];
	/*   If distance is <= tolerance, add entry to array s,
	     and skip to end of this segment  */
	if (d<=pick_tolerance) {
	  add_to_pick_array(s, n, segnum);
	  if (seglen==0)		/* length=0 --> this is the current */
	    addr = last+1;		/* segment. So go to end of MF */
	  else			/* Otherwise skip to next segment */
	    addr = segaddr + seglen;
	}
	else
	  addr += lgd_dim;
      }
    }
  } /* end of while loop 		*/

  /* Before returning, save ptr to array s so that it can be freed
     at beginning of next call */
  last_pick_array = *s;

}   /* end of function dpu_pick_segment */

/*-----------------------------------------------------------------------
 * Function:	gr_set_batch_mode
 * Description:	set the "batching status"
 * Args  IN:	status: 1 for batching, 0 for no batching
 * Notes:	This procedure gets called by GR in response to the
 *		user picking the "Drawing Mode" cycle in the GR
 *		view control panel.
 */
gr_set_batch_mode(status)
int status;
{
  batching = status;
}

/***********************************************************************
 *			  PRIVATE PROCEDURES                           *
 ***********************************************************************/

/*-----------------------------------------------------------------------
 * Function:     execute_mf
 * Description:  Execute a section of the MetaFile
 * Arguments IN: first: first address to execute
 *               last: last address to execute
 * Returns:      nothing
 */
STATIC
execute_mf( first, last )
     register mfa first, last;
{
  mfa i;
  mfe ivec[MAXDIM], entry;
  int screenvec[2];

  while (first<=last) {
    entry=mf_peek(first);
    if (mf_is_command(entry)) {
      if (entry==MF_begin_segment) {
	/* Check visibility:	*/
	if (mf_peek(first+VIS_OS))	/* (visible)	 */
	  first += lgd_header_length;
	else			        /* (invisible) */
	  first += mf_peek(first+LEN_OS);
	continue;
      }
      if (entry==MF_end_segment) {	/* skip trailer	*/
	first += lgd_trailer_length;
	continue;
      }
      if (entry==MF_pen_up) {	/* raise pen	*/
	dpu_pen_status = UP;
	++first;
	continue;
      }
      if (entry==MF_pen_down) {
	/* lower pen	*/
	dpu_pen_status = DOWN;
	++first;
	continue;
      }
      if (entry==MF_point) {	/* draw a point	*/
	dev_point();
	++first;
	continue;
      }
      if (entry==MF_line_style) {
	/* change style	*/
	dev_style((int)mf_peek(first+1));
	first += 2;
	continue;
      }
      if (entry==MF_color) {	/* change color	*/
	dev_color((int)mf_peek(first+1));
	first += 2;
	continue;
      }
      /* Invalid entry	*/
      dpu_internal_error = E_bad_mf_entry;
      goto abort;
    }  /* End of "if (mf_is_command..."		*/
    
    else {		/* We have a vector		*/
      /*   Get vector from MF:	*/
      for (i=0; i<lgd_dim; ++i)
	ivec[i] = mf_peek(first+i);
      /*   convert to screen coords */
      screen_coord( screenvec, screenvec+1, ivec );
      /*   And finally send to device	*/
      if (dpu_pen_status==DOWN)
	dev_draw( screenvec );
      else
	dev_move( screenvec );
      first += lgd_dim;
    }
  }  /* end of while loop 				*/

 abort:
  return;
}

/*-----------------------------------------------------------------------
 * Function:     screen_coord
 * Description:  project world coord vector to screen coords
 * Arguments IN: v: world coord vector t be projected
 *          OUT: (ndv1,ndv2): coord of projection
 * Returns:      nothing
 * Notes:        This is the main 3D to 2D projection routine
 */
STATIC
screen_coord( x, y, v )
     int *x, *y;
     mfe v[];
{
  register double w, xtemp, ytemp;
  
  xtemp = tmat_00 * v[0]
    + tmat_01 * v[1]
      + tmat_02 * v[2]
	+ tmat_03;
  ytemp = tmat_10 * v[0]
    + tmat_11 * v[1]
      + tmat_12 * v[2]
	+ tmat_13;
  w     = tmat_30 * v[0]
    + tmat_31 * v[1]
      + tmat_32 * v[2]
	+ tmat_33;
  *x = (int) ( xtemp/w + .5 );
  *y = (int) ( ytemp/w + .5 );
}

/*-----------------------------------------------------------------------
 * Function:     flush_mf
 * Description:  flush out (execute) unexecuted part of MF
 * Arguments:    (none)
 * Returns:      nothing
 * Notes:        The part executed is that between last_executed
 *               and the current end of the MF.
 */
STATIC
flush_mf()
{
  execute_mf( last_executed+1, mf_last_addr() );
  if (dpu_internal_error==NULL)
    last_executed = mf_last_addr();
}

/*-----------------------------------------------------------------------
 * Function:     redraw_mf
 * Description:  redraw (execute) the entire MF
 * Arguments:    (none)
 * Returns:      nothing
 */
STATIC
redraw_mf()
{
  compute_tmat();
  last_executed = mf_first_addr() - 1;
  dev_erase();
  flush_mf();
}

/*-----------------------------------------------------------------------
 * Function:     compute_3d_rmat
 * Description:  Compute rotation matrix part of the current
 *                 viewwing transformation
 * Arguments OUT: rmat: the computed matrix
 * Returns:      nothing
 */
STATIC
compute_3d_rmat( rmat )
     double	rmat[3][3];
{
  double u[3];
  
#define	v	rmat[0]
#define	r	rmat[1]
#define	e	rmat[2]
  
  LGD_sub_vec( e, current_view.eye, current_view.focus );
  LGD_unit_vec( e );
  LGD_sub_vec( u, current_view.up, current_view.focus );
  LGD_unit_vec( u );
  LGD_cross_vec( v, u, e );
  LGD_unit_vec( v );
  LGD_cross_vec( r, e, v );
  
#undef	v
#undef	r
#undef	e
}

/*-----------------------------------------------------------------------
 * Function:    compute_tmat()
 * Description: Compute the current viewwing transformation matrix
 * Args:        (none)
 * Returns:     nothing
 * Notes:       The matrix is stored in the static global variables
 *              tmat_xx.
 *
 * 		This version is experimental.  It may need to be sped
 * 		up by putting in direct formulas for computing the
 * 		matrix, rather than calling the matrix multiplicaton
 * 		procedure.
 *
 * 		It may also help to convert this entirely to integer
 * 		coords.  Think about it:  the metafile contains integer
 * 		coords, and the screen space is in integer coords, so
 * 		why not work entirely in integers??!!
 */
STATIC
compute_tmat()
{
  int i;
  double e[3], u[3], v[3], exv[3];
  Matrix4x4 M1_M0, M2_M1_M0, M3_M2_M1_M0, M4_M3_M2_M1_M0;
  Matrix4x4 M5_M4_M3_M2_M1_M0;

  /* The entries in the following matrices which are initialized to
   * 0 or 1 retain their value of 0 or 1 always.  The entries which
   * are initialized to 2 get changed; the value 2 is not significant
   * and is never actually used.  */
  static Matrix4x4 M0 = { 2, 0, 0, 2,
			  0, 2, 0, 2,
			  0, 0, 2, 2,
			  0, 0, 0, 1 };
  static Matrix4x4 M1 = { 1, 0, 0, 2,
			  0, 1, 0, 2,
			  0, 0, 1, 2,
			  0, 0, 0, 1 };
  static Matrix4x4 M2 = { 2, 2, 2, 0,
			  2, 2, 2, 0,
			  2, 2, 2, 0,
			  0, 0, 0, 1 };
  static Matrix4x4 M3 = { 1, 0, 0, 0,
			  0, 1, 0, 0,
			  0, 0, 0, 0,
			  0, 0, 2, 1 };
  static Matrix4x4 M4 = { 2, 0, 0, 2,
			  0, 2, 0, 2,
			  0, 0, 2, 2,
			  0, 0, 0, 1 };
  static Matrix4x4 M5 = { 2, 0, 0, 2,
			  0, 2, 0, 2,
			  0, 0, 1, 0,
			  0, 0, 0, 1 };

  LGD_sub_vec(e, current_view.eye, current_view.focus);/* e = eye - focus */
  LGD_unit_vec(e, e);				       /* e = e / ||e||   */
  LGD_sub_vec(u, current_view.up, current_view.focus); /* u = eye - focus */
  LGD_unit_vec(u, u);				       /* u = u / ||u||   */
  LGD_cross_vec(v, u, e);			       /* v = u X e       */
  LGD_unit_vec(v, v);				       /* v = v / ||v||   */
  LGD_cross_vec(exv, e, v);			       /* exv = e X v     */

  for (i=0; i<3; ++i) {
    M0[i][i] = 1 / lgd_wscale[i];
    M0[i][3] = lgd_wbox[0][i];
    M1[i][3] = -current_view.focus[i];
    M2[0][i] = v[i];
    M2[1][i] = exv[i];
    M2[2][i] = e[i];
  }

  M3[3][2] = - 1.0 / LGD_L2dist_vec(current_view.eye, current_view.focus);

  M4[0][0] = 1 / (current_view.u2 - current_view.u1);
  M4[1][1] = 1 / (current_view.v2 - current_view.v1);
  M4[2][2] = 1 / (current_view.h2 - current_view.h1);
  M4[0][3] = -current_view.u1 * M4[0][0];
  M4[1][3] = -current_view.v1 * M4[1][1];
  M4[2][3] = -current_view.h1 * M4[2][2];

  M5[1][1] = - (M5[0][0] = M5[1][3] = canvas_height);
  M5[0][3] = (canvas_width - canvas_height) / 2;

  LGD_mult_matrix4x4(            M1_M0, M1,             M0);
  LGD_mult_matrix4x4(         M2_M1_M0, M2,          M1_M0);
  LGD_mult_matrix4x4(      M3_M2_M1_M0, M3,       M2_M1_M0);
  LGD_mult_matrix4x4(   M4_M3_M2_M1_M0, M4,    M3_M2_M1_M0);
  LGD_mult_matrix4x4(M5_M4_M3_M2_M1_M0, M5, M4_M3_M2_M1_M0);

  tmat_00 = M5_M4_M3_M2_M1_M0[0][0];
  tmat_01 = M5_M4_M3_M2_M1_M0[0][1];
  tmat_02 = M5_M4_M3_M2_M1_M0[0][2];
  tmat_03 = M5_M4_M3_M2_M1_M0[0][3];
  tmat_10 = M5_M4_M3_M2_M1_M0[1][0];
  tmat_11 = M5_M4_M3_M2_M1_M0[1][1];
  tmat_12 = M5_M4_M3_M2_M1_M0[1][2];
  tmat_13 = M5_M4_M3_M2_M1_M0[1][3];
  tmat_20 = M5_M4_M3_M2_M1_M0[2][0];
  tmat_21 = M5_M4_M3_M2_M1_M0[2][1];
  tmat_22 = M5_M4_M3_M2_M1_M0[2][2];
  tmat_23 = M5_M4_M3_M2_M1_M0[2][3];
  tmat_30 = M5_M4_M3_M2_M1_M0[3][0];
  tmat_31 = M5_M4_M3_M2_M1_M0[3][1];
  tmat_32 = M5_M4_M3_M2_M1_M0[3][2];
  tmat_33 = M5_M4_M3_M2_M1_M0[3][3];

}
/*-----------------------------------------------------------------------
 * Function:     add_to_pick_array
 * Description:  add an entry to an array of picked segments
 * Arguments IN: s: pointer to array to add to
 *              *n: current length of array s
 *          segnum: number of segment to add   
 *          OUT: s: (possibly modified) pointer to array
 *              *n: new length of array s
 * Returns:      nothing
 * Notes:     1. The array pointed to by s is, in general, moved during
 *               the call to this procedure in order to make room for the
 *               new entry.
 */
STATIC
add_to_pick_array(s, n, segnum)
int **s, *n, segnum;
{
  if (*s==NULL) {  /* array is empty -- so start it */
    *s = (int *) malloc(sizeof(int));
    *n = 1;
  }
  else {	  /* array is not empty -- so reallocated it */
    ++*n;
    *s = (int *) realloc(*s, (unsigned)((*n)*sizeof(int)));
  }
  (*s)[*n-1] = segnum;
}

/*-----------------------------------------------------------------------
 * Function:     edist_r2vect
 * Description:  compute distance between two points in R2
 * Arguments IN: a: first point
 *               b: second point
 * Returns:      the (Euclidean) dist from a to b
 */
STATIC
double edist_r2vect(a,b)
R2vect a,b;
{
  register double xdiff, ydiff;
  double answer;

  xdiff = a[0] - b[0];
  ydiff = a[1] - b[1];
  answer = sqrt( SQR(xdiff) + SQR(ydiff) );
  return(answer);
}

/*-----------------------------------------------------------------------
 * Function:     point_segment_dist
 * Description:  compute distance from a point to a segment
 * Arguments IN: p: the point
 *               a: first endpoint of segment
 *               b: second endpoint of segment
 * Returns:      (Euclidean) distance from the point to the segment
 */
STATIC
double point_segment_dist(p, a, b)
R2vect p, a, b;
{
  R2vect pma, amb;
  double d;

  sub_r2vect( pma, p, a );
  sub_r2vect( amb, a, b );
  if ( dotprod_r2vect(pma,amb) >= 0 )
    d = enorm_r2vect( pma );
  else
    d = point_ray_dist( p, b, amb );
  return(d);
}

/*-----------------------------------------------------------------------
 * Function:     point_ray_dist
 * Description:  distance between a point and a ray
 * Arguments IN: p: the point
 *               a: endpoint of the ray
 *		 v: direction vector of the ray
 * Returns:      the distance
 */
STATIC
double point_ray_dist( p, a, v )
R2vect p, a, v;
{
  R2vect pma;
  double d;

  sub_r2vect( pma, p, a );
  if ( dotprod_r2vect(pma,v) <= 0 )
    d = enorm_r2vect( pma );
  else
    d = point_line_dist( p, a, v );
  return(d);
}

/*-----------------------------------------------------------------------
 * Function:     point_line_dist
 * Description:  distance between point and line
 * Arguments IN: p: the point
 *               a: a point on the line
 * 		 v: direction vector of the line
 * Returns:      the distance
 */
STATIC
double point_line_dist( p, a, v )
R2vect p, a, v;
{
  R2vect pma;
  double d, vdotpma;

  sub_r2vect( pma, p, a );
  vdotpma = dotprod_r2vect( v, pma );
  d = sqrt( (double) (SQR(pma[0]) + SQR(pma[1])
	   - SQR(vdotpma) / ( SQR(v[0]) + SQR(v[1]) ) ) );
  return(d);
}

STATIC
sub_r2vect(c, a, b)
R2vect c, a, b;
{
  c[0] = a[0] - b[0];
  c[1] = a[1] - b[1];
}

STATIC
double dotprod_r2vect(a,b)
R2vect a,b;
{
  double answer;
  answer = a[0]*b[0] + a[1]*b[1];
  return(answer);
}

STATIC
double enorm_r2vect(v)
R2vect v;
{
  double answer;
  answer = sqrt( SQR(v[0]) + SQR(v[1]) );
  return(answer);
}

