/* $Id: gr.c,v 2.3 89/09/20 17:01:26 mbp Exp $
 *
 * gr.c: main GR procedures
 */

/************************************************************************
 *		Copyright (C) 1989 by Mark B. Phillips                  *
 * 									*
 * Permission to use, copy, modify, and distribute this software and    *
 * its documentation for any purpose and without fee is hereby granted, *
 * provided that the above copyright notice appear in all copies and    *
 * that both that copyright notice and this permission notice appear in *
 * supporting documentation, and that the name of Mark B. Phillips or   *
 * the University of Maryland not be used in advertising or publicity   *
 * pertaining to distribution of the software without specific, written *
 * prior permission.  This software is provided "as is" without express *
 * or implied warranty.                                                 *
 ************************************************************************/

#include <stdio.h>
#include <malloc.h>
#include <math.h>
#include "gr.h"
#include "internal.h"

gr_Menu	GR_current_menu;
double	GR_x1, GR_y1, GR_x2, GR_y2, GR_xfactor, GR_yfactor;
int	GR_window_open = 0;
char	*(*GR_coordinate_display_proc)() = NULL;

/* NOTE: the following specifies how points are drawn.  This default
 * can be overriden by a specification in the SunView defaults
 * database, which currently is read only by the 3D version of GR
 * (in the file window.c). */
int	GR_point_list_default[] = {0,0, 1,0, -1,0, 0,1, 0,-1};
int	*GR_point_list = GR_point_list_default;
int	GR_point_list_count =
  sizeof(GR_point_list_default)	/ (2 * sizeof(int));

#define	YES	1
#define	NO	0

#define MAX(x,y) ( (x>y) ? (x) : (y) )

static int plotting=0;
static FILE *postscript_fp=NULL;
static int batching=0;
static int current_canvas_x = 0;
static int current_canvas_y = 0;

/*-----------------------------------------------------------------------
 * Function:     gr_initialize
 * Description:  Initialize the package
 * Arguments:	 argc, argv: generic Suntools command line argument list
 *		 (list of -W options)
 * Notes:        Must be called before any gr_ procedures other than
 *		 ones beginning with gr_preset_.
 */
int
gr_initialize(argc, argv)
     int argc;
     char **argv;
{
  GR_current_menu.title = NULL;
  gr_coordinate_system(0.0, 0.0, 1.0, 1.0);
  return( GR_initialize_windows(argc, argv) );
}

/*-----------------------------------------------------------------------
 * Function:     gr_set_frame_icon
 * Description:  Specify a frame icon
 * Arguments IN: *icon: the icon to use
 * Returns:      success status
 */
int
gr_set_frame_icon( icon )
Icon *icon;
{
  if ( (GR_base_frame==NULL) || (icon==NULL) )
    return(1);
  else {
    window_set(GR_base_frame, FRAME_ICON, icon, 0);
    return(0);
  }
}

/*-----------------------------------------------------------------------
 * Function:     gr_set_frame_label
 * Description:  Set the frame's label
 * Arguments IN: label: the label to use
 * Returns:      success status
 */
int
gr_set_frame_label( label )
char *label;
{
  if ( (GR_base_frame==NULL) || (label==NULL) )
    return(1);
  else {
    window_set(GR_base_frame, FRAME_LABEL, label, 0);
    return(0);
  }

}

/*-----------------------------------------------------------------------
 * Function:	gr_set_menu
 * Description:	Set the current (application) menu
 * Args  IN:	*menu: the new menu
 * Notes:	
 */
int
gr_set_menu(new_menu)
gr_Menu *new_menu;
{
  int i, len, maxlen;
  
  if (new_menu==NULL) return(1);

  /* If a menu has been specified before, free up its space */
  if (GR_current_menu.title!=NULL) {
    delete_menu(&GR_current_menu);
  }

  /* Compute max length of entries in new menu */
  maxlen = strlen(new_menu->title);
  for (i=0; i<new_menu->nopts; ++i) {
    if ( (len=strlen(new_menu->list[i].string)) > maxlen)
      maxlen = len;
  }

  /* Install centered title and entries into current menu */
  GR_current_menu.title = malloc((maxlen+1)*sizeof(char));
  strcpy(GR_current_menu.title, GR_center_string(new_menu->title,maxlen));
  GR_current_menu.nopts = new_menu->nopts;
  GR_current_menu.list =
    (gr_Opt_pair*)malloc(new_menu->nopts*sizeof(gr_Opt_pair));
  for (i=0; i<GR_current_menu.nopts; ++i) {
    GR_current_menu.list[i].string = malloc((maxlen+1)*sizeof(char));
    strcpy(GR_current_menu.list[i].string,
	   GR_center_string(new_menu->list[i].string, maxlen));
    GR_current_menu.list[i].proc = new_menu->list[i].proc;
  }
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	gr_coordinate_system
 * Description:	Establishes coordinate system of graphics canvas
 * Args IN:	(x1, y1): coords of lower left corner
 *		(x2, y2): coords of upper right corner
 * Notes:   1.	x1 != x2 and y1 != y2 must hold or else and error occurrs.
 *	    2.	This procedure can be called more than once; the coordinate
 *		system changes each time.
 *	    3.	The default coordinate system is with (0,0) in the lower
 *		left and (1,1) in the upper right.
 */
int
gr_coordinate_system(x1, y1, x2, y2)
double x1, y1, x2, y2;
{
  if ( (x1 == x2) || (y1 == y2) )
    return(1);
  GR_x1 = x1;
  GR_y1 = y1;
  GR_x2 = x2;
  GR_y2 = y2;
  GR_xfactor = GR_canvas_width / (GR_x2 - GR_x1);
  GR_yfactor = GR_canvas_height / (GR_y1 - GR_y2);
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	gr_main_loop
 * Description:	Enter the main Suntools notifier loop
 * Args:	(none)
 * Notes:	This call returns only when one of the following happens:
 *		1. The user clicks the "Quit" button
 *		2. The user picks the "Quit" option from the frame menu.
 *		3. The function gr_done() is called.
 */
int
gr_main_loop()
{
  if (GR_base_frame == NULL) {
    GR_error("You must call gr_init before calling gr_main_loop !");
    return(1);
  }
  else {

    /* For development, we use panel_edit: */
    /* paneledit_init(GR_base_frame); */

    GR_window_open = 1;
    window_main_loop(GR_base_frame);
    return(0);
  }
}

/*-----------------------------------------------------------------------
 * Function:	gr_done
 * Description:	closes the window and causes gr_main_loop to return
 * Args:	(none)
 */
int
gr_done()
{
  window_done(GR_base_frame);
}

/*-----------------------------------------------------------------------
 * Function:	gr_canvas_move
 * Description:	move to a new (x,y) in integer canvas coordinates
 * Args  IN:	x,y: new point to move to
 */
int
gr_canvas_move(x,y)
     int x,y;
{
  if (plotting)
    return( GR_postscript_move( x, y ) );
  else {
    current_canvas_x = x;
    current_canvas_y = y;
    return(0);
  }
}	

/*-----------------------------------------------------------------------
 * Function:	gr_point
 * Description:	draw a point at current point
 * Args:	(none)
 * Notes:	Points on canvas are drawn according to the coordinates
 *		in the global variable GR_point_list.  This contains
 *		a list of pixels, in integer coordinates relative to
 *		the current position, to be plotted.  The number of
 *		coordinates in GR_point_list is given by the value
 *		of GR_point_list_count.
 */
int
gr_point()
{
  int i;

  if (plotting) 
    return( GR_postscript_point() );
  else
    for (i=0; i<GR_point_list_count; ++i)
      pw_put(GR_canvas_pw,
	     current_canvas_x+GR_point_list[2*i],
	     current_canvas_y+GR_point_list[2*i+1],
	     1);
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	gr_canvas_draw
 * Description:	draw to a new (x,y) in canvas integer coords
 * Args  IN:	x,y: coords of new point
 * Notes:	(x,y) becomes the new current point
 */
int
gr_canvas_draw(x,y)
     int x,y;
{
  if (plotting)
    return( GR_postscript_draw(x,y) );
  else {
    pw_vector(GR_canvas_pw,
	      current_canvas_x, current_canvas_y, x, y,
	      PIX_SRC, 1);
    current_canvas_x = x;
    current_canvas_y = y;
    return(0);
  }
}

/* -----------------------------------------------------------------------
 * Function:	gr_canvas_arc
 * Description:	draw a circular arc using integer canvas coordinates
 * Args  IN:	xc,yc: center of arc
 * 		x1,y1: initial point of arc
 * 		x2,y2: terminal point of arc
 * Notes:	The arc is drawn counterclockwise.  If (x1,y1) and
 * 		(x2,y2) are the same point, an entire circle is drawn.
 * 		The current point is left unchanged.  (x2,y2) is used
 * 		only to determine the final angle of the arc.  The arc
 * 		is drawn by drawing line segments at angle_inc
 * 		increments.  This increment is computed in terms of
 * 		the radius of the arc, so that a circle whose radius
 * 		is half the width of the canvas is drawn with an
 * 		increment of FULL_ANGLE_INC degrees.
 *
 *		Yes, I know there are much more efficent ways of drawing
 *		arcs.  Dontations of such code will be greatly apprectiated.
 */
int
  gr_canvas_arc( xc,yc,  x1,y1,  x2,y2 )
int xc,yc,x1,y1,x2,y2;
{
#define FULL_ANGLE_INC 2.5

  int x,y, xdiff,ydiff, xfinal,yfinal;
  double r1,r2, th1, th2, angle;
  double angle_inc;
  static double angle_factor = FULL_ANGLE_INC * M_PI / (180.0 * 2.0);
  static double max_angle_inc = 15.0 * M_PI / 180.0;
  
  if (plotting) 
    return( GR_postscript_arc(xc,yc,  x1,y1,  x2,y2 ) );
  else {
    /* Always batch around arc drawing */
    if (!batching) pw_batch_on(GR_canvas_pw);

    /* Compute radius r1 of arc */
    xdiff = x1-xc;  ydiff = y1-yc;
    r1 = sqrt( (double)(xdiff*xdiff + ydiff*ydiff) );

    /* Compute initial and terminal angles */
    th1 = atan2( (double)ydiff, (double)xdiff );
    xdiff = x2-xc;  ydiff = y2-yc;
    r2 = sqrt( (double)(xdiff*xdiff + ydiff*ydiff) );
    th2 = atan2( (double)ydiff, (double)xdiff );

    /* Compute exact coordinates of final point */
    xfinal = xc + (int)( r1 * ( (double)xdiff / (double)r2 ) );
    yfinal = yc + (int)( r1 * ( (double)ydiff / (double)r2 ) );

    /* Make sure angles are correctly oriented */
    /* if (th2<=th1) th2 += 2*M_PI; */
    if (th2>=th1) th2 -= 2*M_PI;

    /* Compute angle increment based on radius r1 */
    angle_inc = angle_factor * ( (double)GR_canvas_width / (double)r1 );
    if (angle_inc>max_angle_inc) angle_inc = max_angle_inc;

    /* now DRAW, buster! */
    gr_canvas_move(x1, y1);
    for (angle=th1-angle_inc; angle>th2; angle-=angle_inc) {
      x = xc + (int) ( r1*cos(angle) );
      y = yc + (int) ( r1*sin(angle) );
      gr_canvas_draw(x, y);
    }
    gr_canvas_draw(xfinal, yfinal);

    /* Turn batching off if it wasn't already on */
    if (!batching) pw_batch_off(GR_canvas_pw);

    return(0);
  }
}

/*-----------------------------------------------------------------------
 * Function:	gr_erase
 * Description:	erase the canvas
 * Args:	(none)
 * Notes:	Has no effect in PostScript mode
 */
int
gr_erase()
{
  if (!plotting) {
    pw_writebackground(GR_canvas_pw, 0,0, 
		       GR_canvas_width, GR_canvas_height, PIX_SRC);
  }
  return(0);
}

/* -----------------------------------------------------------------------
 * Function:	   gr_get_canvas_xy
 * Description:	   Get a point in canvas integer coords from the user
 * Args	 IN:	   bmask: tells which buttons to recognize
 *	OUT:	   x,y: coords of point selected by user
 *		   button: button pressed by user
 * Notes:	   bmask should be a logical OR (|) of GR_LEFT,
 *		   GR_MIDDLE, GR_RIGHT, GR_SHIFT_LEFT, GR_SHIFT_MIDDLE,
 *		   GR_SHIFT_RIGHT, or GR_ESC.  It must include at least
 *		   one of these, or else an error occurrs.
 */
int
gr_get_canvas_xy(x, y, button, bmask)
     int *x, *y;
     gr_Button *button, bmask;
{
  return( GR_get_canvas_xy(x, y, button, bmask) );
}

/*-----------------------------------------------------------------------
 * Function:	gr_get_xy
 * Description:	Get a point in FP coords from the user
 * Notes:	Exactly like gr_get_canvas_xy except that the point is
 *		converted to FP coords.
 */
int
gr_get_xy(x, y, button, bmask)
     double *x, *y;
     gr_Button *button, bmask;
{
  int ix, iy, r;

  r = GR_get_canvas_xy(&ix, &iy, button, bmask);
  *x = GR_SAVNAC_X(ix);
  *y = GR_SAVNAC_Y(iy);
  return(r);
}

/* -----------------------------------------------------------------------
 * Function:	gr_set_coordinate_display_proc
 * Description:	specify a procedure to use for displaying coordinates
 *		  when user is selecting a point
 * Args	 IN:	proc: pointer to coordinate display function
 * Notes:	proc should be declared as follows:
 *			char *proc(x,y)
 *			int x,y;
 *		It should return a pointer to a string to be displayed
 *		in the coordinate display area of the application panel.
 */
int
gr_set_coordinate_display_proc( proc )
char *(*proc)();
{
  if (proc == NULL)
    return(1);
  else {
    GR_coordinate_display_proc = proc;
    return(0);
  }
}

/*-----------------------------------------------------------------------
 * Function:	gr_show_canvas_cursor
 * Description:	Move cursor to a given location in canvas integer coords
 * Args  IN:	x,y: coords of point to move cursor to
 * Notes:	
 */
int
gr_show_canvas_cursor(x,y)
     int x,y;
{
  window_set(GR_canvas, WIN_MOUSE_XY, x, y, 0);
}

/* -----------------------------------------------------------------------
 * Function:	gr_button_message
 * Description:	display button messages in application panel
 * Args  IN:	Lmsg:   message for left mouse button
 * 		Mmsg:   message for middle mouse button
 * 		Rmsg:   message for right mouse button
 * 		SLmsg:  message for shifted left mouse button
 * 		SMmsg:  message for shifted middle mouse button
 * 		SRmsg:  message for shifted right mouse button
 * 		ESCmsg: message for ESCAPE key
 * Notes:	The image corresponding to a button is displayed only
 * 		if that button's message has nonzero length, i.e. is
 * 		other than "".
 */
int
gr_button_message(Lmsg, Mmsg, Rmsg, SLmsg, SMmsg, SRmsg, ESCmsg)
     char  *Lmsg, *Mmsg, *Rmsg, *SLmsg, *SMmsg, *SRmsg, *ESCmsg;
{
  return( GR_show_mouse_message(0, Lmsg)  |
	  GR_show_mouse_message(1, Mmsg)  |
	  GR_show_mouse_message(2, Rmsg)  |
	  GR_show_mouse_message(3, SLmsg) |
	  GR_show_mouse_message(4, SMmsg) |
	  GR_show_mouse_message(5, SRmsg) |
	  GR_show_mouse_message(6, ESCmsg)  );
}

/*-----------------------------------------------------------------------
 * Function:	gr_button_message_clear
 * Description:	clear all button messages
 * Args:	(none)
 */
int
gr_button_message_clear()
{
  return( gr_button_message( "", "", "", "", "", "", "" ) );
}

/*-----------------------------------------------------------------------
 * Function:	gr_message
 * Description:	display a message in one of the application message lines
 * Args  IN:	s: the message string
 *		n: the number of the message line to use (1,2)
 * Notes:	The message numbers here are 1,2. The message numbers
 *		used in window.c, however, are 0,1.
 */
int
gr_message(s, n )
     char *s;
     int n;
{
  return( GR_show_app_message(n-1, s) );
}

/*-----------------------------------------------------------------------
 * Function:	gr_message_clear
 * Description:	clear all application panel messages
 * Args:	(none)
 */
int
gr_message_clear()
{
  int i, r;

  r = 0;
  for (i=0; i<4; ++i) r |= gr_message("", i);
  return(r);
}

/*-----------------------------------------------------------------------
 * Function:	gr_get_string
 * Description:	get a string from the user
 * Args  IN:	s: pointer to location where string is to be stored
 * Notes:	The maximum length of a string returned by this
 *		procedure will be GR_INPUT_STRING_LENGTH, not including
 *		the terminal null character.
 */
int
gr_get_string(s)
{
  return( GR_app_panel_get_string(s) );
}

/*-----------------------------------------------------------------------
 * Function:	gr_batch_on
 * Description:	turn graphics batching on
 * Args:	(none)
 */
int
gr_batch_on()
{
  batching = YES;
  pw_batch_on(GR_canvas_pw);
}

/*-----------------------------------------------------------------------
 * Function:	gr_batch_off
 * Description:	turn graphics batching off
 * Args:	(none)
 */
int
gr_batch_off()
{
  batching = NO;
  pw_batch_off(GR_canvas_pw);
}

/*-----------------------------------------------------------------------
 * Function:	gr_set_help_file
 * Description:	specify the name of the file to display in the help window
 * Args  IN:	filename: the file's name
 * Notes:	This does not change the visibility status of the
 *		help window
 */
gr_set_help_file(filename)
char *filename;
{
  return( GR_set_help_file(filename) );
}

/*-----------------------------------------------------------------------
 * Function:	gr_aspect_ratio
 * Description:	get the aspect ratio of the canvas
 * Args:	(none)
 * Returns:	the ratio ( canvas_width / canvas_height )
 */
double
gr_aspect_ratio()
{
  return( (double)GR_canvas_width / (double)GR_canvas_height );
}

/*-----------------------------------------------------------------------
 * Function:	gr_canvas_width
 * Description:	get the width of the canvas
 * Args:	(none)
 * Returns:	the canvas width in pixels
 */
int
gr_canvas_width()
{
  return(GR_canvas_width);
}

/*-----------------------------------------------------------------------
 * Function:	gr_canvas_height
 * Description:	get the height of the canvas
 * Args:	(none)
 * Returns:	the canvas height in pixels
 */
int
gr_canvas_height()
{
  return(GR_canvas_height);
}

/*-----------------------------------------------------------------------
 * Function:	gr_user_confirm
 * Description:	force user to answer yes or no to a question
 * Args  IN:	s: the question to ask
 * Returns:	1 for yes, 0 for no
 * Notes:	This procedure blocks everything until the user answers
 */
int
gr_user_confirm(s)
{
  return(GR_user_confirm(s));
}

/*-----------------------------------------------------------------------
 * Function:	gr_postscript_begin
 * Description:	begin a PostScript file
 * Args  IN:	fname: the name of the file to write PostScript to
 *		logo: string to use as program logo on picture
 *		label: string to use as picture's label
 *		time: string to use as time stamp on picture
 * Notes:	Any of logo, label, and time may be NULL, in which case
 *		it does not appear on the picture.
 */
int
gr_postscript_begin(fname, logo, label, time)
     char *fname, *logo, *label;
{
  if (plotting) return(1);
  postscript_fp = fopen(fname, "w");
  if (postscript_fp == NULL) return(1);
  if (GR_begin_postscript_file(postscript_fp, logo, label, time) != 0)
    return(1);
  plotting = YES;
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	gr_postscript_end
 * Description:	quit writing a PostScript file
 * Args:	(none)
 */
int
gr_postscript_end()
{
  int r;

  if (!plotting) return(1);
  r = GR_end_postscript_file(postscript_fp);
  r |= fclose(postscript_fp);
  plotting = NO;
  return(r);
}

/*-----------------------------------------------------------------------
 * Function:	gr_print_button
 * Notes:	See GR_print_button (in window.c) for details
 */
int
gr_print_button(status, redraw_proc, logo)
     int status;
     int (*redraw_proc)();
     char *logo;
{
  return( GR_print_button(status,redraw_proc,logo) );
}

/*-----------------------------------------------------------------------
 * Function:	gr_signal
 * Description:	set a trap for a Unix signal
 * Args  IN:	sig: the signal to catch
 *		func: the function to call when the signal is caught
 * Notes:	GR applications should call this instead of signal(3)
 *		in order to peacefully exist with the notifier.
 */
gr_signal(sig, func)
int sig, (*func)();
{
  notify_set_signal_func(GR_base_frame, func, sig, NOTIFY_ASYNC);
}

/*-----------------------------------------------------------------------
 * Function:	gr_set_input_func
 * Description:	specify a function to call when input is available from
 *		a pipe
 * Args  IN:	func: the function to call
 *		fd: file descriptor of pipe
 */
gr_set_input_func(func, fd)
int (*func)(), fd;
{
  notify_set_input_func(GR_base_frame, func, fd);
}

/*-----------------------------------------------------------------------
 * Function:	gr_iconic
 * Description:	change whether the window is iconic
 * Args  IN:	state: 0 for non-iconic, 1 for iconic
 */
gr_iconic(state)
int state;
{
  if (state==1)
      window_set( GR_base_frame, FRAME_CLOSED, TRUE, 0 );
  else
      window_set( GR_base_frame, FRAME_CLOSED, FALSE, 0 );
}

/*-----------------------------------------------------------------------
 * Function:     GR_file_openable
 * Description:  Find out if a file can be opened with a given mode
 * Arguments IN: filename: the name of the file
 *		 mode: the mode to check for
 * Returns:      1 means yes, 0 means no
 */
int
GR_file_openable(filename, mode)
     char *filename, *mode;
{
  FILE *fp;

  if ( (fp=fopen(filename,mode)) == NULL )
    return(0);
  else {
    fclose(fp);
    return(1);
  }
}

/*-----------------------------------------------------------------------
 * Function:     GR_center_string
 * Description:  center a string in a string of a given length
 * Arguments IN: s: string to be centered
 *          OUT: n: length of string to center s in; this does NOT include
 *		   the final null character
 * Returns:      the centered string
 * Notes:        The returned string is stored in dynamic storage which
 *               is freed on the next call. If n <= strlen(s), the first
 *               n chars of s are returned.
 */
char *
GR_center_string(s,n)
     char *s;
     int n;
{
  static char *answer = NULL;
  int i,len,padlen;

  if (answer!=NULL) free(answer);
  answer = malloc((n+1)*sizeof(char));
  for (i=0; i<n; ++i) answer[i] = ' ';
  len = strlen(s);
  padlen = MAX( (n-len)/2, 0 );
  for (i=0; (i<len) && (i+padlen<n); ++i) answer[padlen+i] = s[i];
  answer[n] = '\0';
  return(answer);
}

#ifdef THREE_D			/* procedures for 3D version only */

int
gr_set_eye_display(eye)
     double *eye;
{
  return(GR_set_view_display(GR_EYE, eye));
}

int
gr_set_focus_display(focus)
     double *focus;
{
  return(GR_set_view_display(GR_FOCUS, focus));
}

int
gr_set_up_display(up)
     double *up;
{
  return(GR_set_view_display(GR_UP,up));
}

int
gr_set_view_plane_window_display(u1,u2, v1,v2)
     double u1,u2,v1,v2;
{
  double values[4];

  values[0] = u1;  values[1] = u2;
  values[2] = v1;  values[3] = v2;
  return(GR_set_view_display(GR_VPW, values));
}

int
gr_get_eye_display(eye)
     double *eye;
{
  return(GR_get_view_display(GR_EYE, eye));
}

int
gr_get_focus_display(focus)
     double *focus;
{
  return(GR_get_view_display(GR_FOCUS, focus));
}

int
gr_get_up_display(up)
     double *up;
{
  return(GR_get_view_display(GR_UP,up));
}

int
gr_get_view_plane_window_display(u1,u2, v1,v2)
     double *u1,*u2,*v1,*v2;
{
  double values[4];
  int r;

  r = GR_get_view_display(GR_VPW, values);
  *u1 = values[0]; *u2 = values[1];
  *v1 = values[2]; *v2 = values[3];
  return(r);
}

#endif			/* End of procedures for 3D version only */

/************************************************************************
 *
 *		 P R I V A T E    P R O C E D U R E S
 *
 *	 Anything below here is for use only within this file
 ************************************************************************/

/*-----------------------------------------------------------------------
 * Function:	delete_menu
 * Description:	Free up the space associated with a gr_Menu struct, and
 *		re-initialize its entries
 * Args  IN:	*m: the menu
 */
static int
delete_menu(m)
     gr_Menu *m;
{
  int i;

  if (m==NULL)
    return(1);
  else {
    if (m->title!=NULL) free(m->title);
    if (m->list!=NULL) {
      for (i=0; i<m->nopts; ++i)
	if (m->list[i].string!=NULL) free(m->list[i].string);
      free(m->list);
    }
    m->nopts = 0;
    return(0);
  }

}

