/* Patrick Worfolk
   January 24, 1990
   Computes slope of the best least squares fit
   to the logs of a set of data.
   As per Numerical Recipes.
   */

#include <stdio.h>
#include <math.h>

static double sqrarg;
#define SQR(a) (sqrarg=(a),sqrarg*sqrarg)

/* Compute least squares fit to the line y=a+bx and
   compute standard deviations and chi square stats
   */
void dims_fit(x,y,n,a,b,siga,sigb,chi2)
double *x,*y,*a,*b,*siga,*sigb,*chi2;
int n;
{
	int i;
	double t,sxoss,sx=0.0,sy=0.0,st2=0.0,ss,sigdat;

	*b=0.0;
	for (i=0;i<n;i++) {
		sx += x[i];
		sy += y[i];
	}
	ss=n;
	sxoss=sx/ss;
	for (i=0;i<n;i++) {
		t=x[i]-sxoss;
		st2 += t*t;
		*b += t*y[i];
	}
	*b /= st2;
	*a=(sy-sx*(*b))/ss;
	*siga=sqrt((1.0+sx*sx/(ss*st2))/ss);
	*sigb=sqrt(1.0/st2);
	*chi2=0.0;
	for (i=0;i<n;i++)
		*chi2 += SQR(y[i]-(*a)-(*b)*x[i]);
	sigdat=sqrt((*chi2)/(n-2));
	*siga *= sigdat;
	*sigb *= sigdat;
}
/* Analyze the dimension data by computing the best
   linear fit to the chosen section of curve.
   */
void dims_analyze()
{
	double *logx,*logy,a,b,siga,sigb,chi2,log2(),*dvector(),*x,*y;
	int i,m,n,dim,hi,lo;
	FILE *fp,*fopen();
	char s[80];
	extern int dims_type_option,dims_lsf_start,dims_lsf_end;


	/* Read input file to array */
	sprintf(s,"kaos.dims.tmpdat");
	fp = fopen(s,"r");

	dim = 2;
	x = dvector(0,dim);
	y = dvector(0,dim);
	n = 0;
	while ( fscanf(fp,"%lf %lf",x+n,y+n) != EOF){
		n++;
		if(n>=dim){
			dim *= 2;
			x = (double *) realloc(x,(unsigned)sizeof(double) * (dim+1));
			y = (double *) realloc(y,(unsigned)sizeof(double) * (dim+1));
		}
	}
	fclose(fp);

	lo=dims_lsf_start;
	hi=dims_lsf_end;

	if (hi>n) {
		system_mess_proc(1,"Cannot compute using all points.");
		hi=n;
	}
	if (lo<1) {
		system_mess_proc(1,"Cannot start before first point.");
		lo=1;
	}
	if (lo>=hi) {
		system_mess_proc(1,"There are no valid data points");
		return;
	}

	m = hi-lo;
	/* Allocate memory */
	logx = dvector(0,m);
	if (!logx) {
		system_mess_proc(1,"(Dim analyze) Insufficient memory");
		return;
	}
	logy = dvector(0,m);
	if (!logy) {
		system_mess_proc(1,"(Dim analyze) Insufficient memory");
		return;
	}

	/* copy to log log data */
	for (i=lo; i<=hi; i++) {
		logx[i-lo] = log2(x[i-1]);
		logy[i-lo] = log2(y[i-1]);
	}

	/* send output */
	if (dims_type_option==0) {
		system_mess_proc(0,"Fractal dimension analysis:");
	}
	else if (dims_type_option==2) {
		system_mess_proc(0,"Information dimension analysis:");
	}

	/* send data to be analyzed */
	if (lo+1==hi) {
		sprintf(s,"Slope is: %le",(logy[1]-logy[0])/(logx[1]-logx[0]));
	}
	else {
		dims_fit(logx,logy,m+1,&a,&b,&siga,&sigb,&chi2);
		sprintf(s,"Best fit slope = %le",b);
		system_mess_proc(0,s);
		sprintf(s,"sd = %le, chi square = %le",sigb,chi2);
	}
	system_mess_proc(0,s);


	/* free memory */
	(void) free_dvector(logx,0,m);
	(void) free_dvector(logy,0,m);
	(void) free_dvector(x,0,dim);
	(void) free_dvector(y,0,dim);

}
