/* $Id$
 *
 * window.c
 */

/**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************/

#include <math.h>
#include "wdefs.h"
#include "window.h"
#include "internal.h"
#include "images.h"
#include "../hcore.h"

#define MAX(x,y)	((x)>(y) ? (x) : (y))
#define	WINDOWERROR	-1
#define OFF		0
#define ON		1

#define FONT1_FILE "/usr/lib/fonts/fixedwidthfonts/cour.r.14"
#define FONT2_FILE "/usr/lib/fonts/fixedwidthfonts/cour.r.12"

static Panel		control_panel, message_panel;
static Canvas		canvas, mini_canvas;
static Panel_item	help_button, print_button, exit_button,
			zoom_button, recenter_button, reset_button,
			zoom_factor_slider, model_cycle,
			model_cycle_label, viewport_x_label,
			viewport_y_label, viewport_x_value,
			viewport_y_value, center_label, center_value,
			drawing_mode_label, drawing_mode_cycle;
static Panel_item	message_msg;
static Menu		canvas_menu;
static Notify_value	FrameEventHandler();
static int		Resize(), ControlPanelEventProc(),
			CanvasEventProc(), ExitProc(), ModelProc(),
			DrawingModeProc();
static int		point_selection_mode = OFF;
static int		canvas_point_selection_mode = OFF;
static int		mini_canvas_point_selection_mode = OFF;
static PFI		xy_return_function = NULL;
static int		canvas_menu_initialized = 0;
static int		model;

Pixfont		        *wFont1, *wFont2;
Frame			wFrame;
Pixwin			*wCanvas_pw, *wMCanvas_pw;
int			wCanvasLength, wMCanvasLength;
int			wDrawingMode;

/************************************************************************
 *			  PUBLIC PROCEDURES				*
 ************************************************************************/

/*-----------------------------------------------------------------------
 * Function:	wInitialize
 * Description:	initialize the windows
 * Args:	(none)
 * Returns:	success status (0 = success, !0 = failure)
 * Author:	mbp
 * Date:	Mon Apr  9 00:09:21 1990
 */
int
  wInitialize()
{
  wFont1 = pf_open(FONT1_FILE);
  wFont2 = pf_open(FONT2_FILE);

  wFrame =
    window_create(NULL, FRAME,
		  FRAME_LABEL,			"Hypercad",
		  WIN_WIDTH,			DEFAULT_FRAME_WIDTH,
		  WIN_HEIGHT,			DEFAULT_FRAME_HEIGHT,
		  FRAME_SUBWINDOWS_ADJUSTABLE,	FALSE,
		  WIN_FONT,			wFont1,
		  0);
  if (wFrame == NULL) return(WINDOWERROR);
  (void)notify_interpose_event_func(wFrame,
				    FrameEventHandler,
				    NOTIFY_SAFE);

  message_panel =
    window_create(wFrame, PANEL,
		  WIN_FONT,	wFont1,
		  0);
  if (message_panel == NULL) return(WINDOWERROR);
  message_msg =
    panel_create_item(message_panel, PANEL_MESSAGE,
		      PANEL_ITEM_X,		MESSAGE_MSG_X,
		      PANEL_ITEM_Y,		MESSAGE_MSG_Y,
		      0);
  if (message_msg == NULL) return(WINDOWERROR);

  control_panel =
    window_create(wFrame, PANEL,
		  PANEL_EVENT_PROC,	ControlPanelEventProc,
		  WIN_FONT,	wFont1,
		  0);
  if (control_panel == NULL) return(WINDOWERROR);

  {
    int errorcode = InitControlPanelGadgets();
    if (errorcode != 0) return(errorcode);
  }

  canvas =
    window_create(wFrame, CANVAS,
		  WIN_EVENT_PROC, CanvasEventProc,
		  WIN_FONT,	  wFont1,
		  0);
  if (canvas == NULL) return(WINDOWERROR);
  wCanvas_pw = canvas_pixwin(canvas);

  mini_canvas =
    window_create(wFrame, CANVAS,
		  WIN_EVENT_PROC, CanvasEventProc,
		  WIN_FONT,	  wFont1,
		  0);
  if (mini_canvas == NULL) return(WINDOWERROR);
  wMCanvas_pw = canvas_pixwin(mini_canvas);

  Resize();

#ifdef PANELEDITINGON
  paneledit_init(wFrame);
#endif

  wSetCanvasViewport(0.0, 0.0, 1.0);
  wSetMiniCanvasViewport(0.0, 0.0, 1.0);

  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	wMainLoop
 * Description:	enter the main interaction loop
 * Args:	(none)
 * Returns:	success status
 * Author:	mbp
 * Date:	Mon Apr  9 00:10:16 1990
 * Notes:	Normally returns only at termination of program.
 */
int
  wMainLoop()
{
  window_main_loop(wFrame);
}

/*-----------------------------------------------------------------------
 * Function:	wDone
 * Description:	exit the main interaction loop
 * Args:	(none)
 * Returns:	success
 * Author:	mbp
 * Date:	Tue Apr 10 10:40:01 1990
 * Notes:	Causes wMainLoop to return.
 */
int
  wDone()
{
  window_done(wFrame);
}

/*-----------------------------------------------------------------------
 * Function:	wGetXY
 * Description:	Enter "point selection mode"
 * Args:	func: function to call when "point selection mode"
 *		  terminates
 * Returns:	success code
 * Notes:	The syntax for func is:
 *			func(x,y,code)
 *			double x,y;
 *			int code;
 *		where (x,y) are the coordinates of the selected point,
 *		and code is either 1 or 0.  1 means that the selection
 *		is valid; 0 means that the selection was aborted and
 *		that the values of x and y are meaningless.
 */
int
  wGetXY(func)
PFI func;
{
  return(wGetXY0(func, SELECT_IN_CANVAS));
}

/*-----------------------------------------------------------------------
 * Function:	wSetInputFunc
 * Description:	specify a function to call when input is available from
 *		a pipe
 * Args  IN:	func: the function to call
 *		fd: file descriptor of pipe
 * Returns:	success code
 * Notes:	Must be called after wInitialize.
 */
int
  wSetInputFunc(func, fd)
PFI func;
int fd;
{
  notify_set_input_func(wFrame, func, fd);
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	wSetViewport
 * Description:	Set the window interface's notion of the current viewport
 *		 and model
 * Args  IN:	x,y,s: the new viewport
 *		m: the new model
 * Returns:	success status
 * Notes:	
 */
int
  wSetViewport(x,y,s, m)
double x,y,s;
int m;
{
  char buf[50];
  int val;
  double mcx, mcy, mcs;
  static double xpts_kp[] = {0.0, 0.0, 0.0, 0.0,	1.0, 0.0, -1.0,  0.0};
  static double ypts_kp[] = {0.0, 0.0, 0.0, 0.0,	0.0, 1.0,  0.0, -1.0};
  static double xpts_uhp[] = {0.0, 0.0, 0.0, 0.0,	0.0};
  static double ypts_uhp[] = {0.0, 0.0, 0.0, 0.0,	0.0};
  
  
  if ( (m != KLEIN) && (m != POINCARE) && (m != UHP) ) return(-1);
  model = m;
  
  val = wSetCanvasViewport(x,y,s);
  if (val != 0) return(val);
  
  sprintf(buf,"[%7.2G,%7.2G]",  x, x+s);
  panel_set(viewport_x_value, PANEL_LABEL_STRING, buf, 0);
  
  sprintf(buf,"[%7.2G,%7.2G]",  y, y+s);
  panel_set(viewport_y_value, PANEL_LABEL_STRING, buf, 0);
  
  sprintf(buf,"(%7.2G,%7.2G)",  x+s/2, y+s/2);
  panel_set(center_value, PANEL_LABEL_STRING, buf, 0);
  
  switch (model) {
  case KLEIN:
    panel_set(model_cycle, PANEL_VALUE, 0, 0);
    break;
  case POINCARE:
    panel_set(model_cycle, PANEL_VALUE, 1, 0);
    break;
  case UHP:
    panel_set(model_cycle, PANEL_VALUE, 2, 0);
    break;
  }
  
  /*
   * Now update the picture in the mini_canvas.  Set the mini_canvas
   * viewport to be about 10% larger than the smallest square containing
   * both the canvas viewport and the unit circle in KLEIN and POINCARE
   * models, or both the canvas viewport and the origin in the UHP model.
   */
  switch (model) {
  case KLEIN:
  case POINCARE:
    xpts_kp[0] = x;   ypts_kp[0] = y;
    xpts_kp[1] = x+s; ypts_kp[1] = y;
    xpts_kp[2] = x+s; ypts_kp[2] = y+s;
    xpts_kp[3] = x;   ypts_kp[3] = y+s;
    wSmallestSquare(&mcx, &mcy, &mcs, xpts_kp, ypts_kp, 8);
    break;
  case UHP:
    xpts_uhp[0] = x;   ypts_uhp[0] = y;
    xpts_uhp[1] = x+s; ypts_uhp[1] = y;
    xpts_uhp[2] = x+s; ypts_uhp[2] = y+s;
    xpts_uhp[3] = x;   ypts_uhp[3] = y+s;
    wSmallestSquare(&mcx, &mcy, &mcs, xpts_uhp, ypts_uhp, 5);
    break;
  }
  mcx -= 0.05*mcs;
  mcy -= 0.05*mcs;
  mcs *= 1.1;
  wSetMiniCanvasViewport(mcx, mcy, mcs);
  
  wMCanvasBatch(1);
  wMCErase();
  switch (model) {
  case KLEIN:
  case POINCARE:
    wMCArc( 0.0, 0.0, 1.0, 0.0, 2*M_PI );
    break;
  case UHP:
    wMCSegment(mcx, 0.0, mcx+mcs, 0.0);
    wMCSegment(0.0, mcy, 0.0, mcy+mcs);
    break;
  }
  wMCSegment( x,   y,    x,   y+s );
  wMCSegment( x,   y+s,  x+s, y+s );
  wMCSegment( x+s, y+s,  x+s, y   );
  wMCSegment( x+s, y,    x,   y   );
  wMCanvasBatch(0);
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	wSetMenu0
 * Description:	Set the window's menu
 * Args  IN:	entries: array of wMenuEntry objects
 *		n: number of objects in 'entries'
 * Returns:	success
 * Author:	mbp
 * Date:	Tue Apr 10 10:41:52 1990
 * Notes:	The hcore should not need to call this procedure
 *		directly, but should instead use the macro 'wSetMenu'.
 */
wSetMenu0(entries, n)
     wMenuEntry entries[];
     int n;
{
  int i;

  if (canvas_menu_initialized)
    menu_destroy(canvas_menu);

  canvas_menu_initialized = 1;
  canvas_menu = menu_create(0);

  for (i=0; i<n; ++i)
    menu_set(canvas_menu, MENU_ACTION_ITEM,
	     entries[i].string, entries[i].proc,
	     0);
}

/*-----------------------------------------------------------------------
 * Function:	wMessage
 * Description:	Print a message for the user to see
 * Args  IN:	msg: the message to print
 * Returns:	success status
 * Author:	mbp
 * Date:	Mon Apr 16 18:50:34 1990
 * Notes:	Message can be at most wMAX_MSG_LEN chars long.  If it
 *		is longer, it is truncated to this length before being
 *		displayed.  This truncation overwrites the contents of
 *		the location pointed to by msg.
 */
int
  wMessage(msg)
char *msg;
{
  char *newmsg;
  int val;

  if (msg == NULL)
    newmsg = "";
  else newmsg = msg;
  if (strlen(newmsg) > wMAX_MSG_LEN) {
    newmsg[wMAX_MSG_LEN] = '\0';
    val = -1;
  }
  else val = 0;
  panel_set(message_msg, PANEL_LABEL_STRING, newmsg, 0);
  return(val);
}


/************************************************************************
 *			  PRIVATE PROCEDURES				*
 ************************************************************************/

/*-----------------------------------------------------------------------
 * Function:	InitControlPanelGadgets
 * Description:	initialize the gadgets in the control panel
 * Args:	(none)
 * Returns:	success
 * Author:	mbp
 * Date:	Mon Apr  9 00:09:21 1990
 */
static int
  InitControlPanelGadgets()
{
  help_button =
    panel_create_item(control_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,	&help_pr,
		      PANEL_NOTIFY_PROC,	wHelp,
		      PANEL_ITEM_X,		HELP_BUTTON_X,
		      PANEL_ITEM_Y,		HELP_BUTTON_Y,
		      0);
  if (help_button == NULL) return(WINDOWERROR);

  print_button =
    panel_create_item(control_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,	&print_pr,
		      PANEL_NOTIFY_PROC,	wPrint,
		      PANEL_ITEM_X,		PRINT_BUTTON_X,
		      PANEL_ITEM_Y,		PRINT_BUTTON_Y,
		      0);
  if (print_button == NULL) return(WINDOWERROR);

  exit_button =
    panel_create_item(control_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,	&exit_pr,
		      PANEL_NOTIFY_PROC,	ExitProc,
		      PANEL_ITEM_X,		EXIT_BUTTON_X,
		      PANEL_ITEM_Y,		EXIT_BUTTON_Y,
		      0);
  if (exit_button == NULL) return(WINDOWERROR);

  zoom_button =
    panel_create_item(control_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,	&zoom_pr,
		      PANEL_NOTIFY_PROC,	wZoom,
		      PANEL_ITEM_X,		ZOOM_BUTTON_X,
		      PANEL_ITEM_Y,		ZOOM_BUTTON_Y,
		      0);
  if (zoom_button == NULL) return(WINDOWERROR);

  recenter_button =
    panel_create_item(control_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,	&recenter_pr,
		      PANEL_NOTIFY_PROC,	wRecenter,
		      PANEL_ITEM_X,		RECENTER_BUTTON_X,
		      PANEL_ITEM_Y,		RECENTER_BUTTON_Y,
		      0);
  if (recenter_button == NULL) return(WINDOWERROR);
  wSetRecenterPixrects(recenter_button, &recenter_pr, &recenter25_pr);

  reset_button =
    panel_create_item(control_panel, PANEL_BUTTON,
		      PANEL_LABEL_IMAGE,	&reset_pr,
		      PANEL_NOTIFY_PROC,	wReset,
		      PANEL_ITEM_X,		RESET_BUTTON_X,
		      PANEL_ITEM_Y,		RESET_BUTTON_Y,
		      0);
  if (reset_button == NULL) return(WINDOWERROR);

  model_cycle_label =
    panel_create_item(control_panel, PANEL_MESSAGE,
		      PANEL_ITEM_X,		MODEL_CYCLE_LABEL_X,
		      PANEL_ITEM_Y,		MODEL_CYCLE_LABEL_Y,
		      PANEL_LABEL_STRING,	"Model:",
		      0);

  model_cycle =
    panel_create_item(control_panel, PANEL_CYCLE,
		      PANEL_ITEM_X,		MODEL_CYCLE_X,
		      PANEL_ITEM_Y,		MODEL_CYCLE_Y,
		      PANEL_LAYOUT,		PANEL_VERTICAL,
		      PANEL_CHOICE_STRINGS,	"Klein",	/* 0 */
						"Poincare",	/* 1 */
		       				"Half Plane",	/* 2 */
						0,
		      PANEL_VALUE,		0,
		      PANEL_NOTIFY_PROC,	ModelProc,
		      0);
  if (model_cycle == NULL) return(WINDOWERROR);

  zoom_factor_slider =
    wCreateZoomFactorSlider(control_panel,
			      ZOOM_FACTOR_SLIDER_X,
			      ZOOM_FACTOR_SLIDER_Y,
			      ZOOM_FACTOR_LABEL_MSG_X,
			      ZOOM_FACTOR_LABEL_MSG_Y,
			      ZOOM_FACTOR_VALUE_MSG_X,
			      ZOOM_FACTOR_VALUE_MSG_Y);
  if (zoom_factor_slider == NULL) return(WINDOWERROR);

  viewport_x_label = 
    panel_create_item(control_panel, PANEL_MESSAGE,
		      PANEL_ITEM_X,		VIEWPORT_X_LABEL_X,
		      PANEL_ITEM_Y,		VIEWPORT_X_LABEL_Y,
		      PANEL_LABEL_STRING,	"Viewport x:",
		      0);

  viewport_x_value =
    panel_create_item(control_panel, PANEL_MESSAGE,
		      PANEL_ITEM_X,		VIEWPORT_X_VALUE_X,
		      PANEL_ITEM_Y,		VIEWPORT_X_VALUE_Y,
		      PANEL_LABEL_STRING,	"",
		      PANEL_LABEL_FONT,		wFont2,
		      0);

  viewport_y_label = 
    panel_create_item(control_panel, PANEL_MESSAGE,
		      PANEL_ITEM_X,		VIEWPORT_Y_LABEL_X,
		      PANEL_ITEM_Y,		VIEWPORT_Y_LABEL_Y,
		      PANEL_LABEL_STRING,	"Viewport y:",
		      0);

  viewport_y_value =
    panel_create_item(control_panel, PANEL_MESSAGE,
		      PANEL_ITEM_X,		VIEWPORT_Y_VALUE_X,
		      PANEL_ITEM_Y,		VIEWPORT_Y_VALUE_Y,
		      PANEL_LABEL_STRING,	"",
		      PANEL_LABEL_FONT,		wFont2,
		      0);

  center_label =
    panel_create_item(control_panel, PANEL_MESSAGE,
		      PANEL_ITEM_X,		CENTER_LABEL_X,
		      PANEL_ITEM_Y,		CENTER_LABEL_Y,
		      PANEL_LABEL_STRING,	"Center: ",
		      0);

  center_value =
    panel_create_item(control_panel, PANEL_MESSAGE,
		      PANEL_ITEM_X,		CENTER_VALUE_X,
		      PANEL_ITEM_Y,		CENTER_VALUE_Y,
		      PANEL_LABEL_STRING,	"",
		      PANEL_LABEL_FONT,		wFont2,
		      0);

  center_label =
    panel_create_item(control_panel, PANEL_MESSAGE,
		      PANEL_ITEM_X,		DRAWING_MODE_LABEL_X,
		      PANEL_ITEM_Y,		DRAWING_MODE_LABEL_Y,
		      PANEL_LABEL_STRING,	"Drawing Mode:",
		      0);

  drawing_mode_cycle =
    panel_create_item(control_panel, PANEL_CYCLE,
		      PANEL_ITEM_X,		DRAWING_MODE_CYCLE_X,
		      PANEL_ITEM_Y,		DRAWING_MODE_CYCLE_Y,
		      PANEL_LAYOUT,		PANEL_VERTICAL,
		      PANEL_CHOICE_STRINGS,	"Batch",	/* 0 */
						"Continuous",	/* 1 */
						0,
		      PANEL_VALUE,		0,
		      PANEL_NOTIFY_PROC,	DrawingModeProc,
		      0);
  wDrawingMode = BATCH;
  if (model_cycle == NULL) return(WINDOWERROR);


  return(0);
}

static int
  ExitProc()
{
  wDone();
  exit(0);
}

/*ARGSUSED*/
static int
  ModelProc(item, value, event)
Panel_item item;
int value;
Event *event;
{
  switch (value) {
  case 0:
    SethModel(KLEIN);
    break;
  case 1:
    SethModel(POINCARE);
    break;
  case 2:
    SethModel(UHP);
    break;
  }
}

/*ARGSUSED*/
static int
  DrawingModeProc(item, value, event)
Panel_item item;
int value;
Event *event;
{
  switch (value) {
  default:
  case 0:
    wDrawingMode = BATCH;
    break;
  case 1:
    wDrawingMode = CONTINUOUS;
    break;
  }
}

/*-----------------------------------------------------------------------
 * Function:	FrameEventHandler
 * Description:	main event handler for frame
 * Args  IN:	frame: the frame handle
 *		event: the event
 *		arg, type: (unused??)
 * Returns:	value returned by next event function
 * Author:	mbp
 * Date:	Tue Apr 10 10:10:45 1990
 * Notes:	We interpose this function to catch all frame
 *		events so we can detect when the user does things
 *		with the frame, such as resizing it.
 */
static Notify_value
  FrameEventHandler(frame, event, arg, type)
Frame frame;
Event *event;
Notify_arg arg;
Notify_event_type type;
{
  /* If this is a resize event, readjust internal params via Resize() */
  if (event_id(event) == WIN_RESIZE)
    Resize();

  /* Now return control to the notifier to continue processing
   * the event */
  return(notify_next_event_func(frame, event, arg, type));
}

/*-----------------------------------------------------------------------
 * Function:	Resize
 * Description:	Adjust internal parameters and subwindow sizes to
 *		fit new frame size
 * Args:	(none)
 * Returns:	success status
 * Author:	mbp
 * Date:	Tue Apr 10 10:16:08 1990
 * Notes:	This may cause the frame itself to resize in order
 *		to meet various sizing restrictions.
 */
static int
  Resize()
{
  int fw,fh, mpw, mph, cpw, cph;

#define cw	canvas_width
#define ch	canvas_height
#define mch	mini_canvas_width
#define mcw	mini_canvas_height

  /*
   * Make sure the frame is at least its minimum allowable size
   */
  if (WINWIDTH(wFrame) < MIN_FRAME_WIDTH)
    window_set(wFrame, WIN_WIDTH, MIN_FRAME_WIDTH, 0);
  if (WINHEIGHT(wFrame) < MIN_FRAME_HEIGHT)
    window_set(wFrame, WIN_HEIGHT, MIN_FRAME_HEIGHT, 0);

  /*
   * Calculate length (width=height) of the canvas.
   */
  {
    int tcw = WINWIDTH(wFrame) - CONTROL_PANEL_WIDTH - 3 * WSEP;
    int tch = WINHEIGHT(wFrame) - WINTOPMARGIN(wFrame) - MESSAGE_PANEL_HEIGHT
      - 2 * WSEP;
    wCanvasLength = MAX(tcw, tch);
  }

  /*
   * Recalculate all dimensions with the new canvas size
   */
  fw = cw + CONTROL_PANEL_WIDTH + 3 * WSEP;
  fh = ch + MESSAGE_PANEL_HEIGHT + WINTOPMARGIN(wFrame) + 2 * WSEP;

  mpw = fw - 2 * WSEP;
  cpw = CONTROL_PANEL_WIDTH;
  mcw = cpw;

  mph = MESSAGE_PANEL_HEIGHT;
  mch = mcw;
  cph = fh - WINTOPMARGIN(wFrame) - mph - mch - 3 * WSEP;

  /*
   * Set the size and position of all subwindows
   */
  window_set(wFrame,
	     WIN_WIDTH,  fw,
	     WIN_HEIGHT, fh,
	     0);
  window_set(message_panel,
	     WIN_WIDTH,  mpw,
	     WIN_HEIGHT, mph,
	     0);
  window_set(canvas,
	     WIN_WIDTH,  cw,
	     WIN_HEIGHT, ch,
	     0);
  window_set(control_panel,
	     WIN_WIDTH,  cpw,
	     WIN_HEIGHT, cph,
	     0);
  window_set(mini_canvas,
	     WIN_WIDTH,  mcw,
	     WIN_HEIGHT, mch,
	     0);

  window_set(message_panel,
	     WIN_X,		0,
	     WIN_Y,		0,
	     0);
  window_set(canvas,
	     WIN_X,		0,
	     WIN_BELOW,		message_panel,
	     0);
  window_set(control_panel,
	     WIN_RIGHT_OF,	canvas,
	     WIN_BELOW,		message_panel,
	     0);
  window_set(mini_canvas,
	     WIN_RIGHT_OF,	canvas,
	     WIN_BELOW,		control_panel,
	     0);

  /*
   * Recompute coordinate system factors using the new canvas sizes
   */
  wComputeCanvasFactors();
  wComputeMiniCanvasFactors();

  /*
   * Redraw the picture in the new canvas
   */
  DrawPicture();

#undef cw
#undef ch
#undef mch
#undef mcw
}

/*-----------------------------------------------------------------------
 * Function:	ControlPanelEventProc
 * Description:	Handle an event in the control panel
 * Args  IN:	item: the item over which the event occurred
 *		event: the event
 * Returns:	nothing
 * Author:	mbp
 * Date:	Tue Apr 10 10:20:19 1990
 */
static int
  ControlPanelEventProc(item, event)
Panel_item item;
Event *event;
{
  /* If we're in point selection mode, do nothing.  Otherwise
   * process the event as usual. */
  if (!point_selection_mode)
    panel_default_handle_event(item, event);
}

/*-----------------------------------------------------------------------
 * Function:	CanvasEventProc
 * Description:	Handle an event from one of the canvases
 * Args  IN:	window: the window receiving the event (should be
 *		  either canvas or mini_canvas)
 *		event: the event
 *		arg: (unused?)
 * Returns:	(nothing)
 * Author:	mbp
 * Date:	Tue Apr 10 11:47:24 1990
 */
/*ARGSUSED*/
static int
  CanvasEventProc(window, event, arg)
Window window;
Event *event;
caddr_t arg;
{
  
  if (!point_selection_mode) {
    if (window == canvas) {
      if ( (event_id(event)==MS_RIGHT) && (event_is_down(event)) ) {
	if (canvas_menu_initialized)
	  menu_show(canvas_menu, window, event, 0);
      }
    }
  }
  
  if (   ((canvas_point_selection_mode)      && (window==canvas     ))
      || ((mini_canvas_point_selection_mode) && (window==mini_canvas)) ) {
    switch (event_id(event)) {
    case MS_LEFT:
      if (event_is_down(event))
	EndPointSelection(event, window, 1);
      break;
    case 27: /* ESC */
      EndPointSelection(event, window, 0);
      break;
    }
  }
}

/*-----------------------------------------------------------------------
 * Function:	EndPointSelection
 * Description:	exit from point selection mode
 * Args  IN:	event: the event which caused us to exit
 *		window: the window (canvas) in which event occurred
 *		valid: validity flag to be passed to xy_return_function
 * Returns:	(nothing)
 * Author:	mbp
 * Date:	Tue Apr 10 11:47:59 1990
 * Notes:	This procedure undoes everything that was done when
 *		point selection mode was entered and calls the
 *		xy_return_function with the x,y coords of the event.
 */
static int
  EndPointSelection(event, window, valid)
Event *event;
Window window;
int valid;
{
  double x,y;

  EventToXY(&x, &y, event, window);
  PointSelectionMode(OFF);
  if (canvas_point_selection_mode)
    CanvasPointSelectionMode(canvas, OFF);
  if (mini_canvas_point_selection_mode)
    CanvasPointSelectionMode(mini_canvas, OFF);
  (*xy_return_function)(x,y,valid);
}

/*-----------------------------------------------------------------------
 * Function:	EventToXY
 * Description:	translate an event to its (x,y) coordinates in user
 *		  (viewport) space.
 * Args  IN:	event: the event
 *		window: the window receiving the event (should be
 *		  either canvas or mini_canvas)
 *      OUT:	*x,*y: the user coordinates of the event
 * Returns:	nothing
 * Author:	mbp
 * Date:	Tue Apr 10 10:24:40 1990
 * Notes:	The coordinates returned are in the system set by
 *		the last call to wSetCanvasViewport (for the canvas)
 *		or wSetMiniCanvasViewport (for the mini-canvas).
 */
static int
  EventToXY(x, y, event, window)
double *x,*y;
Event *event;
Window window;
{
  int cx, cy;

  /* Translate to canvas pixel coords */
  EventToCanvasXY(&cx, &cy, event, window);

  /* Translate canvas pixel coords to user coords */
  if (window == canvas) {
    *x = SAVNAC_X_COORD(cx);
    *y = SAVNAC_Y_COORD(cy);
  }
  else if (window == mini_canvas) {
    *x = MINI_SAVNAC_X_COORD(cx);
    *y = MINI_SAVNAC_Y_COORD(cy);
  }
}

/*-----------------------------------------------------------------------
 * Function:	EventToCanvasXY
 * Description:	translate an event to its (x,y) coordinates in canvas
 *		  pixel coordinates
 * Args  IN:	event: the event
 *		canvas: the canvas (should be canvas or mini_canvas)
 *      OUT:	*x,*y: the coords of the event in the given canvas
 * Returns:	(nothing)
 * Author:	mbp
 * Date:	Tue Apr 10 10:28:55 1990
 */
static int
  EventToCanvasXY(x, y, event, canvas)
int *x,*y;
Event *event;
Canvas canvas;
{
  Event *event_in_canvas_space = canvas_event(canvas, event);
  *x = event_x(event_in_canvas_space);
  *y = event_y(event_in_canvas_space);
}

/*-----------------------------------------------------------------------
 * Function:	PointSelectionMode
 * Description:	set the point selection mode
 * Args  IN:	op: ON or OFF
 * Returns:	(nothing)
 * Author:	mbp
 * Date:	Tue Apr 10 10:31:41 1990
 * Notes:	Normally point selection mode is off.  When it is on,
 *		  most event handling is disabled  while the window waits
 *		  for the user to select a point.
 */
static int
  PointSelectionMode(op)
int op;
{
  point_selection_mode = op;
}

/*-----------------------------------------------------------------------
 * Function:	CanvasPointSelectionMode
 * Description:	Turn "point selection mode" on or off for a canvas
 * Args  IN:	canv: the canvas to turn it on or off in
 *		op: ON or OFF
 * Returns:	nothing
 * Notes:	Normally "point selection mode" is off.  Turning it
 *		  on means that the cursor is changed to a small
 *		  crosshair and acceptance of ascii events is enabled.
 */
static int
  CanvasPointSelectionMode(canv, op)
Canvas canv;
int op;
{
  Cursor cursor;

  if (canv == canvas)
    canvas_point_selection_mode = op;
  if (canv == mini_canvas)
    mini_canvas_point_selection_mode = op;

  cursor = window_get(canv, WIN_CURSOR );

  switch (op) {
  case ON:
    cursor_set(cursor,
	       CURSOR_SHOW_CURSOR,		FALSE,
	       CURSOR_SHOW_CROSSHAIRS,		TRUE,
	       CURSOR_CROSSHAIR_THICKNESS,	1,
	       CURSOR_CROSSHAIR_COLOR,		3,
	       CURSOR_CROSSHAIR_LENGTH, 	8,
	       CURSOR_CROSSHAIR_GAP,		2,
	       CURSOR_CROSSHAIR_OP, 		PIX_NOT(PIX_DST),
	       0);
    window_set(canv,
	       WIN_CONSUME_KBD_EVENT, WIN_ASCII_EVENTS,
	       WIN_CURSOR, cursor,
	       0 );
    break;
  case OFF:
    cursor_set(cursor,
	       CURSOR_SHOW_CURSOR,	TRUE,
	       CURSOR_SHOW_CROSSHAIRS,	FALSE,
	       0);
    window_set(canv,
	       WIN_IGNORE_KBD_EVENT,	WIN_ASCII_EVENTS,
	       WIN_CURSOR,		cursor,
	       0);
    break;
  }
}

int
  wGetXY0(func, canvas_spec)
PFI func;
int canvas_spec;
{
  if (func == NULL) return(-1);
  xy_return_function = func;
  PointSelectionMode(ON);
  CanvasPointSelectionMode(canvas, ON);
  if (canvas_spec == SELECT_IN_BOTH)
      CanvasPointSelectionMode(mini_canvas, ON);
  return(0);
}

