(* $Id: Tessellations.m.tpl,v 1.4 90/07/11 13:08:32 mbp Exp Locker: mbp $
 *
 * Tessellations.m: Functions for computing and drawing tessellations
 *   by discrete groups in RH2.
 *)

(**************************************************************************
 *     Copyright (C) 1990 by Mark B. Phillips and Robert R. Miner	  *
 * 									  *
 * Permission to use, copy, modify, and distribute this software, its	  *
 * documentation, and any images it generates for any purpose and without *
 * fee is hereby granted, provided that					  *
 * 									  *
 * (1) the above copyright notice appear in all copies and that both that *
 *     copyright notice and this permission notice appear in supporting	  *
 *     documentation, and that the names of Mark B.  Phillips, Robert R.  *
 *     Miner, or the University of Maryland not be used in advertising or *
 *     publicity pertaining to distribution of the software without	  *
 *     specific, written prior permission.				  *
 *									  *
 * (2) Explicit written credit be given to the authors Mark B.  Phillips  *
 *     and Robert R. Miner in any publication which uses part or all of	  *
 *     any image produced by this software.				  *
 *									  *
 * This software is provided "as is" without express or implied warranty. *
 **************************************************************************)

BeginPackage["Tessellations`", "Dirichlet`", "GroupTheory`", "RH2`"]

DirichletDomain::usage = "DirichletDomain[ p, gens, wordlength ]
returns a list of line segments which form the boundary of an
approximation to the Dirichlet fundamental domain based at p for the
group generated by gens.  Specifically, the line segments form the
boundary of the intersection of the half-spaces corresponding to
points in the orbit of p up to word length wordlength.  The option
OrbitTolerance may be used to explicitly set the tolerance to within
which the orbit is computed."

DirichletSemiDomain::usage = "DirichletSemiDomain is like
DirichletDomain but works with semigroup generators instead of group
generators."

OrbitTolerance::usage "OrbitTolerance is an option for DirichletDomain
and which specifies a tolerance to use in the call to GeometricSemiOrbit."

DirichletTessellation::usage = "DirichletTesselllation usage message not
written yet."

DirichletSemiTessellation::usage = "DirichletTesselllation usage message not
written yet."

GeometricOrbit::usage = "GeometricOrbit[p, gens, wordlength, tol]
returns a list {points, words} of two sublists.  The points sublist
consists of the points in the orbit of the point p under words up to
word length wordlength in the group generated by the matrices gens.
The sublist is pruned so that the hyperbolic distance between any pair
of points in it is at least tol.  If tol is the integer 0, the pruning
step is omitted; since we don't check for relations among the
generators, this will in general have redundancies in it.  The words
sublist consists of the group elements (matrices) corresponding to the
points sublist; the image of p under words[[i]] is points[[i]]. p
should be a kPoint, pPoint, uPoint or r21Vector, and gens should be a
list of matrices in O(2,1).  In all cases the first element of the
points sublist is p, and the first element of the words sublist is the
identity matrix. The remaining elements of these lists are in no
special order."

GeometricSemiOrbit::usage = "GeometricSemiOrbit is like Geometric
Orbit but works with semigroup generators instead of group
generators."

SeparateIndices::usage = "SeparateIndices[{x1,...,xn}, epsilon]
returns a list of the indices of an epsilon-separated sublist of
{x1,...,xn}.  A list is epsilon-separated if the distance between
every pair of points in it is at least epsilon.  SeparateIndices works
by eliminating the second of any pair of points which are withing
epsilon of each other.  The xi may be kPoints, pPoints, uPoints, or
r21Vectors."

Separate::usage = "Separate[{x1,...,xn}, epsilon] returns an
epsilon-separated sublist of the the points {x1,...,xn}.  See also:
SeparateIndices."

Begin["`private`"]

SeparateProgram = "SEPARATEPATH"

SeparateIndices[{points___kPoint}, eps_] :=
  SeparatekPointListIndices[{points}, eps]

SeparateIndices[{points___pPoint}, eps] :=
  Map[ pPoint, SeparatekPointListIndices[ Map[ kPoint, {points} ], eps ] ]

SeparateIndices[{points___uPoint}, eps] :=
  Map[ uPoint, SeparatekPointListIndices[ Map[ kPoint, {points} ], eps ] ]

SeparateIndices[{points___r21Vector}, eps] :=
  Map[ r21Vector, SeparatekPointListIndices[ Map[ kPoint, {points} ], eps ] ]

kPoint/: CForm[ kPoint[x_, y_] ] := {CForm[x], CForm[y]}

SeparatekPointListIndices[list_List, eps_] :=
  RunThrough[ StringJoin[SeparateProgram, " -epsilon ", ToString[CForm[eps]]],
	      Map[ CForm, list ] ]

(*
 * internal version - slow as mud!:
 *
 * SeparatekPointListIndices[list_List, eps_] :=
 *   Block[ {factor, test, flags, i, j, len, x, y},
 *     len = Length[list];
 *     flags = Table[ True, {len} ];
 *     factor = Cosh[eps]^2;
 *     Do[ If[ flags[[i]],
 * 	    test[kPoint[x_,y_]] :=
 *               ( Release[ x list[[i,1]] + y list[[i,2]] - 1 ]^2
 * 		  < ( factor * (x^2 + y^2 - 1)
 * 		      * Release[ list[[i,1]]^2 + list[[i,2]]^2 - 1 ] ) );
 * 	    Do[ If[ flags[[j]],
 * 		    If[ test[ list[[j]] ],
 * 			flags[[j]] = False ] ],
 * 		{j, i+1, len} ] ],
 * 	{i, 1, len-1} ];
 *     Return[ Flatten[ Position[ flags, True ] ] ]
 *     ]
 *)

Separate[list_List, eps_] :=
  list[[ SeparateIndices[ list, eps ] ]]

GeometricOrbit[p_?hPointQ, gens_, wordlength_, tol_] :=
  GeometricSemiOrbit[p, Join[gens,Map[Inverse,gens]], wordlength, tol]

GeometricSemiOrbit[p_?hPointQ, gens_, wordlength_, 0] :=
  Block[ {words},
    (* We prepend the id matrix to make sure it's first; this
     * introduces at least one duplication, since the list returned by
     * EnumerateSemiGroup already has the id in it somewhere.  But we
     * won't fuss over this, since in general the list will have lots
     * of redundancies in it, which will be eliminated at a later
     * time. *)
    words = Prepend[ EnumerateSemiGroup[IdentityMatrix[3],
			            gens, Dot, wordlength],
		     IdentityMatrix[3] ];
    Return[ { Map[ hApply[#, p]&, words ], words } ]
    ]

GeometricSemiOrbit[p_?hPointQ, gens_, wordlength_, tol_] :=
  Block[ {points, words, indices},
    {points, words} = GeometricSemiOrbit[p, gens, wordlength, 0];
    indices = SeparateIndices[ points, tol ];
    Return[ { points[[indices]], words[[indices]] } ]
    ]

DirichletDomain/:
  Options[DirichletDomain] := {OrbitTolerance->0.01}

DirichletDomain[p_?hPointQ, gens_List, wordlength_, opts___] :=
  DirichletSemiDomain[p, Join[gens, Map[Inverse,gens]], wordlength, opts]

DirichletSemiDomain[p_?hPointQ, gens_List, wordlength_, opts___] :=
  Block[ {points, tol},
    tol = OrbitTolerance /. {opts} /. Options[DirichletDomain];
    points = GeometricSemiOrbit[p, gens, wordlength, tol][[1]];
    Return[ DirichletRegion[ First[points], Rest[points] ] ]
    ]

DirichletTessellation[p_?hPointQ, gens_List, wordlength_, opts___] :=
  DirichletSemiTessellation[p, Join[gens,Map[Inverse,gens]], wordlength, opts]

DirichletSemiTessellation[p_?hPointQ, gens_List, wordlength_, opts___] :=
  Block[ {points, words, tol, dirdom},
    tol = OrbitTolerance /. {opts} /. Options[DirichletDomain];
    {points, words} = GeometricSemiOrbit[p, gens, wordlength, tol];
    dirdom = DirichletRegion[ First[points], Rest[points] ];
    Return[ DistributeOp[ hApply, words, dirdom ] ]
    ]

End[]

EndPackage[]
