/* $Id$
 *
 * surface.c: procedures for generating surface data (meshes, lists
 *  of quadrilaterals, etc.)
 */

/************************************************************************
 *  Copyright (C) 1989 by Mark B. Phillips and William M. Goldman       *
 * 									*
 * Permission to use, copy, modify, and distribute this software and	*
 * its documentation for any purpose and without fee is hereby		*
 * granted, provided that the above copyright notice appear in all	*
 * copies and that both that copyright notice and this permission	*
 * notice appear in supporting documentation, and that the names of	*
 * Mark B. Phillips, William M. Goldman, or the University of Maryland	*
 * not be used in advertising or publicity pertaining to distribution	*
 * of the software without specific, written prior permission.  This	*
 * software is provided "as is" without express or implied warranty.	*
 ************************************************************************/

#include "heisenberg.h"
#include <stdio.h>

/*-----------------------------------------------------------------------
 * Function:	chain_list
 * Description: create a list of points on a chain
 * Args IN:	*c: the chain to draw
 *		n: number of points to put in list
 *	OUT:	list: the returned list
 * Returns:	nothing
 * Notes:	If the chain is not vertical, list[n-1] is the same as
 *		list[0].
 *
 *		I adapted this procedure from draw_chain on 7/17/89 - mbp.
 */
chain_list(c,n,list)
     Chain *c;
     int n;
     double list[][3];
{
  double *X;
  double th, thinc, v, vinc;
  int i;

  if (c->rad==CHAIN_VERTICAL) {	/* It's a vertical chain ... */
    vinc = (vert_chain_max - vert_chain_min) / (n-1);
    for(i=0,v=vert_chain_min;
	i<n;
	++i,v+=vinc ) {
      X = list[i];
      X[0] = c->cen.hor.re;
      X[1] = c->cen.hor.im;
      X[2] = v;
    }
  }
  else {   /* It's not a vertical chain, ... */

    /* points 1 thru n-1 (list[0] thru list[n-2]): */
    thinc = 2 * M_PI / (n-1);
    for(i=0,th=0;
	i<n-1;
	++i,th+=thinc) {
      X = list[i];
      X[0] = c->cen.hor.re + c->rad * cos(th); 
      X[1] = c->cen.hor.im + c->rad * sin(th); 
      X[2] = chain_v_coord(&(c->cen),X[0],X[1]);
    }

    /* point n (list[n-1]) is same as point 1 (list[0]): */
    X = list[n-1];
    for (i=0; i<3; ++i) X[i] = list[0][i];
  }
}


/*-----------------------------------------------------------------------
 * Function:	mesh_spinal
 * Description:	write a mesh representing a spinal to a file
 * Args	 IN:	*sp: the spinal to draw
 *		*fp: the file to write to
 * Returns:     (nothing)
 * Notes:       adapted from triangulate_spinal on 11/6/89
 *
 */
mesh_spinal(sp, fp)
     Spinal *sp;
     FILE *fp;
{
  Chain slice;
  Cvector Z1,Z2;
  Complex t0;
  double latitude, latinc;
  int j, success;
  int n;  /* n is the number of points per chain to use */
  double *list1, *list2, *templist;
  
  n = chain_resolution + 1;
  fprintf(fp, "MESH\n%1d %1d\n", n, slice_count+2);
  
  hpoint_to_cvector(Z1,&(sp->p1));
  hpoint_to_cvector(Z2,&(sp->p2));
  
  /* Normalize Z1 and Z2 to have 3rd coord 1 */
  for (j=0; j<3; j++){  
    C_PUSH_S(Z1[j]); C_PUSH_S(Z1[2]); c_div_s(); C_POP_S(Z1[j]); 
    C_PUSH_S(Z2[j]); C_PUSH_S(Z2[2]); c_div_s(); C_POP_S(Z2[j]); 
  } 

  /* compute t0 */
  compute_spinal_t0(&t0, Z1,Z2);
  
  /* First vertex: latitude = 0.0 */
  compute_spinal_slice(&slice,0.0,&t0,Z1,Z2,&success);
  if (success)
    write_chain_list(fp, &slice, n);
  
  /* Intermediate chains: */
  latinc = M_PI / (slice_count + 1);
  for(latitude=latinc; latitude<M_PI; latitude+=latinc) {
    compute_spinal_slice(&slice,latitude,&t0,Z1,Z2,&success);
    if (success)
      write_chain_list(fp, &slice, n);
  }

  /* Final vertex: latitude = Pi */
  compute_spinal_slice(&slice,M_PI,&t0,Z1,Z2,&success);
  if (success)
    write_chain_list(fp, &slice, n);
}

/*-----------------------------------------------------------------------
 * Function:	write_chain_list
 * Description:	Write a list of points on a chain to a file
 * Args  IN:	fp: file to write to
 *		*c: the chain
 *		n: number of points on chain to use
 * Returns:	nothing
 */
write_chain_list(fp, c, n)
     FILE *fp;
     Chain *c;
     int n;
{
  double *list, *listptr;
  int i;

  list = (double*)malloc(3*n*sizeof(double));
  chain_list(c, n, list);
  for (i=0, listptr=list; i<n; ++i,listptr+=3)
    fprintf(fp, "%f %f %f\n", *listptr, *(listptr+1), *(listptr+2));
  free(list);
}

/*-----------------------------------------------------------------------
 * Function:	triangulate_spinal
 * Description:	write a sequence of triangles which tesselate a spinal
 * Args	 IN:	*sp: the spinal to draw
 *		*fp: the file to write to
 * Returns:     (nothing)
 * Notes:       adapted from draw_spinal_c on 7/17/89 - mbp.
 *
 */
triangulate_spinal(sp, fp)
     Spinal *sp;
     FILE *fp;
{
  Chain slice;
  Cvector Z1,Z2;
  Complex t0;
  double latitude, latinc;
  int j, success;
  int n;  /* n is the number of points per chain to use as vertices of	*/
  	  /* triangles; this is chain_resolution + 1			*/
  double *list1, *list2, *templist;
  
  n = chain_resolution + 1;
  
  /* Allocate space for lists of points: */
  list1 = (double*)malloc(3*n*sizeof(double));
  list2 = (double*)malloc(3*n*sizeof(double));
  
  hpoint_to_cvector(Z1,&(sp->p1));
  hpoint_to_cvector(Z2,&(sp->p2));
  
  /* Normalize Z1 and Z2 to have 3rd coord 1 */
  for (j=0; j<3; j++){  
    C_PUSH_S(Z1[j]); C_PUSH_S(Z1[2]); c_div_s(); C_POP_S(Z1[j]); 
    C_PUSH_S(Z2[j]); C_PUSH_S(Z2[2]); c_div_s(); C_POP_S(Z2[j]); 
  } 

  /* compute t0 */
  compute_spinal_t0(&t0, Z1,Z2);
  
  /* First vertex: latitude = 0.0 */
  compute_spinal_slice(&slice,0.0,&t0,Z1,Z2,&success);
  if (success) chain_list(&slice, n, list1);
  
  /* Intermediate chains: */
  latinc = M_PI / (slice_count + 1);
  for(latitude=latinc; latitude<M_PI; latitude+=latinc) {
    compute_spinal_slice(&slice,latitude,&t0,Z1,Z2,&success);
    if (success) {
      /* Build next list2 */
      chain_list(&slice, n, list2);
      
      /* Generate triangles for list1 and list2 */
      write_triangle_strip(list1,list2,n,fp);
      
      /* Interchange list1 and list2 */
      templist = list2;
      list2 = list1;
      list1 = templist;
    }
  }

  /* Final vertex: latitude = Pi */
  compute_spinal_slice(&slice,M_PI,&t0,Z1,Z2,&success);
  if (success) {
    chain_list(&slice, n, list2);
    write_triangle_strip(list1,list2,n,fp);
  }

  free((char*)list1);
  free((char*)list2);
}

/*-----------------------------------------------------------------------
 * Function:	list_spinal
 * Description:	write a sequence of points on a spinal to a file
 * Args	 IN:	*sp: the spinal to draw
 *		*fp: the file to write to
 * Returns:     (nothing)
 * Notes:       writes one point per line, with a blank line
 *		  between chains
 */
list_spinal(sp, fp)
     Spinal *sp;
     FILE *fp;
{
  Chain slice;
  Cvector Z1,Z2;
  Complex t0;
  double latitude, latinc;
  int j, success, sliceno;
  int n;  /* n is the number of points per chain */
  double *list1, *templist;
  
  n = chain_resolution + 1;

  /* Allocate space for list of points: */
  list1 = (double*)malloc(3*n*sizeof(double));
  
  hpoint_to_cvector(Z1,&(sp->p1));
  hpoint_to_cvector(Z2,&(sp->p2));
  
  /* Normalize Z1 and Z2 to have 3rd coord 1 */
  for (j=0; j<3; j++){  
    C_PUSH_S(Z1[j]); C_PUSH_S(Z1[2]); c_div_s(); C_POP_S(Z1[j]); 
    C_PUSH_S(Z2[j]); C_PUSH_S(Z2[2]); c_div_s(); C_POP_S(Z2[j]); 
  } 

  /* compute t0 */
  compute_spinal_t0(&t0, Z1,Z2);
  
  /* First vertex: latitude = 0.0 */
  compute_spinal_slice(&slice,0.0,&t0,Z1,Z2,&success);
  if (success)
    chain_list(&slice, n, list1);
  else
    fprintf(stderr, "list_spinal: error computing first slice\n");

  write_list(fp, list1, n);
  
  /* Intermediate slices: */
  latinc = M_PI / (slice_count + 1);
  for(sliceno=0,latitude=latinc;
      sliceno<slice_count;
      ++sliceno,latitude+=latinc) {
    compute_spinal_slice(&slice,latitude,&t0,Z1,Z2,&success);
    if (success) {
      chain_list(&slice, n, list1);
      write_list(fp, list1, n);
    }
    else
      fprintf(stderr, "list_spinal: error computing slice %1d\n",sliceno);
  }

  /* Final vertex: latitude = Pi */
  compute_spinal_slice(&slice,M_PI,&t0,Z1,Z2,&success);
  if (success) {
    chain_list(&slice, n, list1);
    write_list(fp, list1, n);
  }

  free((char*)list1);
}

/*-----------------------------------------------------------------------
 * Function:	quad_spinal
 * Description:	write a sequence of quadrilaterals which tesselate a spinal
 * Args	 IN:	*sp: the spinal to draw
 *		*fp: the file to write to
 *		strip_list: list of strips to write
 *		strip_count: length of strip_list
 * Returns:     (nothing)
 * Notes:       if strip_count < 0, draws all strips
 */
quad_spinal(sp, fp, strip_list, strip_count)
     Spinal *sp;
     FILE *fp;
     int *strip_list, strip_count;
{
  Chain slice;
  Cvector Z1,Z2;
  Complex t0;
  Complex Z2Z1Herm;
  double latitude, latinc;
  int j, success, sliceno;
  int n;  /* n is the number of points per chain to use as vertices of	*/
  	  /* quadrilaterals; this is chain_resolution + 1		*/
  double *list1, *list2, *templist;
  int stripno;
  
  n = chain_resolution + 1;
  fprintf(fp, "QUAD\n");
  
  /* Allocate space for lists of points: */
  list1 = (double*)malloc(3*n*sizeof(double));
  list2 = (double*)malloc(3*n*sizeof(double));
  
  hpoint_to_cvector(Z1,&(sp->p1));
  hpoint_to_cvector(Z2,&(sp->p2));
/*  
  * Normalize Z1 and Z2 to have 3rd coord 1 *
  for (j=0; j<3; j++){  
    C_PUSH_S(Z1[j]); C_PUSH_S(Z1[2]); c_div_s(); C_POP_S(Z1[j]); 
    C_PUSH_S(Z2[j]); C_PUSH_S(Z2[2]); c_div_s(); C_POP_S(Z2[j]); 
  } 

  * compute t0 *
  compute_spinal_t0(&t0, Z1,Z2);
*/
  
  /* Normalize Z1 and Z2 so that <Z1,Z2> = 2 */
  Herm(&Z2Z1Herm, Z2, Z1);
  for (j=0; j<3; ++j) {
    /* Z2[j] = 2 Z2[j] / Z2Z1Herm */
    C_PUSH_S(Z2[j]); c_sca_mul_s(2.0); 
    C_PUSH_S(Z2Z1Herm); c_div_s(); C_POP_S(Z2[j]);
  } 

  /* First vertex: latitude = 0.0 */
  new_compute_spinal_slice(&slice,0.0,Z1,Z2, &success);
  if (success) chain_list(&slice, n, list1);
  
  /* Intermediate slices: */
  stripno = 0;
  latinc = M_PI / (2 * (slice_count + 1));
  for(sliceno=1,latitude=latinc;
      sliceno<slice_count;
      ++sliceno,latitude+=latinc) {
    new_compute_spinal_slice(&slice,latitude,Z1,Z2, &success);
    ++stripno;
    if (success) {
      /* Build next list2 */
      chain_list(&slice, n, list2);
      
      /* Generate quads for list1 and list2 */
      if ((strip_count < 0) || in_array(stripno, strip_list, strip_count))
	write_quad_strip(list1,list2,n,fp);
      
      /* Interchange list1 and list2 */
      templist = list2;
      list2 = list1;
      list1 = templist;
    }
  }

  /* Final vertex: latitude = Pi/2 */
  new_compute_spinal_slice(&slice,M_PI/2,Z1,Z2, &success);
  ++stripno;
  if (success) {
    chain_list(&slice, n, list2);
    if ((strip_count < 0) || in_array(stripno, strip_list, strip_count))
      write_quad_strip(list1,list2,n,fp);
  }

  free((char*)list1);
  free((char*)list2);
}

/*-----------------------------------------------------------------------
 * Function:	in_array
 * Description:	tests whether a number is in an array
 * Args  IN:	n: the number to test for
 *		array: the array
 *		len: length of array
 * Returns:	0 or 1
 */
in_array(n, array, len)
     int n, *array, len;
{
  while (len > 0)
    if (n == array[--len]) return(1);
  return(0);
}

/*-----------------------------------------------------------------------
 * Function:	write_triangle_strip
 * Description:	write a strip of triangles to a file
 * Args  IN:	list1,list2: lists of vertices along the edges of
 *		  the strip
 *		n: length of lists list1 and list2
 *		fp: file to write to
 * Returns:	nothing
 * Notes:	Assumes fp is already open for writing.
 *		Writes one triangle per line, in the format
 *		  x1 y1 z1  x2 y2 z2  x3 y3 z3
 * 
 *         0   1   2   3   4   5   6   7   8
 * list1   *---*---*---*---*---*---*---*---*
 *         |\  |\  |\  |\  |\  |\  |\  |\  |
 *         | \ | \ | \ | \ | \ | \ | \ | \ | (n = 9 in this example)
 *         |  \|  \|  \|  \|  \|  \|  \|  \|
 * list2   *---*---*---*---*---*---*---*---*
 * 
 * for i = 1 thru n-1 {
 *   {list1[i-1] list1[i] list2[i]}
 *   {list2[i-1] list1[i-1] list2[i]}
 * }
 */
write_triangle_strip(list1,list2,n,fp)
     double list1[][3], list2[][3];
     int n;
     FILE *fp;
{
  int i;

  for (i=1; i<n; ++i) {
    write_point(list1[i-1], fp); fprintf(fp, " ");
    write_point(list1[i], fp); fprintf(fp, " ");
    write_point(list2[i], fp); fprintf(fp, "\n");

    write_point(list2[i-1], fp); fprintf(fp, " ");
    write_point(list1[i-1], fp); fprintf(fp, " ");
    write_point(list2[i], fp); fprintf(fp, "\n");
  }
}

/*-----------------------------------------------------------------------
 * Function:	write_quad_strip
 * Description:	write a strip of quadrilaterals to a file
 * Args  IN:	list1,list2: lists of vertices along the edges of
 *		  the strip
 *		n: length of lists list1 and list2
 *		fp: file to write to
 * Returns:	nothing
 * Notes:	Assumes fp is already open for writing.
 *		Writes one quadrilateral per 2 lines, in the format
 *		  x1 y1 z1  x2 y2 z2
 *			x3 y3 z3  x4 y4 z4
 *		Writes a blank line after the strip.
 * 
 *         0   1   2   3   4   5   6   7   8
 * list1   *---*---*---*---*---*---*---*---*
 *         |   |   |   |   |   |   |   |   |
 *         |   |   |   |   |   |   |   |   | (n = 9 in this example)
 *         |   |   |   |   |   |   |   |   |
 * list2   *---*---*---*---*---*---*---*---*
 * 
 * for i = 0 thru n-2 {
 *   list1[i] list1[i+1] list2[i+1] list2[i]
 * }
 */
write_quad_strip(list1,list2,n,fp)
     double *list1, *list2;
     int n;
     FILE *fp;
{
  int i, nm1=n-1;

  for (i=0; i<nm1; ++i) {
    write_point(list1+3*i,     fp); fprintf(fp, "  ");
    write_point(list1+3*(i+1), fp); fprintf(fp, "\n\t");
    write_point(list2+3*(i+1), fp); fprintf(fp, "  ");
    write_point(list2+3*i,     fp); fprintf(fp, "\n");
  }
  fprintf(fp, "\n");
}

/*-----------------------------------------------------------------------
 * Function:	write_point
 * Description:	write a point to a file
 * Args  IN:	x: the point
 *		fp: file to write to
 * Returns:	nothing
 * Notes:	assumes fp is already open for writing
 */
write_point(x, fp)
     double *x;
     FILE *fp;
{
  fprintf(fp, "%f %f %f", *x, *(x+1), *(x+2));
}

write_list(fp, list, n)
     FILE *fp;
     double *list;
     int n;
{
  int i;

  for (i=0; i<n; ++i)
    fprintf(fp, "%f %f %f\n", *(list+3*i), *(list+3*i+1), *(list+3*i+2));
  fprintf(fp, "\n");
}

/*-----------------------------------------------------------------------
 * Function:	new_compute_spinal_slice
 * Description:	compute a slice of a spinal using the 'new' method
 * Args  IN:	t: 'latiude' of slice to compute (0 <= t <= M_PI/2)
 *		Z1,Z2: vertices of spinal, normalized so that <Z1,Z2>=2
 *      OUT:	*slice: the computed slice
 * Returns:	nothing
 * Notes:	The 'new' method is the linear combination
 *				Z1/tan(t) + tan(t) Z2
 */
new_compute_spinal_slice(slice, t, Z1, Z2, success)
     Chain *slice;
     double t;
     Cvector Z1, Z2;
     int success;
{
  Cvector Z;
  int j;
  double tt;

  if (t==0.0)
    cvector_to_chain(slice, Z1, success);
  else if (t==M_PI/2)
    cvector_to_chain(slice, Z2, success);
  else {
    tt = tan(t);
    for (j=0; j<3; ++j) {
      C_PUSH_S(Z1[j]); c_sca_mul_s(1.0/tt);
      C_PUSH_S(Z2[j]); c_sca_mul_s(tt);
      c_add_s(); C_POP_S(Z[j]);
    }
    cvector_to_chain(slice, Z, success);
  }
}

/*-----------------------------------------------------------------------
 * Function:	write_rcircle_list
 * Description:	Write a list of points on a rcircle to a file
 * Args  IN:	fp: file to write to
 *		*rc: the rcircle
 *		n: number of points on rcircle to use
 * Returns:	nothing
 */
write_rcircle_list(fp, rc)
     FILE *fp;
     Rcircle *rc;
{
  double *list, *listptr;
  int i,n;

  n = rcircle_resolution;
  list = (double*)malloc(3*n*sizeof(double));
  rcircle_list(rc, list);
  for (i=0, listptr=list; i<n; ++i,listptr+=3)
    fprintf(fp, "%f %f %f\n", *listptr, *(listptr+1), *(listptr+2));
  free(list);
}


/*-----------------------------------------------------------------------
 * Function:	rcircle_list
 * Description: create a list of points on a rcircle
 * Args IN:	*c: the rcircle to draw
 *		n: number of points to put in list
 *	OUT:	list: the returned list
 * Returns:	nothing
 * Notes:	I adapted this procedure from draw_rcircle on 11/10/89 - mbp.
 *
 *		This only works for finite Rcircles!
 */
rcircle_list(rc,list)
     Rcircle *rc;
     double list[][3];
{
  double th, thinc;
  Hpoint p1,p2,p3;
  int half_rcircle_resolution,n,sign;
  Complex sq_c_rad, temp;
  double P0[3], D[3], rr;
  int i;
  
  i = 0;
  C_SQRT(sq_c_rad,rc->c_rad);
  /* Force rcircle_resolution to be even: */
  half_rcircle_resolution = (rcircle_resolution + 1) / 2;
  thinc = M_PI / (2 * half_rcircle_resolution);

  sign = -1;
  {

    th = M_PI/4;
    compute_standard_rcircle( &p1, th, sign );
    hpoint_dilate( &p2, &p1, &sq_c_rad);
    hpoint_translate( &p3, &p2, &(rc->cen));
    list[i][0] = p3.hor.re;
    list[i][1] = p3.hor.im;
    list[i][2] = p3.ver;
    ++i;
    for (n=0, th = M_PI/4 + thinc;
	 n < half_rcircle_resolution;
	 ++n, th += thinc) {
      compute_standard_rcircle( &p1, th, sign );
      hpoint_dilate( &p2, &p1, &sq_c_rad);
      hpoint_translate( &p3, &p2, &(rc->cen));
      list[i][0] = p3.hor.re;
      list[i][1] = p3.hor.im;
      list[i][2] = p3.ver;
      ++i;
    }
  }

  sign = 1;
  {
    for (n=0, th = 3*M_PI/4 - thinc;
	 n < half_rcircle_resolution;
	 ++n, th -= thinc) {
      compute_standard_rcircle( &p1, th, sign );
      hpoint_dilate( &p2, &p1, &sq_c_rad);
      hpoint_translate( &p3, &p2, &(rc->cen));
      list[i][0] = p3.hor.re;
      list[i][1] = p3.hor.im;
      list[i][2] = p3.ver;
      ++i;
    }
  }


}

