(* $Id: Heisenberg.m,v 2.0 89/10/03 21:38:35 mbp Exp $
 *
 * Heisenberg.m: Interface to external Heisenberg graphics program
 *)

(***************************************************************************
 *                          Copyright (C) 1990 by                          *
 *        Mark B. Phillips, William M. Goldman, and Robert R. Miner        *
 *                                                                         *
 *  Permission to use, copy, modify, and distribute this software, its     *
 *  documentation, and any images it generates for any purpose and without *
 *  fee is hereby granted, provided that                                   *
 *                                                                         *
 *  (1) the above copyright notice appear in all copies and that both that *
 *      copyright notice and this permission notice appear in supporting   *
 *      documentation, and that the names of Mark B.  Phillips, William M. *
 *      Goldman, Robert R.  Miner, or the University of Maryland not be    *
 *      used in advertising or publicity pertaining to distribution of the *
 *      software without specific, written prior permission.               *
 *                                                                         *
 *  (2) Explicit written credit be given to the authors Mark B. Phillips,  *
 *      William M. Goldman, and Robert R. Miner in any publication which   *
 *      uses part or all of any image produced by this software.           *
 *                                                                         *
 * This software is provided "as is" without express or implied warranty.  *
 ***************************************************************************)

BeginPackage["Heisenberg`", "CH2`", "ExternalGraphics`", "GraphicsNames`"]

Grid::usage = "Grid is a Heisenberg graphics primitive consisting
of the X-Y grid and V-axis."

Color::usage = "Color is an option to gDraw which specifies the color
in which to draw.  It should be one of Black, White, Red, Green,
Yellow, Blue, Magenta, or Cyan.  NOTE: Black always stands for the
background color (even if the background is actually white), and White
stands for the opposite of the background color.  So White is always
visible, and Black is invisible."

Black::usage = "Black is a value for the Color option to gDraw."
White::usage = "White is a value for the Color option to gDraw."
Red::usage = "Red is a value for the Color option to gDraw."
Green::usage = "Green is a value for the Color option to gDraw."
Yellow::usage = "Yellow is a value for the Color option to gDraw."
Blue::usage = "Blue is a value for the Color option to gDraw."
Magenta::usage = "Magenta is a value for the Color option to gDraw."
Cyan::usage = "Cyan is a value for the Color option to gDraw."

SpinalDrawType::usage = "SpinalDrawType is an option to gDraw which
specifies how spinal hyperspheres are to be drawn.  It should be one
of Chains, Rcircles, or Both."

Chains::usage = "Chains is a value for the SpinalDrawType option to
gDraw."

Rcircles::usage = "Rcircles is a value for the SpinalDrawType option
to gDraw."

Both::usage = "Both is a value for the SpinalDrawType option to
gDraw."

ChainResolution::usage = "ChainResolution is an option to gDraw which
determines how Chains are drawn.  A chain is drawn with
ChainResolution line segments."

VerticalChainLimits::usage = "VerticalChainLimits is an option to
gDraw; It is a list of two values which give the min and max vertical
coordinates used in drawing vertical (infinite) chains."

SliceCount::usage = "SliceCount is an option to gDraw which tells the
number of slices (not including the vertices) to use when drawing a
Spinal by Chains."

MeridianCount::usage = "MeridianCount is an option to gDraw which
tells the number of meridians (Rcircles) to use when drawing a Spinal by
Rcircles."

RcircleResolution::usage = "RcircleResolution is an option to gDraw
which determines how Rcircles are drawn.  Rcircles are always drawn
with an even number of line segments: RcircleResolution line segments
are drawn if this value is even, and RcircleResolution + 1 line
segments are drawn if it is odd."

(* NOTE: Fudge should probably be taken out.  All fudging should
 * probably be done in Mathematica rather than in the graphics
 * program.  But since the current graphics code still makes use of
 * fudge, we leave this here for now: *)

Fudge::usage = "Fudge is an option to gDraw; it is used in the
external Heisenberg graphics program to determine if two floating
point values are equal. Values which differ by less than Fudge are
considered equal."

Begin["`Private`"]

(* ErrorFlag is the string returned by the graphics program to
   signal an error *)
ErrorFlag = "-1"

gSetErrorFlag[ErrorFlag]

ColorCommand[Black]	:= "col 0"
ColorCommand[White]	:= "col 1"
ColorCommand[Red]	:= "col 2"
ColorCommand[Green]	:= "col 3"
ColorCommand[Yellow]	:= "col 4"
ColorCommand[Blue]	:= "col 5"
ColorCommand[Magenta]	:= "col 6"
ColorCommand[Cyan]	:= "col 7"

SpinalDrawCommand[Chains]	:= "sdr 0"
SpinalDrawCommand[Rcircles]	:= "sdr 1"
SpinalDrawCommand[Both]		:= "sdr 2"

gDeclareDrawOptions[
  {{Color->White,			ColorCommand},
   {SpinalDrawType->Chains,		SpinalDrawCommand},
   {ChainResolution->36, 		StringJoin["cres ", ToString[#]]&},
   {VerticalChainLimits->{-10,10},	StringJoin["vcl ",
					  ToString[#[[1]]], " ",
					  ToString[#[[2]]]]&},
   {SliceCount->30,			StringJoin["ssc ", ToString[#]]&},
   {MeridianCount->9,			StringJoin["smc ", ToString[#]]&},
   {RcircleResolution->36,		StringJoin["rres ", ToString[#]]&},
   {Fudge->0.000001,			StringJoin["fud ", gN[#]]&}}
  ]

HpointToString[Hpoint[z_,v_]] :=
  StringJoin[gN[Re[z]], " ", gN[Im[z]], " ", gN[v]]
HpointToString[Hpoint[Infinity]] := "0 0 99999"
  
gDeclarePrimitive[Grid, "g"]

gDeclarePrimitive[p_Hpoint,  StringJoin["p ", HpointToString[p]]]

gDeclarePrimitive[Chain[c_Hpoint, r_?NumberQ],
  StringJoin["c ", HpointToString[c], " ", gN[r]]]

gDeclarePrimitive[Chain[c_Hpoint, Vertical],
  StringJoin["c ", HpointToString[c], " -1"]]

VerticalHpointQ[x_Hpoint] := TrueQ[Chop[Abs[N[x[[1]]]]]==0]

gDeclarePrimitive[Rcircle[x_?VerticalHpointQ, DirectedInfinity[z_]],
  StringJoin["r ", HpointToString[x], " ", gN[Re[z]], " ", gN[Im[z]], " 0"]]

gDeclarePrimitive[Rcircle[x_Hpoint, DirectedInfinity[z_:0]],
    StringJoin["r ", HpointToString[x],  " 0 0 0"]]

gDeclarePrimitive[Rcircle[x_Hpoint, z_],
  StringJoin["r ", HpointToString[x], " ", gN[Re[z]], " ", gN[Im[z]], " 1"]]

gDeclarePrimitive[s_Spinal,
  StringJoin["s ", HpointToString[s[[1]]], " ", HpointToString[s[[2]]]]]

gSetEraseCommand["e"]

gSetHideCommand["h"]

gSetUnhideCommand["d"]

gSetClearCommand["k"]

End[]

EndPackage[]

Print[""]
Print["Heisenberg; October 3, 1989"]
Print["Copyright 1989 Mark B. Phillips and William M. Goldman"]
Print["University of Maryland Department of Mathematics"]
Print[""]
(*Print["Type 'gConnect[]' to start graphics program."]*)

{gConnect, gDraw, gHide, gUnhide, gErase, gClear, gDisconnect}
