/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

#include "include.h"
#include "f2c.h"

struct verlist { 
                 vertex_id v_id; /* which vertex this is */
                 int freedom;    /* degrees of freedom */
                 int nbrs;       /* degrees of freedom of self and nbrs */
                 int alloc;      /* columns used so far */
                 int rownum;     /* starting row in sparse matrix  */
                 struct entry *row;
                 REAL **proj;    /* projection matrix to constraints */
                 REAL normal[MAXCOORD]; /* normal to surface */
               };

/* for constructing sparse matrix */
int total_rows;
struct entry { int col;   /* which column */
               int row;   /* which row */
               REAL value;
             };
struct entry *entry_start;
        
struct index { int count;   /* number of entries in row */
               struct entry *entry_head;   /* row of entries */
             };

int bodyrowstart;  /* first row of body volume constraints */

#ifndef NOPROTO
void newton_ysmp(struct index *,REAL *,int,int,int,struct index *);
#endif

/**************************************************************
*
*  Function: global_adjust()
*
*  Purpose:  Global minimization of energy by finding
*            critical point of quadratic approximation
*            to total energy function.
*            Using only lower left triangle of symmetric matrix.
*            Uses YSMP sparse matrix solver, which unfortunately
*            adds stuff to diagonal of non positive definite matrix.
*            Could correct a few such additions, but there seem to
*            be a lot of them for volume constraints.
*
*/

void global_adjust(normal_flag)
int normal_flag;  /* whether to do motion only along surface normal */
{
  
  edge_id e_id;
  vertex_id v_id;
  int i,j,k;
  struct verlist *vhead = NULL;  /* main vertex list */
  struct verlist *v;           /* current  vertex */
  struct verlist *vc;           /* current  vertex */
  int n;
  int vcount;        /* total number of vertices */
  REAL **volrows = NULL;      /* volume constraint rows of matrix */
  REAL *rhs = NULL;             /* right side of augmented matrix  */
  facet_id f_id;
  int currow = 0;
  struct index *array;
  body_id b_id,bb_id;
  int total_entries;
  double energy0 = web.total_energy;
  REAL **self2,**otherD,**o1,**o2,**p1,**p2;
  REAL **grad;
  int A_total; /* entries for A matrix */
  int A_rows;  /* rows for A matrix */

  if ( web.modeltype != LINEAR )
    error("Cannot do Hessian method on quadratic model.",RECOVERABLE);
  if ( (web.dimension != 2) || /* web.concount ||*/ web.surfen_count
      || web.quantity_count || web.diffusion_flag || web.gravflag
      || web.pressflag || web.symmetry_flag || web.pressure_flag
      || web.wulff_flag || web.bodycount || web.metric_flag )
      error("Surface has features which prevent Hessian method.\n",
        RECOVERABLE);

  save_coords();  /* in case this doesn't work */
  total_entries = 0;
  vcount = web.skel[VERTEX].max_ord;
  vhead = (struct verlist *)temp_calloc(vcount,sizeof(struct verlist));

  /* populate vertex list */
  grad = dmatrix(0,MAXCOORD-1,0,MAXCOORD-1);
  FOR_ALL_VERTICES(v_id)
    { MAP conmap;
      int oncount;
      struct constraint *constr;

      if ( get_vattr(v_id) & FIXED ) continue;
      v = vhead + ordinal(v_id);
      v->v_id = v_id;
      v->rownum  = currow;
      if ( (conmap = get_v_constraint_state(v_id)) != 0 )
        { /* calculate projection matrix to fewer degrees of freedom */
          for ( j = 0, oncount = 0; conmap ; j++,conmap >>= 1 )
          { REAL dummy;
            if ( !(conmap & 1) ) continue;
            constr = get_constraint(j);
            eval_all(constr->formula,get_coord(v_id),web.sdim,
              &dummy, grad[oncount]);  /* constraint value and derivs */
            oncount++;
          }
          v->freedom = web.sdim - oncount;
          v->proj = dmatrix(0,web.sdim-1,0,v->freedom-1);
          kernel_basis(grad,v->proj,oncount,web.sdim);
        }
      else if ( normal_flag )
       {
         v->freedom = 1;
         v->proj = dmatrix(0,web.sdim-1,0,v->freedom-1);
         calc_vertex_normal(get_vertex_fe(v_id),v->normal);
         for ( j = 0 ; j < web.sdim ; j++ )
           v->proj[j][0] = v->normal[j];
       }
      else
        v->freedom = web.sdim;
      v->nbrs += v->freedom;
      currow += v->freedom;
    }

  /* allocate matrix */
  A_rows = bodyrowstart = currow;
  total_rows = currow + web.bodycount;
  array = (struct index *)temp_calloc(total_rows,sizeof(struct index));

  /* count neighbors, so can define starts of neighbor lists */
  FOR_ALL_EDGES(e_id)
   {
     vertex_id headv,tailv;

     headv = get_edge_headv(e_id);
     tailv = get_edge_tailv(e_id);

     if ( ordinal(headv) > ordinal(tailv) )
       vhead[ordinal(headv)].nbrs += vhead[ordinal(tailv)].freedom;
     else
       vhead[ordinal(tailv)].nbrs += vhead[ordinal(headv)].freedom;
     
   }

  /* total entries for vertices */
  for ( n = 0 ; n < vcount ; n++ )
     total_entries += vhead[n].freedom*vhead[n].nbrs;
  A_total = total_entries;

  /* count vertices around each body, using Euler's formula */
  if ( web.bodycount )
   FOR_ALL_FACETS(f_id)
    {
      b_id = get_facet_body(f_id);

      if ( valid_id(b_id) ) array[bodyrowstart + ordinal(b_id)].count++;

      b_id = get_facet_body(inverse_id(f_id));
      if ( valid_id(b_id) ) array[bodyrowstart + ordinal(b_id)].count++;
    }
  FOR_ALL_BODIES(b_id)
   { struct index *current = array + bodyrowstart + ordinal(b_id);
      
     current->count = 3 + current->count/2;  /* Euler */
     current->count *= web.sdim;
     total_entries += current->count;
     set_body_volume(b_id,0.0);
   }

  /* allocate matrix */
  entry_start = (struct entry *)temp_calloc(total_entries,sizeof(struct entry));
  for ( n = 0, total_entries = 0 ; n < vcount ; n++ )
   { for ( i = 0 ; i < vhead[n].freedom ; i++ )
      { struct index *curinx = array + vhead[n].rownum + i;
      
        curinx->entry_head = entry_start + total_entries;
        curinx->count = vhead[n].nbrs;
        total_entries += curinx->count;
        for ( j = 0 ; j < vhead[n].freedom ; j++ )
          /* allocate self-self entries */
          curinx->entry_head[j].col = vhead[n].rownum + j;
       }
      vhead[n].alloc = vhead[n].freedom;
    } 
  FOR_ALL_BODIES(b_id)
   { struct index *current = array + bodyrowstart + ordinal(b_id);
      
     current->entry_head = entry_start + total_entries;
     total_entries += current->count;
     current->count = 0;  /* reset */
   }

  /* mark slots for vertices joined by edges */
  FOR_ALL_EDGES(e_id)
   {
     vertex_id headv,tailv;
     struct verlist *hv,*tv;

     headv = get_edge_headv(e_id);
     tailv = get_edge_tailv(e_id);

     hv = vhead + ordinal(headv);
     tv = vhead + ordinal(tailv);

     if ( hv > tv )
       { for ( i = 0 ; i < hv->freedom ; i++ )
          for ( j = 0 ; j < tv->freedom ; j++ )
           array[hv->rownum+i].entry_head[hv->alloc+j].col = tv->rownum + j;
         hv->alloc += tv->freedom;
       }
     else
       { for ( i = 0 ; i < hv->freedom ; i++ )
          for ( j = 0 ; j < tv->freedom ; j++ )
          array[tv->rownum+j].entry_head[tv->alloc+i].col = hv->rownum + i;
         tv->alloc += hv->freedom;
       }

   }

  /* mark slots for body volume constraints */
  FOR_ALL_FACETS(f_id)
    {
      b_id = get_facet_body(f_id);
      bb_id = get_facet_body(inverse_id(f_id));

      if ( valid_id(b_id) )
       { facetedge_id fe = get_facet_fe(f_id);
         for ( j = 0 ; j < FACET_VERTS ; j++, fe = get_next_edge(fe) )
           { vertex_id v_id = get_fe_tailv(fe);
             struct entry *head;
             int count;
             int currentrow = bodyrowstart + ordinal(b_id);
             struct verlist *v = vhead + ordinal(v_id);
             int didflag = 0;
 
             head = array[currentrow].entry_head;           
             count = array[currentrow].count;           
             for ( k = 0 ; k < count ; k++ )
               if ( head[k].col == v->rownum )
                 { didflag = 1;  break; }
             if ( didflag ) continue;

             /* have to allocate */
             for ( i = 0 ; i < v->freedom ; i++ )
               head[array[currentrow].count++].col = v->rownum + i;
           }
        }

      if ( valid_id(bb_id) )
       { facetedge_id fe = get_facet_fe(f_id);
         for ( j = 0 ; j < FACET_VERTS ; j++, fe = get_next_edge(fe) )
           { vertex_id v_id = get_fe_tailv(fe);
             struct entry *head;
             int count;
             int currentrow = bodyrowstart + ordinal(bb_id);
             struct verlist *v = vhead + ordinal(v_id);
             int didflag = 0;
 
             head = array[currentrow].entry_head;           
             count = array[currentrow].count;           
             for ( k = 0 ; k < count ; k++ )
               if ( head[k].col == v->rownum )
                 { didflag = 1;  break; }
             if ( didflag ) continue;

             /* have to allocate */
             for ( i = 0 ; i < v->freedom ; i++ )
               head[array[currentrow].count++].col = v->rownum + i;
           }
        }
    }

  /* allocate right hand side of augmented matrix */
  rhs = (REAL *)temp_calloc(total_rows,sizeof(REAL));

  /* fill in sparse matrix rows and volume constraint rows */
  p1 = dmatrix(0,MAXCOORD-1,0,MAXCOORD-1);
  p2 = dmatrix(0,MAXCOORD-1,0,MAXCOORD-1);
  o1 = dmatrix(0,MAXCOORD-1,0,MAXCOORD-1);
  o2 = dmatrix(0,MAXCOORD-1,0,MAXCOORD-1);
  self2 = dmatrix(0,MAXCOORD-1,0,MAXCOORD-1);
  otherD = dmatrix(0,MAXCOORD-1,0,MAXCOORD-1);
  FOR_ALL_FACETS(f_id)
    {
      REAL side[FACET_EDGES][MAXCOORD];   /* 3 sides of facet */
      REAL ss[FACET_EDGES];     /* squares of sides */
      REAL sd[FACET_EDGES];   /* dot of side with next side */
      facetedge_id fe_id;       
      struct verlist *v[FACET_VERTS];
      REAL two_area;   /* twice area of facet */
      REAL first[FACET_VERTS][MAXCOORD]; /* first partials */
      REAL *coord[FACET_VERTS];  /* for volume calculation */

      b_id = get_facet_body(f_id);
      bb_id = get_facet_body(inverse_id(f_id));

      fe_id = get_facet_fe(f_id);
      for ( i = 0 ; i < FACET_EDGES ; i++ )
        { v[i] = vhead + ordinal(get_fe_tailv(fe_id));
          coord[i] = get_coord(get_fe_tailv(fe_id));
          get_fe_side(fe_id,side[i]);
          fe_id = get_next_edge(fe_id);
        }

      /* body volume constraints */
      if ( valid_id(b_id) || valid_id(bb_id) )
      { if ( valid_id(b_id) )
          set_body_volume(b_id,get_body_volume(b_id) +
                         triple_prod(coord[0],coord[1],coord[2])/6);
        if ( valid_id(bb_id) )
          set_body_volume(bb_id,get_body_volume(bb_id) -
                         triple_prod(coord[0],coord[1],coord[2])/6);
        for ( i = 0 ; i < FACET_VERTS ; i++ )
        {
          struct entry *head;
          int count;
          int currentrow;
          REAL g[MAXCOORD],gg[MAXCOORD],*ggg;

          cross_prod(coord[(i+1)%3],coord[(i+2)%3],g);
          if ( v[i]->proj )
            { vec_mat_mul(g,v[i]->proj,gg,web.sdim,v[i]->freedom);
              ggg = gg;
            }
          else ggg = g;
          if ( valid_id(b_id) )
           {
             currentrow = bodyrowstart + ordinal(b_id);
             head = array[currentrow].entry_head;           
             count = array[currentrow].count;           
             for ( k = 0 ; k < count ; k++, head++ )
               if ( head->col == v[i]->rownum ) break;
             if ( k == count ) 
               error("Cannot find body in global_adjust. \n",RECOVERABLE);
             for ( j = 0 ; j < v[i]->freedom ; j++ )
               (head++)->value += ggg[j]/6;
           }
          if ( valid_id(bb_id) )
           {
             currentrow = bodyrowstart + ordinal(bb_id);
             head = array[currentrow].entry_head;           
             count = array[currentrow].count;           
             for ( k = 0 ; k < count ; k++, head++ )
               if ( head->col == v[i]->rownum ) break;
             if ( k == count ) 
               error("Cannot find body in global_adjust. \n",RECOVERABLE);
             for ( j = 0 ; j < v[i]->freedom ; j++ )
               (head++)->value -= ggg[j]/6;
           }
        }
      }

      for ( i = 0 ; i < FACET_EDGES ; i++ )
        {
          ss[i] = dot(side[i],side[i],web.sdim);
          sd[i] = -dot(side[(i+1)%FACET_EDGES],side[(i+2)%FACET_EDGES],web.sdim);
        }
      two_area = sqrt(ss[1]*ss[2] - sd[0]*sd[0]);

      /* first derivatives of area */
      for ( i = 0 ; i < FACET_EDGES ; i++ )
        {
          int ii,jj;
          REAL g[MAXCOORD];

          if ( v[i]->freedom == 0 ) continue;
          ii = (i+2)%FACET_EDGES;  /* side previous vertex i */
          jj = (i+1)%FACET_EDGES;  /* side opposite vertex i, next vertex */
          for ( k = 0 ; k < web.sdim ; k++ )
            { first[i][k] = (side[ii][k]*ss[jj] - sd[i]*(-side[jj][k]))
                                                                /2/two_area;
            }
          if ( v[i]->proj )
            { vec_mat_mul(first[i],v[i]->proj,g,web.sdim,
                               v[i]->freedom);
              for ( k = 0 ; k < v[i]->freedom ; k++ )
                rhs[v[i]->rownum+k] -= g[k];
            }
          else
            for ( k = 0 ; k < v[i]->freedom ; k++ )
                rhs[v[i]->rownum+k] -= first[i][k];
        }

      /* second derivatives */
      for ( i = 0 ; i < FACET_EDGES ; i++ )
        {
          struct entry *second[MAXCOORD],*other1[MAXCOORD],*other2[MAXCOORD];
          REAL self,other;
          REAL **pp,**oo;
          int ii,jj;

          if ( v[i]->freedom == 0 ) continue;

          ii = (i+2)%FACET_EDGES;  /* side previous vertex i */
          jj = (i+1)%FACET_EDGES;  /* side opposite vertex i, next vertex */

          /* find sparse matrix entries */
          second[0] = other1[0] = other2[0] = NULL;
          for ( n = 0 ; n < v[i]->alloc ; n++ ) 
            { if ( array[v[i]->rownum].entry_head[n].col == v[i]->rownum )
                 for ( k = 0 ; k < v[i]->freedom ; k++ )
                   second[k] = array[v[i]->rownum+k].entry_head + n;
            }
          if ( second[0] == NULL )
            error("Cannot find self second partial.",RECOVERABLE);

          if ( v[jj]->freedom ) 
            if ( v[i]->rownum > v[jj]->rownum )
              {
                for ( n = 0 ; n < v[i]->alloc ; n++ ) 
                 if ( array[v[i]->rownum].entry_head[n].col == v[jj]->rownum )
                  for ( k = 0 ; k < v[i]->freedom ; k++ )
                    other1[k] = array[v[i]->rownum+k].entry_head + n;
                if ( (other1[0] == NULL) &&  v[jj]->freedom )
                 error("Cannot find other1 second partial.",RECOVERABLE);
               }
             else
               {  
                 for ( n = 0 ; n < v[jj]->alloc ; n++ )
                  if ( array[v[jj]->rownum].entry_head[n].col == v[i]->rownum )
                   for ( k = 0 ; k < v[jj]->freedom ; k++ )
                     other2[k] = array[v[jj]->rownum+k].entry_head + n;
                 if ( (other2[0] == NULL) &&  v[jj]->freedom )
                  error("Cannot find other2 second partial.",RECOVERABLE);
               }

          /* for now, all degrees of freedom are coordinate */
          for ( j = 0 ; j < web.sdim ; j++ )
            for ( k = 0 ; k < web.sdim ; k++ )
              { self = -(side[jj][j]*side[jj][k])/2;
                if ( j == k ) self += ss[jj]/2;
                self2[j][k] = 
                           (self - 2*first[i][j]*first[i][k])/two_area;

                if ( v[jj]->freedom == 0 ) continue;
                other = -side[ii][j]*side[jj][k] 
                            - side[jj][j]*(-side[ii][k])/2;
                if ( j == k ) other -= sd[i]/2;
                otherD[j][k] = (other - 2*first[i][j]*first[jj][k])/two_area;
              }
          if ( v[i]->proj )
            { tr_mat_mul(v[i]->proj,self2,p1,web.sdim,
                    v[i]->freedom,web.sdim);
              mat_mult(p1,v[i]->proj,p2,v[i]->freedom,
                    web.sdim,v[i]->freedom);
              pp = p2;
              tr_mat_mul(v[i]->proj,otherD,o1,web.sdim,
                    v[i]->freedom,web.sdim);
              oo = o1;
            }
          else { pp = self2; oo = otherD; }
          if ( v[jj]->proj )
            { mat_mult(oo,v[jj]->proj,o2,v[i]->freedom,
                    web.sdim,v[jj]->freedom);
              oo = o2;
            }
          for ( j = 0 ; j < v[i]->freedom ; j++ )
            for ( k = 0 ; k < v[i]->freedom ; k++ )
              second[j][k].value += pp[j][k];
          for ( j = 0 ; j < v[i]->freedom ; j++ )
            for ( k = 0 ; k < v[jj]->freedom ; k++ )
              if ( v[i]->rownum > v[jj]->rownum )
                 other1[j][k].value += oo[j][k];
              else
                 other2[k][j].value += oo[j][k];
         }
    }

  /* rhs for body constraint rows */
  FOR_ALL_BODIES(b_id)
   { REAL *current = rhs + bodyrowstart + ordinal(b_id);
     *current = (get_body_fixvol(b_id) - get_body_volume(b_id));
   }

  /* solve stuff */
  newton_ysmp(array,rhs,A_rows,A_total,web.bodycount,array+bodyrowstart);

  /* move vertices */
  for ( n = 0, vc = vhead ; n < vcount ; n++, vc++ )
    {
      REAL *coord;

      if ( vc->freedom == 0 ) continue;
      coord = get_coord(vc->v_id);
      for ( j = 0 ; j < web.sdim ; j++ )
        if ( vc->proj )
          coord[j] += dot(vc->proj[j],rhs+vc->rownum,vc->freedom);
        else
          coord[j] += rhs[vc->rownum + j];
    }

  calc_content();
  calc_pressure();
  calc_energy();  /* energy after motion */
  
  /* in case didn't work */
  if ( web.total_energy > energy0 )
    { double lambda; 
      double minlambda=1.0,minenergy=1e99;
      struct verlist *vc;

      error("Global optimization didn't reduce energy.\n",WARNING);
      fprintf(stderr,"    Blew up to %f\n",web.total_energy);
      /* check what happens along motion */
      for ( lambda = 1.0 ; lambda > .001 ; lambda /= 2 )
        {
           restore_coords();
           /* move vertices */
           for ( n = 0, vc = vhead ; n < vcount ; n++, vc++ )
             {
               REAL *coord;

               if ( vc->freedom == 0 ) continue;
               coord = get_coord(vc->v_id);
               for ( j = 0 ; j < web.sdim ; j++ )
                 if ( vc->proj )
                   coord[j] += lambda*dot(vc->proj[j],rhs+vc->rownum,
                                                            vc->freedom);
                 else
                   coord[j] += lambda*rhs[vc->rownum + j];
             }
           calc_content();
           calc_pressure();
           calc_energy();  /* energy after motion */
           printf("lambda %f   energy %20.15f \n",lambda,web.total_energy);
           if ( web.total_energy < minenergy ) 
             { minenergy = web.total_energy;
               minlambda = lambda;
             }
        }
      restore_coords();
      if ( minenergy < energy0 )
        {
           /* move vertices */
           for ( n = 0, vc = vhead ; n < vcount ; n++, vc++ )
             {
               REAL *coord;

               if ( vc->freedom == 0 ) continue;
               coord = get_coord(vc->v_id);
               for ( j = 0 ; j < web.sdim ; j++ )
                 if ( vc->proj )
                   coord[j] += minlambda*dot(vc->proj[j],rhs+vc->rownum,
                                                            vc->freedom);
                 else
                   coord[j] += minlambda*rhs[vc->rownum + j];
             }
           calc_content();
           calc_pressure();
           calc_energy();  /* energy after motion */
        }
    }
  for ( j = 0 ; j < vcount ; j++ )
    if ( vhead[j].proj ) free_matrix(vhead[j].proj);
  if ( oldcoord ) unsave_coords();
  if ( vhead ) temp_free((char *)vhead);
  if ( entry_start ) temp_free((char *)entry_start);
  if ( array ) temp_free((char *)array);
  if ( volrows ) free_matrix(volrows);
  if ( self2 ) free_matrix(self2);
  if ( otherD ) free_matrix(otherD);
  if ( o1 ) free_matrix(o1);
  if ( o2 ) free_matrix(o2);
  if ( p1 ) free_matrix(p1);
  if ( p2 ) free_matrix(p2);
  if ( grad ) free_matrix(grad);

  return;

}

/*********************************************************************
*
* function: newton_ysmp()
*
* purpose: uses Yale sparse matrix routine to solve Hessian
*
*/

/*  C declarations of the YSMP routines  */

#ifdef NOPROTO
int odrv_();
int sdrvmd_();
void sdrv_flag_check();
void odrv_flag_check();
#else
int odrv_( integer *, integer *,integer *,double *, integer *,integer *,
integer *,integer *, integer *, integer *);

int sdrvmd_( integer *, integer *,integer *, integer *,integer *,double *,
        double *,double *, integer *,integer *,double *,integer *,
        integer *, integer *, double *);
                                        
void sdrv_flag_check(integer , integer , integer );
void odrv_flag_check(integer , integer );
#endif
#ifdef FULLPROTO
void odrv_( int *N, int *IA,int *JA,double *A, int *P,int *IP,
int *NSP,int *ISP, int *PATH, int *FLAG );

void sdrvmd_( int *N, int *P,int *IP, int *IA,int *JA,double *A,
        double *B,double *Z, int *NSP,int *ISP,double *RSP,int *ESP,
        int *PATH, int *FLAG, double *EMAX );
                                        
void sdrv_flag_check(integer ESP, integer FLAG, integer N);
void odrv_flag_check(integer FLAG, integer N);
#endif

/* comparison function for sorting columns */
int qcomp(a,b)
struct entry *a,*b;
{ return a->col - b->col;
}

/* comparison function for sorting into transpose order */
int trancomp(a,b)
struct entry *a,*b;
{ int q = a->col - b->col;
  if ( q ) return q;
  return a->row - b->row;
}

void newton_ysmp(array,rhs,n,total_entries,concount,conarray)
struct index *array;       /* pointers to rows */
REAL *rhs;                 /* right hand side of system */
int n;                  /* size of system */
int total_entries;       /* matrix entries */
int concount;   /* number of global constraints */
struct index *conarray;  /* pointers to constraint gradients */
{
  integer *IA=0, *JA=0, *P=0, *IP=0, NSP, *ISP=0;
  doublereal *A=0;
  integer i,j, PATH, FLAG;
  integer total;
  doublereal *B;
  integer N = n;
  struct entry *entry;
  int currow;
  REAL *T1,*T2,*T3,**C,**CA,**CAC;
  double *RSP;
  integer ESP;
  double EMAX;

  /* allocate storage for arrays to be passed to ysmp */
  IA = (integer *)temp_calloc(N+1,sizeof(integer));
  JA = (integer *)temp_calloc(total_entries,sizeof(integer));
   A = (double *)temp_calloc(total_entries,sizeof(double));
   P = (integer *)temp_calloc(N,sizeof(integer));
  IP = (integer *)temp_calloc(N,sizeof(integer));
  NSP = 8*N + 16*total_entries;
  ISP = (integer *)temp_calloc(NSP,sizeof(integer));
/*  B  = (double *)temp_calloc(N,sizeof(double));  */

  /* put in row labels for sorting */
  for ( i = 0 ; i < n ; i++ )
   {
     entry = array[i].entry_head;
     for ( j = 0 ; j < array[i].count ; j++, entry++ )
       entry->row = i;
   }
  /* sort entries into transposed order */
  qsort((char*)entry_start,total_entries,sizeof(struct entry),trancomp);

  /* set up indexing and sparse matrix */
  currow = 0;
  IA[currow] = 0+1;
  total = 0;
  for ( i = 0, entry = entry_start ; i < total_entries ; i++, entry++ )
    { if ( entry->col > entry->row ) continue;
      if ( entry->value == 0.0 ) continue;
      while ( currow < entry->col ) IA[++currow] = total + 1;
      JA[total] = entry->row+1;
      A[total]  = entry->value;
      total++;
    }
  while ( currow < n ) IA[++currow] = total + 1;
     /* may have some empty rows at end due to constraints */
     /* and IA[N] also needs to be set */
  B = rhs;

#ifdef GDEBUG
/* some debug printing */
{ REAL x;
for ( i = 0 ; i < n ; i++ ) 
  { x = rhs[i]; printf("  %f  b[%d] \n",x,i); }
printf("IA: ");
for ( i = 0 ; i <= n ; i++ ) printf(" %d",IA[i]);
printf("\nJA: ");
for ( i = 0 ; i < total ; i++ ) printf(" %d",JA[i]);
printf("\n");
for ( i = 0 ; i < n ; i++ ) 
  { int j,k,m;
    for ( m = 0 ; m < i ; m++ ) printf("         ");
    for ( m = i,  j = 0, k = IA[i]-1 ; m < n /* j < IA[i+1]-IA[i] */; m++ )
      if ( (m == JA[k]-1) && (k < IA[i+1]-1) )
        { printf(" %f",A[k]); k++; j++; }
      else printf(" %f",0.0);
    printf("\n");
  }
}
#endif

  /*  Call ODRV to perform the reordering on A */
      PATH = 2;        
      odrv_( &N, IA,JA,A, P,IP, &NSP,ISP, &PATH, &FLAG );
      odrv_flag_check(FLAG,N);

  /*  Call SDRVMD to compute the solution */
     {  RSP = (double *) ISP;
        PATH = 1;
        sdrvmd_(&N,P,IP,IA,JA,A,B,B,&NSP,ISP,RSP,&ESP, &PATH,&FLAG,&EMAX);
        sdrv_flag_check(ESP,FLAG,N);
     }


  /* do constraints */
  if ( concount == 0 ) goto free_stuff;
  C = dmatrix(0,concount-1,0,n-1);
  CA = dmatrix(0,concount-1,0,n-1);
  CAC = dmatrix(0,concount-1,0,concount-1);
  T1 = vector(0,concount-1);
  T2 = vector(0,concount-1);
  T3 = vector(0,n-1);
  for ( i = 0 ; i < concount ; i++ )
    { /* fill in C  (really C transpose) and solve for inverses */
      entry = conarray[i].entry_head;
      for ( j = 0 ; j < conarray[i].count ; j++, entry++ )
          C[i][entry->col] = entry->value;
      PATH = 3;
      sdrvmd_(&N,P,IP,IA,JA,A,C[i],CA[i],&NSP,ISP,RSP,&ESP, &PATH,&FLAG,&EMAX);
      sdrv_flag_check(ESP,FLAG,N);
    }
  mat_mul_tr(CA,C,CAC,concount,n,concount);
  mat_inv(CAC,concount);
  matvec_mul(C,B,T1,concount,n);
  matvec_mul(CAC,T1,T2,concount,concount);
  vec_mat_mul(T2,CA,T3,concount,n);
  for ( i = 0 ; i < n ; i++ )
    B[i] -= T3[i];

for ( i = 0 ; i < n ; i++ ) 
printf(" %f x[%d]  \n",rhs[i],i);

  free_matrix(C);
  free_matrix(CA);
  free_matrix(CAC);
  free_vector(T1,0,concount-1);
  free_vector(T2,0,concount-1);
  free_vector(T3,0,N-1);

  /* free stuff */
free_stuff:
  temp_free((char *)IA);
  temp_free((char *)A);
  temp_free((char *)JA);
  temp_free((char *)B);
  temp_free((char *)ISP);
  temp_free((char *)IP);
  temp_free((char *)P);
}


void odrv_flag_check(FLAG, N)
integer FLAG,N;
{
if (!FLAG) return;

/*  Fatal error */
sprintf(errmsg,"\n*** after ODRV:  N = %d  FLAG = %d -- ",N,FLAG);
if ( 9*N<FLAG && FLAG<=10*N )
sprintf(errmsg+strlen(errmsg),"Insufficient storage in MD, K = %d\n",FLAG-9*N);
else if ( FLAG == 10*N + 1 )
strcat(errmsg,"Insufficient storage in ODRV\n");
else if ( FLAG == 11*N + 1 )
strcat(errmsg,"Illegal path specification\n");
else
strcat(errmsg,"Mysterious value of FLAG\n");
error(errmsg,RECOVERABLE);
}

void sdrv_flag_check(ESP, FLAG, N)
integer ESP,FLAG,N;
{ char *c;
if (ESP<=0)
  { sprintf(errmsg,"\n*** after SDRVMD:  Storage shortage:  ESP = %d\n",ESP);
    error(errmsg,RECOVERABLE); 
  }
if (FLAG==0) return;
if (FLAG<0)
  error("SDRVMD: Hessian not positive definite.\n",RECOVERABLE);

/*  Fatal error:  print message and die */
sprintf(errmsg,"\n*** after SDRVMD:  N = %d  FLAG = %d -- ",N,FLAG);
c = errmsg + strlen(errmsg);
if ( 2*N < FLAG && FLAG <= 3*N )
sprintf(c,"Duplicate entry in A at row %d\n",FLAG-2*N);
else if ( 6*N < FLAG && FLAG <= 7*N )
sprintf(c,"Insufficient storage in SSF at row %d\n",FLAG-6*N);
else if ( FLAG == 7*N + 1 )
sprintf(c,"Insufficient storage in SNF\n");
else if ( 8*N < FLAG && FLAG <= 9*N )
sprintf(c,"Zero pivot at row %d\n",FLAG-8*N);
else if ( FLAG == 10*N + 1 )
sprintf(c,"Insufficient storage in SDRV\n");
else if ( FLAG == 11*N + 1 )
sprintf(c,"Illegal path specification\n");
else printf(c,"Mysterious value of FLAG: %d\n",FLAG);
error(errmsg,RECOVERABLE);
}

