/* lib.d file lg.h */
/*
This is the header file giving the structure of a labelled graph and its
vertices, and listing al the functions which operate on them.
A  labelled graph consists of a finite alphabet and a finite
set of vertices and a finite set of labelled directed edges
(called  arrows).
An arrow has a  source and a  target.
For each vertex |v| and each element |g| of the alphabet, there is at most one
arrow with this vertex as source and this label.
Warning All our labelled graphs have a basepoint, and it is assumed that any
vertex in a labelled graph can be
found by proceeding from the basepoint, in the correct direction along
arrows.
There are various different kinds of labelled graphs; to accommodate the
differences we have various different kinds of vertices. All vertices have
a common basic structure, but each vertex also contains a "magic pocket" where
information specific to the vertices of that particular type is stored.
In this file the functions and definitions relating to the basic structure,
and thus essentially to all types of labelled graph, appear first, with the 
specific structures being dealt with at the end of the file.
*/
 
/*
Definitions common to all types of labelled graph.
The magic pocket function table associated with each vertex is used to hold
certain basic vertex functions, needed for all kinds of vertices, but whose
definition needs a knowledge of the internal structure of the magic
pocket.
*/
typedef struct mp_fntab {
	int (*mp_sgn)(); /*compare two vertices*/
	dp (*mp_create)(); /*allocate space for magic pocket*/
	void (*mp_kill)();	
		 /*deallocate space used by magic pocket*/
	void (*mp_print)();
	/* print out the contents of the magic pocket */
	int (*mp_gethpindex)(); /*get the heap index of a vertex*/
	void (*mp_sethpindex)();
		/*set the heap index of a vertex*/ 
}
mp_fntab;

typedef struct vertex {
	struct {
		unsigned visit :1;
		unsigned print :1; /* for printing*/
	} bits;
	mp_fntab * fntab;
	int index;
	int degree;
	gen back; /* the last edge in a path from the basepoint. 
		This is only ever set away from the basepoint in trees, but in other
	graphs it identifies the basepoint */
	int category; /* used only for fsa's */
	dp mp; /*magic pocket, a pointer to other information, specific to
			a particular type of vertex */
	struct vertex **adj; 
}
vertex;
typedef vertex * vindex ; /*the address of a vertex*/
/*We try to program so that a |vindex| could be some other indexing
system for vertices.  But I don't suppose this attempt succeeds*/

typedef struct {
	vindex basepoint;
	int num_verts;
	int degree;
	int type;
	boolean locked; /*TRUE if a breadth first search is in progress*/
	list discards; /* list of vertices which have been identified
		with other vertices, and therefore need to be thrown away*/
	mp_fntab * fntab;
}
lg; /*labelled graph*/

/*
Possible  graph types.
*/
#define BASIC_WTREE 0
#define KMP_WTREE 1 /*Knuth Morris Pratt failure pointers*/
#define MF_WTREE 2 /*Fast Merge Find*/
#define CON_WTREE 3 /*Considered and inverse considered linked list*/
#define BASIC_FSA 4
#define OPT_FSA 5


/* This function initializes the structures of a labelled graph.
The second argument is the (magic pocket) function tableindex, which 
determines the type of the labelled graph.
*/
extern void lg_init PARMS((lg * , mp_fntab * fntab, int type, int degree));
extern void lg_clear PARMS((lg *));

extern boolean lg_eq PARMS((lg *, lg *));
extern void lg_cpy PARMS((lg *, lg *));

/* Space is assigned for a vertex using |lg_vertex_create(&lg_name)|.
The way space is assigned will depend on |lg_name|. After the space is
set aside, it is initialized.
The procedure returns a pointer to the space allocated.
*/
extern vindex lg_vertex_create PARMS((lg * lg_namep));
extern void lg_vertex_kill PARMS ((lg *, vindex));

/* The basepoint is reset to a different (already existing) vindex.
*/
extern void lg_reset_basepoint PARMS ((lg *, vindex));

extern lg * lg_read PARMS((mp_fntab * fntab, int type,FILE * file));
extern void lg_print PARMS((FILE * wfile,lg * ));

extern void lg_set_indices PARMS ((lg * lgp));
extern void lg_erase_indices PARMS ((lg *));

#define LINEAR 0
#define CIRCULAR 1
extern void fsa_language_enumerate PARMS((FILE *,lg * fsa, int n, int order ,int
nsyms));
extern void fsa_opt PARMS((int ncats, lg * fsa));
extern lg * fsa_or PARMS((lg * fsa1, lg * fsa2));
extern lg  *fsa_and PARMS((lg * fsa1, lg * fsa2));
extern lg * fsa_not PARMS((lg * fsa ));
extern lg * fsa_exists PARMS((lg * fsa ,int coordinate)); 
	/*  coordinate is 1 or 2 */
extern lg * fsa_quantify_y PARMS((lg * fsa ,int i));
extern lg * fsa_exists_x2y1 PARMS((lg * fsa1, lg * fsa2 ));
extern lg * fsa_diag PARMS ((lg * fsa));
extern boolean fsa_is_diag PARMS ((lg * fsa));
extern lg * fsa_eosclosure PARMS ((lg * fsa));
extern lg * fsa_eosinsert PARMS ((lg * fsa));
extern lg * fsa_eosdelete PARMS ((lg * fsa));
extern lg * fsa_swapxy PARMS ((lg * fsa));
extern lg *fsa_preprocessor1 PARMS ((VOID));
extern lg *fsa_preprocessor2 PARMS ((VOID));
extern lg * fsa_false PARMS ((int degree));
extern lg * fsa_prod PARMS ((lg * fsa1, lg * fsa2));
extern lg * fsa_doubleprod PARMS ((lg * fsa1, lg * fsa2));
extern lg * fsa_amalgcomp PARMS ((lg * fsa1, lg * fsa2, lg * fsa3));
extern vindex wtree_word_trace PARMS((lg * lgp, word * wp));

#define get_num_verts(lgp) (lgp)->num_verts
#define set_num_verts(lgp,n) (lgp)->num_verts = n
#define get_degree(lgp) (lgp)->degree
#define set_degree(lgp,k) (lgp)->degree = k
#define basept(lgp) (lgp)->basepoint



/* Function table for list of addresses of labelled graphs, the largest
labelled graph coming first in the list.
*/
extern elt_fntab LGP_fntab;
#define LGP (&LGP_fntab)

/* Apply a word |*wp| to the vertex |*s|. If the word can be traced out
in the labelled graph, the function returns the vindex at the end of the 
path; otherwise it returns |UNDEFINED|.
*/
extern vindex lg_word_find  PARMS ((lg * lgp, word * wp));


/* |vindex2word| is essentially inverseto |lg_word_find|, but it only makes
sense in graphs with back pointers defined. It sets |*wp| equal to the word
which labels the path from the basepoint to |v| that is marked by the back
pointers.
*/
extern void vindex2word PARMS ((vindex v,word *wp));

/* Is the labelled graph trivial (a basepoint and no arrows)?
*/
extern boolean non_trivial PARMS (( lg *lgp));

/* Our standard method for traversing a labelled graph is by a breadth
first search. This involves being able to mark each vertex as it is
visited, and to maintain a list of vertices which have been visited,
but not yet dealt with.
The structure below enables a breadth first search to be carried out.
*/
typedef struct {
	lg * lgp;/*which labelled graph is being searched?*/
	vindex * verts; /*the start of an array of |vindex|'s*/
	int start_fringe; /*the position in the array where we first have
		a vertex which has not yet been dealt with*/
	int end; /*the position in the array of the last meaningful entry*/
}
bfs_traverser;

/* To initialize an |bfs_traverser|, we supply as arguments the address
of what we are initializing and the address of the labelled graph we
plan to search. The labelled graph itself changes during the search,
because we have to record in the labelled graph the fact that there 
is a search going on.
Another change to the labelled graph is that during the search, some
vertices are marked as visited. But this marking is erased at the end
of the search.
It is not permissible to have two searches going on at once.
The |bfs_traverser| also changes during the search, which should be no
surprise.
*/
extern void bfs_init PARMS((bfs_traverser *, lg * ));

/* 
|bfs_clear()| deallocates space needed for a |bfs_traverser|.
The labelled graph will also be changed, because we have to record the
fact that the search has been completed.
*/
extern void bfs_clear PARMS((bfs_traverser * ));

/* |bfs_next()| puts the address of the next vertex into the slot given by
the second argument. This function marks the vertices as ``visited''.
The successors of |w| are marked as
visited immediately |bfs_next()| is  next called.
(Of course, some of these successors may have been marked
as visited during previous calls of |bfs_next()|.)
*/
extern boolean bfs_next PARMS((bfs_traverser * , vindex * ));
#define set_visited(v,b)		(v)->bits.visit = (b)
#define get_visited(v)		(v)->bits.visit

/* The function table which allows us to make lists of
addresses of vertices.
*/
extern elt_fntab VINDEX_fntab;
#define VINDEX (&VINDEX_fntab)
extern elt_fntab ASINT_VINDEX_fntab;
#define ASINT_VINDEX &ASINT_VINDEX_fntab

/* 
The arguments in the next lot of functions are |dp|'s because the functions
are used in list handling. In practice. they will be pointers to
|vindex|'s (i.e. of type |vertex **| = |vindex *|).
*/ 
extern int vindex_sgn_dp PARMS (( dp vindex_ptr1, dp vindex_ptr2));
extern void vindex_cpy_dp PARMS ((dp old_vindex_ptr, dp new_vindex_ptr));
				/*Copy the |vindex| only, not the whole vertex structure.  */
extern vindex * vindex_create PARMS (()); /* create space for a vindex */
extern void vindex_print_dp PARMS ((FILE * wfile,dp vindex_ptr));
extern int get_vindex_hpindex_dp PARMS ((dp vindex_ptr));
extern void set_vindex_hpindex_dp PARMS ((dp vindex_ptr, int i));

extern void vertex_print PARMS ((FILE * wfile,vindex, lg*));
extern void vertex_cprint PARMS ((FILE* wfile, vindex, lg*));
extern void vertex_print_dp PARMS ((FILE *,dp));
extern boolean is_basepoint PARMS ((vindex));

/* Let |v| and |w| be two vertices of a labelled graph.
|set(v,g,w)| sets the value of the arrow with source |v| and label
|g| equal to |w|. |unset()| sets the arrow equal to |UNDEFINED|. 
*/
#define UNDEFINED 0
#define get(v,g)	((v)->adj)[(unsigned int)(g)]
#define set(v,g,w)	((v)->adj)[(unsigned int)(g)]=w
#define unset(v,g)  ((v)->adj)[(unsigned int)(g)]=0

#define get_index(v)	((v)->index)
#define set_index(v,i)	((v)->index = i)

#define set_back_ptr(v,g)		(v)->back = (g)
#define backg(v)			(v)->back		
#define set_state(v,w)	(v)->adj[0] = (w)
#define get_state(v)	(v)->adj[0]
/* the space at |adj[0]| may be used to store the address of a vertex (or
state) of a finite state automata associated with a particular vertex of a
word tree */

typedef struct vtxlink{
	struct vtxlink * next;
	vindex v;
} vtxlink;

typedef struct { 
	vtxlink ** array; 
	int maxhash;
} vtxhashlist; 

extern void vtxhashlist_init PARMS ((vtxhashlist * hlp,int size));
/* size is maxhash+1 */
extern void vtxhashlist_insert PARMS((vtxhashlist * hlp, vindex * v, int hash));
extern boolean vtxhashlist_find PARMS ((vtxhashlist * hlp, vindex * in, vindex
* out, int hash));
extern void vtxhashlist_clear PARMS ((vtxhashlist * hlp));

#define NONACCEPTSTATE 0
#define ACCEPTSTATE 1
#define get_category(v)	(v)->category
#define set_category(v,i)	(v)->category=i

/* Return the distance from the basepoint.  */
extern int dist PARMS ((vindex));
#define IMPOSSIBLE_DISTANCE   -1 /*distance between two vertices*/


/*
|root| identifies the vertex that currently represents |v|. |v| itself is
being discarded.  */
extern vindex root PARMS ((vindex ));

#define set_archive_image(u,v)		(u)->adj[0] = (v)
#define	set_rhs(u,v)			(u)->adj[0] = (v)
#define archive_image(u)		(u)->adj[0]
#define	get_rhs(u)			(u)->adj[0]


extern mp_fntab GENERIC_fntab;
extern mp_fntab KMP_fntab;
extern mp_fntab WDIFF_fntab;
extern mp_fntab ACCEPTOR_fntab;
extern mp_fntab FSA_OPT_fntab;
extern mp_fntab PAIR_fntab;
extern mp_fntab TRIPLE_fntab;
extern mp_fntab LIST_fntab;
#define GENERIC &GENERIC_fntab
#define KMP &KMP_fntab
#define WDIFF &WDIFF_fntab
#define ACCEPTOR  &ACCEPTOR_fntab
#define FSA_OPT &FSA_OPT_fntab
#define PAIR &PAIR_fntab
#define TRIPLE &TRIPLE_fntab
#define LIST &LIST_fntab

/* The next few definitions make it possible to use function tables
without making it too apparent to the user. This will enable the
mechanism for object oriented coding to be changed if necessary (i.e.
using some other approach than function tables).
*/
#define get_type(lgp)	(lgp)->type
#define set_type(lgp,i)	((lgp)->type = (i))

/*
The different structures hidden in the magic pockets.
In a |KMP| the |failure_pointer| at v  gives the vertex corresponding to the
longest possible proper suffix of $s$ which is a path in the tree of acceptable
words.
*/
typedef struct {
	unsigned lhs :1;
	unsigned lhs_tree :1;
	unsigned inv_lhs :1;
	unsigned inv_lhs_tree :1;
	unsigned fails_to_reducible :1;
	unsigned under_consideration :1;
	unsigned under_inv_con :1;
	unsigned considered :1;
	unsigned considered_tree :1;
	unsigned inv_considered :1;
	unsigned inv_considered_tree :1;
	unsigned overlap :1;
	unsigned wdiff :1;
}
status_type;

typedef struct {
	status_type status;
	int priority;
	int heap_index;
	vindex failure;
#ifdef BAD
	vindex nextc;
#endif
}
KMPstruct;


typedef struct WDIFFstruct {
	list *arrows;
}
WDIFFstruct;

typedef struct ACCEPTORstruct {
	list better; /*a list of word differences which come from reduced
		words of the same length which are lexicographically ahead*/
	list worse; /*a list of word differences which come from reduced
		words of the same length which are lexicographically behind.*/
	list behind; /* a list of word differences of length one which
		come from reduced words of length one less.*/
}
ACCEPTORstruct;

typedef struct FSA_OPTstruct {
   vertex *orig; /* pointer to original state */
   struct FSA_OPTstruct *member_l; /* link to next member of this block */
   struct FSA_OPTstruct **member_b; /* pointer to link field that points to us */
   struct block *blk; /* pointer to the block this is in */
   struct FSA_OPTstruct *split_l; /* the link to the next split-off state */
   struct inv_array * inverse; /* interleaved arrays, indexed by sym */
} FSA_OPTstruct;

typedef struct block {
   int count; /* count of elements in the block */
   struct block *blk_l, *wait_l, *j_l; /* links */
   struct FSA_OPTstruct *member_h; /* pointer to first member */
   struct FSA_OPTstruct *split_h; /* pointer to first member belonging to inverse */
} block;

typedef struct inv_array {
	struct FSA_OPTstruct *h, *l;
} inv_array;

typedef struct PAIRstruct {
	vindex state1;
	vindex state2;
}
PAIRstruct;

typedef struct TRIPLEstruct {
	vindex state1;
	vindex state2;
	vindex state3;
}
TRIPLEstruct;

typedef struct LISTstruct {
	list * listp; 
}
LISTstruct;

/*
Functions specific to |KMP|s.
*/

/* Identify two vertices in a KMP tree. The function returns |TRUE| if the 
two vertices were previously distinct, |FALSE| otherwise.
*/
extern boolean
coin PARMS ((lg *lgp, vindex keep, vindex discard));

/*
The function |lift_vertex_characteristics| is used to lift the magic pocket
information in a |KMP| from one vertex to another (e.g to a vertex with
which it is being identified).
*/
#define lift_vertex_characteristics(v) \
			( \
				((KMPstruct *)(root(v)->mp))->status.lhs |= \
					((KMPstruct *)((v)->mp))->status.lhs, \
				((KMPstruct *)(root(v)->mp))->status.lhs_tree |= \
					((KMPstruct *)((v)->mp))->status.lhs_tree, \
				((KMPstruct *)(root(v)->mp))->status.inv_lhs |= \
					((KMPstruct *)((v)->mp))->status.inv_lhs, \
				((KMPstruct *)(root(v)->mp))->status.inv_lhs_tree |= \
					((KMPstruct *)((v)->mp))->status.inv_lhs_tree, \
				((KMPstruct *)(root(v)->mp))->status.fails_to_reducible |= \
					((KMPstruct *)((v)->mp))->status.fails_to_reducible, \
				((KMPstruct *)(root(v)->mp))->status.under_consideration |= \
					((KMPstruct *)((v)->mp))->status.under_consideration, \
				((KMPstruct *)(root(v)->mp))->status.under_inv_con |= \
					((KMPstruct *)((v)->mp))->status.under_inv_con, \
				((KMPstruct *)(root(v)->mp))->status.considered |= \
					((KMPstruct *)((v)->mp))->status.considered, \
				((KMPstruct *)(root(v)->mp))->status.considered_tree |= \
					((KMPstruct *)((v)->mp))->status.considered_tree, \
				((KMPstruct *)(root(v)->mp))->status.inv_considered |= \
					((KMPstruct *)((v)->mp))->status.inv_considered, \
				((KMPstruct *)(root(v)->mp))->status.inv_considered_tree |= \
					((KMPstruct *)((v)->mp))->status.inv_considered_tree, \
				((KMPstruct *)(root(v)->mp))->status.overlap |= \
					((KMPstruct *)((v)->mp))->status.overlap \
			)

#define get_failure(v)	((KMPstruct *)((v)->mp))->failure
#define set_failure(v,f)	((KMPstruct *)((v)->mp))->failure=f
#define get_priority(v)		((KMPstruct *)((v)->mp))->priority
#define set_priority(v,i)		((KMPstruct *)((v)->mp))->priority=i
#define is_lhs_tree(v) 	(((KMPstruct *)((v)->mp))->status.lhs_tree)
#define does_fail_to_reducible(v)	(((KMPstruct *)((v)->mp))->\
		status.fails_to_reducible)
#define is_reducible(v)		(archive_image(v)? \
	get_rhs(archive_image((v))) != archive_image((v)): FALSE)
#ifdef BAD
extern void set_next PARMS ((vindex,vindex));
#define get_next(u)			((KMPstruct *)((u)->mp))->nextc
#else
#define set_next(u,v)		set_failure((u),(v))
#define get_next(u)			get_failure((u))
#endif
#define CON_TREE(v)\
	(get_status((v),CONSIDERED_TREE) || get_status((v),INV_CONSIDERED_TREE))
#define CON_LEAF(v)\
	(get_status((v),CONSIDERED) || get_status((v),INV_CONSIDERED))
#define set_parent(child,parent)	set_failure(child,parent)
#define get_parent(child)		get_failure(child)


boolean get_status PARMS((vindex v, int status));
void set_status PARMS((vindex v, int status, boolean b));

/* 
In a |KMP|, the vertices in a specific subtree can be marked.
|mark_downwards(v)| sets the |status| of |v| to be |TRUE| and  marks the 
vertices between |v| and the basepoint (tracing
back along the back pointers). |unmark_downwards(v)| sets the |status| to
be |FALSE| and undoes the marking from v, analogously.
*/
extern void mark_downwards PARMS ((vindex v, int status));
extern void unmark_downwards PARMS (( vindex , int status));

/* Assume that |remain| points to a |KMP| with an empty magic pocket,
and |disappear| points to a |KMP| possibly with magic pocket information.
|disappear| is absorbed into |remain|, and is left as a
trivial labelled graph with one vertex and no arrows.
*/
extern void kill_and_devour PARMS (( lg *remain, lg *disappear));


/* A special create function for |to_add|, which maps the basepoint
into |archive|.
*/
extern void to_add_create PARMS ((VOID));

extern void refine PARMS ((VOID));
extern void comp_failure PARMS (( vindex ));
/* phase */
#define BUILD						0

#define LH							3
#define LHS							5
#define RHS							6
#define LS							7
#define REDUCIBLE					9
#define FAILS_TO_REDUCIBLE 10
#define WORD_DIFF						11
#define ON_PQ						14
#define BASEPOINT					15
 /* phases */
#define LH_REBUILD					17
#define LS_REBUILD					18
#define RHS_TREE					20
#define LHS_TREE					21
#define UNDER_CONSIDERATION			22
#define INV_LHS						23
#define INV_LHS_TREE				25
#define CONSIDERED 					26
#define CONSIDERED_TREE				27
#define	posn(i,j) 			((i)*(num_gens+1) + (j)) 


typedef struct ggv_triple {
	gen g1;
	gen g2;
	vindex v;
}
ggv_triple;

/* Return the number of word differences and install any new
differences.
*/
extern int wdiffs_from_wpair PARMS (( word *, word *));

extern int num_wdiffs;
extern void kill_arrows PARMS ((vindex));
#define get_arrows(v)		((WDIFFstruct*)((v)->mp))->arrows
#define unset_arrows(v)		((WDIFFstruct*)((v)->mp))->arrows = 0
extern void mk_ggv_triple PARMS ((gen, gen, vindex, ggv_triple*));
extern list *mk_arrows PARMS ((vindex));
extern void WD_create PARMS ((VOID));

extern elt_fntab GGV_TRIPLE_fntab;
#define GGV_TRIPLE (&GGV_TRIPLE_fntab)

#define INV_CONSIDERED				28
#define INV_CONSIDERED_TREE			29
#define OVERLAP						30
#define UNDER_INV_CON				31

#define BACK_FROM_BASE				-2




extern list *behindp PARMS ((vindex));
extern list *betterp PARMS ((vindex));
extern list *worsep PARMS ((vindex));
extern int statelist_hash PARMS ((list *lp, int m));


#define get_state1(v)	((PAIRstruct *)((v)->mp))->state1
#define get_state2(v)	((PAIRstruct *)((v)->mp))->state2
#define set_state1(v,v1)	((PAIRstruct *)((v)->mp))->state1=v1
#define set_state2(v,v2)	((PAIRstruct *)((v)->mp))->state2=v2
#define get_1state(v)	((TRIPLEstruct *)((v)->mp))->state1
#define get_2state(v)	((TRIPLEstruct *)((v)->mp))->state2
#define get_3state(v)	((TRIPLEstruct *)((v)->mp))->state3
#define set_1state(v,v1)	((TRIPLEstruct *)((v)->mp))->state1=v1
#define set_2state(v,v2)	((TRIPLEstruct *)((v)->mp))->state2=v2
#define set_3state(v,v3)	((TRIPLEstruct *)((v)->mp))->state3=v3

#define get_listp(v) ((LISTstruct *)((v)->mp))->listp
#define set_listp(v,lp) ((LISTstruct *)((v)->mp))->listp=lp


