/*** FILE "mapbase.h" ********************************************** 110693 ***/
#ifndef _mapbase_h_
#define _mapbase_h_

#include "fnd/fnd.h"

/*** pix, spix ****************************************************************/

typedef n16 pix;   //pix  = unsigned pixel unit
typedef z16 spix;  //spix =   signed pixel unit

/*** Class hieracy: ***********************************************************/
/*

MapBase ---->*----> MapPCBase
             |
             *----> MapSCBase

MapBase:   Bitmap. The possible primitive operations reach beond the usual
           set-/get-pixel operations. There is no limit for the use of a
           MapBase; it's intended use is for textures and other kinds of
           images, it specially serves as a minimal interface for a viewport,
           i.e. a map that is shown on the screen. All kinds of hybrids are
           possible.

MapPCBase: Pseudo-color map. A subclass of MapBase, and can therefore be used
           as an ordinary MapBase class, i.e. be used as if it's just a bitmap!
           Colors have to be allocated before they can be used. There are two
           colors, that can't be allocated this way, -well, that's not quite
           true; they are preallocated-, which are refered to as Black and
           White. Even though they are called Black and White, and are often
           used as if they are black and white, the colors of Black and White
           doesn't have to be black and white! Black is always the palette
           entry 0, White can be any palette entry, -usually the last-, which
           can be retrieved by the GetWhite operation. Black is refered to as
           the background color.

MapSCBase: Static-color map. Again a subclass of MapBase, with the exact same
           functionality as an ordinary MapBase class. No matter have many
           colors that are actually used, -always at least two, of course-, the
           user is supposed to use it, as if all combinations of red, green and
           blue are available; a MapSCBase class always maps a requested color
           triple to what it thinks, is the best match.

Note: To build a subclass of MapBase, MapPCBase or MapSCBase, exactly the pure
      virtual procedures have to be binded (together with the construction/
      destruction, of course). The virtual, but not pure procedures may
      eventually be overrided. This is typically done to gain speed, when
      drawing capabilities exists in hardware. However, default implementations
      exist. Be aware, that if a pixel-for-pixel reproducibility is required,
      the default implementations should not be overrided.
*/
/*** NC = Non-Clipping/-Checking **********************************************/
/*
Note: Some of the following map-operations have the extension NC for
      Non-Clipping/-Checking, which means, they should be used with great care,
      since improper use may/will result in corruption of data. All NC
      operations are considered time critical; boundary checks can't be
      afforded!
*/
/*** PPoint *******************************************************************/

class PPoint  //Polygon-point, used when specifying polygons.
{
public:
  pix x, y;
};

/*** MapBase ******************************************************************/

class MapBase
{
public:
  virtual pix SizeX() const=0;  //Returns the size of the map in the x...
  virtual pix SizeY() const=0;  //...and the y direction.

  virtual pix MaxX() const;  //Returns the maximum valid pix value in the x...
  virtual pix MaxY() const;  //...and in the y direction.
                             //Note: The MaxX/MaxY value is one less than the
                             //      SizeX/SizeY value!
                             //As default calls SizeX/SizeY.

  virtual void BitColor(n1 c)=0;         //Sets current color (0=Black,
                                         //1=White).
  virtual n1 GetBitColorNC(pix x, pix y) const;  //Reads current color (Black or
                                                 //White only, that is, _all_
                                                 //pixels are reported as either
                                                 //Black or White).
                                                 //As default calls GetPixelNC.
  /* Note: All maps can be considered to be bitmaps, where each pixel has the
   *       value 0 or 1. Therefore the above two operations applies to any kind
   *       of map.
   */

  void Black();  //Sets current color to Black.
  void White();  //Sets current color to White.
                 //As default calls Color.
  /* Note: There are always two color values, that will be called Black and
   *       White, and that will be used by the system as if they are actually
   *       black and white, even though that might not be the case!
   *       Terminology: Black=the system color black, not nessessarily the
   *                          color black.
   *                    black=the color black.
   *                    White=the system color white.
   *                    white=the color white.
   */

  virtual void Clear();  //Clears the hole map using the color Black.
                         //As default calls ClearNC.
  virtual void ClearNC(pix x1, pix y1, pix x2, pix y2);  //Clears a window using
                                                         //the color Black.
                                                         //As default calls
                                                         //FillBoxNC.

  virtual void PixelNC(pix x, pix y)=0;  //Sets pixel value to the current
                                         //color.
  virtual void GetPixelNC(pix x, pix y,
                          n16 &red, n16 &green, n16 &blue) const=0;
  //Reads the value of a pixel.
  /* Note: It makes good sence to let all map-classes have this get-pixel
   *       operation; when doing texture-mapping, f.ex., this is the essential
   *       map function - having it frees the texture-mapper from distinguishing
   *       between the different kinds of maps, and that's clever!
   *       Extremly usefull in a number of other cases too.
   *       It does not necessarily return (0,0,0) for Black and
   *       (MAX_R,MAX_G,MAX_B) for White!
   */

  virtual void HLineNC(pix x1, pix x2, pix y);  //Draws horisontal line.
  virtual void VLineNC(pix x, pix y1, pix y2);  //Draws vertical line.
  virtual void DLineNC(pix x1, pix y1, pix x2, pix y2);  //Draws line, draws
                                                         //horisontal/vertical
                                                         //_slower_ than the
                                                         //above two functions!
  virtual void LineNC(pix x1, pix y1, pix x2, pix y2);  //Draws line by calling
                                                        //HLineNC/VLineNC/
                                                        //DLineNC.
  /* Note: The above line-drawing functions all draw lines of width "0",
   *       that is, of width one pix.
   */

  virtual void BoxNC(pix x1, pix y1, pix x2, pix y2);  //Draws an outlined
                                                       //rectangle.
                                                       //As default calls
                                                       //HLineNC and VLineNC.
  /* Note: Even though this can't be called a "primitive" operation,
   *       experience tells me, it's nice to have!
   */

  virtual void FillBoxNC(pix x1, pix y1, pix x2, pix y2);  //Draws a filled
                                                           //rectangle.

  virtual void FillTriangleNC(PPoint points[]);
  //Draws a filled triangle, as descrip by the points 'points', which, -of
  //course-, should contain exactly three points.

  virtual void FillConvexPolygonNC(PPoint points[], word n);
  //Draws a filled polygon, as descriped by the points 'points' and the number
  //of points 'n'. If the specified polygon isn't convex, or n<3, the result
  //can't be predicted. Can't handle concave, non-simple polygons.
  //Note: There is no limit on the size of n!

  virtual void FillPolygonNC(PPoint points[], word n);
  //Same as FillConvexPolygon, except that it can handle cancave polygons too.
  //Can't handle non-simple polygons.
  //Note: There is no limit on the size of n!

  virtual void Flush();  //Supposed to flush all internal buffers.
                         //As default does nothing.
  /* Note: It's important to use this function correctly! The use of any
   *       drawing-primitive doesn't necessarily imply, that the change will
   *       take effect, before after flush has been called!!!
   *       This feature exists primarily to be able to communicate effectively
   *       with certain window-environments.
   *       The use of any class, that can't be specified more precisely than a
   *       MapBase, MapPCBase, or MapSCBase class, should always imply, that
   *       flush is used as proscribed here!
   */

#define MIN_ASPECTRATIO (1/1000)
#define MAX_ASPECTRATIO 1000

  virtual void AspectRatio(double f);  //Sets the pixel aspect ratio factor
                                       //(size of pixel in the x-direction)/
                                       //(size of pixel in the y-direction).
                                       //The allowed range is from
                                       //MIN_ASPECTRATIO to MAX_ASPECTRATIO.
                                       //This restriction is primarily to avoid
                                       //ranges, that can't be guaranteed to be
                                       //covered by the double-type.
                                       //Despites, it makes good sence.
                                       //Range-check should be performed.
                                       //As default does nothing.
  virtual double GetAspectRatio() const;  //Returns the pixel aspect ratio
                                          //factor.
                                          //As default returns 1.
/*** EXPERIMENTAL AREA ONLY ***/

  virtual void Copy(MapBase& M, pix x1, pix y1, pix x2, pix y2);
  //COPYING EASY!

/******************************/
};

/*** MapPCBase ****************************************************************/

class MapPCBase: virtual public MapBase
{
public:
  void BitColor(n1 c);  //Calls ColorNC.

  virtual n16 GetWhite() const=0;  //Returns the palette entry for White.
                                   //Note: There isn't an equivalent function
                                   //      for Black, since it always has the
                                   //      entry 0!

  virtual n16 MaxPaletteEntry() const=0;  //Returns the maximum valid palette
                                          //entry, that is, the number of
                                          //palette entries -1.

  virtual void InitColorAllocation()=0;  //Marks all palette entries as
                                         //deallocated, except Black and White.
  virtual n16 FreeColors() const=0;      //Returns the number of colors, which
                                         //are free for allocation.
  virtual n16 AllocColor()=0;            //Returns allocated palette entry,
                                         //returns 0 if allocation could not
                                         //take place; palette entry 0 can't be
                                         //allocated this way; it's Black!
  virtual void DeallocColor(n16 c)=0;    //Deallocates allocated color.
  virtual n1 AllocatedColor() const=0;   //Indicates, if the current palette
                                         //entry is allocated.
  /* Note: Terminology: current color=the color that will be used by the next
   *                                  drawing primitive, that is, a color
   *                                  triple (red,green,blue).
   *                    current palette entry=the palette entry, that holds
   *                                          the value of the current color.
   */

  virtual void ColorNC(n16 c)=0;  //Selects the current palette entry.
  virtual void PaletteEntry(n16 red, n16 green, n16 blue)=0;  //Sets the current
                                                              //palette entry.
  virtual void GetPaletteEntry(n16 &red, n16 &green, n16 &blue) const=0;
  //Reads the current palette entry.

  virtual n16 GetPixelNC(pix x, pix y) const=0;  //Reads palette entry value at
                                                 //pixel (x,y).

/*** EXPERIMENTAL AREA ONLY ***/

  virtual void Copy(MapBase& M, pix x1, pix y1, pix x2, pix y2,
                    n16 colors=65534); //colors=maximum number of colors that
                                       //may be used, if possible (never Black
                                       //or White!).
  //COPYING NOT EASY HERE! QUANTIZATION USED!

/******************************/
};

/*** MapSCBase ****************************************************************/

class MapSCBase: virtual public MapBase
{
public:
  void BitColor(n1 c);  //Calls Color.

  virtual void Color(n16 red, n16 green, n16 blue)=0;  //Sets the current color.

/*** EXPERIMENTAL AREA ONLY ***/

  virtual void Copy(MapBase& M, pix x1, pix y1, pix x2, pix y2);
  //COPYING EASY!

  virtual void Blend(MapBase& M, pix x1, pix y1, pix x2, pix y2);

/******************************/
};

#endif
/******** "mapbase.h" *********************************************************/

