/* module: 4dd_render.c   iris 4d rendering */

static char SccsId[] = "@(#)4dd_render.c	1.2";

#include <stdio.h>
#include <math.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <gl.h>
#include <device.h>

#include "4dd_echo.h"
#include "4dd_cyfile.h"
#include "4dd_gt.h"

extern int errno;
extern struct wireinfo wire;
extern struct renderinfo render;

unsigned int rgb_mapcolor(int *rgbimage, struct Vertex *vtx, int lg, int lt);

#define TRIANGLE_MESH		1			/* patch mode */
#define QUADRILATERAL_MESH	2			/* patch mode */

#define RENDER_GOURAUD				1			/* lighting model mode */
#define RENDER_VALUE_COLORMAP		2			/* colormaped mode */
#define RENDER_VALUE_RGB			3			/* rgb mode */
#define RENDER_GOURAUD_RGB			4			/* shaded rgb mode */
                                    

/* values returned by menu */

#define ROTATE			(long)10
#define ZOOM			(long)11
#define PAN				(long)12
#define PERSP			(long)13
#define TWIST			(long)14
#define WIRE_XFRM		(long)15
#define ORTHO_FLAG		(long)16

#define RENDER			(long)20
#define METER			(long)21
#define ERASE			(long)22
#define OPTIONS			(long)23
#define EXIT			(long)24
#define CLIP			(long)25
#define SNAP			(long)26

#define LTMIN_CLIP		(long)30
#define LTMAX_CLIP		(long)31
#define LGMIN_CLIP		(long)32
#define LGMAX_CLIP		(long)33
#define LTMIN_UNCLIP	(long)34
#define LTMAX_UNCLIP	(long)35
#define LGMIN_UNCLIP	(long)36
#define LGMAX_UNCLIP	(long)37

#define HIRES			(long)41
#define LORES			(long)42
#define DRAWLT			(long)43
#define DRAWLG			(long)44
#define COLOR			(long)45
#define DOUBLEBUF		(long)46
#define SINGLEBUF		(long)47
#define DEPTHCUE		(long)48

#define TRIANGLES		(long)51
#define QUADRILATERALS	(long)52
#define PUP_GOURAUD		(long)53
#define PUP_VALUE_COLORMAP		(long)54
#define PUP_VALUE_RGB			(long)55
#define PUP_GOURAUD_RGB			(long)56

#define MATERIAL_SKIN		(long)61
#define MATERIAL_PLASTIC	(long)62
#define MATERIAL_DEFAULT	(long)63

#define FUNCTION_RADIUS		(long)71
#define FUNCTION_SMOOTHNESS	(long)72
#define FUNCTION_RGBCOLOR	(long)73

#define RELIEF_SIZE			(long)81
#define RELIEF_MAKE			(long)82
#define OSCILLATE			(long)83

Matrix idmat = {1., 0., 0., 0.,
				0., 1., 0., 0.,
				0., 0., 1., 0.,
				0., 0., 0., 1.};

float lm_material_skin[] = {
	SPECULAR, .3, .3, .3,
	DIFFUSE, 1.00, .933, .890,
	SHININESS, 90.,
	LMNULL
};

float lm_material_plastic[] = {
	SPECULAR, 1., 1., 1.,
	DIFFUSE, .46, .92, 1.,
	SHININESS, 128.,
	LMNULL
};

float lm_light_default[] = {
	AMBIENT, 0.0, 0.0, 0.0,
	LCOLOR, 1.0, 1.0, 1.0,
	POSITION, 0.5, 1.0, 1.0, 0.0,
	LMNULL
};

static long material_index_save;



render_event(vtx, event, value)

struct Vertex *vtx;
Device event;
short value;
{
	long pupvalue;
	char string[128];
	Boolean first_time = TRUE;

	switch (event) {
	default:
		break;
	case CREATE:

		prefposition(
			render.position_x, render.size_x + render.position_x,
			render.position_y, render.size_y + render.position_y
		);
		render.vtx = vtx;
		render.gid = winopen(render.name);
		wintitle(render.name);
		keepaspect(render.size_x, render.size_y);
		winconstraints();
		if (render.doublebuffer) {
			doublebuffer();
		} else {
			singlebuffer();
		}
		RGBmode();
		overlay((long)2);
		gconfig();
		zbuffer(TRUE);
		mmode(MVIEWING);
		lmdef(DEFMATERIAL, 1, 0, NULL);
		lmdef(DEFMATERIAL, 2, 11, lm_material_skin);
		lmdef(DEFMATERIAL, 3, 11, lm_material_plastic);
		lmdef(DEFLIGHT, 1, 0, NULL);
		lmdef(DEFLIGHT, 1, 14, lm_light_default);
		lmdef(DEFLMODEL, 1, 0, NULL);
		render.menu = render_defpup(&render);
		cpack(render.color_background);
		frontbuffer(TRUE);
		clear();
		frontbuffer(FALSE);
		if (render.render_firsttime) {
			render_redraw(&render);
		}
		break;
	case REDRAW:
		render_redraw(&render);
		break;
	case DESTROY:
		lmbind(LIGHT0, 0);
		winclose(render.gid);
		break;
	case WRITECONFIG:
		render_putdefaults(&render);
		break;
	case MENUBUTTON:
		help_event("Hold button, select item, release button",
			(Device)REDRAW, 0);
		if (getbutton(MENUBUTTON)) {
			pupvalue = dopup(render.menu);
			switch (pupvalue) {
			default:
				break;
			case RENDER:
				qenter((short)REDRAW, render.gid);
				break;
			case ROTATE:
				render_rotate(&render);
				break;
			case ZOOM:
				render_zoom(&render);
				break;
			case PAN:
				render_pan(&render);
				break;
			case PERSP:
				render_persp(&render);
				break;
			case TWIST:
				render_twist(&render);
				break;
			case OSCILLATE:
				render_oscillate(&render);
				break;
			case WIRE_XFRM:
				render.wire_transform = !render.wire_transform;
				render.menu = render_defpup(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case ORTHO_FLAG:
				render.ortho_flag = !render.ortho_flag;
				render.menu = render_defpup(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case METER:
				render_meter(&render);
				break;
			case LTMIN_CLIP:
				vtx->ltmin = render.index_lt;
				qenter((short)REDRAW, render.gid);
				break;
			case LTMAX_CLIP:
				vtx->ltmax = render.index_lt;
				qenter((short)REDRAW, render.gid);
				break;
			case LGMIN_CLIP:
				vtx->lgmin = render.index_lg;
				qenter((short)REDRAW, render.gid);
				break;
			case LGMAX_CLIP:
				vtx->lgmax = render.index_lg;
				qenter((short)REDRAW, render.gid);
				break;
			case LTMIN_UNCLIP:
				vtx->ltmin = 0;
				qenter((short)REDRAW, render.gid);
				break;
			case LTMAX_UNCLIP:
				vtx->ltmax = vtx->nlt - 1;
				qenter((short)REDRAW, render.gid);
				break;
			case LGMIN_UNCLIP:
				vtx->lgmin = 0;
				qenter((short)REDRAW, render.gid);
				break;
			case LGMAX_UNCLIP:
				vtx->lgmax = vtx->nlg - 1;
				qenter((short)REDRAW, render.gid);
				break;
			case MATERIAL_DEFAULT:
				render.material_index = 1;
				render.menu = render_defpup(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case MATERIAL_SKIN:
				render.material_index = 2;
				render.menu = render_defpup(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case MATERIAL_PLASTIC:
				render.material_index = 3;
				render.menu = render_defpup(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case HIRES:
				if (--render.ltresolution < 1) render.ltresolution = 1;
				--render.lgresolution;
				if (--render.lgresolution < 1) render.lgresolution = 1;
				render.menu = render_defpup(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case LORES:
				if (++render.ltresolution > 4) render.ltresolution = 4;
				++render.lgresolution;
				if (++render.lgresolution > 8) render.lgresolution = 8;
				render.menu = render_defpup(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case TRIANGLES:
				render.patch_mode = TRIANGLE_MESH;
				render.menu = render_defpup(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case QUADRILATERALS:
				render.patch_mode = QUADRILATERAL_MESH;
				render.menu = render_defpup(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case PUP_GOURAUD:
				/* draw using Gouraud shading, no data values */
				render.material_index = material_index_save;
				render.render_mode = RENDER_GOURAUD;
				render.menu = render_defpup(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case PUP_GOURAUD_RGB:
				/* draw using Gouraud shaded colors from data array */
				render.material_index = material_index_save;
				render.render_mode = RENDER_GOURAUD_RGB;
				render.menu = render_defpup(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case PUP_VALUE_COLORMAP:
				/* draw using RGB colors from data array */
				material_index_save = render.material_index;
				render.material_index = 0;		/* disable lighting */
				render.render_mode = RENDER_VALUE_COLORMAP;
				render.menu = render_defpup(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case PUP_VALUE_RGB:
				/* draw using colors mapped from the colormap by data array */
				material_index_save = render.material_index;
				render.material_index = 0;		/* disable lighting */
				render.render_mode = RENDER_VALUE_RGB;
				render.menu = render_defpup(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case FUNCTION_RADIUS:
				function_radius(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case FUNCTION_SMOOTHNESS:
				function_smoothness(&render);
				qenter((short)REDRAW, render.gid);
				break;
			case DOUBLEBUF:
				render.doublebuffer = TRUE;
				winset(render.gid);
				doublebuffer();
				gconfig();
				render.menu = render_defpup(&render);
				break;
			case SINGLEBUF:
				render.doublebuffer = FALSE;
				winset(render.gid);
				singlebuffer();
				gconfig();
				render.menu = render_defpup(&render);
				break;
			case EXIT:
				help_event("Exiting...", (Device)REDRAW, 0);
				edit_exit = TRUE;
				break;
			} /* end switch */
		} /* end if */
		break;
	} /* end switch */
	return(render.gid);
}


render_defpup(ri)

struct renderinfo *ri;
{
	static long pup = -1;
	long pup_clip, pup_unclip, pup_material, pup_resol, pup_xfrm;
	long pup_options, pup_functions;
	char string[256], *s;

	if (pup != -1) {
		freepup(pup);
		pup = -1;
	}

	s = string;
	s += sprintf(s, "View Transforms %%t");
	if (ri->wire_transform) {
		s += sprintf(s, "| Use Local Transform %%x%d", WIRE_XFRM);
	} else {
		s += sprintf(s, "| Rotate %%x%d", ROTATE);
		s += sprintf(s, "| Zoom %%x%d", ZOOM);
		s += sprintf(s, "| Pan %%x%d", PAN);
		s += sprintf(s, "| Perspec %%x%d", PERSP);
		s += sprintf(s, "| Twist %%x%d", TWIST);
		s += sprintf(s, "| Oscillate %%x%d", OSCILLATE);
		s += sprintf(s, "| Wireframe Transform %%x%d", WIRE_XFRM);
	}
	if (ri->ortho_flag) {
		s += sprintf(s, "| Use Perspective %%x%d", ORTHO_FLAG);
	} else {
		s += sprintf(s, "| No Perspective %%x%d", ORTHO_FLAG);
	}
	pup_xfrm = defpup(string);

	s = string;
	s += sprintf(s, "Clip Data %%t");
	s += sprintf(s, "| Latitude Min %%x%d", LTMIN_CLIP);
	s += sprintf(s, "| Latitude Max %%x%d", LTMAX_CLIP);
	s += sprintf(s, "| Longitude Min %%x%d", LGMIN_CLIP);
	s += sprintf(s, "| Longitude Max %%x%d", LGMAX_CLIP);
	pup_clip = defpup(string);

	s = string;
	s += sprintf(s, "UnClip Data %%t");
	s += sprintf(s, "| Latitude Min %%x%d", LTMIN_UNCLIP);
	s += sprintf(s, "| Latitude Max %%x%d", LTMAX_UNCLIP);
	s += sprintf(s, "| Longitude Min %%x%d", LGMIN_UNCLIP);
	s += sprintf(s, "| Longitude Max %%x%d", LGMAX_UNCLIP);
	pup_unclip = defpup(string);

	s = string;
	s += sprintf(s, "Material %%t");
	if (ri->material_index == 1) {
		s += sprintf(s, "| >> Default << %%x%d", MATERIAL_DEFAULT);
	} else {
		s += sprintf(s, "| Default %%x%d", MATERIAL_DEFAULT);
	}
	if (ri->material_index == 2) {
		s += sprintf(s, "| >> Skin    << %%x%d", MATERIAL_SKIN);
	} else {
		s += sprintf(s, "| Skin %%x%d", MATERIAL_SKIN);
	}
	if (ri->material_index == 3) {
		s += sprintf(s, "| >> Plastic << %%x%d", MATERIAL_PLASTIC);
	} else {
		s += sprintf(s, "| Plastic %%x%d", MATERIAL_PLASTIC);
	}
	pup_material = defpup(string);

	s = string;
	s += sprintf(s, "Resolution %%t");
	if (ri->ltresolution >= 1) {
		s += sprintf(s, "| Raise Resolution %%x%d", HIRES);
	} else {
		s += sprintf(s, "| ");
	}
	if (ri->ltresolution <= 4) {
		s += sprintf(s, "| Lower Resolution %%x%d", LORES);
	} else {
		s += sprintf(s, "| ");
	}
	pup_resol = defpup(string);

	s = string;
	s += sprintf(s, "Render Options %%t");
	if (ri->patch_mode == TRIANGLE_MESH) {
		s += sprintf(s, "| Quadrilaterals %%x%d", QUADRILATERALS);
	} else {
		s += sprintf(s, "| Triangles %%x%d", TRIANGLES);
	}
	s += sprintf(s, "| Gouraud Shading %%x%d", PUP_GOURAUD);
	s += sprintf(s, "| Gouraud RGB Shading %%x%d", PUP_GOURAUD_RGB);
	s += sprintf(s, "| Function/Colormap Shading %%x%d",PUP_VALUE_COLORMAP);
	s += sprintf(s, "| Function/RGB Shading %%x%d", PUP_VALUE_RGB);
	if (ri->doublebuffer) {
		s += sprintf(s, "| Single Buffer %%x%d", SINGLEBUF);
	} else {
		s += sprintf(s, "| Double Buffer %%x%d", DOUBLEBUF);
	}
	pup_options = defpup(string);

	s = string;
	s += sprintf(s, "Value Map Functions %%t");
	s += sprintf(s, "| Radius %%x%d", FUNCTION_RADIUS);
	s += sprintf(s, "| Smoothness %%x%d", FUNCTION_SMOOTHNESS);
	pup_functions = defpup(string);

	s = string;
	s += sprintf(s, "Rendering %%t");
	s += sprintf(s, "| Redraw %%x%d", RENDER);
	s += sprintf(s, "| Transform %%m");
	s += sprintf(s, "| Meter Points %%x%d", METER);
	s += sprintf(s, "| Clip Data %%m");
	s += sprintf(s, "| UnClip Data %%m");
	s += sprintf(s, "| Materials %%m");
	s += sprintf(s, "| Resolution %%m");
	s += sprintf(s, "| Render Options %%m");
	s += sprintf(s, "| Value Map Functions %%m");
	s += sprintf(s, "| Exit Edit %%x%d", EXIT);
	pup = defpup(string, pup_xfrm, pup_clip, pup_unclip,
		pup_material, pup_resol, pup_options, pup_functions);
	return(pup);
}


render_redraw(ri)

struct renderinfo *ri;
{
	help_event("Busy...", (Device)REDRAW, 0);
	winset(ri->gid);
	drawmode(NORMALDRAW);
	getorigin(&ri->position_x, &ri->position_y);
	getsize(&ri->size_x, &ri->size_y);
	viewport(0, ri->size_x - 1, 0, ri->size_y - 1);
	lsetdepth(0x100, 0x7fffff);
	zclear();
	cpack(ri->color_background);
	clear();
	if (ri->render_mode == RENDER_GOURAUD) {
		lmcolor(LMC_COLOR);
	} else if (ri->render_mode == RENDER_GOURAUD_RGB) {
		lmcolor(LMC_AD);
	}
	render_transform(ri, FALSE, ri->wire_transform);
	render_draw(ri);
	swapbuffers();
}


render_rotate(ri)

struct renderinfo *ri;
{
	short val;					/* qread event value */
	long dev;
	short base_x, base_y;
	Angle azim;
	Angle inc;

	base_x = getvaluator(MOUSEX);
	base_y = getvaluator(MOUSEY);
	azim = ri->azim;
	inc = ri->inc;

	qdevice((Device)MOUSEX);
	qdevice((Device)MOUSEY);
	help_event("Move mouse, left button stops", (Device)REDRAW, 0);
	for (;;) {
		switch (qread(&val)) {
		case MENUBUTTON:
			qenter((short)MENUBUTTON, ri->gid);
		case LEFTMOUSE:
			unqdevice((Device)MOUSEX);
			unqdevice((Device)MOUSEY);
			return;
		case MOUSEX:
		case MOUSEY:
			while ((dev = qtest()) == MOUSEX || dev == MOUSEY) {
				qread(&val);
			}
			ri->inc = inc - ((getvaluator(MOUSEY) - base_y) << 3);
			ri->azim = azim + ((getvaluator(MOUSEX) - base_x) << 3);
			render_redraw(ri);
			break;
		}
	}
}



render_oscillate(ri)

struct renderinfo *ri;
{

	long dev;
	int azim_base = ri->azim;
	int inc_base = ri->inc;
	struct itimerval timer;
	float time;
	float time_angle;
	short val;

	help_event("Left button stops", (Device)REDRAW, 0);
	/* set the timer to count down for a long time */
	timer.it_interval.tv_sec = 1000000;
	timer.it_interval.tv_usec = 0;
	timer.it_value.tv_sec = 1000000;
	timer.it_value.tv_usec = 0;
	setitimer(ITIMER_REAL, &timer, NULL);

	while (1) {
		if (qtest()) {
			dev = qread(&val);
			if (dev == MENUBUTTON || dev == LEFTMOUSE) {
				break;
			}
		}
		/* get timer value and convert to modulo 15000 milliseconds */
		getitimer(ITIMER_REAL, &timer);
		time = (float)timer.it_value.tv_sec;
		time += (float)timer.it_value.tv_usec * 1e-6;
		time = fmod(time, 15.);		/* time = 0 to 14.999 seconds */
		/* time angle rotates thru 360 degrees (2 pi) each 15 seconds */
		time_angle = M_PI * 2. * (float)time / 15.;

		ri->azim = azim_base + (450 * fcos(time_angle));
		ri->inc = inc_base + (100 * fsin(2. * time_angle));

		render_redraw(ri);
	}
	return;
}



render_zoom(ri)

struct renderinfo *ri;
{
	short val;					/* qread event value */
	short base_y;
	float zoom;

	base_y = getvaluator(MOUSEY);
	zoom = ri->zoom;
	qdevice((Device)MOUSEY);
	help_event("Move mouse, left button stops", (Device)REDRAW, 0);
	for (;;) {
		switch (qread(&val)) {
		case MENUBUTTON:
			qenter((short)MENUBUTTON, ri->gid);
		case LEFTMOUSE:
			unqdevice((Device)MOUSEY);
			return;
		case MOUSEY:
			while (qtest() == MOUSEY) {
				qread(&val);
			}
			ri->zoom = zoom + (getvaluator(MOUSEY) - base_y) / 100.;
			ri->zoom = MIN(MAX(ri->zoom, .1), 10.);
			render_redraw(ri);
			break;
		}
	}
}


render_pan(ri)

struct renderinfo *ri;
{
	short val;					/* qread event value */
	long dev;
	short base_x, base_y;
	float pan_x;
	float pan_y;

	base_x = getvaluator(MOUSEX);
	base_y = getvaluator(MOUSEY);
	pan_x = ri->panx;
	pan_y = ri->pany;
	qdevice((Device)MOUSEX);
	qdevice((Device)MOUSEY);
	help_event("Move mouse, left button stops", (Device)REDRAW, 0);
	for (;;) {
		switch (qread(&val)) {
		case MENUBUTTON:
			qenter((short)MENUBUTTON, ri->gid);
		case LEFTMOUSE:
			unqdevice((Device)MOUSEX);
			unqdevice((Device)MOUSEY);
			return;
		case MOUSEX:
		case MOUSEY:
			while ((dev = qtest()) == MOUSEX || dev == MOUSEY) {
				qread(&val);
			}
			ri->panx = pan_x + (getvaluator(MOUSEX) - base_x)/800.;
			ri->pany = pan_y + (getvaluator(MOUSEY) - base_y)/800.;
			render_redraw(ri);
			break;
		}
	}
}


render_persp(ri)

struct renderinfo *ri;
{
	short val;					/* qread event value */
	short base_y;
	float persp;

	base_y = getvaluator(MOUSEY);
	persp = ri->dist;
	qdevice((Device)MOUSEY);
	help_event("Move mouse, left button stops", (Device)REDRAW, 0);
	for (;;) {
		switch (qread(&val)) {
		case MENUBUTTON:
			qenter((short)MENUBUTTON, ri->gid);
		case LEFTMOUSE:
			unqdevice((Device)MOUSEY);
			return;
		case MOUSEY:
			while (qtest() == MOUSEY) {
				qread(&val);
			}
			ri->dist = persp + (getvaluator(MOUSEY) - base_y) / 100.;
			ri->dist = MAX(ri->dist, ri->clipz + .01);
			render_redraw(ri);
			break;
		}
	}
}


render_twist(ri)

struct renderinfo *ri;
{
	short val;					/* qread event value */
	short base_x;
	float twist;

	base_x = getvaluator(MOUSEY);
	twist = ri->twist;
	qdevice((Device)MOUSEX);
	help_event("Move mouse, left button stops", (Device)REDRAW, 0);
	for (;;) {
		switch (qread(&val)) {
		case MENUBUTTON:
			qenter((short)MENUBUTTON, ri->gid);
		case LEFTMOUSE:
			unqdevice((Device)MOUSEY);
			return;
		case MOUSEX:
			while (qtest() == MOUSEX) {
				qread(&val);
			}
			ri->twist = twist + ((getvaluator(MOUSEX) - base_x) << 3);
			render_redraw(ri);
			break;
		}
	}
}

render_transform(ri, cursor_mode, use_wire_xfrm)

struct renderinfo *ri;
Boolean cursor_mode;
Boolean use_wire_xfrm;
{
	if (use_wire_xfrm) {
		/* use the same transforms as the wireframe */
		ri->clipx = wire.clipx;
		ri->clipy = wire.clipy;
		ri->clipz = wire.clipz;
		ri->panx = wire.panx;
		ri->pany = wire.pany;
		ri->panz = wire.panz;
		ri->zoom = wire.zoom;
		ri->dist = wire.dist;
		ri->azim = wire.azim;
		ri->inc = wire.inc;
		ri->twist = wire.twist;
	}

	/* projection transforms */
	loadmatrix(idmat);

	if (ri->ortho_flag) {
		ortho(
			ri->panx - ri->clipx/ri->zoom, ri->panx + ri->clipx/ri->zoom,
			ri->pany - ri->clipy/ri->zoom, ri->pany + ri->clipy/ri->zoom,
			-ri->clipz, ri->clipz
		);
	} else {
		window(
			ri->panx - ri->clipx/ri->zoom, ri->panx + ri->clipx/ri->zoom,
			ri->pany - ri->clipy/ri->zoom, ri->pany + ri->clipy/ri->zoom,
			ri->dist - ri->clipz, ri->dist + ri->clipz
		);
	}
		
	/* viewing transforms */
	polarview(ri->dist, ri->azim, ri->inc, ri->twist);

	/* world transforms */
	rotate(900, 'x');  /* make inc and azim work nice by swapping x/y axes */

	/* lighting model */
	lmbind(MATERIAL, ri->material_index);
	lmbind(LIGHT0, 1);
	lmbind(LMODEL, ri->model_index);
}


render_draw(ri)

struct renderinfo *ri;
{
	int lg;
	struct Vertex *vtx = ri->vtx;
	int incr = ri->lgresolution;

	for (lg = vtx->lgmin; lg <= vtx->lgmax-incr; lg += incr) {
		render_drawlg(ri, lg, lg+incr);
	}
	/* if cylindrical, and all longitudes in range, close gap */
	if (!(vtx->gs->flags & FLAG_CARTESIAN) &&
		vtx->lgmin == 0 && (vtx->lgmax == vtx->nlg-1)) {
		render_drawlg(ri, lg, 0);
	}
	if (ri->render_mode == RENDER_GOURAUD) {
		lmcolor(LMC_COLOR);
	} else if (ri->render_mode == RENDER_GOURAUD_RGB) {
		lmcolor(LMC_DIFFUSE);
	}
}


render_drawlg(ri, lg0, lg1)

struct renderinfo *ri;
int lg0, lg1;
{
	int lt;
	int i;
	int incr = ri->ltresolution;
	struct Vertex *vtx = ri->vtx;
	unsigned data;

	if (ri->patch_mode == TRIANGLE_MESH) {
		switch (ri->render_mode) {
		default:
		case RENDER_GOURAUD:
			i = 256;
			bgntmesh();
			for (
				lt = vtx->ltmin-1 + incr;
				lt <= vtx->ltmax+1 - incr;
				lt += incr
			) {
				n3f(&vtx->pnt[lg0][lt][NX]);
				v3f(&vtx->pnt[lg0][lt][LX]);
				n3f(&vtx->pnt[lg1][lt][NX]);
				v3f(&vtx->pnt[lg1][lt][LX]);
				if ((i-=2) <= 0) {
					endtmesh();
					lt -= incr;
					i = 256;
					bgntmesh();
				}
			}
			endtmesh();
			break;
		case RENDER_GOURAUD_RGB:
			i = 256;
			bgntmesh();
			for (
				lt = vtx->ltmin-1 + incr;
				lt <= vtx->ltmax+1 - incr;
				lt += incr
			) {
				cpack(vtx->data[lg0][lt]);
				n3f(&vtx->pnt[lg0][lt][NX]);
				v3f(&vtx->pnt[lg0][lt][LX]);
				cpack(vtx->data[lg1][lt]);
				n3f(&vtx->pnt[lg1][lt][NX]);
				v3f(&vtx->pnt[lg1][lt][LX]);
				if ((i-=2) <= 0) {
					endtmesh();
					lt -= incr;
					i = 256;
					bgntmesh();
				}
			}
			endtmesh();
			break;
		case RENDER_VALUE_COLORMAP:
			i = 256;
			bgntmesh();
			for (lt = vtx->ltmin + incr; lt <= vtx->ltmax - incr; lt += incr) {
				data = vtx->data[lg0][lt];
				if (data < 128) {
					cpack(255 << 8 | data*2);
				} else {
					cpack((~data*2&0xff) << 8 | 255);
				}
				v3f(&vtx->pnt[lg0][lt][LX]);
				v3f(&vtx->pnt[lg1][lt][LX]);
				if ((i-=2) <= 0) {
					endtmesh();
					lt -= incr;
					i = 256;
					bgntmesh();
				}
			}
			endtmesh();
			break;
		case RENDER_VALUE_RGB:
			i = 256;
			bgntmesh();
			for (lt = vtx->ltmin + incr; lt <= vtx->ltmax - incr; lt += incr) {
				cpack(vtx->data[lg0][lt]);
				v3f(&vtx->pnt[lg0][lt][LX]);
				cpack(vtx->data[lg1][lt]);
				v3f(&vtx->pnt[lg1][lt][LX]);
				if ((i-=2) <= 0) {
					endtmesh();
					lt -= incr;
					i = 256;
					bgntmesh();
				}
			}
			endtmesh();
			break;
		}
	} else {
		for (lt = vtx->ltmin + incr; lt <= vtx->ltmax; lt += incr) {
			if (vtx->data[lg0][lt-incr] != VTX_VOID &&
				vtx->data[lg1][lt-incr] != VTX_VOID &&
				vtx->data[lg1][lt] != VTX_VOID &&
				vtx->data[lg0][lt] != VTX_VOID) {

				bgnpolygon();
				n3f(&vtx->pnt[lg0][lt-incr][NX]);
				v3f(&vtx->pnt[lg0][lt-incr][LX]);
				n3f(&vtx->pnt[lg1][lt-incr][NX]);
				v3f(&vtx->pnt[lg1][lt-incr][LX]);
				n3f(&vtx->pnt[lg1][lt][NX]);
				v3f(&vtx->pnt[lg1][lt][LX]);
				n3f(&vtx->pnt[lg0][lt][NX]);
				v3f(&vtx->pnt[lg0][lt][LX]);
				endpolygon();
			}
		}
	}
}



render_putdefaults(ri)

struct renderinfo *ri;
{
	return(0);
}



render_getdefaults(ri)

struct renderinfo *ri;
{
	char *cp;

	strcpy(ri->name, "Echo Demo Rendered Display");
	ri->color_background = 0x00000034;
	ri->color_meterlines = 0xff0000ff;
	ri->color_erase = 0xff00ff00;
	ri->position_x = 408;
	ri->position_y = 123;
	ri->size_x = 838;
	ri->size_y = 838;
	ri->clipx = .17;
	ri->clipy = .17;
	ri->clipz = .17;
	ri->panx = 0.;
	ri->panz = 0.;
	ri->zoom = 1.;
	ri->dist = 1.;
	ri->azim = 0;
	ri->inc = 900;
	ri->twist = 0;
	ri->ltresolution = 1;
	ri->lgresolution = 1;
	ri->index_lt = -1;
	ri->index_lg = -1;
	ri->wire_transform = TRUE;
	ri->ortho_flag = FALSE;
	ri->patch_mode = TRIANGLE_MESH;
	ri->render_mode = RENDER_GOURAUD;
	ri->material_index = 2;
	material_index_save = ri->material_index;
	ri->model_index = 1;
	ri->doublebuffer = FALSE;
	ri->render_firsttime = TRUE;
	return(0);
}



render_meter(ri)

struct renderinfo *ri;
{
	short val;					/* qread event value */
	float margin = 20.;			/* margin around window frame, for sanity */
	int cursor_x, cursor_y;		/* current cursor, in screen space */
	int cursor_lt, cursor_lg;	/* current cursor, in data space */
	int offset_lt, offset_lg;	/* screenspace to window space */
	float factor_lt, factor_lg;	/* data_space / screen_pixels */
	long dev;					/* event device */
	int r;						/* radius of the selected point */
	Boolean flag = TRUE;		/* loop exit flag, on LEFTMOUSE */
	short lgoffset;				/* used to invert X mouse action */
	struct Vertex *vtx = ri->vtx;

	help_event("Move mouse, middle picks, left stops", (Device)REDRAW, 0);

	winset(ri->gid);
	winpop();
	drawmode(OVERDRAW);
	mapcolor(1, 255, 0, 0);
	linewidth(2);
	frontbuffer(TRUE);
	getorigin(&ri->position_x, &ri->position_y);
	getsize(&ri->size_x, &ri->size_y);
	viewport(0, ri->size_x - 1, 0, ri->size_y - 1);
	lsetdepth(0x0, 0x7ffeff);
	render_transform(ri, TRUE, FALSE);

	/* transform the valuator space to data index space */
	offset_lg = ri->position_x + margin;
	offset_lt = ri->position_y + margin;
	factor_lt = (float)(vtx->ltmax - vtx->ltmin) / (ri->size_y - 2. * margin);
	factor_lg = (float)(vtx->lgmax - vtx->lgmin) / (ri->size_x - 2. * margin);

	/* display the surface cursor */
	cursor_x = getvaluator(MOUSEX);
	cursor_y = getvaluator(MOUSEY);
	if (!(ri->vtx->gs->flags & FLAG_CARTESIAN)) {
		cursor_lg = vtx->lgmax - (cursor_x - offset_lg) * factor_lg;
		cursor_lt = (cursor_y - offset_lt) * factor_lt + vtx->ltmin;
	} else {
		cursor_lg = vtx->lgmin + (cursor_x - offset_lg) * factor_lg;
		cursor_lt = (cursor_y - offset_lt) * factor_lt + vtx->ltmin;
	}
	cursor_lg = MAX(MIN(cursor_lg, vtx->lgmax), vtx->lgmin);
	cursor_lt = MAX(MIN(cursor_lt, vtx->ltmax), vtx->ltmin);
	render_cursor(ri, cursor_lt, cursor_lg, TRUE);

	/* column headings */
	if (ri->vtx->gs->flags & FLAG_CARTESIAN) {
		puts(" lt   lg      x         y         z    ");
		puts("---- ---- --------- --------- ---------");
	} else {
		puts(" lt   lg     r      theta       x         y         z    ");
		puts("---- ---- ------- --------- --------- --------- ---------");
	}
	qdevice(MIDDLEMOUSE);
	qdevice(MOUSEX);
	qdevice(MOUSEY);
	while (flag) {
		switch (qread(&val)) {
		case LEFTMOUSE:
			flag = FALSE;
			break;
		case MIDDLEMOUSE:
			if (val) {
				ri->index_lt = cursor_lt;
				ri->index_lg = cursor_lg;
				render_meter_display(ri, cursor_lt, cursor_lg);
			}
			break;
		case MOUSEX:
		case MOUSEY:
			while ((dev = qtest()) == MOUSEX || dev == MOUSEY) {
				qread(&val);
			}
			render_cursor(ri, cursor_lt, cursor_lg, FALSE);
			cursor_x = getvaluator(MOUSEX);
			cursor_y = getvaluator(MOUSEY);
			if (!(ri->vtx->gs->flags & FLAG_CARTESIAN)) {
				cursor_lg = vtx->lgmax - (cursor_x - offset_lg) * factor_lg;
				cursor_lt = (cursor_y - offset_lt) * factor_lt + vtx->ltmin;
			} else {
				cursor_lg = vtx->lgmin + (cursor_x - offset_lg) * factor_lg;
				cursor_lt = (cursor_y - offset_lt) * factor_lt + vtx->ltmin;
			}
			cursor_lg = MAX(MIN(cursor_lg, vtx->lgmax), vtx->lgmin);
			cursor_lt = MAX(MIN(cursor_lt, vtx->ltmax), vtx->ltmin);
			render_cursor(ri, cursor_lt, cursor_lg, TRUE);
			break;
		}
	}
	unqdevice(MOUSEX);
	unqdevice(MOUSEY);
	unqdevice(MIDDLEMOUSE);
	render_cursor(ri, cursor_lt, cursor_lg, FALSE);
	drawmode(NORMALDRAW);
	frontbuffer(FALSE);
	linewidth(1);
}



render_cursor(ri, cursor_lt, cursor_lg, flag)

struct renderinfo *ri;
int cursor_lt, cursor_lg;
Boolean flag;
{
	int lg, lt;
	int limit;
	struct Vertex *vtx = ri->vtx;

	color((flag)?(1):(0));

	bgnline();
	lt = MAX(cursor_lt - 4, vtx->ltmin);
	limit = MIN(cursor_lt + 4, vtx->ltmax);
	lg = cursor_lg;
	v3f(&vtx->pnt[lg][lt][LX]);
	for (++lt; lt <= limit; ++lt) {
		v3f(&vtx->pnt[lg][lt][LX]);
	}
	endline();

	bgnline();
	lg = MAX(cursor_lg - 6, vtx->lgmin);
	limit = MIN(cursor_lg + 6, vtx->lgmax);
	lt = cursor_lt;
	v3f(&vtx->pnt[lg][lt][LX]);
	for (++lg; lg <= limit; ++lg) {
		v3f(&vtx->pnt[lg][lt][LX]);
	}
	endline();
}


render_meter_display(ri, lt, lg)

struct renderinfo *ri;
int lt, lg;
{
	GSPEC *gs = ri->vtx->gs;
	int r;
	real x, y, z, t, radius;

	if (gs->flags & FLAG_CARTESIAN) {
		if ((r = GETR(gs, lt, lg)) != VOID) {
			x = (real)gs->lgincr * 1e-6 * (real)lg;
			y = (real)gs->ltincr * 1e-6 * (real)lt;
			z = (real)r * 1e-6;
#if 0
			printf("%4.1d %4.1d % 8.6f % 8.6f % 8.6f\n",
				lt, gs->lgmax - lg, x, y, z);
#else
			printf("%4.1d %4.1d % 8.6f % 8.6f % 8.6f\n",
				lt, lg, x, y, z);
#endif
		} else {
			printf("%4.1d %4.1d % 8.6f % 8.6f      VOID\n",
				lt, lg, x, y);
		}
	} else {
		if ((r = GETR(gs, lt, lg)) != VOID) {
			radius = (real)r * 1e-6;
			t = URTOR(gs->lgincr * lg);
			x = sin(t) * radius;
			y = (real)gs->ltincr * 1e-6 * (real)lt;
			z = cos(t) * radius;
			t = URTOD(gs->lgincr * lg);
			printf("%4.1d %4.1d %7d  % 8.2f % 8.6f % 8.6f % 8.6f\n",
				lt, lg, r, t, x, y, z);
		} else {
			y = (real)gs->ltincr * 1e-6 * (real)lt;
			t = URTOD(gs->lgincr * lg);
			printf("%4.1d %4.1d    VOID  % 8.2f      VOID % 8.6f      VOID\n",
				lt, lg, t, y);
		}
	}
}



function_radius(ri)

struct renderinfo *ri;
{
	struct Vertex *vtx = ri->vtx;
	GSPEC *gs = vtx->gs;
	int lg, lt, r;

	for (lg = vtx->lgmin; lg <= vtx->lgmax; ++lg) {
		for (lt = vtx->ltmin; lt <= vtx->ltmax; ++lt) {
			r = GETR(gs, lt, lg);
			if (r != VOID) {
				vtx->data[lg][lt] = (r/50)%256;
			}
		}
	}
}


function_smoothness(ri)

struct renderinfo *ri;
{
	struct Vertex *vtx = ri->vtx;
	GSPEC *gs = vtx->gs;
	int lg, lt;
	int sum;
	int diff;

	for (lg = vtx->lgmin+1; lg <= vtx->lgmax-1; ++lg) {
		for (lt = vtx->ltmin+1; lt <= vtx->ltmax-1; ++lt) {
			sum = 0;
			sum += GETR(gs, lt-1, lg-1);
			sum += GETR(gs, lt-1, lg);
			sum += GETR(gs, lt-1, lg+1);
			sum += GETR(gs, lt, lg-1);
			sum += GETR(gs, lt, lg+1);
			sum += GETR(gs, lt+1, lg-1);
			sum += GETR(gs, lt+1, lg);
			sum += GETR(gs, lt+1, lg+1);
			diff = abs((sum/8)-GETR(gs, lt, lg));
			vtx->data[lg][lt] = MIN(diff, 255);
		}
	}
}
