/*
  new
  9/18/1992
  ScianHwuFiles.c
  ReadSTFFile routine

  Tzong-Yow Hwu

  The file format is:
  
      RANK <rank>
      DIMENSIONS <xdim> <ydim> ....
      SCALAR|VECTOR <nComponents>
      [ BOUNDS     <xmin> <xmax> <ymin> <ymax> .... ] {required when no grid}
      [ INTERLACED|NONINTERLACED ]  { required only when fieldType is vector }  
      [ ORDER COLUMN|ROW ]
      [ NAME <datasetName> ]
      [ TIME <time> ]
      DATA
      <values> ......
      .
      .
      .
      VECTOR <nComponents>
      INTERLACED|NONINTERLACED
      [ ORDER COLUMN|ROW ]
      GRID
      <values> ......
      .
      .
      .
      END
      .
      .
      .
      .

  Grid is optional for any dataset, if a grid is not present then regular
  dataform is assumed.  The order of grid and data for a dataset is 
  arbitrary in the data file.  However, rank and dimensions are required 
  before either value is processed and the rank must precede dimensions. 
  If no grid is specified, then bounds is required.  
  If no order specified for a set of values, then the default of 
  column major is assumed.
  If no dataset name is specified for the dataset, then it inherits from the
  previous dataset in the file.  If this happens to the first dataset, then
  it takes the name of the file.
*/

#include "Scian.h"
#include "ScianTypes.h"
#include "ScianArrays.h"
#include "ScianIcons.h"
#include "ScianWindows.h"
#include "ScianObjWindows.h"
#include "ScianVisWindows.h"
#include "ScianVisObjects.h"
#include "ScianControls.h"
#include "ScianColors.h"
#include "ScianDialogs.h"
#include "ScianFiles.h"
#include "ScianFileSystem.h"
#include "ScianLists.h"
#include "ScianPictures.h"
#include "ScianErrors.h"
#include "ScianTimers.h"
#include "ScianDatasets.h"
#include "ScianFilters.h"
#include "ScianTextBoxes.h"
#include "ScianTitleBoxes.h"
#include "ScianButtons.h"
#include "ScianSliders.h"
#include "ScianScripts.h"
#include "ScianIDs.h"
#include "ScianVisContours.h"
#include "ScianStyle.h"
#include "ScianSpaces.h"
#include "ScianMethods.h"
#include "ScianObjFunctions.h"
#include "ScianHwuFiles.h"

#define MAXHEADER 9
#define MAXNAMELEN 256

/* field type */
#define SCALAR 0
#define VECTOR 1

/* vector field data permutation type */
#define NONINTERLACED 0
#define INTERLACED 1

#define COLUMNMAJOR 0
#define ROWMAJOR 1

/* for datasetType */
#define ERROR -1
#define END 0
#define DATA 1
#define GRID 2

/* for specFlag */
#define ISTYPE 1
#define ISDIMS 2
#define ISBOUNDS 4
#define ISNAME 8
#define ISORDER 16
#define ISRANK 32 
#define ISPERMUTATION 64 
#define ISTIME 128

typedef struct  {
     int fieldType;   /* SCALAR or VECTOR */
     long *dimensions; /* xdim, ydim, zdim .... */
     real *bounds;  /* bounds array */
     int nBounds; /* number of bounds provided by the user if any */
     char datasetName[MAXNAMELEN];
     int order; /* COLUMNMAJOR(default) or ROWMAJOR */
     int rank;
     int permutation; /* INTERLACED or NONINTERLACED */
     int nComponents;  /* number of components for vector data set */
     real time;
     } spec;

#ifdef PROTO
static int parseHeader(FILE *inFile, int *specFlag, spec *datasetSpec)
#else
static int parseHeader(inFile, specFlag, datasetSpec)
FILE *inFile;
int *specFlag;
spec *datasetSpec;
#endif
{
  int datasetType = END; /* DATA, GRID, ERROR, or END */
  int headerLine = 0;
  char dirStr[20]; 
  int counter;
  int c;

  for (SkipBlanks(inFile); (c = getc(inFile)) >= 0 && headerLine < MAXHEADER;
       SkipBlanks(inFile))
  {
    ungetc(c, inFile);

    if (1 == fscanf(inFile, "%s", dirStr))
      {
        if (*dirStr == '#')
          {
	    /* a comment */
	    ReadLn(inFile);
	    continue;
	  }



        if (0 == strcmp2(dirStr, "RANK"))
          {
	    if ( *specFlag & ISRANK )
	      {
                FileFormatError("ReadSTFFile", "Error reading header: double RANKs");
		datasetType = ERROR;
	        break;
	      }
	    SkipBlanks(inFile);
            if ( 1 != fscanf( inFile, "%d", &(datasetSpec->rank) ) )
	      {
	        FileFormatError("ReadSTFFile", "Error reading header: RANK value");
		datasetType = ERROR;
	        break;
	      }
            if ( datasetSpec->rank < 1  )
              /* rank must be greater than or equal to 1 */
              {
                FileFormatError("ReadSTFFile", "Error reading header: RANK value");
		datasetType = ERROR;
	        break;
              }

	     ReadLn(inFile);
	     headerLine++;
	     *specFlag |= ISRANK;
             continue; 
           }



        if (0 == strcmp2(dirStr, "TIME"))
          {
	    if ( *specFlag & ISTIME )
	      {
                FileFormatError("ReadSTFFile", "Error reading header: double TIMEs");
		datasetType = ERROR;
	        break;
	      }
	    SkipBlanks(inFile);
            if ( 1 != fscanf( inFile, "%g", (float *) &(datasetSpec->time) ) )
	      {
	        FileFormatError("ReadSTFFile", "Error reading header: TIME value");
		datasetType = ERROR;
	        break;
	      }

	     ReadLn(inFile);
	     headerLine++;
	     *specFlag |= ISTIME;
             continue; 
           }



         if (0 == strcmp2(dirStr, "SCALAR"))
	   {
	     if ( *specFlag & ISTYPE )
	       {
                 FileFormatError("ReadSTFFile", "Error reading header: double field types");
		 datasetType = ERROR;
		 break;
	       }

	     headerLine++;

	     ReadLn(inFile);
	     datasetSpec->fieldType = SCALAR;
	     datasetSpec->nComponents = 0;
	     *specFlag |= ISTYPE;	  
	     continue;
	   }


	 if (0 == strcmp2(dirStr, "VECTOR"))
	   {
	     if ( *specFlag & ISTYPE )
	       {
                 FileFormatError("ReadSTFFile", "Error reading header: double field types");
		 datasetType = ERROR;
		 break;
	       }

	     SkipBlanks(inFile);
             if ( 1 != fscanf(inFile, "%d", &(datasetSpec->nComponents)) )
	       {
		 FileFormatError("ReadSTFFile", "Error reading header: vector component number missing");
		 datasetType = ERROR;
		 break;
	       }
             if ( datasetSpec->nComponents < 1 )
               {
                 FileFormatError("ReadSTFFile", "Error reading header: invalid vector component value");
		 datasetType = ERROR;
		 break;
               }

	     headerLine++;

	     ReadLn(inFile);
	     datasetSpec->fieldType = VECTOR;
	     *specFlag |= ISTYPE;	  
	     continue;
	   }



	 if (0 == strcmp2(dirStr, "INTERLACED"))
	   {
	     if ( *specFlag & ISPERMUTATION )
	       {
                 FileFormatError("ReadSTFFile", "Error reading header: double permutation types");
		 datasetType = ERROR;
		 break;
	       }

	     headerLine++;
	     ReadLn(inFile);
	     datasetSpec->permutation = INTERLACED;
	     *specFlag |= ISPERMUTATION;
	     continue;
	   }



	 if (0 == strcmp2(dirStr, "NONINTERLACED"))
	   {
	     if ( *specFlag & ISPERMUTATION )
	       {
                 FileFormatError("ReadSTFFile", "Error reading header: double permutation types");
		 datasetType = ERROR;
		 break;
	       }

	      headerLine++;
	      ReadLn(inFile);
	      datasetSpec->permutation = NONINTERLACED;
	      *specFlag |= ISPERMUTATION;
	      continue;
	   }



	 if (0 == strcmp2(dirStr, "NAME"))
	   {
             if ( *specFlag & ISNAME )
	       {
		 FileFormatError("ReadSTFFile", "Error reading header: double NAME");
		 datasetType = ERROR;
		 break;
               }

	     headerLine++;
	     SkipBlanks(inFile);

	     if ( 1 != fscanf(inFile, "%s", datasetSpec->datasetName))
	       {
		 FileFormatError("ReadSTFFile", "Error reading header: NAME FORMAT");
		 datasetType = ERROR;
		 break;
               }
	     ReadLn(inFile);
             *specFlag |= ISNAME;
	     continue;
	   }




	 if (0 == strcmp2(dirStr, "ORDER"))
	   {
             if ( *specFlag & ISORDER )
	       {
		 FileFormatError("ReadSTFFile", "Error reading header: double ORDER");
		 datasetType = ERROR;
		 break;
               }

	     headerLine++;
	     SkipBlanks(inFile);

	     if ( 1 != fscanf(inFile, "%s", dirStr))
	       {
		 FileFormatError("ReadSTFFile", "Error reading header: ORDER FORMAT");
		 datasetType = ERROR;
		 break;
               }
	     if ( 0 == strcmp2(dirStr, "COLUMN") )
	       {
		 datasetSpec->order = COLUMNMAJOR;
               }
	     else if ( 0 == strcmp2(dirStr, "ROW") )
	       {
		 datasetSpec->order = ROWMAJOR;
               }
	     else
	       {
		 FileFormatError("ReadSTFFile", "Error reading header: ORDER FORMAT");
		 datasetType = ERROR;
		 break;
               }

	     ReadLn(inFile);
             *specFlag |= ISORDER;
	     continue;
	   }

	 if (0 == strcmp2(dirStr, "DIMENSIONS"))
	   {
	     if ( !(*specFlag & ISRANK) )
	       {
		 FileFormatError("ReadSTFFile", "Error reading header: RANK must precede DIMENSIONS");
		 datasetType = ERROR;
		 break;
               }
		 
             if ( *specFlag & ISDIMS )
	       {
		 FileFormatError("ReadSTFFile", "Error reading header: double DIMENSIONS");
		 datasetType = ERROR;
		 break;
               }

	     headerLine++;

            /* allocate storage for dimensions*/
            if ( !(datasetSpec->dimensions = (long *) malloc(datasetSpec->rank * sizeof(long))) )
              {
	        FileFormatError("ReadSTFFile", "Unable to allocate memory");
		datasetType = ERROR;
		break;
              }

	     *specFlag |= ISDIMS;

	     /* read all the dimensions */
	     for(counter = 0; counter < datasetSpec->rank; counter++)
		{
		  SkipBlanks(inFile);
		  if ( 1 != fscanf(inFile, "%ld", datasetSpec->dimensions + counter) )
		    {
		      FileFormatError("ReadSTFFile", "Error reading header: DIMENSIONS FORMAT");
		      datasetType = ERROR;
		      break;
		    }
                }

             if (datasetType == ERROR)
	       break;

	     ReadLn(inFile);
	     continue;
	   }



	 if (0 == strcmp2(dirStr, "BOUNDS"))
	   {
	     if ( *specFlag & ISBOUNDS )
	       {
		 FileFormatError("ReadSTFFile", "Error reading header: double BOUNDS");
		 datasetType = ERROR;
		 break;
               }

	     headerLine++;

             /* allocate storage for bounds*/
             if ( !(datasetSpec->bounds = (real *) malloc(sizeof(real))) )
               {
	         FileFormatError("ReadSTFFile", "Unable to allocate memory");
		 datasetType = ERROR;
		 break;
               }

	     *specFlag |= ISBOUNDS;

	     /* read all the bounds */
	     for( counter = 0, SkipBlanks(inFile);
		  (c = getc(inFile))!= '\n' && c >= 0;
		  counter++, SkipBlanks(inFile)
		)
	        {
		  ungetc(c, inFile);
		  if ( !( datasetSpec->bounds = (real *) realloc(datasetSpec->bounds, (counter+1)*sizeof(real))) )
		    {
	              FileFormatError("ReadSTFFile", "Unable to allocate memory");
		      datasetType = ERROR;
		      break;
		    }
		  if ( 1 != fscanf(inFile, "%f", (float *) (datasetSpec->bounds + counter)) )
		    {
		      FileFormatError("ReadSTFFile", "Error reading header: BOUNDS FORMAT");
		      datasetType = ERROR;
		      break;
		    }
                 }
             
             if (datasetType == ERROR)
	       /* this break gets out of the outer for loop */
	       break;

	     /* the number of bounds must be an even number */
	     if ( counter == 0 || counter % 2 != 0 )
	       {
		 FileFormatError("ReadSTFFile", "Error reading header: BOUNDS FORMAT");
		 datasetType = ERROR;
		 break;
	       }


	     datasetSpec->nBounds = counter;
	     continue;
	   }



	 if (0 == strcmp2(dirStr, "DATA"))
	   {
	     headerLine++;
	     ReadLn(inFile);
	     datasetType = DATA;
	     break;
	   }



	 if (0 == strcmp2(dirStr, "GRID"))
	   {
	     headerLine++;
	     ReadLn(inFile);
	     datasetType = GRID;
	     break;
	   }



	 if (0 == strcmp2(dirStr, "END"))
	   {
	     ReadLn(inFile);
	     break;
	   }



	 FileFormatError("ReadSTFFile", "Error reading header: unkown spec");
	 datasetType = ERROR;
	 break;

       }
     else
       {
	 /* a blank line*/
	 ReadLn(inFile);
	 continue;
       }
  }




  if ( datasetType == END )
    {
      if ( headerLine != 0 )
        {
          FileFormatError("ReadSTFFile",
	  "Error reading header: Incomplete header spec for dataset");
	  datasetType = ERROR;
        }
      else
	{
	  return datasetType; 
	}
    }



  if ( 
       (datasetType == ERROR) || 

       /* rank and dimensions are always required */
       ( !(*specFlag & ISRANK) || !(*specFlag & ISDIMS) ) ||

       /* for grid data set, permutation and fieldtype are required */
       /* and its fieldtype must be vector */
       ( (datasetType == GRID) && ( !(*specFlag & ISPERMUTATION) || !(*specFlag & ISTYPE) || (datasetSpec->fieldType != VECTOR) ) ) ||
       
       /* for field data set, fieldtype is required */
       /* and for vector dataset, permutation is required */
       ( (datasetType == DATA) && ( !(*specFlag & ISTYPE) || ( (datasetSpec->fieldType == VECTOR) && !(*specFlag & ISPERMUTATION) ) ) ) 

     )
    {
      switch ( datasetType )
	{
	  case ERROR:
	    /* error message was broadcasted */
	    break;
          case DATA:
	    if ( !(*specFlag & ISRANK) )
	      {
                FileFormatError("ReadSTFFile",
		"Error reading header: RANK missing before DATA");
	      }
            else if ( !(*specFlag & ISDIMS) )
	      {
                FileFormatError("ReadSTFFile",
		"Error reading header: DIMENSIONS missing before DATA");
	      }
	    else if ( !(*specFlag & ISTYPE) )
	      {
                FileFormatError("ReadSTFFile",
		"Error reading header: TYPE missing before DATA");
	      }
            else if ( !(*specFlag & ISPERMUTATION) )
	      {
                FileFormatError("ReadSTFFile",
		"Error reading header: PERMUTATION missing before vector DATA");
	      }
	    break;
          case GRID:
	    if ( !(*specFlag & ISRANK) )
	      {
                FileFormatError("ReadSTFFile",
		"Error reading header: RANK missing before GRID");
	      }
            else if ( !(*specFlag & ISDIMS) )
	      {
                FileFormatError("ReadSTFFile",
		"Error reading header: DIMENSIONS missing before GRID");
	      }
	    else if ( !(*specFlag & ISTYPE) )
	      {
                FileFormatError("ReadSTFFile",
		"Error reading header: TYPE missing before GRID");
	      }
            else if ( !(*specFlag & ISPERMUTATION) )
	      {
                FileFormatError("ReadSTFFile",
		"Error reading header: PERMUTATION missing before GRID");
	      }
	    else if ( datasetSpec->fieldType != VECTOR )
	      {
                FileFormatError("ReadSTFFile",
		"Error reading header: GRID must be vector type");
	      }
	    break;
          default:
              FileFormatError("ReadSTFFile",
	      "Error reading header: header format error");
	    break;
	}
      datasetType = ERROR;
    }


  return datasetType;
} /* parseHeader */



#ifdef PROTO
static real *readValue(FILE *inFile, spec *datasetSpec, ObjPtr dataset)
#else
static real *readValue(inFile, datasetSpec, dataset)
FILE *inFile;
spec *datasetSpec;
ObjPtr dataset;
#endif
/* return pointer to bounds of the dataset just read */
/* return null if an error occurred */
{

  real *bounds;
  long *index;
  int counter;
  int nComponents;

  
  /* determine number of components first */
  if ( datasetSpec->fieldType == SCALAR )
    {
      nComponents = 1;
    }
  else
    {
      nComponents = datasetSpec->nComponents;
    }



  /* alloc bounds array and init it to contain 0 values */
  if ( !(bounds = (real *) malloc( 2*nComponents*sizeof(real) )) )
    {
      FileFormatError("ReadSTFFile", "Unable to allocate memory");
      return NULL;
    }
  for ( counter = 0; counter < 2*nComponents; counter++)
     {
       bounds[counter] = 0.0;
     }


  /* alloc index array and init it to contain 0 values */
  if ( !(index = (long *) malloc( datasetSpec->rank*sizeof(long) )) )
    {
      FileFormatError("ReadSTFFile", "Unable to allocate memory");
      return 0;
    }
  for ( counter = 0; counter < datasetSpec->rank; counter++)
     {
       index[counter] = 0L;
     }


    /* prepare to read in the field values from file */
    if ( !SetCurField(FIELD1, dataset) ) 
      {
        free(index);
	free(bounds);
	return NULL;
      }

    /* adjust the index and read in the corresponding field value */ 
    {
      real curData;
      char dataStr[32];
      int n; /* counter */

      
      counter = 0;

      if ( datasetSpec->fieldType == SCALAR || datasetSpec->permutation == NONINTERLACED )
        {
         for ( n = 0; n < nComponents; n++ )
           {
             counter = 0;
             for(SkipBlanks(inFile); 1 == fscanf(inFile, "%s", dataStr); SkipBlanks(inFile))
	     {
	        if ( !ParseReal( &curData, dataStr ) )
	          {
                    FileFormatError("ReadSTFFile", "Invalid Data Value");
	            /* clean up and return an NULL ptr */
                    free(index);
                    free(bounds);
	            return NULL;
                  }

	        PutFieldComponent(FIELD1, n, index, (real) curData);

		/* look for bounds */
		if ( curData < bounds[2*n] )
		  {
		    bounds[2*n] = curData;
                  }
                else if ( curData > bounds[2*n+1] )
		  {
		    bounds[2*n+1] = curData;
		  }

	        /* adjust the index according to column or row major */
	        if( datasetSpec->order == COLUMNMAJOR)
	          {
	            for(counter = 0; counter < datasetSpec->rank; counter++)
	              {
	                index[counter]++;
                        if (index[counter] == datasetSpec->dimensions[counter])
		          {
		           index[counter] = 0;
		           continue;
		          }
                        break;
                      }
                    if (counter == datasetSpec->rank)
	              {
	                /* This signifies the nth component data is all read */
	                break;
	              }
	          }
                else
	          {
	            for(counter = datasetSpec->rank - 1; counter > -1 ; counter--)
	              {
	                index[counter]++;
                        if (index[counter] == datasetSpec->dimensions[counter])
		          {
		           index[counter] = 0;
		           continue;
		          }
                        break;
                      }
                    if (counter == -1)
	              {
	                /* This signifies the nth component value is all read */
	                break;
	              }
	          }

	     } /* for scanf */

           }  /* for ( n = 0; n < nComponents; n++ ) */

         /* this indicates that the data number is less than dimensions */
         if ( ( datasetSpec->fieldType == VECTOR && n != nComponents) ||
	      ( datasetSpec->order == COLUMNMAJOR && counter != datasetSpec->rank )||
	      ( datasetSpec->order == ROWMAJOR && counter != -1 ) )
           {
            FileFormatError("ReadSTFFile", "Incosistent Data File");
            free(index);
	    free(bounds);
            return NULL;
           }

        } /* if fieldType == SCALAR || permutation == NONINTERLACED */
      else /* fieldType is VECTOR and permutation is INTERLACED */ 
        {
           n = 0;
           counter = 0;
           for(SkipBlanks(inFile); 1 == fscanf(inFile, "%s", dataStr); SkipBlanks(inFile))
	   {
	      if ( !ParseReal( &curData, dataStr ) )
	        {
                  FileFormatError("ReadSTFFile", "Invalid Data Value");
	          /* clean up and return NULL */
                  free(index);
		  free(bounds);
	          return NULL;
                }

	      PutFieldComponent(FIELD1, n, index, (real) curData);


	      /* look for bounds */
	      if ( curData < bounds[2*n] )
	        {
	          bounds[2*n] = curData;
                }
              else if ( curData > bounds[2*n+1] )
		{
		  bounds[2*n+1] = curData;
		}

	      n++;

	      if ( n == nComponents ) /* time to update index */
                {
	         /* adjust the index according to column or row major */
	         if( datasetSpec->order == COLUMNMAJOR)
	           {
	             for(counter = 0; counter < datasetSpec->rank; counter++)
	               {
	                 index[counter]++;
                         if (index[counter] == datasetSpec->dimensions[counter])
		           {
		            index[counter] = 0;
		            continue;
		           }
                         break;
                       }
                     if (counter == datasetSpec->rank)
	               {
	                 /* This signifies the field data is all read */
	                 break;
	               }
	           }
                 else
	           {
	             for(counter = datasetSpec->rank - 1; counter > -1 ; counter--)
	               {
	                 index[counter]++;
                         if (index[counter] == datasetSpec->dimensions[counter])
		           {
		            index[counter] = 0;
		            continue;
		           }
                         break;
                       }
                     if (counter == -1)
	               {
	                 /* This signifies the field data is all read */
	                 break;
	               }
	           }
 
                 n = 0;

                } /* if n == nComponents */

	     } /* for fscanf */

         /* this indicates that the data number is less than dimensions */
         if ( ( datasetSpec->order == COLUMNMAJOR && counter != datasetSpec->rank )||
	      ( datasetSpec->order == ROWMAJOR && counter != -1 ) )
           {
            FileFormatError("ReadSTFFile", "Incosistent Data File");
            free(index);
	    free(bounds);
            return NULL;
           }

        } /* else where fieldType is VECTOR and INTERLACED */


    }

    free(index);
    return bounds;

}/* readValue */


#ifdef PROTO
static ObjPtr ReadSTFFile(char *name)
#else
static ObjPtr ReadSTFFile(name)
char *name;
#endif
{
    int datasetType; /* DATA, GRID, END or ERROR */
    spec datasetSpec;
    int specFlag = 0;

    FILE *inFile;

    real *gridBounds = NULL;
    real *dataMinmaxes = NULL;

    int isGrid = 0, isData = 0;
    int gridComponents; /* the number of components for the grid if any */
    int counter;
    int more = 1;

    ObjPtr dataForm;
    ObjPtr gridDataset;
    ObjPtr retVal;

    inFile = fopen(name, "r");
    if ( !inFile )
      {
        Error("ReadSTFFile", OPENFILEERROR, name);
	return NULLOBJ;
      }
     

    /* set default dataset specs */
    datasetSpec.rank = 0;
    datasetSpec.nComponents = 0;
    datasetSpec.order = COLUMNMAJOR;
    strcpy(datasetSpec.datasetName, name);



    while( more ) 
      {
        /* parse header lines */
        datasetType = parseHeader(inFile, &specFlag, &datasetSpec);

        switch ( datasetType )
          {
	    case ERROR: 
	       FileFormatError("Dataset Name", datasetSpec.datasetName);
	       more = 0;
	       break;
            case GRID:
	       /* if isGrid then error, double grid */
	       if ( isGrid )
		 {
		   FileFormatError("ReadSTFFile", "Multiple Grid Spec");
	           FileFormatError("Dataset Name", datasetSpec.datasetName);
	           more = 0;
	           break;
                 }

	       gridComponents = datasetSpec.nComponents;

	       /* create the grid dataset */
               gridDataset = NewStructuredDataset(datasetSpec.datasetName, datasetSpec.rank, datasetSpec.dimensions, gridComponents);

	       /* something wrong with grid value format */
               if ( !(gridBounds = readValue( inFile, &datasetSpec, gridDataset)))
		 {
	           FileFormatError("Dataset Name", datasetSpec.datasetName);
	           more = 0;
	           break;
		 }

	       /* reset specFlag */
               specFlag &= (~ISTYPE & ~ISPERMUTATION & ~ISORDER);
	       isGrid = 1;
               break;
            case DATA:
	       /* if isData then error, double data */
	       if ( isData )
	         {
		   FileFormatError("ReadSTFFile", "Multiple Data Spec");
	           FileFormatError("Dataset Name", datasetSpec.datasetName);
	           more = 0;
	           break;
                 }

	       /* create the dataset */
               retVal = NewStructuredDataset(datasetSpec.datasetName, datasetSpec.rank, datasetSpec.dimensions, datasetSpec.nComponents);

	       /* something wrong with data value format */
               if ( !(dataMinmaxes = readValue( inFile, &datasetSpec, retVal)) )
	         {
	           FileFormatError("Dataset Name", datasetSpec.datasetName);
	           more = 0;
	           break;
		 }
               else
		 {
		   /* has no use with dataMinmaxes */
		   free(dataMinmaxes);
		 }

	       /* reset specFlag */
               specFlag &= (~ISTYPE & ~ISPERMUTATION & ~ISORDER);
	       isData = 1;
	       break;
            case END:
	       /* if isData && isGrid then use curvilinear dataForm */
	       /* if isData && !isGrid then regular dataForm */
	       /* for this case, the bounds are required, error if no */
	       /* if !isData && isGrid the error */
	       /* if !isData && !isGrid then means end of file */
		       
	       if ( !isData && !isGrid )
	         /* end of dataset file reached */
		 {
		   more = 0;
		   break;
		 }

	       if ( !isData && isGrid )
		 {
		   FileFormatError("ReadSTFFile", "NO DATA");
	           FileFormatError("Dataset Name", datasetSpec.datasetName);
	           more = 0;
	           break;
		 }

	       if ( specFlag & ISBOUNDS )
		 {
		   int nBounds; /* number of bounds */

		   /* get rid of calculated grid bounds if any and 
		      use the bounds provided by the user */
		   /* determine the number of bounds */
		   if ( isGrid )
		     {
		       free(gridBounds);
		       gridBounds = NULL;
		       nBounds = 2 * gridComponents;
		     }
                   else
		     {
		       nBounds = 2 * datasetSpec.rank;
		     }

		   /* test if the user provide correct bounds */
		   if ( nBounds != datasetSpec.nBounds )
		     {
		       FileFormatError("ReadSTFFile", "BOUNDS format error");
	               FileFormatError("Dataset Name", datasetSpec.datasetName);
	               more = 0;
	               break;
		     }

                   gridBounds = datasetSpec.bounds;
                 }


	       if ( isData && isGrid )
                 /* for this case, bounds are optional */
		 {
		   dataForm = NewCurvilinearDataForm(datasetSpec.datasetName, datasetSpec.rank, datasetSpec.dimensions, gridBounds, gridDataset);
		 }
               else
		 {
		   /* isData && !isGrid */	
		   /* for this case bounds are required */
		   if ( !(specFlag & ISBOUNDS) )
		     {
		       FileFormatError("ReadSTFFile", "BOUNDS required for dataset with no grid");
	               FileFormatError("Dataset Name", datasetSpec.datasetName);
	               more = 0;
	               break;
		     }
		   dataForm = NewRegularDataForm(datasetSpec.datasetName, datasetSpec.rank, datasetSpec.dimensions, gridBounds);
		 }

	       SetDatasetForm(retVal, dataForm);
	       if ( specFlag & ISTIME )
		 {
		   RegisterTimeDataset(retVal, datasetSpec.time);
		 }
               else
		 {
	           RegisterDataset(retVal);
		 }

	       isGrid = 0;
	       isData = 0;
	       specFlag = 0;
	       free(datasetSpec.dimensions);
	       free(gridBounds);
	       gridBounds = NULL;

               /* set default dataset specs */
               datasetSpec.rank = 0;
               datasetSpec.nComponents = 0;
               datasetSpec.order = COLUMNMAJOR;

	       break;
            default:     
	       /* dummy case */
	       break;
          } /* switch */
      } /* while */

   if ( specFlag & ISDIMS )
     {
       free(datasetSpec.dimensions);
     }

   if ( specFlag & ISBOUNDS )
     {
       free(datasetSpec.bounds);
     }
   if ( isGrid )
     {
       free(gridBounds);
     }
   fclose(inFile);
   return NULLOBJ;
}



#ifdef PROTO
void InitHwuFiles(void)
#else
void InitHwuFiles()
#endif
{
  DefineFormat("STF", "stf", ReadSTFFile, 0);
}

#ifdef PROTO
void KillHwuFiles(void)
#else
void KillHwuFiles()
#endif
{
}


