/*ScianPictures.h
  External stuff and data structures for ScianPictures.c
*/

#define CARDBASIS	1	/*Basis for cardinal spline*/

#define SOLIDLINE	0	/*Solid line*/
#define DASHEDLINE	1	/*Line style for dashed line*/

/*Values for picture item type.  Must go from 0 up by ones*/
#define POLYGON		0	/*Type for a polygon*/
#define FRUSTUM		1	/*Conical frustum*/
#define SPHERE		2	/*Sphere*/
#define RECTMESH	3	/*Rectangular mesh*/
#define PLANARPOLYGON	4	/*Planar polygon*/
#define POLYGONS	5	/*A bunch of polygons*/
#if 0
#define POLYLINE	6	/*A polyline*/
#endif
#define NPICITEMTYPES	6	/*Number of pic types*/

/*Vertex tuple*/
typedef struct
    {
	float normal[3];	/*Normal at that vertex*/
	float position[3];	/*Position of that vertex*/
	float alpha;		/*Alpha transparency value*/
	int colorIndex;		/*Color within color table*/
    } VertexTuple, *VertexPtr;

/*Item in a picture*/
typedef struct picitem
    {
	struct picitem *next;	/*Next item in list*/
	short type;		/*Type of item.  Based on this, it's really
				  interpreted as something else*/
	short colorShading;	/*Color shading*/
	short lightShading;	/*Light shading*/
    } PicItem, *PicItemPtr;

/*Don't change the values of these defines!*/
/*Values for color shading*/
#define NOCOLOR		0
#define MONOCOLOR	1
#define SMOOTHCOLOR	2

/*Values for light shading*/
#define NOLIGHT		0
#define MONOLIGHT	1
#define SMOOTHLIGHT	2

/*Values for current color mode*/
#define CMODECMAP	0
#define CMODERGB	1

/*Polygon*/
typedef struct
    {
	PicItem item;		/*The item part*/
	long nVertices;		/*Number of vertices*/
	VertexTuple vertices[1];/*The vertices*/
    } Polygon, *PolyPtr;

/*A bunch of polygons, all with the same colorShading and lightShading*/
typedef struct
    {
	PicItem item;		/*The item part*/
	Bool enclosed;		/*True iff totally enclosed*/
	PolyPtr polygons;	/*The polygons*/
	PolyPtr lastItem;
    } Polygons, *PolysPtr;

/*Conical frustum*/
typedef struct
    {
	PicItem item;		/*The item part*/
	float end1[3];		/*First endpoint*/
	float rad1;		/*Radius at first endpoint*/
	float end2[3];		/*Second endpoint*/
	float rad2;		/*Radius at second endpoint*/
	int colorIndex;		/*Color index*/
    } Frustum, *FrustumPtr;

/*Sphere*/
typedef struct
    {
	PicItem item;		/*The item part*/
	float center[3];	/*The center of the sphere*/
	float radius;		/*The radus of the sphere*/
	int colorIndex;		/*The color indices*/
    } Sphere, *SpherePtr;

/*Rectangular mesh*/
typedef struct
    {
	PicItem item;		/*The item part*/
	Bool inCenter;		/*Nodes are in center rather than around*/
	long xDim;		/*Y dimension*/
	long yDim;		/*X dimension*/
	VertexTuple vertices[1];/*The vertices*/
    } RectMesh, *RectMeshPtr;

/*Picture*/
typedef struct
    {
	Thing thing;
	PicItemPtr items;	/*The list of actual items*/
	PicItemPtr lastItem;
    } Picture, *PicPtr;

#ifdef PROTO
void AppendItemToPicture(PicPtr, PicItemPtr);
Bool AppendPolyToPicture(ObjPtr, int, VertexTuple[]);
Bool AppendPolyToPolys(PolysPtr, int, VertexTuple[]);
Bool AppendFrustumToPicture(ObjPtr, float end1[3], float, float end2[3], float, int);
void ConvertFrustumOntoPicture(ObjPtr, float end1[3], float, float end2[3], float, int, Bool);
Bool AppendSphereToPicture(ObjPtr, float center[3], float, int);
ObjPtr NewPicture(void);
void DrawPicture(ObjPtr, Bool, ObjPtr);
void GetPictureBounds(ObjPtr, real bounds[6]);
void InitPictures(void);
void KillPictures(void);
void ColorPictureByObject(ObjPtr, ObjPtr, Bool);
void UnColorPicture(ObjPtr);
RectMeshPtr NewRectMesh(long, long, Bool);
void SetPicColorShading(ObjPtr pic, int shading);
void SetPicLightShading(ObjPtr pic, int shading);
PolysPtr NewPolygons(void);
void SetRectMeshVPosition(RectMeshPtr, long, long, real, real, real);
void SetRectMeshVNormal(RectMeshPtr, long, long, real, real, real);
void GetRectMeshVPosition(RectMeshPtr, long, long, real *, real *, real *);
void GetRectMeshVNormal(RectMeshPtr, long, long, real *, real *, real *);
void SetRectMeshVColor(RectMeshPtr, long, long, real, real, real);
void SetRectMeshVColorIndex(RectMeshPtr, long, long, int);
void SetRectMeshVAlpha(RectMeshPtr, long, long, real);
void SetRectMeshCPosition(RectMeshPtr, long, long, real, real, real);
void SetRectMeshCNormal(RectMeshPtr, long, long, real, real, real);
void GetRectMeshCPosition(RectMeshPtr, long, long, real *, real *, real *);
void GetRectMeshCNormal(RectMeshPtr, long, long, real *, real *, real *);
void SetRectMeshCColor(RectMeshPtr, long, long, real, real, real);
void SetRectMeshCColorIndex(RectMeshPtr, long, long, int);
void SetRectMeshCAlpha(RectMeshPtr, long, long, real);
void InterpRectCenters(RectMeshPtr);
void InterpRectVertices(RectMeshPtr);
void CalcRectNormals(RectMeshPtr);
ObjPtr ConvertPicture(ObjPtr, ObjPtr);
ObjPtr ConvertOntoPicture(ObjPtr, ObjPtr, ObjPtr);
PolysPtr SphereToPolys(PicItemPtr, int);
void ColorItemsByObject(PicItemPtr curItem, ObjPtr colorObj, Bool);
void ColorItemByObject(PicItemPtr curItem, ObjPtr colorObj, Bool);
void ColorItemsWithIndex(PicItemPtr curItem, int);
void ColorItemWithIndex(PicItemPtr curItem, int);
void ColorPictureByPicture(ObjPtr, ObjPtr, ObjPtr);
PicItemPtr ColorItemsByItems(PicItemPtr destItem, ObjPtr owner, PicItemPtr curItem);
#else
void AppendItemToPicture();
Bool AppendPolyToPicture();
Bool AppendPolyToPolys();
Bool AppendFrustumToPicture();
void ConvertFrustumOntoPicture();
Bool AppendSphereToPicture();
ObjPtr NewPicture();
void DrawPicture();
void GetPictureBounds();
void InitPictures();
void KillPictures();
void ColorPictureByObject();
void UnColorPicture();
RectMeshPtr NewRectMesh();
void SetPicColorShading();
void SetPicLightShading();
PolysPtr NewPolygons();
void SetRectMeshVPosition();
void SetRectMeshVNormal();
void GetRectMeshVPosition();
void GetRectMeshVNormal();
void SetRectMeshVColor();
void SetRectMeshVColorIndex();
void SetRectMeshVAlpha();
void SetRectMeshCPosition();
void GetRectMeshCPosition();
void GetRectMeshCNormal();
void SetRectMeshCNormal();
void SetRectMeshCColor();
void SetRectMeshCColorIndex();
void SetRectMeshCAlpha();
void InterpRectCenters();
void InterpRectVertices();
void CalcRectNormals();
ObjPtr ConvertPicture();
ObjPtr ConvertOntoPicture();
PolysPtr SphereToPolys();
void ColorItemsByObject();
void ColorItemByObject();
void ColorItemsWithIndex();
void ColorItemWithIndex();
void ColorPictureByPicture();
PicItemPtr ColorItemsByItems();
#endif
