/* ****************************************************************
 *                         illum_mod.h
 * ****************************************************************
 * include file for the illumination models
 */
#ifndef ILLUM_H
#define ILLUM_H
#include "geo.h"

/* The material structure maintains the description of a material
 *  interface.  An interface between a conductor or dielectric and
 *  air is characterized by loading the properties of the material
 *  and an index of refraction of 1 for the outside material.  An
 *  interface between two materials is characterized by generating
 *  a pseudo-material as described in appendix III.5.1, Approxima-
 *  tion Methodology.
 *
 * In filling the material structure, the reflected direction is
 *  the 'outside' of the material.  That is, for an interface
 *  between air and glass, for example, the the reflected direction
 *  or 'outside' is air (Ar_spectral = NULL, nr=1.0), and the
 *  transmitted direction is glass (nt=1.5, etc.)
 *
 * Individual spectral components of the material are characterized
 *  by the sampled spectral values and a multiplier to scale these
 *  values (as discussed in appendix II.1.5, Selecting Ka, Ka, Ks,
 *  Ft and Fr)
 */

typedef struct {
    double  Ka_scale;       /* ambient multiplier */
    double  *Ka_spectral;   /* sampled ambient spectral curve */
    double  Kd_scale;       /* diffuse multiplier */
    double  *Kd_spectral;   /* sampled diffuse spectral curve */
    double  Ks_scale;       /* specular multiplier */
    double  *Ks_spectral;   /* sampled specular spectral curve */
    double  Kt_scale;       /* transmitted multiplier            */
    double  *Kt_spectral;   /* sampled specular transmitted      */
			    /*  curve.  The Kt_ properties are   */
			    /* used for the Whitted model only   */
    double  *Ar_spectral;   /* sampled filter spectral curve.    */
    double  *At_spectral;   /* sampled filter spectral curve     */
			    /*  These are used in the Hall model */
			    /*  only.  Filter attenuation is     */
			    /*  ignored if a NULL pointer is     */
			    /*  specified.                       */
    double  beta;           /* roughness indicator */
    double  (*D)();         /* microfacet distribution */
    double  D_coeff;        /* the coefficient for the           */
			    /*  microfacet distribution function */
			    /*  computed from by microfacet      */
			    /*  distribution init function       */
    double  (*G)();         /* geometric attenuation function */
    double  G_coeff;        /* the coefficient for the geometric */
			    /*  attenuation function (m_2 for    */
			    /*  the Sancer function, unused for  */
			    /*  the Torrance-Sparrow function)   */
    double  Ro;             /* average reflectance */
    double  nr;             /* index of refraction (incident) */
    double  nt;             /* index of refraction (transmitted) */
    double  k;              /* absorption coefficient */
    int     conductor;      /* flags the specular material as a  */
			    /*  conductor */
    int     transparent;    /* flags whether the material is     */
			    /*  transparent --- note, composite  */
			    /*  materials have a dielectric      */
			    /*  specular component but may not   */
			    /*  be transparent                   */
    int     r_is_air;       /* flags that the 'outside' or       */
			    /*  reflect side of the interface is */
			    /*  air                              */
} ILLUM_MATL;

typedef struct {
    double  I_scale;        /* illumination multiplier */
    double  *I_spectral;    /* sampled source spectral curve */
    POINT3  center;         /* center of the light source */
} ILLUM_LGT;

int     IM_init();
double  *IM_bouknight();
double  *IM_phong();
double  *IM_blinn();
double  *IM_whitted();
double  *IM_hall();
int     IM_exit();

#endif CLR_H
/* ************************************************************* */
