/****************************************************************************
 * phigs_ren.c
 * Author Joel Welling
 * Copyright 1990, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
#include <stdio.h>
#include <math.h>
#ifdef sun
#include <phigs/phigs.h>
#endif
#ifdef ultrix
#include <PHIGS/phigs.h>
#include <PHIGS/phigs_defs.h>
#endif
#ifdef VMS
#include "SYS$LIBRARY:PHIGS.H"
#include "SYS$LIBRARY:PHIGS_DEFS.H"
#endif
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "ren.h"
#include "assist.h"
 
/* Notes-
   -some implementations might use phigs_ws_type_create to get double buffering
   -some implementations may offer other HLHSR modes worth trying
*/

/* 
The following definitions are needed to work around variances in 
Phigs implementations between platforms.  The basic version was done
on a Sun under Sun Phigs.
*/
#if ( VMS || ultrix )  /* DEC Phigs case */
#define PRGB PCM_RGB
#define PDEFAULT_MEM_SIZE 0
#define PHIGS_HLHSR_MODE_ZBUFF PPHLHSR_MODE_ZBUFFER
#define PHIGS_HLHSR_ID_ZBUFF PPHLHSR_INDEX_ZBUFFER
#define cbndl_r( thisbundle ) (thisbundle).rgb.r
#define cbndl_g( thisbundle ) (thisbundle).rgb.g
#define cbndl_b( thisbundle ) (thisbundle).rgb.b
#else  /* Sun Phigs case */
#define cbndl_r( thisbundle ) (thisbundle).x
#define cbndl_g( thisbundle ) (thisbundle).y
#define cbndl_b( thisbundle ) (thisbundle).z
#endif


/* Needed external definitions */
extern char *malloc(), *realloc();

/* 
Variables to provide external hooks to the phigs workstation.
See ren_setup(...) for usage.  phigs_conn_id is used only there;
phigs_wsid is the workstation id used throughout the renderer.
*/
Pconnid phigs_conn_id;
Pint phigs_wsid= 1;

/* Phigs identifier constants */
#define ROOT -1
#define HEAD -2
#define VIEWINDEX 1

/* Symbols defined by setup and used in parsing attribute lists */
static Symbol depth_cue_symbol, color_symbol, backcull_symbol, 
	     text_height_symbol;

static Pvector3 *get_phigs_vector(avect)
Vector avect;
{
	Pvector3 *result;

	result = (Pvector3 *) malloc(sizeof(Pvector3));
	if (!result) 
		ger_fatal("get_phigs_vector: can't allocate a Pvector3!");
	result->x= (Pfloat)vector_x( avect );
	result->y= (Pfloat)vector_y( avect );
	result->z= (Pfloat)vector_z( avect );

	return(result);
}

static Ppoint3 *get_phigs_point(apoint)
Point apoint;
{
	Ppoint3 *result;

	result = (Ppoint3 *) malloc(sizeof(Ppoint3));
	if (!result) 
		ger_fatal("get_phigs_point: can't allocate a Ppoint3!");
	result->x= (Pfloat)point_x( apoint );
	result->y= (Pfloat)point_y( apoint );
	result->z= (Pfloat)point_z( apoint );

	return(result);
}

char *ren_def_material(material)
Material material;
/* This routine fills the slot for a material definition routine */
{
  ger_debug("ren_def_material: doing nothing");
  return((char *)0);
}

void ren_free_material(thismat)
Material thismat;
/* This routine fills the slot for a material deletion routine */
{
  ger_debug("ren_free_material: doing nothing");
}

/* This routine dumps a Phigs object. */
void ren_dump(thisgob)
Gob thisgob;
{
	int thisid;
 
	thisid= gob_idnum( thisgob );
	fprintf(stderr,"ren_dump: Dumping gob number %d\n",thisid);
fprintf(stderr,"Dump routine not yet implemented\n");

}

static Ppoint3 *get_loc_vtx( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine transfers vertices containing only locations to a vertex
data array.
*/
{
	Vertex vertex;
	Ppoint3 *vertexlist, *vtx_copy;
                              
	ger_debug("get_loc_vtx: getting vertex list, locs only.");
 
	if ( !(vertexlist= (Ppoint3 *)malloc( vertexcount*sizeof(Ppoint3) ) ))
		ger_fatal( "get_loc_vtx: unable to allocate %d Ppoint3's!",
			vertexcount );
 
	vtx_copy= vertexlist;
	while ( vertexcount-- )
		{
		vertex= first_vertex( vlist );
		vlist= rest_of_vertices( vlist );
		vtx_copy->x= (Pfloat)vertex_x(vertex);
		vtx_copy->y= (Pfloat)vertex_y(vertex);
		vtx_copy->z= (Pfloat)vertex_z(vertex);
		vtx_copy++;
		}
 
	return( vertexlist );
} 

static Pcobundl *get_vtx_clrs( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine returns a list of colors extracted from the nodes of
a vertex list.
*/
{
	Vertex vertex;
	Pcobundl *colorlist, *clr_copy;
                              
	ger_debug("get_vtx_clrs: getting vertex list colors.");
 
	if ( null( vertex_color( first_vertex(vlist) ) ) )
	  ger_fatal("get_vtx_clrs: vertex list has no colors!\n");

	if ( !(colorlist= (Pcobundl *)malloc( vertexcount*sizeof(Pcobundl) ) ))
		ger_fatal( "get_vtx_clrs: unable to allocate %d Pcobundl's!",
			vertexcount );
 
	clr_copy= colorlist;
	while ( vertexcount-- )
		{
		vertex= first_vertex( vlist );
		vlist= rest_of_vertices( vlist );
		cbndl_r(*clr_copy)= (Pfloat)color_red(vertex_color(vertex));
		cbndl_g(*clr_copy)= (Pfloat)color_green(vertex_color(vertex));
		cbndl_b(*clr_copy)= (Pfloat)color_blue(vertex_color(vertex));
		clr_copy++;
		}
 
	return( colorlist );
} 

static void set_average_color( vlist, count )
Vertex_list vlist;
int count;
/* This routine sets the drawing colors to match the average color of
 * the vertex list passed it.
 */
{
        float r, g, b, a;
	int index;

        ger_debug("set_average_color");

	ast_vlist_ave_color( vlist, count, &r, &g, &b, &a );
	index= ast_ctb_bestindex( r, g, b );
	psetlinecolourind( index );
	psetmarkercolourind( index );
	psetintcolourind( index );
	/* Since text doesn't use vlists, no need to set text color */
}

void ren_sphere()
/*
This routine constructs a sphere object.  The sphere is of radius 1,
and centered at the origin.
*/
{
	ger_debug("ren_sphere");

	ast_sphere(ren_mesh);
}

void ren_cylinder()
/*
This routine constructs a cylinder object.  The cylinder is of radius
1, and is aligned with the z axis.  The ends of the cylinder are at z=0.0
and z=1.0 .
*/
{
	ger_debug("ren_cylinder");

	ast_cylinder(ren_mesh);
}

void ren_torus(bigradius, smallradius)
float bigradius, smallradius;
/*
This routine constructs a torus object from the args passed it.  The 
arguments should be the major and minor radii of the torus respectively.  
The torus is drawn so that the hole lies along the z axis.
*/
{
	ger_debug("ren_torus");

	ast_torus( bigradius, smallradius, ren_mesh );
}

void ren_text(txtpoint,uvec,vvec,txtstring)
Point txtpoint;
Vector uvec, vvec;
char *txtstring;
/*
This routine constructs a text object from the args passed it.  
The argument list should contain a point, two vectors to define the 
u and v directions of the text plane, and a text string.
*/
{
        Ppoint3 *text_pt;
	Pvector3 dir[2];

	ger_debug("ren_text");

	text_pt= get_phigs_point(txtpoint);
	dir[0].x= vector_x(uvec);
	dir[0].y= vector_y(uvec);
	dir[0].z= vector_z(uvec);
	dir[1].x= vector_x(vvec);
	dir[1].y= vector_y(vvec);
	dir[1].z= vector_z(vvec);
	ptext3( text_pt, dir, txtstring );

	free( (char *)text_pt );
}

void ren_polyline( vlist, count )
Vertex_list vlist;
int count;
/*
This routine constructs a polyline object from the vertex list
passed it.  The parameter count is the number of vertices in vlist.
The object constructed is a single polyline, which may be concave but 
should not be self-intersecting.  The attributes (color and normal) at 
all vertices are assumed to be the same as those of the first vertex.  
*/
{
	Ppoint3 *vertices;
 
	ger_debug("ren_polyline");

	/* Set average color if present */
	if ( !null( vertex_color(first_vertex(vlist)) ) )
	  set_average_color(vlist,count);

	/* Allocate memory for the vertices */
	vertices= get_loc_vtx( vlist, count );

	/* 
	Create the polyline.  It goes into the already open structure.
	*/
	ppolyline3( count, vertices );

	/* Free memory */
	free( vertices );
}
 
void ren_polymarker( vlist, count )
Vertex_list vlist;
int count;
/*
This routine constructs a polymarker object from the vertex list
passed it.  The parameter count is the number of vertices in vlist.
The object constructed is a single polymarker.  The attributes (color 
and normal) at all vertices are assumed to be the same as those of the 
first vertex.  
*/
{
	Vertex thisvertex;
	Ppoint3 *vertices;
 
	ger_debug("ren_polymarker");
 
	/* Set average color if present */
	if ( !null( vertex_color(first_vertex(vlist)) ) )
	  set_average_color(vlist,count);

	/* Allocate and extract vertex information */
	vertices= get_loc_vtx( vlist, count );
	/* 
	Create the points.  It goes into the already open structure.
	*/
	ppolymarker3( count, vertices );

	/* Free memory */
	free( (char *)vertices );
}
 
void ren_polygon( vlist, count )
Vertex_list vlist;
int count;
/*
This routine constructs a polygon object from the vertices
in the list passed it.  The parameter count contains the number of
vertices in the list.  The object constructed is a single
polygon, which may be concave but should not be self-intersecting.
The attributes (color and normal) at all vertices are assumed to be the
same as those of the first vertex.  
*/
{
	Vertex thisvertex;
	Ppoint3 *vertices;
 
	ger_debug("ren_polygon");

	/* Set average color if present */
	if ( !null( vertex_color(first_vertex(vlist)) ) )
	  set_average_color(vlist,count);

	/* Allocate memory and extract point data */
	vertices= get_loc_vtx( vlist, count );

	/* 
	Create the polygon.  It goes into the already open structure.
	*/
	pfillarea3( count, vertices );

	/* Free memory */
	free( (char *)vertices );
}
 
void ren_triangle( vlist, count )
Vertex_list vlist;
int count;
/*
This routine causes the renderer to construct a triangle mesh object from 
the args (all vertices) in the list passed it.  The object constructed is a 
triangle strip, with each adjacent triple of vertices specifying one triangle.
The attributes (color and normal) at all vertices are assumed to be the
same as those of the first vertex.
*/
{
	Ppoint3 *vertices;
	Pcobundl *colors;
	int i, has_colors= 0;
	float r, g, b;

	ger_debug("ren_triangle");

	/* Check if there are enough vertices */
	if (count<3) {
	  ger_error("ren_triangle: %d is not enough vertices; call ignored.",
		    count);
	  return;
	}

	/* Check if the vertices have colors */
	if ( !null( vertex_color( first_vertex(vlist) ) ) ) has_colors= 1;

	/* Allocate memory and extract point data */
	vertices= get_loc_vtx( vlist, count );
	if (has_colors) colors= get_vtx_clrs( vlist, count );

	/* 
	Create a series of triangles.  Since phigs does no lighting
	calculation, the fact that we should be flipping the ordering
	of every second vertex to get the normal correct is irrelevant.
	*/
	if (has_colors) {
	  r= 0.0; g= 0.0; b= 0.0;
	  for (i=0; i<3; i++) { 
	    r += cbndl_r(colors[i]);
	    g += cbndl_g(colors[i]);
	    b += cbndl_b(colors[i]);
	  }
	  for (i=0; i<count-3; i++) {
	    psetintcolourind( ast_ctb_bestindex( r, g, b ) );
	    pfillarea3( 3, vertices+i );
	    r= r + cbndl_r(colors[i+3]) - cbndl_r(colors[i]);
	    g= g + cbndl_g(colors[i+3]) - cbndl_g(colors[i]);
	    b= b + cbndl_b(colors[i+3]) - cbndl_b(colors[i]);
	  }
	  psetintcolourind( ast_ctb_bestindex( r, g, b ) );
	  pfillarea3( 3, vertices+count-3 );
	}
	else for (i=0; i<count-2; i++) pfillarea3( 3, vertices+i );

	/* Free memory */
	free( (char *)vertices );
	if (has_colors) free( (char *)colors );
}

void ren_mesh( vlist, vcount, flist, fcount )
Vertex_list vlist;
Facet_list flist;
int vcount, fcount;
/* 
This routine causes the renderer to construct a general mesh object from 
the args passed it.  The attributes (color and normal) at all vertices 
are assumed to be the same as those of the first vertex.
*/
{
	Vertex_list thisfacet;
	int facet_vertices;
 
	ger_debug("ren_mesh: %d vertices, %d facets", vcount, fcount);

	if (fcount<0) {
	  ger_error("ren_mesh: invalid facet count!");
	  return;
	}

	while (fcount--) {
	  thisfacet= first_facet( flist );
	  flist= rest_of_facets( flist );
	  facet_vertices= list_length( thisfacet );
	  ren_polygon( thisfacet, facet_vertices );
	}
}

void ren_bezier( vlist, count )
Vertex_list vlist;
int count;
/*
This routine causes the renderer to construct a Bezier patch object from 
the args (16 vertices) in the list passed it.  The object constructed is a 
4 by 4 bicubic Bezier patch.  The attributes (color and normal) at all 
vertices are assumed to be the same as those of the first vertex.
*/
{
	ger_debug("ren_bezier");
	ast_bezier( vlist, ren_mesh );
}

static void add_trans( trans )
Transformation trans;
/* Add the transformation in this gob to the current structure. */
{
	float *matrix;
 
	ger_debug("add_trans: adding transformation matrix");

	matrix= array2d_to_c( trans );
	psetlocaltran3( (Pmatrix3 *)matrix, PPOSTCONCATENATE );
	free( (char *)matrix );
}
 
static void add_attr( attrlist )
Attribute_list attrlist;
/* Add the attributes in this gob to the current structure. */
{
        Color thiscolor;
        int tempint;
	float tempfloat;
	Pair thispair;
 
	ger_debug("add_attr:");
 
	/* check for 'color attribute */
 	if ( !null( thispair= symbol_attr( color_symbol, attrlist ) ) ) {
	        thiscolor= pair_color( thispair );
		ger_debug("           color RGB= (%f %f %f)",
			  color_red(thiscolor), color_green(thiscolor),
			  color_blue(thiscolor) );
		tempint= ast_ctb_bestindex( color_red(thiscolor),
			      color_green(thiscolor), color_blue(thiscolor) );
		psetlinecolourind( tempint );
		psettextcolourind( tempint );
		psetmarkercolourind( tempint );
		psetintcolourind( tempint );
		};
	/* check for 'text-height attribute */
 	if ( !null( thispair= symbol_attr( text_height_symbol, attrlist ) ) ) {
		tempfloat= pair_float( thispair );
		ger_debug("           Text height= %f", tempfloat);
		psetcharheight( tempfloat );
		};
}
 
static void add_kids( kidlist )
Child_list kidlist;
/*  
This routine adds the gobs in the 'children' slot of the given gob
to the current structure.
*/
{
	int thisid;
	Gob thiskid;
 
	ger_debug("add_kids: adding children");
 
 	while ( !null(kidlist) )
		{
		thiskid= first_child( kidlist );
		kidlist= rest_of_children( kidlist );
	    	thisid= gob_idnum( thiskid );
		ger_debug("            adding gob %d as child.",thisid);
		pexecutestruct( thisid );
		};
}

void ren_gob(current_gob, trans, attr, primitive, children )
int current_gob;
Transformation trans;
Attribute_list attr;
Primitive primitive;
Child_list children;
/*
This routine sees all gobs as they are defined.  It adds them to
the Phigs database, and constructs the gob dag as appropriate.
*/
{
	ger_debug("ren_gob: Defining gob %d.", current_gob);
 
	/*   Open the new structure.  The struct id is the gob id. */
	popenstruct( current_gob ); 

	/* If a transform is present, add it. */
	if ( !null(trans) ) add_trans(trans);
	
	/* If attribute modifiers are present, add them. */
	if ( !null(attr) ) add_attr(attr);

	/* 
	If this gob has primitive operators, add them.  Otherwise,
	add the children of this gob to the current structure.
	*/
	if ( !null(primitive) ) eval_function( primitive_op(primitive) );
	else add_kids( children );
 
	/* Close the new structure, adding it to the gob list. */
	pclosestruct();
}
 
void ren_light( location, lightcolor )
Point location;
Color lightcolor;
/*                                             
This routine constructs a light object, first adding a translation
and a color if necessary.  Since plain phigs does no lighting, this
routine does nothing.
*/
{
	ger_debug("ren_light: doing nothing");
}

void ren_ambient(lightcolor)
Color lightcolor;
/*                                             
This routine constructs a ambient light object, first adding a 
color if necessary.  Since plain phigs does no lighting, this
routine does nothing.
*/
{
	ger_debug("ren_ambient");
}

void ren_camera( lookat, lookfrom, lookup, fov, hither, yon, background )
Point lookat, lookfrom;
Vector lookup;
float fov, hither, yon;
Color background;
/*
This routine defines a camera.  The camera goes into its own structure.
That structure is not a gob, so it doesn't go in the gob table.
*/
{
        Pviewmapping3 map;
	Pvector3 vpn;
	Pvector3 *vup;
	Ppoint3  *vrp;
	Pviewrep3 rep;
	Pint error= 0;
	float range_x, range_y, range_z, range;
 	Pcobundl color_rep;

	ger_debug("ren_camera");

	range_x= point_x(lookat) - point_x(lookfrom);
	range_y= point_y(lookat) - point_y(lookfrom);
	range_z= point_z(lookat) - point_z(lookfrom);
	range= sqrt( range_x*range_x + range_y*range_y + range_z*range_z );

	/* Set up viewing parameters */
	map.proj= PPERSPECTIVE;
	map.prp.x= 0.0;
	map.prp.y= 0.0;
	map.prp.z= range;
	map.window.xmin= -range*tan( DegtoRad * fov/2.0 );
	map.window.xmax= -map.window.xmin;
	map.window.ymin= map.window.xmin;
	map.window.ymax= map.window.xmax;
	map.front_plane= map.prp.z + hither;
	map.back_plane= map.prp.z + yon;
	map.view_plane= 0.0;

	rep.clip_limit.xmin= map.viewport.xmin= 0.0;
	rep.clip_limit.xmax= map.viewport.xmax= 1.0;
	rep.clip_limit.ymin= map.viewport.ymin= 0.0;
	rep.clip_limit.ymax= map.viewport.ymax= 1.0;
	rep.clip_limit.zmin= map.viewport.zmin= 0.0;
	rep.clip_limit.zmax= map.viewport.zmax= 1.0;
	rep.clip_xy= rep.clip_back= rep.clip_front= PCLIP;

	vpn.x= -range_x;
	vpn.y= -range_y;
	vpn.z= -range_z;
	vup= get_phigs_vector( lookup );
	vrp= get_phigs_point( lookat );

	/* Calculate view orientation matrix and view mapping matrix */
	pevalvieworientationmatrix3( vrp, &vpn, vup, &error,
				    rep.orientation_matrix );
	if (error) {
	  ger_error("ren_camera: pevalvieworientationmatrix3 error %d", error);
	  error= 0;
	}
	pevalviewmappingmatrix3( &map, &error, rep.mapping_matrix );
	if (error) {
	  ger_error("ren_camera: pevalviewmappingmatrix3 error %d",error);
	  error= 0;
	}

	/* set display update state to PWAIT, to prevent re-rendering
	 * until ren_render is called.
	 */
	psetdisplayupdatest(phigs_wsid, PWAIT, PNIVE);

	/* Set the view representation */
	psetviewrep3(phigs_wsid, VIEWINDEX, &rep);

	/* set background color to the desired background color */
	cbndl_r( color_rep )= color_red( background );
	cbndl_g( color_rep )= color_green( background );
	cbndl_b( color_rep )= color_blue( background );
	psetcolourrep( phigs_wsid, 0, &color_rep );

	/* Clean up */
	free( (char *)vup );
	free( (char *)vrp );
}

void ren_render(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* 
This routine renders the object given, in the current rendering environment.
 */
{
	int thisid;
 
	thisid= gob_idnum( thisgob );

	ger_debug("ren_render: rendering object given by gob %d", thisid);

	/* set display update state to PWAIT */
	psetdisplayupdatest(phigs_wsid, PWAIT, PNIVE);

	/* Open and empty HEAD structure, then refill and close it. */
	popenstruct( HEAD );
	pemptystruct( HEAD );
	if (!null(thistrans)) add_trans(thistrans);
	if (!null(thisattrlist)) add_attr(thisattrlist);
	pexecutestruct( thisid );
	pclosestruct();

	/* set display update state to PASAP, causing redraw */
	psetdisplayupdatest(phigs_wsid, PASAP, PNIVE);
}

void ren_traverselights(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* 
This routine adds the given object to the lights environment. 
Since plain phigs does no lighting, this routine does nothing.
*/
{
	ger_debug("ren_traverselights: doing nothing");
}

void ren_free(thisgob)
Gob thisgob;
/* 
This routine frees memory associated with the given gob.  Note that any 
future call referencing this gob or its children will be impossible unless 
it is somehow redefined first. 
*/
{
	int thisid;

	thisid= gob_idnum( thisgob );

	ger_debug("ren_free:  freeing gob %d",thisid);

	/* set display update state to PWAIT, to prevent re-rendering
	 * as the deletion occurs.
	 */
	psetdisplayupdatest(phigs_wsid, PWAIT, PNIVE);

	pdelstruct( thisid );
}

static void color_setting_function( index, red, green, blue )
int index;
float red, green, blue;
/* This routine is passed to ast_ctb_init, to set the color table */
{
        Pcobundl colorinfo;

        /* no debugging, called too often */

	cbndl_r(colorinfo)= red;
	cbndl_g(colorinfo)= green;
	cbndl_b(colorinfo)= blue;
        psetcolourrep( phigs_wsid, index, &colorinfo );
}

static void color_table_init()
/* This routine sets up the workstation color table */
{
        int error= 0, size=0, size_return, first_index, num_indices;
	char *buffer= (char *)0;
	Pconnid conn_id;
	Pwstype ws_type;
	Pwstables lengths;
	Pcobundl color_rep;

        ger_debug("color_table_init");

	psetcolourmodel( phigs_wsid, PRGB );

	/* Get the workstation type (may be known only to user interface).
         * We don't even need the conn_id, but must go through getting it
	 * to get the workstation type.
	 */
	pinqwsconntype( phigs_wsid, size, &error, buffer, &conn_id,
		       &size_return, &ws_type ); /* call to get buffer size */
#if ( !(VMS || ultrix) ) /* DEC Phigs always produces an error here */
	if (error) ger_error(
              "color_table_init: error %d on first call to pinqwsconntype!",
			     error);
#endif
	buffer= (char *)malloc( size_return*sizeof(char) );
	if (!buffer) ger_fatal("color_table_init: can't allocate %d bytes!",
			       size_return);
	size= size_return;
	pinqwsconntype( phigs_wsid, size, &error, buffer, &conn_id,
		       &size_return, &ws_type ); /* call to get buffer size */
	if (error) ger_fatal(
              "color_table_init: error %d on second call to pinqwsconntype!",
			     error);

	/* Pick appropriate table size and offset */
	pinqwssttable(ws_type, &error, &lengths);
	if (error) ger_fatal("color_table_init: error %d from pinqwssttable!",
			     error);
	first_index= 1; /* leave one for background color */
#if ( !(VMS || ultrix) )
	if ( lengths.colour>8 ) {  /* attempt to spare other windows */
	  first_index= 4;
	}
#endif
	num_indices= lengths.colour-first_index;

	/* Initialize the color table assist module */
	ast_ctb_init(first_index,num_indices,color_setting_function);

	/* set background color to black */
	cbndl_r( color_rep )= 0.0;
	cbndl_g( color_rep )= 0.0;
	cbndl_b( color_rep )= 0.0;
	psetcolourrep( phigs_wsid, 0, &color_rep );
}



void ren_setup(renderer,device,open_device,outfile,hints)
char *renderer, *device, *outfile;
int open_device;
Attribute_list hints;
/* This routine initializes, if it is not already initialized. */
{
	static int initialized=0; /* to hold initialization state */
	Pair thispair;
	Pwstype ws_type;
	Pconnid conn_id;

	ger_debug("ren_setup: initializing phigs; renderer %s, device %s",
		renderer, device);

	/* If the renderer was already initialized, return */
	if (initialized) {
		ger_debug("ren_setup: called twice; this call ignored.");
		return;
		}
	else initialized= 1;

	/* 
	Generate some symbols to be used later in attribute list
	parsing.
	*/
	depth_cue_symbol= create_symbol("depth-cue");
	color_symbol= create_symbol("color");
	backcull_symbol= create_symbol("backcull");
	text_height_symbol= create_symbol("text-height");

	/* 
	If open_device is false, someone else has initialized Phigs.
	Otherwise, initialize Phigs.
	*/
	if (open_device) {
	  /* set up phigs and workstation */
	  popenphigs( (Pchar*)NULL, PDEFAULT_MEM_SIZE );
#ifdef sun
	  if ( !device || !strcmp(device,"sun_tool") ) {
	    ws_type= phigs_ws_type_sun_tool;
	    conn_id= 0;
	  }
	  else if ( !strcmp(device,"sun_canvas") ) {
	    ws_type= phigs_ws_type_sun_canvas;
	    conn_id= phigs_conn_id; 
	          /* global variable; must be externally set */
	  }
	  else if ( !strcmp(device,"cgm") ) {
	    ws_type= phigs_ws_type_cgm_out;
	    conn_id= outfile ? outfile : "phigs_ren.cgm";
	  }
#endif
#if (ultrix || VMS)
	  if ( !device || !strcmp(device,"dec_defaults") ) {
	    ws_type= PHIGS$K_WSTYPE_DEFAULT;
	    conn_id= PHIGS$K_CONID_DEFAULT;
	  }
	  else if ( !strcmp(device,"dec_decwindows_output") ) {
	    /* DecWindows device comes from 'outfile' for user convenience */
	    ws_type= PHIGS$K_DECWINDOWS_OUTPUT;
#ifdef ultrix
	    conn_id= outfile ? outfile : "0:0"; 
#else
	    conn_id= outfile ? outfile : PHIGS$K_CONID_DEFAULT;
#endif
          }
	  else if ( !strcmp(device,"dec_decwindows_drawable") ) {
	    ws_type= PHIGS$K_DECWINDOWS_DRAWABLE;
	    conn_id= phigs_conn_id; 
	          /* global variable; must be externally set */
          }
	  else if ( !strcmp(device,"dec_pex_output") ) {
	    /* DecWindows device comes from 'outfile' for user convenience */
	    ws_type= PHIGS$K_PEX_OUTPUT;
#ifdef ultrix
	    conn_id= outfile ? outfile : ":0.0"; 
#else
	    conn_id= outfile ? outfile : PHIGS$K_CONID_DEFAULT;
#endif
          }
	  else if ( !strcmp(device,"dec_pex_drawable") ) {
	    ws_type= PHIGS$K_PEX_DRAWABLE;
	    conn_id= phigs_conn_id; 
	          /* global variable; must be externally set */
          }
#endif
	  else ger_fatal("ren_setup: unknown phigs device %s",device);
	  popenws( phigs_wsid, conn_id, ws_type );
	  psetdisplayupdatest( phigs_wsid, PWAIT, PNIVE );
	  psethlhsrmode( phigs_wsid, PHIGS_HLHSR_MODE_ZBUFF );
	}

	/* set up workstation color table */
	color_table_init();

	/* create and post root structure */
	popenstruct( ROOT );
	psetviewind( VIEWINDEX );
	psethlhsrid( PHIGS_HLHSR_ID_ZBUFF );
	psetlinetype(PLN_SOLID);
	psetlinewidth(1.0);
	psetmarkertype(PMK_POINT);
	psetintstyle(PSOLID);
	pexecutestruct( HEAD ); /* implicitly creates HEAD struct */
	pclosestruct();
	ppoststruct( phigs_wsid, ROOT, 1.0 );
}

void ren_reset( hard )
int hard;
/* This routine resets the renderer */
{
	ger_debug("ren_reset: resetting renderer; hard= %d", hard);

        /* Hard resets require recreation of lisp-side variables */
        if (hard) {
	  depth_cue_symbol= create_symbol("depth-cue");
	  color_symbol= create_symbol("color");
	  backcull_symbol= create_symbol("backcull");
	  text_height_symbol= create_symbol("text-height");
	}
}

void ren_shutdown()
/* This routine shuts down the renderer */
{
	ger_debug("ren_shutdown");

	/* Close workstation and phigs */
	pclosews( phigs_wsid );
	pclosephigs();
}
