/****************************************************************************
 * parse.c
 * Author Chris Nuuja
 * Copyright 1991, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/

#include <stdio.h>
#include <string.h>
#include "alisp.h"
#include "parse.h"

#define MAXPRINTLENGTH 6  /* Max number of list elements that get displayed
			     when printing a list.  This is necessary so as
			     not to spend all your time printing out lists
			     that are thousands of elements long.  Such lists
			     are typically returned by user defined functions
			     , and must therefore be displayed to stderr.
                          */
typedef void (*funptr)();

static int SZ_PARSE_STK = 16384, SZ_LEX_STK = 16384; 
static NODE **PARSESTACK,**STARTPARSE,**PARSEMAX;
static funptr *LEXSTACK,*STARTLEX,*LEXMAX;
static NODE *LPAREN_MARKER;
  
#define push_parse(token)  \
   *(++PARSESTACK) = token; \
   if (PARSESTACK == PARSEMAX)  grow_parsestack()
#define pop_parse() --PARSESTACK
#define TOP_PARSE *PARSESTACK

#define push_lex(token)  \
   *(++LEXSTACK) = token; \
   if (LEXSTACK == LEXMAX)  grow_lexstack()
#define pop_lex() --LEXSTACK
#define TOP_LEX *LEXSTACK

static void key_list(), key_binder(), key_list2(),possible_key(),var_list(),
    if_end(), do_end(),do_binder2(),do_binder(),let_binder(),Cond_list(),
    ok_rparen(), right_paren(),ok_lparen(),left_paren(),list(),body(),
    cond_body(),s(), symbol(), End(),make_defstruct(),fix_defstruct(),eval_s(),
    p();


void setup_parse()
{
    /* intialize the file stack that the parser uses */
    f_stack = (G_FILES *)malloc(sizeof(G_FILES));
    f_stack->print_flag= 1;
    f_stack->nxt= 0;   /* f_stack->nxt == 0 is used to find the end of stack */
    if(gargc == 1)
      {
      CURRENT_FILE = "standard input";
      LINE_NUMBER = 1;
      f_stack->infile = stdin;
      }
    else 
      {
      if((f_stack->infile = fopen(gargv[gargc-1],"r"))==NULL)
         {
         fprintf(stderr,"Error, could not open file:%s!!\n",gargv[gargc-1]);
         exit(1);
         }
      CURRENT_FILE = gargv[gargc-1];
      LINE_NUMBER=1;
      }

    /*  set PRINT_FLAG so that result are echoed to standard out */
    PRINT_FLAG=1;

   STARTPARSE = (NODE **) malloc(SZ_PARSE_STK*sizeof(NODE *));
   PARSESTACK = STARTPARSE-1;
   PARSEMAX = STARTPARSE + SZ_PARSE_STK - 1;
   STARTLEX = (funptr *) malloc(SZ_LEX_STK*sizeof(funptr));
   LEXSTACK = STARTLEX-1;
   LEXMAX = STARTLEX + SZ_LEX_STK - 1;
   LPAREN_MARKER = new_node(N_TRUE);
}

void cleanup_parse()
{
    /*  free last file */
    if((Infile(f_stack)!=NULL) && (Infile(f_stack)!=stdin) 
	&& (fclose(Infile(f_stack))==EOF))
    {
	fprintf(stderr,"Error closing input file");
	exit(1);
    }
    if (f_stack);
    	free(f_stack);
    free(STARTPARSE);
    free(STARTLEX);
}

static void grow_lexstack()
{
   int currentplace;

   currentplace = LEXSTACK - STARTLEX;
   SZ_LEX_STK = SZ_LEX_STK*3/2;
   STARTLEX = (funptr *) realloc(STARTLEX,SZ_LEX_STK*sizeof(funptr));
   LEXMAX = STARTLEX + SZ_LEX_STK - 1;
   LEXSTACK = STARTLEX + currentplace;
   if (!STARTLEX)
      {
      fprintf(stderr,"Error, Ran out of Lexer stack space:%d\n",SZ_LEX_STK);
      exit(1);
      }
}

static void grow_parsestack()
{
   int currentplace;

   currentplace = PARSESTACK - STARTPARSE;
   SZ_PARSE_STK = (SZ_PARSE_STK + 1)*3/2;
   STARTPARSE = (NODE **) realloc(STARTPARSE,SZ_PARSE_STK*sizeof(NODE *));
   PARSEMAX = STARTPARSE + SZ_PARSE_STK - 1;
   PARSESTACK = STARTPARSE + currentplace;
   if (!PARSESTACK)
      {
      fprintf(stderr,"Error, Ran out of PARSE stack space:%d\n",LEXMAX);
      exit(1);
      }
}

static void make_accessor(name,var,num)
NODE *name,*var;
int num;
{
   NODE *result,*symbol;
   char buffer[128];

   result = cons(cons( get_symbolrep("quote",K_QUOTE,6),
		       cons(name,NIL)),NIL);
   result = cons(get_integerrep(num),result);
   result = cons(var,result);
   result = cons(get_symbolrep("access",K_SYSTEM,7),result);
   result = cons(result,NIL);
   result = cons(cons(var,NIL),result);
   sprintf(buffer,"%s",symbol_name(name));
   strcat(buffer,"-");
   strcat(buffer,symbol_name(var));
   symbol = get_symbolrep(buffer,K_NORMAL,68);
   result = cons(symbol,result);
   result = cons(get_symbolrep("defun",K_DEFUN,6),result);
   push_parse(result);
   result = cons(get_integerrep(num),NIL);
   result = cons(cons( get_symbolrep("quote",K_QUOTE,6),
		       cons(symbol,NIL)),result);
   result = cons(get_symbolrep("add-setf-method",K_SYSTEM,16),result);
   push_parse(result);
}

static void make_defstruct()
{
   NODE *result,*temp,*varlist,*vlist,*name;
   int numvars;
   char buffer[128];

   varlist = TOP_PARSE;
   pop_parse();
   name = TOP_PARSE;
   pop_parse();
   if (TOP_PARSE != LPAREN_MARKER)
      {
      fprintf(stderr,"FOILED!\n");
      print_out(TOP_PARSE);
      exit(1);
      }
   pop_parse();   /*  Removes LPAREN_MARKER at start of (defstruct */
   temp =new_node(N_LIST);
   result = temp;
   numvars=1;
   for(vlist=cdr(varlist);!null(cdr(vlist));vlist=cdr(vlist))
      {
      car(temp) = car(car(vlist));
      incr_ref(car(temp));
      cdr(temp) = new_node(N_LIST);
      incr_ref(cdr(temp));
      temp = cdr(temp);
      make_accessor(name,car(car(vlist)),numvars);
      numvars++;
      }
   car(temp) = car(car(vlist));
   incr_ref(car(temp));
   cdr(temp) = NIL;
   incr_ref(cdr(temp));
   make_accessor(name,car(car(vlist)),numvars);
   numvars++;
   result = cons(cons( get_symbolrep("quote",K_QUOTE,6),
		       cons(name,NIL)),result);
   result = cons(get_integerrep(numvars),result);
   result = cons(get_symbolrep("make-structure",K_SYSTEM,15),result);
   result = cons(result,NIL);
   result = cons(varlist,result);
   sprintf(buffer,"make-");
   strcat(buffer,symbol_name(name));
   result = cons(get_symbolrep(buffer,K_NORMAL,68),result);
   result = cons(get_symbolrep("defun",K_DEFUN,6),result);

   result = cons(result,NIL);   /* maker */
   for(vlist=cdr(varlist);!null(vlist);vlist=cdr(vlist))
      {
      result = cons(TOP_PARSE,result);   /* accessors */
      pop_parse();
      result = cons(TOP_PARSE,result);   /* add-setf-method */
      pop_parse();
      }
   result = cons(get_symbolrep("progn",K_PROGN,6),result);
   push_parse(result);
}

static void trans_string()
{
    NODE *string;

    string = new_node(N_STRING);
    symbol_name(string) = (char *) malloc(CHAR_NUM+1*sizeof(char));
    strcpy(symbol_name(string),token);
    push_parse(string);
}

static void trans_symbol()
{
   switch(SYMBOL_TYPE)
   {
      case K_TRUE:  push_parse(TRUE_NODE); break;
      case K_NIL:   push_parse(NIL); break;
      default: 	    push_parse(get_symbolrep(token,SYMBOL_TYPE,CHAR_NUM));
   }
}

static void trans_float()
{
   float floatval;

   scan_float(token,&floatval);
   push_parse(get_floatrep(floatval));
}

static void trans_int()
{
   int intval;

   scan_int(token,&intval);
   push_parse(get_integerrep(intval));
}

static void parse_error(place)
char *place;
{
   static int numerrors=0;

   fprintf(stderr,"Error in file %s:\n",CURRENT_FILE);
   fprintf(stderr,"At linenumber %d:  %s is invalid %s\n",LINE_NUMBER,
	   token,place);
   if (++numerrors == 10)
      {
      char line[128];

      fprintf(stderr,"Many errors have occured,");
      fprintf(stderr,"Do you wish to continue? (y/n)\n");
      if ( !fgets(line,128,stdin) ) {
	fprintf(stderr,"Cannot get reply; exiting!\n");
	exit(1);
      }
      while (line[0] != 'y' && line[0] != 'n') {
	fprintf(stderr,"Please enter a 'y' or a 'n'\n");
	if ( !fgets(line,128,stdin) ) {
	  fprintf(stderr,"Cannot get reply; exiting!\n");
	  exit(1);
	}
      }
      if (line[0] == 'n')
	exit(1);
      numerrors=0;
      }
}

static void End()
{
   fprintf(stderr,"Implementation error! this function 'End' should never have been called\n");
   exit(0);
       
}

static void p()
{
   get_token();
   switch(TOKEN_TYPE)
     {
     case SYMBOL_TK: 
       trans_symbol(); 
       push_lex(p);
       push_lex(eval_s);
       break;
     case STRING_TK:
       trans_string();
       push_lex(p);
       push_lex(eval_s);
       break;
     case INT_TK: 
       trans_int();
       push_lex(p);
       push_lex(eval_s);
       break;
     case FLOAT_TK: 
       trans_float();
       push_lex(p);
       push_lex(eval_s);
       break;
     case LEFT_PAREN_TK:
       get_token();
       push_parse(LPAREN_MARKER);
       push_lex(p);
       push_lex(eval_s);
       push_lex(list);
       break;
     case QUOTEMACRO_TK:
       get_token();
       push_parse(LPAREN_MARKER);  /* Fake '('  */
       push_parse(get_symbolrep("quote",K_QUOTE,6));  /* Fake quote */
       push_lex(p);
       push_lex(eval_s);
       push_lex(ok_rparen);   /* Fake ')'  */
       push_lex(s);
       break;
     case EOF_TK:
       break;
     default:
       parse_error("at program top level");
       fprintf(stderr,"warning: skipping %s\n",token);
       push_lex(p);
       break;
     }
}

static void eval_s()
{
   ilisp(TOP_PARSE);
   pop_parse();
}

static void fix_defstruct()
{
    push_parse(LPAREN_MARKER);
    push_parse(get_symbolrep("&key",K_KEY,5));
    push_lex(make_defstruct);
    push_lex(key_list2);
}

static void symbol()
{
   if (TOKEN_TYPE == SYMBOL_TK)
	{
	trans_symbol();
	}
     else
	{
	parse_error("where a symbol was expected");
        fprintf(stderr,"warning: skipping %s\n",token);
	get_token();
	push_lex(symbol);
	}
}

static void s()
{
    switch(TOKEN_TYPE)
      {
      case SYMBOL_TK: 
        trans_symbol(); break;
      case STRING_TK:
	trans_string(); break;
      case INT_TK: 
	trans_int(); break;
      case FLOAT_TK: 
	trans_float(); break;
      case LEFT_PAREN_TK:
        get_token();
        push_parse(LPAREN_MARKER);
	push_lex(list);
	break;
      case QUOTEMACRO_TK:
	get_token();
	push_parse(LPAREN_MARKER);  /* Fake '('  */
	push_parse(get_symbolrep("quote",K_QUOTE,6));  /* Fake quote */
	push_lex(ok_rparen);   /* Fake ')'  */
	push_lex(s);
	break;
      default:
	parse_error("as first token of an S-expression");
        fprintf(stderr,"warning: skipping %s\n",token);
	get_token();
	push_lex(s);
	break;
      }
}

static void cond_body()
{
    switch(TOKEN_TYPE)
     {
     case SYMBOL_TK: 
       trans_symbol(); 
       get_token();
       push_lex(body);
       break;
     case STRING_TK:
       trans_string();
       get_token();
       push_lex(body);
       break;
     case INT_TK: 
       trans_int();
       get_token();
       push_lex(body);
       break;
     case FLOAT_TK: 
       trans_float();
       get_token();
       push_lex(body);
       break;
     case LEFT_PAREN_TK:
       push_parse(LPAREN_MARKER);
       get_token();
       push_lex(body);
       push_lex(get_token);
       push_lex(list);
       break;
     case QUOTEMACRO_TK:
       push_parse(LPAREN_MARKER);  /* Fake '('  */
       push_parse(get_symbolrep("quote",K_QUOTE,6));  /* Fake quote */
       get_token();
       push_lex(body);
       push_lex(get_token);
       push_lex(ok_rparen);
       push_lex(s);
       break;
     case RIGHT_PAREN_TK:
       push_parse(NIL);  /* Add the implied NIL consequence */
       push_lex(ok_rparen);
       break;
     default:
       parse_error("in cond body");
       fprintf(stderr,"warning: skipping %s\n",token);
       get_token();
       push_lex(cond_body);
       break;
     }
}

static void body()
{
     switch(TOKEN_TYPE)
     {
     case SYMBOL_TK: 
       trans_symbol(); 
       get_token();
       push_lex(body);
       break;
     case STRING_TK:
       trans_string();
       get_token();
       push_lex(body);
       break;
     case INT_TK: 
       trans_int();
       get_token();
       push_lex(body);
       break;
     case FLOAT_TK: 
       trans_float();
       get_token();
       push_lex(body);
       break;
     case LEFT_PAREN_TK:
       push_parse(LPAREN_MARKER);
       get_token();
       push_lex(body);
       push_lex(get_token);
       push_lex(list);
       break;
     case QUOTEMACRO_TK:
       push_parse(LPAREN_MARKER);  /* Fake '('  */
       push_parse(get_symbolrep("quote",K_QUOTE,6));  /* Fake quote */
       get_token();
       push_lex(body);
       push_lex(get_token);
       push_lex(ok_rparen);
       push_lex(s);
       break;
     case RIGHT_PAREN_TK:
       push_lex(ok_rparen);
       break;
     default:
       parse_error("in statement body");
       fprintf(stderr,"warning: skipping %s\n",token);
       get_token();
       push_lex(body);
       break;
     }
}

static void list()
{
     switch(TOKEN_TYPE)
       {
       case SYMBOL_TK:
         trans_symbol();
         switch(SYMBOL_TYPE)
           {
           case K_COND:
      	     push_lex(Cond_list);
      	     break;
    	   case K_LET: case K_LETSTAR:
	     push_lex(body);
	     push_lex(get_token);
	     push_lex(let_binder);
	     push_lex(left_paren);
	     break;
	   case K_DO: case K_DOSTAR:
	     push_lex(body);
	     push_lex(get_token);
	     push_lex(do_end);
	     push_lex(left_paren);
	     push_lex(get_token);
	     push_lex(do_binder);
	     push_lex(left_paren);
	     break;
	   case K_IF:
	     push_lex(if_end);
	     push_lex(get_token);
	     push_lex(s);
	     push_lex(get_token);
	     push_lex(s);
	     break;
	   case K_LAMBDA:
	     push_lex(body);
	     push_lex(get_token);
	     push_lex(var_list);
	     push_lex(left_paren);
	     break;
	   case K_DEFUN:
	     push_lex(body);
	     push_lex(get_token);
	     push_lex(var_list);
	     push_lex(left_paren);
	     push_lex(get_token);
	     push_lex(symbol);
	     break;
	   case K_DEFSTRUCT:
	     pop_parse();
	     push_lex(fix_defstruct);
	     push_lex(get_token);
	     push_lex(symbol);
	     break;
	   case K_AND: case K_OR: case K_PROGN:
	     push_lex(body);
	     break;
	   case K_NOT:
	     push_lex(right_paren);
	     push_lex(get_token);
	     push_lex(s);
	     break;
	   default:
	     push_lex(body);   /* Same as S BODY production */
	     break;
	   }
	 get_token();
	 break;
       case LEFT_PAREN_TK: case RIGHT_PAREN_TK: case INT_TK: 
       case FLOAT_TK: case STRING_TK: case QUOTEMACRO_TK:
	   push_lex(body);
	   break;
       default:
	   parse_error(" First element of list ");
	   fprintf(stderr,"TYPE:%d\n",(int)TOKEN_TYPE);
	   exit(0);
	   break;
       }
}

static void left_paren()
{
     if (TOKEN_TYPE != LEFT_PAREN_TK)
       {
       parse_error("where a left parenthesis '(' was expected\n");
       fprintf(stderr,"Warning, inserting a '('\n");
       push_parse(LPAREN_MARKER);
       }
     else
       {
       push_parse(LPAREN_MARKER);
       get_token();
       }
}

static void ok_lparen()
{
      push_parse(LPAREN_MARKER);
      get_token();
}

static void right_paren()
{
     if (TOKEN_TYPE != RIGHT_PAREN_TK)
       {
       parse_error("where a ')' was expected");
       fprintf(stderr,"Warning, inserting a ')' \n");
       }
     ok_rparen();
}

static void ok_rparen()
{
  NODE *result;

     result=NIL; 
     while(TOP_PARSE != LPAREN_MARKER)  
        {
        result = cons(TOP_PARSE,result);
        pop_parse();
        }
     pop_parse();   /* remove '(' marker */
     push_parse(result);
}

static void Cond_list()
{
     switch(TOKEN_TYPE)
       {
       case RIGHT_PAREN_TK:
	 push_lex(ok_rparen);
	 break;
       case LEFT_PAREN_TK:
	 push_lex(Cond_list);
	 push_lex(get_token);
	 push_lex(cond_body);
	 push_lex(get_token);
	 push_lex(s);
	 push_lex(ok_lparen);
	 break;
       default:
	 parse_error("in (case consequence) pair list of COND statement");
	 fprintf(stderr,"warning: skipping %s\n",token);
	 get_token();
	 push_lex(Cond_list);
	 break;
       }
}

static void let_binder()
{
     switch(TOKEN_TYPE)
       {
       case RIGHT_PAREN_TK:
	 push_lex(ok_rparen);
	 break;
       case LEFT_PAREN_TK:
	 push_lex(let_binder);
	 push_lex(get_token);
	 push_lex(right_paren);
	 push_lex(get_token);
	 push_lex(s);
	 push_lex(get_token);
	 push_lex(symbol);
	 push_lex(ok_lparen);
	 break;
       default:
	 parse_error("in (variable value) pair list of LET statement");
	 fprintf(stderr,"warning: skipping %s\n",token);
	 get_token();
	 push_lex(let_binder);
	 break;
       }
}

static void do_binder()
{
     switch(TOKEN_TYPE)
       {
       case RIGHT_PAREN_TK:
	 push_lex(ok_rparen);
	 break;
       case LEFT_PAREN_TK:
	 push_lex(do_binder);
	 push_lex(get_token);
	 push_lex(do_binder2);
	 push_lex(get_token);
	 push_lex(s);
	 push_lex(get_token);
	 push_lex(symbol);
	 push_lex(ok_lparen);
	 break;
       default:
	 parse_error("in (variable init-val stepper-val) triple list for DO statement");
	 fprintf(stderr,"warning: skipping %s\n",token);
	 get_token();
	 push_lex(do_binder);
	 break;
       }
}

static void do_binder2()
{
   NODE *result;

   switch(TOKEN_TYPE)
     {
       case RIGHT_PAREN_TK:  /* might not be specified, then it */
	 result = TOP_PARSE;
	 push_parse(result);  /* doesn't change per step */
	 push_lex(ok_rparen);
	 break;
       case LEFT_PAREN_TK: case SYMBOL_TK: case INT_TK: case FLOAT_TK:
       case STRING_TK: case QUOTEMACRO_TK:
	 push_lex(right_paren);
	 push_lex(get_token);
	 push_lex(s);
	 break;
       default:
	 fprintf(stderr,"Bad token type:%d for %s in DOBINDER2, exiting\n",
		    (int)TOKEN_TYPE,token);
	 exit(0);
	 break;
     }
}

static void do_end()
{
   switch(TOKEN_TYPE)
     {
     case SYMBOL_TK: case INT_TK: case FLOAT_TK: case LEFT_PAREN_TK: 
     case STRING_TK: case QUOTEMACRO_TK:
       push_lex(body);
       push_lex(get_token);
       push_lex(s);
       break;
     default:
       parse_error("in end-test for DO statement");
       fprintf(stderr,"warning: skipping %s\n",token);
       get_token();
       push_lex(do_end);
       break;
     }
}

static void if_end()
{
   switch(TOKEN_TYPE)
    {
    case RIGHT_PAREN_TK:
      push_parse(NIL);  /* ADD implied Nil Consequence value */
      push_lex(ok_rparen);
      break;
    case SYMBOL_TK: case INT_TK: case FLOAT_TK: case LEFT_PAREN_TK: 
    case STRING_TK: case QUOTEMACRO_TK:
      push_lex(right_paren);
      push_lex(get_token);
      push_lex(s);
      break;
    default:
      parse_error("in consequence part of IF statement");
      fprintf(stderr,"warning: skipping %s\n",token);
      get_token();
      push_lex(if_end);
      break;
    }
}

static void var_list()
{
   switch(TOKEN_TYPE)
     {
       case RIGHT_PAREN_TK:
	 push_lex(ok_rparen);
	 break;
       case SYMBOL_TK:
	 switch(SYMBOL_TYPE)
	   {
	   case K_KEY:
	     trans_symbol();
	     get_token();
	     push_lex(key_list);
	     push_lex(get_token);
	     push_lex(key_binder);
	     break;
	   case K_REST:
	     trans_symbol();
	     get_token();
	     push_lex(possible_key);
	     push_lex(get_token);
	     push_lex(symbol);
	     break;
	   default:
	     push_lex(var_list);
	     push_lex(get_token);
	     push_lex(symbol);
	     break;
	   }
	 break;
       default:
	 parse_error("in variable declaration list");
	 fprintf(stderr,"warning: skipping %s\n",token);
	 get_token();
	 push_lex(var_list);
	 break;
     }
}

static void possible_key()
{
   switch(TOKEN_TYPE)
	{
	case RIGHT_PAREN_TK:
	  push_lex(ok_rparen);
	  break;
	case SYMBOL_TK:
	  if (SYMBOL_TYPE == K_KEY)
	    {
	    trans_symbol();
	    get_token();
	    push_lex(key_list2);
	    push_lex(get_token);
	    push_lex(key_binder);
	    }
	  else
	    {
	    parse_error("after a &rest <var> (expeced a &key or a ')')"
		       );
	    fprintf(stderr,"warning: skipping %s\n",token);
	    get_token();
	    push_lex(possible_key);
	    }
	  break;
        default:
	  parse_error("after a &rest <var> (expeced a &key or a ')')");
	  fprintf(stderr,"warning: skipping %s\n",token);
	  get_token();
	  push_lex(possible_key);
	  break;
	}
}

static void key_list2()
{
   switch(TOKEN_TYPE)
     {
     case RIGHT_PAREN_TK:
       push_lex(ok_rparen);
       break;
     case SYMBOL_TK: case LEFT_PAREN_TK:
       push_lex(key_list2);
       push_lex(get_token);
       push_lex(key_binder);
       break;
     default:
       /*  
	   A &rest should be treated as a normal variable after
	   the first &rest 
       */
       parse_error("in variable list following &key");
       fprintf(stderr,"warning: skipping %s\n",token);
       get_token();
       push_lex(key_list2);
       break;
     }
}

static void key_binder()
{
   switch(TOKEN_TYPE)
     {
     case SYMBOL_TK:
       push_parse(LPAREN_MARKER);
       trans_symbol();
       push_parse(NIL);
       push_lex(ok_rparen);
       break;			 
     case LEFT_PAREN_TK:
       push_lex(right_paren);
       push_lex(get_token);
       push_lex(s);
       push_lex(get_token);
       push_lex(symbol);
       push_lex(ok_lparen);
       break;
     default:
	parse_error("in keyword variable declaration list following &key(was expecting a symbol or '(')");
	fprintf(stderr,"warning: skipping %s\n",token);
	get_token();
	push_lex(key_binder);
        break;
     }
}

static void key_list()
{
   switch(TOKEN_TYPE)
     {
     case RIGHT_PAREN_TK:
       push_lex(ok_rparen);
       break;
     case LEFT_PAREN_TK:
       push_lex(key_list);
       push_lex(get_token);
       push_lex(key_binder);
       break;
     case SYMBOL_TK:
       if (SYMBOL_TYPE == K_REST)
	 {
	 trans_symbol();
	 get_token();
	 push_lex(right_paren);
         push_lex(get_token);
	 push_lex(symbol);
	 }
       else
	 {
	 push_lex(key_list);
         push_lex(get_token);
	 push_lex(key_binder);
	 }
       break;
     default:
	parse_error("in keyword variable declaration list");
	fprintf(stderr,"warning: skipping %s\n",token);
	get_token();
	push_lex(key_list);
        break;
     }
}

/*  Assumes a file has been pushed onto the file stack, and that it will
    be popped off after this function completes.  setup_parse initializes
    the file stack to either stdin or to the file given at the command line.
    This function will still be executing when a load primitive is
    evaluated.  apply_load will push_filestack(the new file), call this func,
    then call pop_filestack() to return the file being read to what it was
    before the load primitive.
*/
void parse()
{
   funptr func;

   push_lex(End);
   push_lex(p);
   while(TOP_LEX != (funptr)End)  /* at End when EOF read */
      {
      func = *LEXSTACK--;
      (*func)();
      if (SESSION_END)
	 {
	 while (TOP_LEX != (funptr)End) pop_lex();
	 PARSESTACK = STARTPARSE;
	 }
      }
   if (TOP_LEX == (funptr)End)   pop_lex();   /* get rid of End marker */
}

/*FUNCTION print_out()
 *it will print out the nodal tree*/

/*WHEN USED?
 *Called by main() in main.c to print out what's returned
 *by parse_stream()*/

/*WHAT'S RETURNED?
 *nothing*/
void print_out(info)
NODE	*info;
{   
    void print_list();

    switch(Type_Of(info)){
	case N_NIL:
	    fprintf(stderr,"NIL ");
	    break;
        case N_TRUE:
	    fprintf(stderr,"t ");
	    break;
	case N_SYMBOL:
	    if (symbol_type(info) == K_SYSTEM)
		fprintf(stderr,"system:%s ",symbol_name(info));
	    else 
	        fprintf(stderr,"%s ",symbol_name(info));
	    break;
	case N_INT:
	    fprintf(stderr,"%d ",getfixnum(info));
	    break;
	case N_REAL:
	    fprintf(stderr,"%f ",getflonum(info));
	    break;
	case N_LIST:
	    fprintf(stderr,"( ");
	    print_list(info,0);
	    fprintf(stderr,") ");
	    break;
	case N_STRING:
	    fprintf(stderr,"\"%s\" ",symbol_name(info));
	    break;
	case N_ARRAY:
	    fprintf(stderr,"ARRAY:%d ",array_max(info));
	    break;
	case N_USERDATA:
	    (*(userdef_methods(info)->print))(userdef_data(info));
	    break;
	default:
	    break;
    }
}

void dprint_out(info)
NODE	*info;
{   
    void dprint_list();

    switch(Type_Of(info)){
	case N_NIL:
	    fprintf(stderr,"NIL ");
	    break;
        case N_TRUE:
	    fprintf(stderr,"t ");
	    break;
	case N_SYMBOL:
	    if (symbol_type(info) == K_SYSTEM)
		fprintf(stderr,"|%d|system:%s ",info->ptr_cnt,symbol_name(info));
	    else 
	        fprintf(stderr,"|%d|%s ",info->ptr_cnt,symbol_name(info));
	    break;
	case N_INT:
	    fprintf(stderr,"|%d|%d ",info->ptr_cnt,getfixnum(info));
	    break;
	case N_REAL:
	    fprintf(stderr,"|%d|%f ",info->ptr_cnt,getflonum(info));
	    break;
	case N_LIST:
	    fprintf(stderr,"|%d|( ",info->ptr_cnt);
	    dprint_list(info);
	    fprintf(stderr,") ");
	    break;
	case N_STRING:
	    fprintf(stderr,"\"%s\" ",symbol_name(info));
	    break;
	case N_ARRAY:
	    fprintf(stderr,"|%d|ARRAY:%d ",info->ptr_cnt,array_max(info));
	    break;
	case N_USERDATA:
	    (*(userdef_methods(info)->print))(userdef_data(info));
	    break;
	default:
	    break;
    }
}
/*list printer called by print_out()*/
void print_list(info,visitnum)
NODE	*info;
int visitnum;
{
	if (null(info))
		return;
	if (!consp(info))
		{
		print_out(info);
		return;
		}
        if (visitnum == MAXPRINTLENGTH)
	   {
	   fprintf(stderr,". . .");
	   return;
	   }
	print_out(car(info));
	print_list(cdr(info),visitnum+1);
}

void dprint_list(info)
NODE	*info;
{
	if (null(info))
		{
		fprintf(stderr,"nil");
		return;
		}
	if (!consp(info))
		{
		dprint_out(info);
		fprintf(stderr,"NE");
		return;
		}
	dprint_out(car(info));
	fprintf(stderr," !%d! ",info->ptr_cnt);
	dprint_list(cdr(info));
}
