/****************************************************************************
 * matrix_ops.c
 * Author Joel Welling and Chris Nuuja
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include <math.h>

/* 
This module contains routines that create or manipulate transformations
in the form of 4 by 4 matrices.
*/

/* Vectors are considered to align if the ratio of their cross product 
 * squared to their dot product squared is less than the following value.
 */
#define epsilon 0.00001

float *make_translate_c(Tx,Ty,Tz)
float Tx,Ty,Tz;
{
	float *newTrans;
	int row,column;
#ifndef NOMALLOCDEF
	char *malloc();
#endif
	
	if ( !(newTrans = (float *) malloc(16*sizeof(float))) )
	        ger_fatal("make_translate_c: unable to allocate 16 floats!");
	for (row=0;row<4;row++)
		for(column=0;column<4;column++)
			{
			if (column == row)	newTrans[4*row+column] = 1.0;
			else newTrans[4*row+column] = 0.0;
			}
	newTrans[3]=Tx;
	newTrans[7]=Ty;
	newTrans[11]=Tz;
	return(newTrans);
}

static float *make_x_rotate_c(angle)
float angle;
/* This routine returns a rotation about the x axis in c form. */
{
	float *newRotat,radAngle;
	int i;
#ifndef NOMALLOCDEF
	char *malloc();
#endif

	radAngle = angle * DegtoRad;
	if ( !(newRotat = (float *) malloc(16*sizeof(float))) )
	        ger_fatal("make_x_rotate_c: can't allocate 16 floats!");
	for (i=0;i<16;i++) newRotat[i]=0.0;

	newRotat[0]= 1.0;
	newRotat[5]= cos(radAngle);
	newRotat[6]= -sin(radAngle);
	newRotat[9]= sin(radAngle);
	newRotat[10]= cos(radAngle);
	newRotat[15]= 1.0;

	return( newRotat );
}

static float *make_y_rotate_c(angle)
float angle;
/* This routine returns a rotation about the y axis in c form. */
{
	float *newRotat,radAngle;
	int i;
#ifndef NOMALLOCDEF
	char *malloc();
#endif

	radAngle = angle * DegtoRad;
	if ( !(newRotat = (float *) malloc(16*sizeof(float))) )
	        ger_fatal("make_y_rotate_c: can't allocate 16 floats!");
	for (i=0;i<16;i++) newRotat[i]=0.0;

	newRotat[0]= cos(radAngle);
	newRotat[2]= sin(radAngle);
	newRotat[5]= 1.0;
	newRotat[8]= -sin(radAngle);
	newRotat[10]= cos(radAngle);
        newRotat[15]= 1.0;

	return( newRotat );
}

static float *make_z_rotate_c(angle)
float angle;
/* This routine returns a rotation about the z axis in c form. */
{
	float *newRotat,radAngle;
	int i;
#ifndef NOMALLOCDEF
	char *malloc();
#endif

	radAngle = angle * DegtoRad;
	if ( !(newRotat = (float *) malloc(16*sizeof(float))) )
	        ger_fatal("make_z_rotate_c: can't allocate 16 floats!");
	for (i=0;i<16;i++) newRotat[i]=0.0;

	newRotat[0]= cos(radAngle);
	newRotat[1]= -sin(radAngle);
	newRotat[4]= sin(radAngle);
	newRotat[5]= cos(radAngle);
	newRotat[10]= 1.0;
	newRotat[15]= 1.0;

	return( newRotat );
}

float *make_scale_c(Sx,Sy,Sz)
float Sx,Sy,Sz;
{
	float *newScale;
	int row,column;
#ifndef NOMALLOCDEF
	char *malloc();
#endif
	
	if ( !(newScale = (float *) malloc(16*sizeof(float))) )
	        ger_fatal("make_scale_c: unable to allocate 16 floats!");
	for (row=0;row<4;row++)
		for(column=0;column<4;column++)
			{
			if (column == row)	newScale[4*row+column] = 1.0;
			else newScale[4*row+column] = 0.0;
			}
	newScale[0]= newScale[0] * Sx;
	newScale[5]= newScale[5] * Sy;
	newScale[10]= newScale[10] * Sz;
	return(newScale);
}

float *matrix_mult_c(M1,M2)
float M1[16], M2[16];
{
	int row,column,i;
	float *newMatrix;
#ifndef NOMALLOCDEF
	char *malloc();
#endif

	if ( !(newMatrix = (float *) malloc(16*sizeof(float))) )
	        ger_fatal("matrix_mult_c: unable to allocate 16 floats!");

	for (i=0;i<16;i++) newMatrix[i]=0.0;
	for (row = 0;row<4;row++)
		for (column= 0;column<4;column++)
			for (i=0;i<4;i++)
				newMatrix[(4*row)+column] += M1[(4*row)+i]*
				   M2[(4*i)+column];
	
	return(newMatrix);
}

float *transpose( matrix )
float *matrix;
/* This routine replaces the given matrix with its transpose. */
{
	register float temp;
	register int i,j;

	for (i=0; i<4; i++)
		for (j=0; j<i; j++)
			if ( i != j) {
				temp= *(matrix + 4*i +j);
				*(matrix + 4*i + j)= *(matrix + 4*j +i);
				*(matrix + 4*j + i)= temp;
				};

	return( matrix );
}

/* This routine generates a translation matrix */
Transformation Make_Translate(theArgs)
NODE *theArgs;
{
	float x,y,z,*matrix;
	Transformation result;

	
	ger_debug("Make_Translate");

	x = (float)get_number(safe_car(theArgs));
	theArgs=safe_cdr(theArgs);
	y = (float)get_number(safe_car(theArgs));
	theArgs=safe_cdr(theArgs);
	z = (float)get_number(safe_car(theArgs));

	matrix = make_translate_c(x,y,z);

	result = array2d_to_lisp(matrix);
	free( (char *)matrix );
	return(result);
}

/* This routine generates a scaling matrix */
Transformation Make_Scale(theArgs)
NODE *theArgs;
{
	float x,y,z,*matrix;
	Transformation result;
	
	ger_debug("Make_Scale");


	x = (float)get_number(safe_car(theArgs));
	theArgs=safe_cdr(theArgs);
	y = (float)get_number(safe_car(theArgs));
	theArgs=safe_cdr(theArgs);
	z = (float)get_number(safe_car(theArgs));
	matrix = make_scale_c(x,y,z);

	result = array2d_to_lisp(matrix);
	free( (char *)matrix );

	return(result);
}

/* 
This routine generates a rotation about an arbitrary axis. See, for 
example, the Ardent Dore Reference Guide.  (That source apparently
has a sign error in the (result+0) element, however).
*/
static float *make_arbitrary_rotate_c( theta, x, y, z )
float theta, x, y, z;
{
	float s, c;
	float *result;

	ger_debug("Make_Arbitrary_Rotate: theta= %f, x= %f, y= %f, z= %f",
		theta, x, y, z);

	s= sin( DegtoRad * theta );
	c= cos( DegtoRad * theta );

	result= (float *)malloc( 16*sizeof(float) );
	if (!result) ger_fatal(
		"Make_Arbitrary_Rotate: unable to allocate 16 floats!");

	*(result+0)=   x*x + (1.0-x*x)*c;
	*(result+1)=   x*y*(1.0-c) - z*s;
	*(result+2)=   x*z*(1.0-c) + y*s;
	*(result+3)=   0.0;
	*(result+4)=   x*y*(1.0-c) + z*s;
	*(result+5)=   y*y + (1.0-y*y)*c;
	*(result+6)=   y*z*(1.0-c) - x*s;
	*(result+7)=   0.0;
	*(result+8)=   x*z*(1.0-c) - y*s;
	*(result+9)=   y*z*(1.0-c) + x*s;
	*(result+10)=  z*z + (1.0-z*z)*c;
	*(result+11)=  0.0;
	*(result+12)=  0.0;
	*(result+13)=  0.0;
	*(result+14)=  0.0;
	*(result+15)=  1.0;

	return( result );
}

/* This routine generates a rotation matrix. */
float *make_rotate_c(angle, xcomp, ycomp, zcomp)
float angle, xcomp, ycomp, zcomp;
{
	float *matrix, norm;

	ger_debug("make_rotate_c: %f degrees about (%f, %f, %f)",
		angle, xcomp, ycomp, zcomp);

	/* 
	Do fall-throughs for most common cases (axis along a coordinate
	direction), or general case.
	*/
	if ( (xcomp==0.0) && (ycomp==0.0) && (zcomp==0.0) ) {
	  ger_error(
	     "make_rotate_c: can't make rotation about a zero-length axis.");
	  matrix = make_scale_c(1.0,1.0,1.0);
	}
	else if ( (ycomp==0.0) && (zcomp==0.0) && (xcomp>0.0) )
	        matrix = make_x_rotate_c(angle);
	else if ( (zcomp==0.0) && (xcomp==0.0) && (ycomp>0.0) )
	        matrix = make_y_rotate_c(angle);
	else if ( (xcomp==0.0) && (ycomp==0.0) && (zcomp>0.0) )
	        matrix = make_z_rotate_c(angle);
	else {
		norm= sqrt( xcomp*xcomp + ycomp*ycomp + zcomp*zcomp );
		matrix= make_arbitrary_rotate_c( angle, 
			xcomp/norm, ycomp/norm, zcomp/norm ); 
		};

	return( matrix );
}

/* This routine generates a rotation matrix in lisp form. */
Transformation Make_Rotate(theArgs)
NODE *theArgs;
{
	float *matrix, angle, xcomp, ycomp, zcomp;
	Vector axis;
	Transformation result;

	ger_debug("Make_Rotate");

	/* Extract axis vector and rotation angle */
	axis = safe_car(theArgs);
	theArgs = safe_cdr(theArgs);
	angle = (float)get_number(safe_car(theArgs));

	/* Calculate components */
	xcomp= vector_x( axis );
	ycomp= vector_y( axis );
	zcomp= vector_z( axis );

	matrix= make_rotate_c( angle, xcomp, ycomp, zcomp );
	result = array2d_to_lisp(matrix);
	free( matrix );

	return(result);
}

/*
This routine multiplies two matrices.
The safe_car of theArgs should be the first matrix. 
The safe_car safe_cdr of theArgs is the second matrix.
*/
Transformation matrix_mult(trans1, trans2)
Transformation trans1, trans2;
{
	float *matrix1,*matrix2,*matrix3;
	Transformation result;
	
	ger_debug("matrix_mult:");

	matrix1 = array2d_to_c(trans1);
	matrix2 = array2d_to_c(trans2);
	matrix3 = matrix_mult_c(matrix1,matrix2);
	result = array2d_to_lisp(matrix3);

	free( matrix1 ); free( matrix2 ); free( matrix3 );
	return(result);
}

/* This routine multiplies a vector by a matrix in C form. */
float *matrix_vector_c(matrix, vector)
float *matrix, *vector;
{
	int i,j;
	float *result;

	ger_debug("matrix_vector_c:");

	if ( !(result=(float *)malloc(4*sizeof(float))) )
		ger_fatal("matrix_vector_c: can't allocate 4 floats!");

	for (i=0; i<4; i++) {
		*(result+i)= 0.0;
		for (j=0; j<4; j++) {
			*(result+i)+= *(matrix+4*i+j) * *(vector+j);
			}
		}

	return( result );
}

/* This routine returns an identity matrix */
Transformation Make_Identity()
{
	float *matrix;
	Transformation result;

	ger_debug("make_ident");

	matrix = make_scale_c(1.0,1.0,1.0);
	result = array2d_to_lisp(matrix);
	return(result);
}

/* This routine returns a rotation which will reverse the given
 * vector (as opposed to a simple inversion, which is not a rotation.
 */
float *make_flip_c( vx, vy, vz )
float vx, vy, vz;
{
  float px= 0.0, py= 0.0, pz= 1.0, dot, cx, cy, cz, normsqr;

  ger_debug("make_flip_c: flipping %f %f %f", vx, vy, vz);

  /* Find a vector not parallel to the given vector */
  dot= px*vx + py*vy + pz*vz;
  cx= py*vz - pz*vy;
  cy= pz*vx - px*vz;
  cz= px*vy - py*vx;
  if ( (cx*cx + cy*cy + cz*cz) < epsilon*dot*dot ) { /* this p won't work */
    px= 1.0; py= 0.0; pz= 0.0;
    dot= px*vx + py*vy + pz*vz;
    };

  /* Extract the normal component of that vector */
  normsqr= vx*vx + vy*vy + vz*vz;
  px= px - vx*dot/normsqr;
  py= py - vy*dot/normsqr;
  pz= pz - vz*dot/normsqr;

  /* Return a 180 degree rotation about that vector */
  return( make_rotate_c( 180.0, px, py, pz ) );
}

/* This routine returns a rotation which will rotate its first
 * parameter vector to align with its second parameter vector.
 */
float *make_aligning_rotation( v1x, v1y, v1z, v2x, v2y, v2z )
float v1x, v1y, v1z, v2x, v2y, v2z;
{
  float ax, ay, az, dot, theta;

  ger_debug("make_aligning_rotation: %f %f %f to %f %f %f",
	    v1x, v1y, v1z, v2x, v2y, v2z);

  ax= v1y*v2z - v1z*v2y;
  ay= v1z*v2x - v1x*v2z;
  az= v1x*v2y - v1y*v2x;
  dot= v1x*v2x + v1y*v2y + v1z*v2z;

  if ( (ax*ax + ay*ay + az*az) < epsilon*dot*dot ) { /* they already align */
    if (dot >= 0.0 ) /* parallel */
      return( make_translate_c( 0.0, 0.0, 0.0 ) );
    else /* anti-parallel */
      return( make_flip_c( v1x, v1y, v1z ) );
    }
  else {
    theta= acos( ( v1x*v2x+v1y*v2y+v1z*v2z ) / 
		( sqrt( v1x*v1x+v1y*v1y+v1z*v1z )
		 * sqrt( v2x*v2x+v2y*v2y+v2z*v2z ) ) );
    return( make_rotate_c( RadtoDeg*theta, ax, ay, az ) );
  }

  /* NOTREACHED */
}
