/****************************************************************************
 * dore_ui.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module interfaces the 'dui' Demo User Interface to the dore renderer.
*/

#include <stdio.h>
#include <dore.h>
#include <dore_proto.h>
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "ui.h"
#include "control.h"
#include "ren.h"
#include "assist.h"

/* 
The following external is a hook into the Dore renderer which returns
the object handle of a given gob.
*/
extern DtObject ren_dore_objhook();

/* 
The following 4 variables must be set in this routine for use in the rest
of the program.  The sole purpose of this routine is to set these 4 variables.
*/
#define NMODELS 600       /* maximum number of models that can be created */

int nmodels = 0;         /* number of models actually created and available */
int firstcycle= 0;       /* model number of first model to display in an
                            'autocycle'. */
int lastcycle= 0;        /* model number of the last model to be displayed
                            in an 'autocycle'. */
DtObject models[NMODELS]; /* array containing the Dore' group 'handle' for
                            each of the models created */

/* Variable to hold initialization state */
static int initialized= 0;

/* 
The main routine is provided by the dui software;  this routine provides
contact with the P3D software.
*/
geom_spec(minmax, argc, argv)
DtVolume *minmax;
int argc;
char *argv[];
{
        ger_debug("geom_spec:");

        if ( !alisp(argc, argv) ) ger_fatal("Alisp error occurred");
}

void ui_setup(renderer, device, outfile, hints, controller, startframe, 
	      framecopies)
char *renderer, *device, *controller, *outfile;
int startframe, framecopies;
Attribute_list hints;
/* This routine initializes the renderer and controller */
{
  /*
   * Since the dui software does most everything, use no device or
   * controller.
   */
  ger_debug(
	    "ui_setup: substituting \"none\" for controller");
  
  if (!initialized) {
    ctrl_setup( "none", 0, 0 );
    ren_setup( "dore", device, 0, outfile, hints );
    initialized= 1;
  }
  else {
    ger_error(
	   "ui_setup: called twice; this call interpreted as a hard reset.");
    ui_reset( 1, startframe, framecopies );
  }
}

void ui_reset( hard, startframe, framecopies )
int hard, startframe, framecopies;
/* This routine resets everything */
{
        ger_debug("ui_reset: resetting renderer and controller;  hard= %d",
		  hard);

	ast_text_reset( hard );
	ast_prim_reset( hard );
	ast_spln_reset( hard );
	ctrl_reset( hard, startframe, framecopies );
	ren_reset( hard );
}

void ui_shutdown()
/* This routine shuts everything down */
{
      ger_debug("ui_shutdown: shutting everything down.\n");

      ren_shutdown();
      ctrl_end();
}

void ui_camera(lookat, lookfrom, lookup, fov, hither, yon, background)
Point lookat, lookfrom;
Vector lookup;
float fov, hither, yon;
Color background;
/* dui handles camera setting, so do nothing */
{
        ger_debug("ui_camera: doing nothing");
}

void ui_render(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* This routine adds the gob to the 'models' list */
{
        ger_debug("ui_render: saving object given by gob %d",
		  gob_idnum(thisgob));

	if (nmodels<NMODELS) {
		lastcycle= nmodels;
	        models[nmodels++]= DsHoldObj( ren_dore_objhook(thisgob) );
		}
}

void ui_traverselights(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* dui handles lighting, so do nothing. */
{
        ger_debug("ui_traverselights: doing nothing");
}

void ui_free(thisgob)
Gob thisgob;
/* 
This routine does nothing, since we don't want the renderer to destroy
the gob before dui is done using it (which won't be until the application
exits).
*/
{
        ger_debug("ui_free: doing nothing");
}
