/****************************************************************************
 * dore_ren.c
 * Author Joel Welling
 * Copyright 1989, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
#include "stdio.h"
#include "dore.h"
#include "dore_proto.h"
#include "alisp.h"
#include "p3d.h"
#include "ge_error.h"
#include "matrix_ops.h"
#include "ren.h"
 
/* Needed external definitions */
extern char *malloc(), *realloc();
extern DtObject ren_dore_objhook();  /* a hook to return object handles */

/* Constants */
#define PI 3.14159265

/* Symbols defined by setup and used in parsing attribute lists */
static Symbol depth_cue_symbol, color_symbol, backcull_symbol, 
	     text_height_symbol, material_symbol;

/* Dore objects needed for device management */
static DtObject device, frame, view;
static DtVolume volume;
static DtReal White[3]= { 1.0, 1.0, 1.0 }, Black[3]= { 0.0, 0.0, 0.0 };

/* Dore objects needed for object, camera and lights management */
static DtObject current_lights, current_camera, current_object;

/* The following generic chunk of code implements the hash table used to
 * translate gob id's into Dore objects.  Things get added to the table
 * by ren_gob, freed from the table by ren_free_gob, and accessed wherever
 * necessary.
 */
/* here the error codes are the same since 0 is the only invalid pointer */
typedef DtObject Hash_result;
#define BAD_HASH_LOOKUP 0 
#define BAD_HASH_ADD 0
/********** Hash table implementation follows **********************/
/* What the int is being hashed to, and failure return code.
   failed lookups return (Hash_result)BAD_HASH_LOOKUP 
   multiple defines of same value return (Hash_result)BAD_HASH_ADD
   */

/* Hash table size */
#define hashsize 4099 /* a prime number */

typedef struct hash_cell_struct {
	int val;
	Hash_result result;
	struct hash_cell_struct *next, *last;
	} Hash_cell;

/* Hash table itself- static allocation assures all the elements will start
 * out null.
 */
static Hash_cell *hashtable[ hashsize ];

static Hash_result hash_lookup(val)
int val;
/* Hash table lookup */
{
  int id;
  Hash_cell *thiscell;

  id= abs( val % hashsize );
  for (thiscell= hashtable[id]; thiscell; thiscell= thiscell->next)
    if ( thiscell->val== val ) return(thiscell->result);
  return( (Hash_result) BAD_HASH_LOOKUP );
}

static Hash_result hash_add( val, result )
int val;
Hash_result result;
/* Add a value-result pair to hash table */
{
  int id;
  Hash_cell *thiscell;

  if ( hash_lookup(val) != (Hash_result)BAD_HASH_LOOKUP )
    return( (Hash_result)BAD_HASH_ADD );

  id= abs( val % hashsize );
  if (!(thiscell= (Hash_cell *)malloc( sizeof(Hash_cell) )))
	ger_fatal("hash_add: unable to allocate %d bytes!\n",
		sizeof(Hash_cell));
  thiscell->val= val;
  thiscell->result= result;
  thiscell->next= hashtable[id];
  if (hashtable[id]) hashtable[id]->last= thiscell;
  thiscell->last= NULL;
  hashtable[id]= thiscell;

  return( result );
}

static void hash_free(val)
int val;
/* Free a value-result pair from the hash table.  Does nothing if the
 * value is not found.
 */
{
  int id;
  Hash_cell *thiscell;

  id= abs( val % hashsize );
  for (thiscell= hashtable[id]; thiscell; thiscell= thiscell->next)
    if ( thiscell->val==val ) {
      if (thiscell->next) thiscell->next->last= thiscell->last;
      if (thiscell->last) thiscell->last->next= thiscell->next;
      else hashtable[id]= thiscell->next;
      free( (char *)thiscell );
    }
}
/****************End of hash table implementation*******************/
static DtObject get_object(gobid)
int gobid;
/* This function does a hash lookup with error checking */
{
	Hash_result result;

	if ( (result= hash_lookup(gobid)) == BAD_HASH_LOOKUP )
		ger_fatal(
		  "get_object: gob %d has no corresponding Dore object!",
		  gobid);
	return( (DtObject)result );
}

static DtReal *get_dore_vector(avect)
Vector avect;
{
	DtReal *result;

	result = (DtReal *) malloc(3*sizeof(DtReal));
	if (!result) 
		ger_fatal("get_dore_vector: can't allocate 3 DtReals!");
	result[0] = (DtReal) vector_x( avect );
	result[1] = (DtReal) vector_y( avect );
	result[2] = (DtReal) vector_z( avect );

	return(result);
}

static DtReal *get_dore_color(clrstruct)
Color clrstruct;
{
	DtReal *result;

	ger_debug("get_dore_color:  getting colors");

	result = (DtReal *) malloc(3*sizeof(DtReal));
	if (!result) ger_fatal("get_dore_color: can't allocate 3 DtReals!");

	result[0] = (DtReal) color_red(clrstruct);
	result[1] = (DtReal) color_green(clrstruct);
	result[2] = (DtReal) color_blue(clrstruct);

	return(result);
}

static DtMatrix4x4 *array2d_c_to_dore( array )
float *array;
/* 
This routine produces a 4 by 4 matrix of type DtMatrix4x4 from a
4 by 4 matrix of floats.  The matrix should be freed when no longer
needed.
*/
{
	register DtReal *val;
	DtMatrix4x4 *result;

	ger_debug("array2d_c_to_dore");

	val = (DtReal *) malloc(16*sizeof(DtReal));
	if (!val) ger_fatal("array2d_to_c: couldn't allocate 16 DtReals!");
	result= (DtMatrix4x4 *)val;

	for ( ; val<(DtReal *)result+16; val++ ) *val= (DtReal) *array++;

	return( result );
}

char *ren_def_material(material)
Material material;
/* This routine fills the slot for a material definition routine */
{
  ger_debug("ren_def_material: doing nothing");
  return((char *)0);
}

void ren_free_material(thismat)
Material thismat;
/* This routine fills the slot for a material deletion routine */
{
  ger_debug("ren_free_material: doing nothing");
}

/* This routine dumps a Dore object. */
void ren_dump(thisgob)
Gob thisgob;
{
	DtObject thisobj;
	DtNameType type, *name_type=&type;
	DtPtr name;
	int thisid;

	thisid= gob_idnum(thisgob);
	fprintf(stderr,"ren_dump: Dumping gob number %d\n",thisid);

	thisobj= get_object(thisid);
	fprintf(stderr,"Dumping object at location %d\n",(int)thisobj);
	switch ((int)DsInqObjStatus(thisobj)) {
		case (int)DcObjectValid: fprintf(stderr,"Object exists\n");
			break;
		case (int)DcObjectDeleted: fprintf(stderr,"Object deleted\n");
			return;
		case (int)DcObjectInvalid: fprintf(stderr,"Object invalid\n");
			return;
		};

	DsInqObjName(thisobj,name_type,&name);
	switch ((int)*name_type) {
		case (int)DcNameInteger: fprintf(stderr,"Object name %d\n",
			*(int *)name); break;
		case (int)DcNameString: fprintf(stderr,"Object name <%s>\n",
			(char *)name); break;
		case (int)DcNameNone: fprintf(stderr,"Object has no name.\n");
			break; };

	fprintf(stderr,"Object is of type %d\n",(int)DsInqObjType(thisobj));

	if (DsInqObjType(thisobj) == DcTypeGroup)
		if (DgCheck(thisobj) == DcGroupBad) 
			fprintf(stderr,"Group failed circularity test\n");
		else fprintf(stderr,"Group OK (I think)\n");

	DsPrintObj( thisobj );
}

static DtReal *get_loc_vtx( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine transfers vertices containing only locations to a vertex
data array.
*/
{
	Vertex vertex;
	DtReal *vertexlist, *vtx_copy;
                              
	ger_debug("get_loc_vtx: getting vertex list, locs only.");
 
	if ( !(vertexlist= (DtReal *)malloc( 3*vertexcount*sizeof(DtReal) ) ))
		ger_fatal( "get_loc_vertex: unable to allocate %d DtReals!",
			3*vertexcount );
 
	vtx_copy= vertexlist;
	while ( !null(vlist))
		{
		vertex= first_vertex( vlist );
		vlist= rest_of_vertices( vlist );
		*vtx_copy++= (DtReal)vertex_x(vertex);
		*vtx_copy++= (DtReal)vertex_y(vertex);
		*vtx_copy++= (DtReal)vertex_z(vertex);
		}
 
	return( vertexlist );
}
 
static DtReal *get_locnrm_vtx( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine transfers vertices containing only locations and normals to 
a vertex data array.
*/
{
	Vertex vertex; 
	Vector normal;
	DtReal *vertexlist, *vtx_copy;
 
	ger_debug("get_loc_vtx: getting vertex list, locs and normals.");
 
	if ( !(vertexlist= (DtReal *)malloc( 6*vertexcount*sizeof(DtReal) ) ))
		ger_fatal( "get_loc_vertex: unable to allocate %d DtReals!",
			6*vertexcount );
 
	vtx_copy= vertexlist;
	while ( !null(vlist) )
		{
		vertex= first_vertex( vlist );
		vlist= rest_of_vertices( vlist );
		*vtx_copy++= (DtReal)vertex_x(vertex);
		*vtx_copy++= (DtReal)vertex_y(vertex);
		*vtx_copy++= (DtReal)vertex_z(vertex);

		if (!(normal = vertex_normal(vertex)))
			ger_error("get_locnrm_vtx: vertex missing normal!");
		*vtx_copy++= (DtReal)vector_x( normal );
		*vtx_copy++= (DtReal)vector_y( normal );
		*vtx_copy++= (DtReal)vector_z( normal );
		}
 
	return( vertexlist );
}
 
static DtReal *get_locclr_vtx( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine transfers vertices containing only locations and colors to 
a vertex data array.
*/
{
	Vertex vertex;
	Color color;
	DtReal *vertexlist, *vtx_copy;

	ger_debug("get_loc_vtx: getting vertex list, locs and colors.");
 
	if ( !(vertexlist= (DtReal *)malloc( 6*vertexcount*sizeof(DtReal) ) ))
		ger_fatal( "get_loc_vertex: unable to allocate %d DtReals!",
			6*vertexcount );
 
	vtx_copy= vertexlist;
	while ( !null(vlist) )
		{
		vertex= first_vertex( vlist );
		vlist= rest_of_vertices( vlist );
		*vtx_copy++= (DtReal)vertex_x(vertex);
		*vtx_copy++= (DtReal)vertex_y(vertex);
		*vtx_copy++= (DtReal)vertex_z(vertex);

		if (!(color = vertex_color(vertex)))
			ger_error("get_locclr_vtx: vertex missing color!");
		*vtx_copy++= (DtReal)color_red( color );
		*vtx_copy++= (DtReal)color_green( color );
		*vtx_copy++= (DtReal)color_blue( color );
		}

	return( vertexlist );
}
 
static DtReal *get_locnrmclr_vtx( vlist, vertexcount )
Vertex_list vlist;
int vertexcount;
/* 
This routine transfers vertices containing only locations and normals to 
a vertex data array.
*/
{
	Vertex vertex;
	Vector normal;
	Color color;
	DtReal *vertexlist, *vtx_copy;
 
	ger_debug(
		"get_loc_vtx: getting vertex list, locs, normals and colors.");
 
	if ( !(vertexlist= (DtReal *)malloc( 9*vertexcount*sizeof(DtReal) ) ))
		ger_fatal( "get_loc_vertex: unable to allocate %d DtReals!",
			9*vertexcount );
 
	vtx_copy= vertexlist;
	while ( !null(vlist) )
		{
		vertex= first_vertex( vlist );
		vlist= rest_of_vertices( vlist );
		*vtx_copy++= (DtReal)vertex_x(vertex);
		*vtx_copy++= (DtReal)vertex_y(vertex);
		*vtx_copy++= (DtReal)vertex_z(vertex);

		if (!(normal = vertex_normal(vertex)))
			ger_error("get_locnrmclr_vtx: vertex missing normal!");
		*vtx_copy++= (DtReal)vector_x( normal );
		*vtx_copy++= (DtReal)vector_y( normal );
		*vtx_copy++= (DtReal)vector_z( normal );

		if (!(color = vertex_color(vertex)))
			ger_error("get_locnrmclr_vtx: vertex missing color!");
		*vtx_copy++= (DtReal)color_red( color );
		*vtx_copy++= (DtReal)color_green( color );
		*vtx_copy++= (DtReal)color_blue( color );
		}

	return( vertexlist );
}

void ren_sphere()
/*
This routine constructs a Dore sphere object.  The sphere is of radius 1,
and centered at the origin.
*/
{
	ger_debug("ren_sphere");

	DgAddObj( DoPrimSurf( DcSphere ));
}

void ren_cylinder()
/*
This routine constructs a Dore cylinder object.  The cylinder is of radius
1, and is aligned with the z axis.  The ends of the cylinder are at z=0.0
and z=1.0 .
*/
{
	ger_debug("ren_cylinder");

	DgAddObj( DoPrimSurf( DcCylinder ));
}

void ren_torus(bigradius, smallradius)
float bigradius, smallradius;
/*
This routine constructs a Dore torus object from the args
passed it.  The arguments should be the major and minor radii of the torus
respectively.  The torus is drawn so that the hole lies along the z axis.
This requires an initial rotation, since Dore produces tori with the axis
along y by default.
*/
{
	ger_debug("ren_torus");

	DgAddObj( DoRotate( DcXAxis, PI/2.0 ) );
	DgAddObj( DoTorus( (DtReal)bigradius, (DtReal)smallradius ) );
}

void ren_text(txtpoint,uvec,vvec,txtstring)
Point txtpoint;
Vector uvec, vvec;
char *txtstring;
/*
This routine constructs a Dore text object from the args passed it.  
The argument list should contain a point, two vectors to define the 
u and v directions of the text plane, and a text string.
*/
{
	DtPoint3 point;
	DtReal *u, *v;

	ger_debug("ren_text");

	point[0]= (DtReal)point_x( txtpoint );
	point[1]= (DtReal)point_y( txtpoint );
	point[2]= (DtReal)point_z( txtpoint );

	u= get_dore_vector( uvec );
	v= get_dore_vector( vvec );

	DgAddObj( DoText( point, u, v, (DtPtr)txtstring ) );

	free( (char *)u ); free( (char *)v );

}

void ren_polyline( vlist, count )
Vertex_list vlist;
int count;
/*
This routine constructs a Dore polyline object from the vertex list
passed it.  The parameter count is the number of vertices in vlist.
The object constructed is a single polyline, which may be concave but 
should not be self-intersecting.  The attributes (color and normal) at 
all vertices are assumed to be the same as those of the first vertex.  
*/
{
	Vertex thisvertex;
	DtVertexType vertextype;
	DtReal *vertices;
 
	ger_debug("ren_polyline");
 
	/* 
	Check the first vertex to see if it has color and/or normals.
	This will determine the vertex type we are dealing with.
	*/
	thisvertex= first_vertex( vlist );
	if ( !null(vertex_color( thisvertex )) )
		if ( !null(vertex_normal( thisvertex )) ) 
			vertextype= DcLocNrmClr;
		else vertextype= DcLocClr;
	else
		if ( !null(vertex_normal( thisvertex )) ) vertextype= DcLocNrm;
		else vertextype= DcLoc;
 
	/* 
	The following calls allocate enough memory for the vertex list
	(depending on vertex type), and copy the vertex data into the
	list. 
	*/
	switch ( (int)vertextype ) {
		case (int)DcLoc:
			vertices= get_loc_vtx( vlist, count ); break; 
		case (int)DcLocNrm:
			vertices= get_locnrm_vtx( vlist, count ); break;
		case (int)DcLocClr:
			vertices= get_locclr_vtx( vlist, count ); break;
		case (int)DcLocNrmClr:
			vertices= get_locnrmclr_vtx( vlist, count );
		};
 
	/* 
	Create the polyline.  It goes into the already open Dore group.
	*/
	DgAddObj( DoPolyline( DcRGB, vertextype, (DtInt)count, vertices) );

	/* Free memory */
	free( vertices );
}
 
void ren_polymarker( vlist, count )
Vertex_list vlist;
int count;
/*
This routine constructs a Dore polymarker object from the vertex list
passed it.  The parameter count is the number of vertices in vlist.
The object constructed is a single polymarker.  The attributes (color 
and normal) at all vertices are assumed to be the same as those of the 
first vertex.  
*/
{
	Vertex thisvertex;
	DtVertexType vertextype;
	DtReal *vertices;
 
	ger_debug("ren_polymarker");
 
	/* 
	Check the first vertex to see if it has color and/or normals.
	This will determine the vertex type we are dealing with.
	*/
	thisvertex= first_vertex( vlist );
	if ( !null(vertex_color( thisvertex )) )
		if ( !null(vertex_normal( thisvertex )) ) 
			vertextype= DcLocNrmClr;
		else vertextype= DcLocClr;
	else
		if ( !null(vertex_normal( thisvertex )) ) vertextype= DcLocNrm;
		else vertextype= DcLoc;
 
	/* 
	The following calls allocate enough memory for the vertex list
	(depending on vertex type), and copy the vertex data into the
	list. 
	*/
	switch ( (int)vertextype ) {
		case (int)DcLoc:
			vertices= get_loc_vtx( vlist, count ); break; 
		case (int)DcLocNrm:
			vertices= get_locnrm_vtx( vlist, count ); break;
		case (int)DcLocClr:
			vertices= get_locclr_vtx( vlist, count ); break;
		case (int)DcLocNrmClr:
			vertices= get_locnrmclr_vtx( vlist, count );
		};
 
	/* 
	Create the points.  It goes into the already open Dore group.
	*/
	DgAddObj( DoPointList( DcRGB, vertextype, count, vertices ) );

	/* Free memory */
	free( (char *)vertices );
}
 
void ren_polygon( vlist, count )
Vertex_list vlist;
int count;
/*
This routine constructs a Dore polygon object from the vertices
in the list passed it.  The parameter count contains the number of
vertices in the list.  The object constructed is a single
polygon, which may be concave but should not be self-intersecting.
The attributes (color and normal) at all vertices are assumed to be the
same as those of the first vertex.  
*/
{
	Vertex thisvertex;
	DtVertexType vertextype;
	DtReal *vertices;
 
	ger_debug("ren_polygon");

	/* 
	Check the first vertex to see if it has color and/or normals.
	This will determine the vertex type we are dealing with.
	*/
	thisvertex= first_vertex( vlist );
	if ( !null(vertex_color( thisvertex )) )
		if ( !null(vertex_normal( thisvertex )) ) 
			vertextype= DcLocNrmClr;
		else vertextype= DcLocClr;
	else
		if ( !null(vertex_normal( thisvertex )) ) vertextype= DcLocNrm;
		else vertextype= DcLoc;
 
	/* 
	The following calls allocate enough memory for the vertex list
	(depending on vertex type), and copy the vertex data into the
	list. 
	*/
	switch ( (int)vertextype ) {
		case (int)DcLoc:
			vertices= get_loc_vtx( vlist, count ); break; 
		case (int)DcLocNrm:
			vertices= get_locnrm_vtx( vlist, count ); break;
		case (int)DcLocClr:
			vertices= get_locclr_vtx( vlist, count ); break;
		case (int)DcLocNrmClr:
			vertices= get_locnrmclr_vtx( vlist, count );
		};
 
	/* 
	Create the polygon.  It goes into the already open Dore group.
	*/
	DgAddObj( DoSimplePolygon( DcRGB, vertextype, count,
			vertices, DcConcave ) );

	/* Free memory */
	free( (char *)vertices );
}
 
void ren_triangle( vlist, count )
Vertex_list vlist;
int count;
/*
This routine causes the renderer to construct a triangle mesh object from 
the args (all vertices) in the list passed it.  The object constructed is a 
triangle strip, with each adjacent triple of vertices specifying one triangle.
The attributes (color and normal) at all vertices are assumed to be the
same as those of the first vertex.
*/
{
	Vertex thisvertex;
	DtVertexType vertextype;
	int trianglecount;
	DtInt *triangles, *tri_copy;
	DtReal *vertices;
	int iloop;
 
	ger_debug("ren_triangle");
 
	/* Calculate number of triangles in this strip */
	trianglecount= count - 2;
	
	/* 
	Check the first vertex to see if it has color and/or normals.
	This will determine the vertex type we are dealing with.
	*/
	thisvertex= first_vertex( vlist );
	if ( !null(vertex_color( thisvertex )) )
		if ( !null(vertex_normal( thisvertex )) ) 
			vertextype= DcLocNrmClr;
		else vertextype= DcLocClr;
	else
		if ( !null(vertex_normal( thisvertex )) ) vertextype= DcLocNrm;
		else vertextype= DcLoc;
 
	/* 
	The following calls allocate enough memory for the vertex list
	(depending on vertex type), and copy the vertex data into the
	list. 
	*/
	switch ( (int)vertextype ) {
		case (int)DcLoc:
			vertices= get_loc_vtx( vlist, count ); break; 
		case (int)DcLocNrm:
			vertices= get_locnrm_vtx( vlist, count ); break;
		case (int)DcLocClr:
			vertices= get_locclr_vtx( vlist, count ); break;
		case (int)DcLocNrmClr:
			vertices= get_locnrmclr_vtx( vlist, count );
		};
 
	/* 
	Allocate and fill the triangle index array.  It should end
	up containing: { 0, 1, 2, 2, 1, 3, 2, 3, 4, 4, 3, 5, ... }
	*/
	if ( !(triangles= (DtInt *)malloc( 3*trianglecount*sizeof(DtInt) ) ) )
		ger_fatal("ren_triangle: unable to allocate %d DtInts!",
			3*trianglecount);
	tri_copy= triangles;
	for (iloop= 0; iloop<trianglecount; iloop++)
		if ( iloop % 2 )	/* interchange vertices */
			{
			*tri_copy++= (DtInt)iloop+1;
			*tri_copy++= (DtInt)iloop;
			*tri_copy++= (DtInt)iloop+2;
			}
		else			/* don't interchange vertices */
			{
			*tri_copy++= (DtInt)iloop;
			*tri_copy++= (DtInt)iloop+1;
			*tri_copy++= (DtInt)iloop+2;
			};
 
	/* 
	Create the triangle mesh.  It goes into the already open Dore group.
	*/
	DgAddObj( DoTriangleMesh( DcRGB, vertextype, (DtInt)count, vertices,
			(DtInt)trianglecount, triangles, DcTrue ) );

	/* Free memory */
	free( vertices );
	free( triangles );
}

void ren_mesh( vlist, vcount, flist, fcount )
Vertex_list vlist;
Facet_list flist;
int vcount, fcount;
/* 
This routine causes the renderer to construct a general mesh object from 
the args passed it.  The attributes (color and normal) at all vertices 
are assumed to be the same as those of the first vertex.
*/
{
	Vertex thisvertex;
	Vertex_list vlist_copy, thisfacet;
	Facet_list flist_copy;
	DtVertexType vertextype;
	DtInt *contours, *cont_copy, *indices, *ind_copy;
	DtReal *vertices;
	int iloop, indexcount, tempcount;
 
	ger_debug("ren_mesh: %d vertices, %d facets", vcount, fcount);
 
	/* 
	Check the first vertex to see if it has color and/or normals.
	This will determine the vertex type we are dealing with.
	*/
	thisvertex= first_vertex( vlist );
	if ( !null(vertex_color( thisvertex )) )
		if ( !null(vertex_normal( thisvertex )) ) 
			vertextype= DcLocNrmClr;
		else vertextype= DcLocClr;
	else
		if ( !null(vertex_normal( thisvertex )) ) vertextype= DcLocNrm;
		else vertextype= DcLoc;
 
	/* 
	The following calls allocate enough memory for the vertex list
	(depending on vertex type), and copy the vertex data into the
	list. 
	*/
	switch ( (int)vertextype ) {
		case (int)DcLoc:
			vertices= get_loc_vtx( vlist, vcount ); break; 
		case (int)DcLocNrm:
			vertices= get_locnrm_vtx( vlist, vcount ); break;
		case (int)DcLocClr:
			vertices= get_locclr_vtx( vlist, vcount ); break;
		case (int)DcLocNrmClr:
			vertices= get_locnrmclr_vtx( vlist, vcount );
		};
 
	/* Traverse the vertex list, assigning an index to each vertex. */
	vlist_copy= vlist;
	for (iloop=0; iloop<vcount; iloop++) {
		(void)set_vertex_index( first_vertex(vlist_copy), iloop );
		vlist_copy= rest_of_vertices( vlist_copy );
		}

	/* Allocate and fill the contours array, counting indices as we go. */
	if (!(contours= (DtInt *)malloc( fcount*sizeof(DtInt) ) ) )
		ger_fatal(
		   "ren_mesh: unable to allocate %d DtInts for contours!", 
		   fcount);
	indexcount= 0;
	cont_copy= contours;
	flist_copy= flist;
	for (iloop=0; iloop<fcount; iloop++) {
		thisfacet= first_facet( flist_copy );
		tempcount=0;
		while ( !null(thisfacet) ) {
			tempcount++;
			thisfacet= rest_of_vertices( thisfacet );
			}
		indexcount+= tempcount;
		*cont_copy++= tempcount;
		flist_copy= rest_of_facets( flist_copy );
		}

	/* Allocate and fill the indices array. */
	if (!(indices= (DtInt *)malloc( indexcount*sizeof(DtInt) ) ) )
		ger_fatal(
		   "ren_mesh: unable to allocate %d DtInts for indices!", 
		   indexcount);
	ind_copy= indices;
	flist_copy= flist;
	for (iloop=0; iloop<fcount; iloop++) {
		thisfacet= first_facet( flist_copy );
		while ( !null(thisfacet) ) {
			*ind_copy++= vertex_index( first_vertex( thisfacet ) );
			thisfacet= rest_of_vertices( thisfacet );
			}
		flist_copy= rest_of_facets( flist_copy );
		}

	/* 
	Create the polygon mesh.  It goes into the already open Dore group.
	*/
	DgAddObj( DoSimplePolygonMesh( DcRGB, vertextype, (DtInt)vcount,
		vertices, fcount, contours, indices, DcConcave, DcTrue ) );

	/* Free memory */
	free( vertices );
	free( contours );
	free( indices );
}

void ren_bezier( vlist, count )
Vertex_list vlist;
int count;
/*
This routine causes the renderer to construct a Bezier patch object from 
the args (16 vertices) in the list passed it.  The object constructed is a 
4 by 4 bicubic Bezier patch.  The attributes (color and normal) at all 
vertices are assumed to be the same as those of the first vertex.
*/
{
	Vertex thisvertex;
	DtVertexType vertextype;
	DtReal *vertices;
 
	ger_debug("ren_bezier");
 
	/* 
	Check the first vertex to see if it has color and/or normals.
	This will determine the vertex type we are dealing with.
	*/
	thisvertex= first_vertex( vlist );
	if ( !null(vertex_color( thisvertex )) )
		if ( !null(vertex_normal( thisvertex )) ) 
			vertextype= DcLocNrmClr;
		else vertextype= DcLocClr;
	else
		if ( !null(vertex_normal( thisvertex )) ) vertextype= DcLocNrm;
		else vertextype= DcLoc;
 
	/* 
	The following calls allocate enough memory for the vertex list
	(depending on vertex type), and copy the vertex data into the
	list. 
	*/
	switch ( (int)vertextype ) {
		case (int)DcLoc:
			vertices= get_loc_vtx( vlist, count ); break; 
		case (int)DcLocNrm:
			vertices= get_locnrm_vtx( vlist, count ); break;
		case (int)DcLocClr:
			vertices= get_locclr_vtx( vlist, count ); break;
		case (int)DcLocNrmClr:
			vertices= get_locnrmclr_vtx( vlist, count );
		};
 
	/* 
	Create the Bezier patch.  It goes into the already open Dore group.
	*/
	DgAddObj( DoPatch( DcRGB, vertextype, 
			  DcBezier4, vertices, DcBezier4 ) );

	/* Free memory */
	free( vertices );
}

static void add_trans( trans )
Transformation trans;
/* Add the transformation in this gob to the current Dore group. */
{
	float *matrix;
	DtMatrix4x4 *dmatrix;
	DtObject newobj;
 
	ger_debug("add_trans: adding transformation matrix");

	matrix= array2d_to_c( trans );
	dmatrix= array2d_c_to_dore( matrix );

	newobj= DoTransformMatrix( *dmatrix, DcPreConcatenate );

	DgAddObj(newobj);

	free( (char *)dmatrix );
	free( (char *)matrix );
}
 
static void add_attr( attrlist )
Attribute_list attrlist;
/* Add the attributes in this gob to the current Dore group. */
{
	DtReal *color;
	DtReal tempfloat;
	Pair thispair;
	Material material;
	DtReal kd, ka;
 
	ger_debug("add_attr:");
 
	/* check for 'color attribute */
 	if ( !null( thispair= symbol_attr( color_symbol, attrlist ) ) ) {
		color= get_dore_color( pair_color( thispair ) );
		ger_debug("           color RGB= (%f %f %f)",
			*(float *)color, *(float *)(color+1),
			*(float *)(color+2) );
		DgAddObj( DoDiffuseColor( DcRGB, color ) );
		free( (char *)color );
		};
	/* check for 'material attribute */
 	if ( !null( thispair= symbol_attr( material_symbol, attrlist ) ) ) {
	  material= pair_material(thispair);
	  ka= material_ka(material);
	  kd= material_kd(material);
	  /* The following 2 lines impose a lower limit on diffuse and
	   * ambient reflection, for the benefit of dynamic renderers,
	   * where highly specular objects may otherwise disappear.
	   */
	  if ( kd < 0.35 ) kd= 0.35;
	  if ( ka < 0.35 ) ka= 0.35;
	  DgAddObj( DoAmbientIntens(ka) );
	  DgAddObj( DoDiffuseIntens(kd) );
	  DgAddObj( DoSpecularIntens(material_ks(material)) );
	  DgAddObj( DoSpecularFactor(material_exp(material)) );
	  DgAddObj( DoRefractionIndex(material_refract(material)) );
	};
	/* check for 'backcull attribute */
 	if ( !null( thispair= symbol_attr( backcull_symbol, attrlist ) ) ) {
		if ( pair_boolean( thispair ) ) {
			ger_debug("               backface culling on");
			DgAddObj( DoBackfaceCullSwitch( DcOn ) );
			}
		else {
			ger_debug("               backface culling off");
			DgAddObj( DoBackfaceCullSwitch( DcOff ) );
			}
		};
	/* check for 'text-height attribute */
 	if ( !null( thispair= symbol_attr( text_height_symbol, attrlist ) ) ) {
		tempfloat= pair_float( thispair );
		ger_debug("           Text height= %f", tempfloat);
		DgAddObj( DoTextHeight( tempfloat ) );
		};
}
 
static void add_kids( kidlist )
Child_list kidlist;
/*  
This routine adds the gobs in the 'children' slot of the given gob
to the current Dore group.
*/
{
	int thisid;
	Gob thiskid;
 
	ger_debug("add_kids: adding children");
 
 	while ( !null(kidlist) )
		{
		thiskid= first_child( kidlist );
		kidlist= rest_of_children( kidlist );
	    	thisid= gob_idnum( thiskid );
		ger_debug("            adding gob %d as child.",thisid);
		DgAddObj( get_object(thisid) );
		};
}

void ren_gob(current_gob, trans, attr, primitive, children )
int current_gob;
Transformation trans;
Attribute_list attr;
Primitive primitive;
Child_list children;
/*
This routine is to see all gobs as they are defined.  It adds them to
the gob list, and constructs the gob dag as appropriate.
*/
{
	ger_debug("ren_gob: Defining gob %d.", current_gob);
 
	/*   Open the new group */
	DoGroup( DcTrue );

	/* If a transform is present, add it. */
	if ( !null(trans) ) add_trans(trans);
	
	/* If attribute modifiers are present, add them. */
	if ( !null(attr) ) add_attr(attr);

	/* 
	If this gob has primitive operators, add them.  Otherwise,
	add the children of this gob to the current Dore group.
	*/
	if ( !null(primitive) ) eval_function( primitive_op(primitive) );
	else add_kids( children );
 
	/* Close the new group, adding it to the gob list. */
	if (hash_add( current_gob, DsHoldObj(DgClose()) ) ==
		(Hash_result)BAD_HASH_ADD) 
		ger_fatal("ren_gob: unable to add gob to hash table!\n");
}
 
void ren_light( location, lightcolor )
Point location;
Color lightcolor;
/*                                             
This routine constructs a Dore light object, first adding a translation
and a color if necessary.  All new Dore objects are added to the already
open group.
*/
{
	DtReal *color;
	static DtPoint3 origin= { 0.0, 0.0, 0.0 };
	static DtVector3 yvec= { 0.0, 1.0, 0.0 };
	static DtVector3 xvec= { 1.0, 0.0, 0.0 };
	DtPoint3 lightpos;
 
	ger_debug("ren_light");

	/* Set the location */
	lightpos[0]= (DtReal)point_x( location );
	lightpos[1]= (DtReal)point_y( location );
	lightpos[2]= (DtReal)point_z( location );

	/* If position is parallel to yvec, use xvec */
	if ((lightpos[0]==0.0) && (lightpos[2]==0.0))
		DgAddObj( DoLookAtFrom( origin, lightpos, xvec ) );
	else
		DgAddObj( DoLookAtFrom( origin, lightpos, yvec ) );
	
	/* Set the color if it's given */
	if (!null(lightcolor)) 
		{
		color= get_dore_color( lightcolor );
		DgAddObj( DoLightColor( DcRGB, color ) );
		free( (char *)color );
		}

	/* Add the light itself */
	DgAddObj( DoLight() );
}

void ren_ambient(lightcolor)
Color lightcolor;
/*                                             
This routine constructs a Dore ambient light object, first adding a 
color if necessary.  All new Dore objects are added to the already
open group.
*/
{
	DtReal *color;
 
	ger_debug("ren_ambient");

	/* Set the light type to ambient */
	DgAddObj( DoLightType( DcLightAmbient ) );

	/* Set the color if it's given */
	if (!null(lightcolor)) 
		{
		color= get_dore_color( lightcolor );
		DgAddObj( DoLightColor( DcRGB, color ) );
		free( (char *)color );
		}

	/* Add the light itself */
	DgAddObj( DoLight() );
}

void ren_camera( lookat, lookfrom, lookup, fov, hither, yon, background )
Point lookat, lookfrom;
Vector lookup;
float fov, hither, yon;
Color background;
/*
This routine defines a Dore camera.  The camera goes into its own group.
That group is not a gob, so it doesn't go in the gob table.  The 'view'
object is modified to reflect the background color of the camera.
*/
{
	DtPoint3 at, from;
	DtReal *up;
	DtReal dcolor[3];
 
	ger_debug("ren_camera");

	/* Open the group */
	DoGroup( DcTrue );

	/* Set up Dore viewing parameters */
	at[0]= (DtReal) point_x( lookat );
	at[1]= (DtReal) point_y( lookat );
	at[2]= (DtReal) point_z( lookat );

	from[0]= (DtReal) point_x( lookfrom );
	from[1]= (DtReal) point_y( lookfrom );
	from[2]= (DtReal) point_z( lookfrom );

	up= get_dore_vector( lookup );

	DgAddObj( DoLookAtFrom( at, from, up ) );
	DgAddObj( DoPerspective( (DtReal)fov, (DtReal)hither, (DtReal)yon ) );
		
	/* Add the camera itself */
	DgAddObj( DoCamera() );

	/* Set the global camera object to the newly-created object. */
	DgReplaceObjInGroup( current_camera, DgClose() );

	/* Set the view background color appropriately. */
	dcolor[0]= color_red( background );
	dcolor[1]= color_green( background );
	dcolor[2]= color_blue( background );
	DvSetBackgroundColor( view, DcRGB, dcolor );

	free( (char *)up );
}

void ren_render(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* 
This routine renders the object given, in the current rendering environment.
 */
{
	int thisid;
 
	thisid= gob_idnum( thisgob );

	ger_debug("render: rendering object given by gob %d", thisid);

	/* 
	Construct an object with the appropriate transform and
	attributes, whose child is the object to be rendered.
	*/
	DoGroup( DcTrue );
	if (!null(thistrans)) add_trans(thistrans);
	if (!null(thisattrlist)) add_attr(thisattrlist);
	DgAddObj( get_object(thisid) );

	/* Replace the old 'current object' with the new object */
	DgReplaceObjInGroup( current_object, DgClose() );

	/* Update the view */
	DvUpdate(view);
}

void ren_traverselights(thisgob, thistrans, thisattrlist)
Gob thisgob;
Transformation thistrans;
Attribute_list thisattrlist;
/* This routine adds the given object to the lights environment. */
{
	int thisid;
	static int lastid= -1; /* impossible id number */

	thisid= gob_idnum( thisgob );
	ger_debug("ren_traverselights: setting lights to gob %d",thisid);
 
	if (thisid != lastid ) { /* Changed since last render */
		/* 
		Construct an object with the appropriate transform and
		attributes, whose child is the object to be rendered.
		*/
		DoGroup( DcTrue );
		if (!null(thistrans)) add_trans(thistrans);
		if (!null(thisattrlist)) add_attr(thisattrlist);
		DgAddObj( get_object(thisid) );

		/* Set the global lights object to the given gob. */
		DgReplaceObjInGroup( current_lights, DgClose() );

		lastid= thisid;
	}
}

void ren_free(thisgob)
Gob thisgob;
/* 
This routine frees memory associated with the given gob.  Note that any 
future call referencing this gob or its children will be impossible unless 
it is somehow redefined first. 
*/
{
	int thisid;

	thisid= gob_idnum( thisgob );

	ger_debug("ren_free:  freeing gob %d",thisid);

	DsReleaseObj( get_object(thisid) );
	hash_free( thisid );
}

void ren_setup(renderer,device,open_device,outfile,hints)
char *renderer, *device, *outfile;
int open_device;
Attribute_list hints;
/* This routine initializes Dore, if it is not already initialized. */
{
	static int initialized=0; /* to hold initialization state */
	Pair thispair;
 
	ger_debug("ren_setup: initializing dore; renderer %s, device %s",
		renderer, device);

	/* If the renderer was already initialized, return */
	if (initialized) {
		ger_debug("ren_setup: called twice; this call ignored.");
		return;
		}
	else initialized= 1;

	/* 
	Generate some symbols to be used later in attribute list
	parsing.
	*/
	depth_cue_symbol= create_symbol("depth-cue");
	color_symbol= create_symbol("color");
	backcull_symbol= create_symbol("backcull");
	text_height_symbol= create_symbol("text-height");
	material_symbol= create_symbol("material");

	/* 
	If open_device is false, someone else is initializing Dore and
	we're done.  Otherwise, initialize Dore.
	*/
	if (open_device) {
		DsInitializeSystem(2);
		device= DoDevice("ardentx11", "-geometry =640x512+0+0");
		DdInqExtent(device,&volume);
		frame= DoFrame();
		DdSetFrame(device,frame);
		DfSetBoundary(frame,&volume);
		view= DoView();
		DgAddObjToGroup(DfInqViewGroup(frame), view);
		DvSetBoundary(view,&volume);

		DgAddObjToGroup( DvInqDisplayGroup(view), 
				DoRepType(DcSurface) );
		DgAddObjToGroup( DvInqDisplayGroup(view), 
			DoInterpType(DcVertexShade) );
		DgAddObjToGroup( DvInqDisplayGroup(view), 
			DoAmbientSwitch(DcOn) );
		DgAddObjToGroup( DvInqDisplayGroup(view), 
			DoDiffuseColor(DcRGB,White) );
		DgAddObjToGroup( DvInqDisplayGroup(view), 
			DoDiffuseIntens(1.0) );
		DgAddObjToGroup( DvInqDisplayGroup(view),
			DoSpecularSwitch(DcOn) );

		current_object= DoGroup( DcFalse );
		current_camera= DoInLineGroup( DcFalse );
		current_lights= DoGroup( DcFalse );
		DgAddObjToGroup( DvInqDefinitionGroup(view), current_camera );
		DgAddObjToGroup( DvInqDefinitionGroup(view), current_lights );
		DgAddObjToGroup( DvInqDisplayGroup(view), current_object );

		/* Add depth cueing if it is requested in the hints list */
 		if ( !null( thispair=symbol_attr(depth_cue_symbol,hints) ) ) {
			if ( pair_boolean( thispair ) ) {
				ger_debug("Depth cueing requested");
				DgAddObjToGroup( DvInqDisplayGroup(view),
				   DoDepthCueSwitch(DcOn) );
				DgAddObjToGroup( DvInqDisplayGroup(view),
				   DoDepthCue(1.0,0.0,1.0,0.5,DcRGB,Black) );
				}
			};

		/* 
		If the specified renderer is appropriate, use the
		Production Time Renderer.
		*/
		if (!strcmp(renderer,"dore-production"))
			DvSetRendStyle(view,DcProductionTime);
		}
}

void ren_reset( hard )
int hard;
/* This routine resets the renderer */
{
	ger_debug("ren_reset: resetting renderer; hard= %d", hard);

        /* Hard resets require recreation of lisp-side variables */
        if (hard) {
	  depth_cue_symbol= create_symbol("depth-cue");
	  color_symbol= create_symbol("color");
	  backcull_symbol= create_symbol("backcull");
	  text_height_symbol= create_symbol("text-height");
	  material_symbol= create_symbol("material");
	}
}

void ren_shutdown()
/* This routine shuts down the renderer */
{
	/* At the moment, this routine does nothing. */
	ger_debug("ren_shutdown: doing nothing");
}

DtObject ren_dore_objhook(thisgob)
Gob thisgob;
/* 
This routine returns a Dore object given a gob, for use by user interfaces.
*/
{
        return( get_object( gob_idnum(thisgob) ) );
}
