/****************************************************************************
 * transform.c
 * Author Joel Welling
 * Copyright 1990, Pittsburgh Supercomputing Center, Carnegie Mellon University
 *
 * Permission use, copy, and modify this software and its documentation
 * without fee for personal use or use within your organization is hereby
 * granted, provided that the above copyright notice is preserved in all
 * copies and that that copyright and this permission notice appear in
 * supporting documentation.  Permission to redistribute this software to
 * other organizations or individuals is not granted;  that must be
 * negotiated with the PSC.  Neither the PSC nor Carnegie Mellon
 * University make any representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 *****************************************************************************/
/*
This module provides some routines for manipulating transforms.
*/

#include <stdio.h>
#include <math.h>
#include "p3dgen.h"
#include "pgen_objects.h"
#include "ge_error.h"
#include "indent.h"

/* Identity transform */
static P_Transform identity_trans_body= {{
  1.0, 0.0, 0.0, 0.0,
  0.0, 1.0, 0.0, 0.0,
  0.0, 0.0, 1.0, 0.0,
  0.0, 0.0, 0.0, 1.0
}};
P_Transform *Identity_trans= &identity_trans_body; 

void dump_trans(P_Transform *trans)
/* This routine dumps a transformation */
{
  int i;
  float *floattrans;

  ger_debug("transform: dump_trans");

  floattrans= trans->d;
  ind_push();
  for (i=0; i<16; i+=4) {
    ind_write("( %f %f %f %f )",floattrans[i],floattrans[i+1],
              floattrans[i+2],floattrans[i+3]);
    ind_eol();
  }  
  ind_pop();
}

P_Transform *allocate_trans( VOIDLIST )
/* This routine allocates a transform */
{
  P_Transform *thistrans;

  ger_debug("transform: allocate_trans");

  if ( !(thistrans= (P_Transform *)malloc(sizeof(P_Transform))) )
    ger_fatal("transform: allocate_trans: unable to allocate %d bytes!",
              sizeof(P_Transform));

  return( thistrans );
}

void destroy_trans( P_Transform *trans )
/* This routine destroys a transform */
{
  ger_debug("transform: destroy_trans");
  free( (P_Void_ptr)trans );
}

void copy_trans( P_Transform *target, P_Transform *source )
/* This routine copies a transform into existing memory */
{
  int i;
  float *f1, *f2;

  ger_debug("transform: copy_trans");

  f1= target->d;
  f2= source->d;
  for (i=0; i<16; i++) 
    *f1++= *f2++;
}

P_Transform *duplicate_trans( P_Transform *thistrans )
/* This routine returns a unique copy of a transform */
{
  P_Transform *dup;

  ger_debug("transform: duplicate_trans");

  dup= allocate_trans();
  copy_trans(dup, thistrans);
  return(dup);
}

P_Transform *transpose_trans( P_Transform *thistrans )
/* This routine takes the transpose of the given transform in place */
{
  register float temp;
  register int i,j;
  register float *matrix;
  
  ger_debug("transform: transpose_trans");

  matrix= thistrans->d;
  for (i=0; i<4; i++)
    for (j=0; j<i; j++)
      if ( i != j) {
	temp= *(matrix + 4*i +j);
	*(matrix + 4*i + j)= *(matrix + 4*j +i);
	*(matrix + 4*j + i)= temp;
      };
  
  return( thistrans );
}

P_Transform *premult_trans( P_Transform *multby, P_Transform *original )
/* This routine premultiplies 'multby' into 'original' in place,
 * and returns a pointer to original.
 */
{
  int i, row, column;
  float *M1, *M2;
  float newMatrix[16];

  ger_debug("transform: premult_trans");

  M1= multby->d;
  M2= original->d;

  for (i=0;i<16;i++) newMatrix[i]=0.0;

  for (row = 0;row<4;row++)
    for (column= 0;column<4;column++)
      for (i=0;i<4;i++)
        newMatrix[(4*row)+column] += M1[(4*row)+i]*
          M2[(4*i)+column];

  M2= original->d;
  for (i=0; i<16; i++) *M2++= newMatrix[i];
  return( original );
}

P_Transform *translate_trans( double x, double y, double z )
/* This function returns a translation transformation */
{
  P_Transform *trans;
  float *fptr;
  int row, column;

  ger_debug("transform: translate_trans");

  trans= allocate_trans();
  fptr= trans->d;
  
  for (row=0;row<4;row++)
    for(column=0;column<4;column++)
      {
	if (column == row) fptr[4*row+column] = 1.0;
	else fptr[4*row+column] = 0.0;
      }
  fptr[3]=x;
  fptr[7]=y;
  fptr[11]=z;

  return(trans);
}

P_Transform *rotate_trans( P_Vector *axis, double angle )
/* This function returns a rotation transformation */
{
  float s, c;
  P_Transform *trans;
  float *result, x, y, z, norm;

  ger_debug("transform: rotate_trans");

  x= axis->x;
  y= axis->y;
  z= axis->z;
  norm= sqrt( x*x + y*y + z*z );
  if (norm==0.0) {
    ger_error("transform: rotate_trans: axis degenerate; using z axis");
    x= 0.0;
    y= 0.0;
    z= 1.0;
  }
  else {
    x= x/norm;
    y= y/norm;
    z= z/norm;
  }
  
  s= sin( DegtoRad * angle );
  c= cos( DegtoRad * angle );

  trans= allocate_trans();
  result= trans->d;
  *(result+0)=   x*x + (1.0-x*x)*c;
  *(result+1)=   x*y*(1.0-c) - z*s;
  *(result+2)=   x*z*(1.0-c) + y*s;
  *(result+3)=   0.0;
  *(result+4)=   x*y*(1.0-c) + z*s;
  *(result+5)=   y*y + (1.0-y*y)*c;
  *(result+6)=   y*z*(1.0-c) - x*s;
  *(result+7)=   0.0;
  *(result+8)=   x*z*(1.0-c) - y*s;
  *(result+9)=   y*z*(1.0-c) + x*s;
  *(result+10)=  z*z + (1.0-z*z)*c;
  *(result+11)=  0.0;
  *(result+12)=  0.0;
  *(result+13)=  0.0;
  *(result+14)=  0.0;
  *(result+15)=  1.0;
  
  return( trans );
}

P_Transform *scale_trans( double x, double y, double z )
/* This function returns a scale transformation */
{
  P_Transform *trans;
  float *fptr;
  int row,column;

  ger_debug("transform: scale_trans");

  trans= allocate_trans();
  fptr= trans->d;

  for (row=0;row<4;row++)
    for(column=0;column<4;column++)
      {
	if (column == row) fptr[4*row+column] = 1.0;
	else fptr[4*row+column] = 0.0;
      }
  fptr[0]= fptr[0] * x;
  fptr[5]= fptr[5] * y;
  fptr[10]= fptr[10] * z;

  return(trans);
}
