@device[postscript]
@make[manual]
@begin[Titlepage]
@majorheading[P3D Description and Definitions]
@blankspace(1 inch)
by

Joel Welling

@value(Date) @value(time)
@copyrightnotice[Pittsburgh Supercomputing Center, Carnegie Mellon University.
Permission is granted to duplicate and distribute this document
without fee, so long as the  above copyright notice is preserved in
all copies. ]
@end[Titlepage]
@section[Introduction]

The Pittsburgh Supercomputing Center 3D Metafile (P3D) is intended to serve
as a storage format for three dimensional models.  The goal of the project
is to develop a representation that will be compact, portable, and flexible,
and can be created and examined or modified by a variety of software packages
on a variety of platforms.  Although P3D is extensible to allow all the
functionality of model languages intended for photorealistic rendering (e.g.
Pixar's proposed Renderman standard), it is not intended for the production of
photorealistic images, or even to insure that different renderers will
produce identical images from a given P3D model.  Rather, it is designed to
provide a format for the storage and transport of models which can be viewed
on a variety of platforms with a variety of rendering capabilities.

In order to provide maximum flexibility, it was decided that P3D would be a
complete programming language.  This allows it to be used in the same fashion
as Postscript, with the creating application determining the structure of
the metafile based on its internal view of the model.  Lisp provides a well-
known language which can be interpreted or compiled and is well-suited to
describing the directed acyclic graphs (DAGs) common in three dimensional
modelling.  Hence, it was decided that P3D would be written as a set of
extensions to Common Lisp.

This document makes no attempt to describe Common Lisp itself.  For details 
on that language, see any of a number of texts, for example "Common Lisp - 
the Language" by Guy Steele Jr. (Digital Press 1984).  In practice, current 
implementations of P3D (as of May 1989) run on a subset of Common Lisp called 
Alisp.  For details on this lisp, see Chris Nuuja's document on Alisp.  This 
provides a basis for P3D that can be easily distributed, plus some benefits 
in security, at some cost in functionality.

@section[Overview]

A P3D model consists of graphical objects, called 'gobs' for short.  These
objects may be primitives like spheres or triangle lists, or they may be built
up of simpler objects.  The internal structure of a gob may (and probably will)
be a directed acyclic graph or DAG, the internal nodes of which are also gobs
and the leaf nodes of which are gobs consisting of geometrical primitives.

P3D supports sphere, cylinder, torus, polygon, polyline, polymarker, 
triangle list, generalized mesh, spline surface, and text geometrical 
primitives.  In addition, light sources are treated as geometrical 
primitives and occupy leaf nodes of the DAG.

Gobs are defined either by involking a function which returns a
primitive gob, or by listing the 'children' and possibly the
attributes the new gob is to have.  A gob can be saved either by
binding it to a name (for example, via a lisp 'setq' function) or by
including it directly into the list of children of another gob.
Color, material type, and backface cullability are examples of
attributes which might be associated with a gob.

P3D also defines primitive structures needed to support simple
geometrical and graphical entities like points, vectors, vertices,
colors, and materials.  There is a structure used to represent a
camera, which can be used within the metafile to define a point of
view.  Transformations are represented as four by four lisp matrices.

Rendering of a given view is triggered by the 'snap' function, which takes two
gobs and a camera structure as parameters.  One gob supplies the geometrical
primitives, while the other supplies the lights;  they can in fact both be
instances of the same gob.  The camera structure defines the viewing
parameters.  One of two possible actions may be triggered to perform rendering,
depending on the functionality of the underlieing renderer.  If the renderer
cannot support a hierarchical geometry, the geometry DAG may be traversed
within the Lisp itself so that calls to the renderer are only made when
geometric primitives are found.  If the renderer will support a hierarchical
geometry database, a call to the renderer can be made as each gob is defined,
so that at rendering time the DAG traversal is done entirely within the
renderer.

When the rendering process begins, a certain set of default attributes for the
model are assumed.  As the DAG is traversed, each new gob may have attributes
or transformations associated with it.  A gob's attributes are added to those
in effect above it on the DAG, superceeding the previous values of attributes
of the same type.  Transformations are multiplied together in such a way that
at any given level a local coordinate system exists, and the transformation
from that coordinate system to the coordinate system of the top-level gob
(where rendering began) is the product of all the transformations in between.
Thus, for example, any gob can set a color to be inherited by its children;
if no gob farther down the DAG resets the color (and the vertices used have no
color of their own) geometrical primitives below that gob will ultimately be
drawn with the color given.  Likewise, a transformation associated with any gob
can be used to change the ultimate position at which the structures below it in
the DAG will be drawn.

Any gob may be instanced arbitrarily many times in the geometry DAG;  each
instance will inherit attributes and transformations independently from its
parents.  Note that the DAG must remain acyclic or severe problems may result.
It is the programmer's responsibility to make sure that no gob is its own
decendent.

@section[Syntax Rules and Special Conventions]

The syntax of P3D is exactly that of Lisp.  Each P3D metafile should begin
with the Lisp comment:
@begin(programexample)
;P3D nnn
@end(programexample)
left justified on the first line of the metafile, where nnn is the release
number of the P3D specification to which the metafile conforms (for example,
2.1).  This serves to identify the metafile as P3D rather than arbitrary
lisp code.

@section[Coordinate Conventions and Orientation Rules]

P3D assumes that all coordinate systems are right-handed.  For example,
if the x axis of a coordinate system points to the right of a viewer and
the y axis points up, the z axis will point toward the viewer.  The signs of
rotations are also assumed positive in the right-handed direction.  For
example, a positive rotation of 90 degrees about the z axis will move a
vector initially pointing along the x axis into alignment with the y axis.

Coordinate transformations are represented by four by four matrices, as 
follows:
@begin(verbatim)
	@b(rotation:)

		[ R11 R12 R13 0 ]
		[ R21 R22 R23 0 ]
		[ R31 R32 R33 0 ]
		[  0   0   0  1 ]

	@b(translation:)

		[  1   0   0  Tx ]
		[  0   1   0  Ty ]
		[  0   0   1  Tz ]
		[  0   0   0  1  ]

	@b(scale:)

		[ Sx   0   0  0 ]
		[  0  Sy   0  0 ]
		[  0   0  Sz  0 ]
		[  0   0   0  1 ]

@end(verbatim)
where R11 through R33 are appropriate rotation components, Tx, Ty, and Tz are
translation components in the x, y, and z directions respectively, and Sx, Sy,
and Sz are scaling factors in the x, y, and z directions respectively.  With
these conventions, a point in three dimensional space will be represented by a
column vector, the first three components of which are the x, y, and z
coordinates of the point and the fourth component of which is always unity. 
Transforming that point to a new coordinate system then corresponds to left
multiplying the column vector by the appropriate transformation matrix.

Note that it is never necessary to explicitly construct these transformation
matrices.  P3D provides functions that construct them, as described in the
section on Geometric Transformations below.

The 'front' face of a polygon is also determined by a right hand rule.  For
example, the front face of the triangle with vertices ( 0, 0, 0), ( 1, 0, 0),
and (0, 1, 0) is the face on the z>0 side.  If the vertices of a polygon have
local normals (see below), and those normals are inconsistant with the facing
direction of the polygon as derived from the right hand rule, unpredictable
results may occur on rendering.  Polygon facing may be used for backface
culling.

All text is drawn in a plane, the orientation of which is specified by the
'u' and 'v' vectors provided in the text primitive.  The front face of the
text is derived by applying the right hand rule to these two vectors.  If
the 'u' vector lies in the x direction and the 'v' vector lies in the y
direction, the front face of the text will be on the positive z side.

@section[Memory Management Rules]

The programmer need not worry about freeing memory associated with most
P3D structures, as the normal garbage collection mechanisms of Lisp
will handle the task.  This is not always possible for gobs, however,
as a gob may have associated with it data structures in the renderer,
and those data structures must be freed when the gob is freed.  Thus,
several special functions are needed to handle the memory management
of gobs.

A gob is freed if and only if the following conditions apply.  It must
have no parents which have not been freed, it must not be 'held' (see
below), and it or its last surviving parent must be explicitly freed.
Functions to hold and release a gob, and to free it, are described in
the section on Geometrical Objects below.  It is important to use
these functions when destroying a gob which is no longer needed, so
that memory within the renderer will be reclaimed.

@section[Primitive Objects]

@subsection[Point]

The structure used to hold the coordinates of a point in 3-space is defined
as follows:
@begin(programexample)

(defstruct point
	(x 0.0)		;x coordinate
	(y 0.0)		;y coordinate
	(z 0.0))	;z coordinate

@end(programexample)
It can be created via the function 'make-point', which can take the options
':x', ':y', and ':z', all of which are floating point numbers defaulting to
0.0.  For example,
@begin(programexample)
(setq origin (make-point))
@end(programexample)
is the definition of the standard symbol 'origin'.

@subsection[Vector]

The structure used to hold a vector in 3-space is defined as follows:
@begin(programexample)

(defstruct vector
	(x 0.0)		;x coordinate
	(y 0.0)		;y coordinate
	(z 0.0))	;z coordinate

@end(programexample)
It can be created via the function 'make-vector', which can take the options
':x', ':y', and ':z', all of which are floating point numbers defaulting to 
0.0.  For example,
@begin(programexample)
(setq x-vec (make-vector :x 1.0 :y 0.0 :z 0.0))
@end(programexample)
is the definition of the standard symbol 'x-vec'.

@subsection[Color]

The structure used to hold a color is defined as follows:
@begin(programexample)

(defstruct color
	(r 0.8)		;red intensity
	(g 0.8)		;green intensity
	(b 0.8)		;blue intensity
	(a 1.0))	;opacity

@end(programexample)
It can be created via the function 'make-color', which can take the options
':r' for red intensity, ':g' for green intensity, ':b' for blue intensity, and
':a' for opacity (sometimes called the alpha channel).  All of these values are
assumed to be floating point values in the range 0.0 to 1.0 inclusive, with 0.0
being no intensity in that color or complete transparency.  The defaults are
0.8 for each of the red, green, and blue values, and 1.0 or complete opacity
for the alpha value.  For example,
@begin(programexample)
(setq red (make-color :r 1.0 :g 0.0 :b 0.0))
@end(programexample)
is the definition of the standard symbol 'red'.

@subsection[Vertex]

The structure used to hold a vertex (a position in 3 dimensional space,
possibly with an associated color and/or normal vector) is defined as
a struct containing at least the following fields:
@begin(programexample)

(defstruct vertex
	(x 0.0)		;x coordinate
	(y 0.0)		;y coordinate
	(z 0.0)		;z coordinate
	(clr nil)	;local color
	(normal nil))	;local surface normal

@end(programexample)
It can be created via the function 'make-vertex', which can take the options
':x' for x coordinate, ':y' for y coordinate, ':z' for z coordinate (these
three being floating point numbers), ':clr' (a color, as returned by
'make-color') for local color, and ':normal' (a vector, as returned by
'make-vector') for local surface normal.  The coordinate values default to 0.0,
so the default location is the origin.  The local color defaults to nil, so
that the vertex will be drawn with a color inherited from its parents in the
DAG.  The local normal also defaults to nil;  the renderer will attempt to
calculate a normal by interpolating the orientations of adjacent polygons if
this is the case.  For example,
@begin(programexample)
(setq red-origin (make-vertex :clr red))
@end(programexample)
will define a vertex located at the origin with a color of 'red' (as defined
above) and no normal.

@subsection[Material]

The structure used to hold a material (a set of properties used
with attributes like color to determine the appearance of an object)
is represented as a structure with at least the following fields:
@begin(verbatim)

        @b(Field)             @b(Type)                @b(Meaning)

        :ka              float          ambient light weighting factor
        :kd              float          diffuse light weighting factor
        :ks              float          specular light weighting factor
        :exp             float          specular exponent
        :reflect         float          reflection coefficient
        :refract         float          index of refraction
        :energy          color          energy density (for radiosity)

@end(verbatim)
Other structure fields may exist, but they are maintained by P3D and
should not be modified by the programmer.  
A material should always be created with the 'def-material' function.
@begin(format)

	( @b(def-material) :ka ka-value :kd kd-value :ks ks-value
                  :exp exp-value :reflect reflect-value 
                  :refract refract-value :energy energy-color )

		parameters:
                        :ka ka-value (optional): ambient light
                                weighting factor
                        :kd kd-value (optional): diffuse light
                                weighting factor
                        :ks ks-value (optional): specular light
                                weighting factor
                        :exp exp-value (optional): specular exponent
                        :reflect reflect-value (optional):
                                reflection coefficient
                        :refract refract-value (optional):
                                index of refraction
                        :energy energy-color (optional): energy
                                density for radiosity
		returns:  material with the given characteristics

@end(format) 
All the keyword-field pairs are optional.  Fields which
are not specified are assigned specific default values; see the
specification of the 'default-material' predefined symbol at the end
of this document for the default values of each field.

A material can be used to describe the way light interacts with an
object, and thus its appearance.  For example,
@begin(programexample)
(setq aluminum-material (def-material
                :ka 0.25 :kd 0.25 :ks 0.75 :exp 6.0 :reflect 0.75))
@end(programexample)
is the definition of the standard material symbol 'aluminum-material'.
This material is quite reflective, but has a relatively low
specular exponent, so it will appear shiny but not polished.

@subsection[Camera]

The structure used to hold a camera is defined as follows:
@begin(programexample)

(defstruct camera
	(lookfrom origin)	;eye point
	(lookat origin)		;point to look at
	(up y-vec)		;view's 'up' direction
	(fovea 56.0)		;view included angle
	(hither -0.01)		;hither clipping distance
	(yon -100.0)		;yon clipping distance 
        (background black))     ;background color

@end(programexample)
Camera structs are used to define viewing characteristics for rendering.  It is
created via the function 'make-camera', which can take the following options:
@begin(verbatim)

	@b(Option)		@b(Type)		@b(Meaning)

	:lookfrom	point		location of camera
	:lookat		point		location at which camera is to point
	:up		vector		direction from bottom to top of view
	:fovea		float		camera opening angle in degrees
	:hither		float		distance to hither clipping plane
	:yon		float		distance to yon clipping plane
        :background     color           background color

@end(verbatim)

Both the lookat and lookfrom points default to the origin, so at least one must
be set to produce a valid camera.  The camera's 'up' direction defaults to the
y direction.  The fovea angle defaults to 56 degrees, and the hither and yon
distances default to -0.01 and -100.0 respectively.  (Note that because the
coordinate system of the camera is required to be right-handed, the ':lookat'
point will lie in the negative z direction from the origin, so the hither and
yon clipping distances must be negative).  The background color defaults
to black.

The following example defines a camera at the coordinates (0.0, 0.0,
20.0) looking at the origin, with all other characteristics left at
their defaults.
@begin(programexample)
(setq this-camera (make-camera 
			:lookat origin
			:lookfrom (make-point :z 20.0)))
@end(programexample)

@section[Geometric Transformations]

P3D provides functions to generate transformation matrices, as described in the
Coordinate Conventions section above.  The functions are:
@begin(format)

	( @b(make-translate) Tx Ty Tz )

		parameters:
			Tx, Ty, Tz:  floating point numbers giving distance
				to translate in the x, y, and z directions
				respectively.
		returns:
			a transformation matrix encoding the translation

	( @b(make-rotate) axis angle )

		parameters:
			axis:  vector providing the axis about which to rotate
			angle:  angle in degrees of the rotation
		returns:
			a transformation matrix encoding the rotation

	( @b(make-scale) Sx Sy Sz )

		parameters:
			Sx, Sy, Sz:  floating point numbers giving factors by
				which to scale the x, y, and z directions
				respectively

		returns:
			a transformation matrix encoding the scaling

	( @b(make-identity) )

		parameters: none

		returns: a transformation matrix encoding the identity
			transformation

	( @b(compose-transform) trans1 trans2 )

		parameters:
			trans1, trans2: transformation matrices

		returns:  trans1 x trans2  (remember that subsequent
			transformations should left multiply existing 
			transformations)

	( @b(compose-transforms) trans1 trans2 trans3 ... transn )

		parameters:
			trans1 ... transn: transformation matrices

		returns:  trans1 x trans2 x ... x transn (remember that
			subsequent transformations should left multiply
			existing transformations).  compose-transforms 
			is somewhat slower than compose-transform.
@end(format)

@section[Geometrical Object (Gob)]

A gob is represented as a structure with at least the following options:
@begin(verbatim)
	@b(Option)		@b(Type)		@b(Meaning)

	:attr		assoc-list	attribute-value pairs for this gob
	:transform	transformation	coordinate transformation
	:children	list		list of gobs to be children
@end(verbatim)
Other structure slots may exist, but they are maintained by P3D and should
not be modified by the programmer.  All of the fields default to nil.

A gob should always be created with 'def-gob', or with one of the geometrical
primitive generators (see below). If 'def-gob' is used, the definition should
include a ':children' option or the gob will have no decendents in the DAG and
thus be useless.

@begin(format)

	( @b(def-gob) :attr attrlist
		  :transform transformation
		  :children childlist )

		parameters:
			:children childlist (required):  list of
				children of this gob
			:transform transformation (optional):  coordinate 
				transformation for this gob
			:attr attrlist (optional):  association list of
				attribute and value pairs for this gob
		returns:  gob with the given children, coordinate
			transformation, and attributes

@end(format)

If old_gob_1 and old_gob_2 are previously created gobs, the following will
create a gob called new_gob which includes a coordinate transformation, and 
has the two other gobs as children.
@begin(programexample)

(setq new_gob (def-gob  :transform (make-translate 1.0 0.0 0.0)
			:children (list old-gob-1 old-gob 2)))
@end(programexample)

As described in the Memory Management section above, a gob can be
'held' to guarantee that it will not be freed, and 'unheld' to permit
reclaimation of its memory.  The 'hold-gob' and 'unhold-gob' functions
have the following syntax:
@begin(format)

        ( @b(hold-gob) gob )

                parameters:
                        gob:  gob to be held

                returns: T

        ( @b(unhold-gob) gob )

                parameters:
                        gob:  gob to be released

                returns: Nil

@end(format)
Applying 'hold-gob' to a gob which is already held is a null
operation, as is applying 'unhold-gob' to a gob which is not held.

A gob which is not held and which has no parents can be freed with
'free-gob'.  This function causes memory associated with the gob in
the renderer to be freed; any future reference to the gob will be an
error.  The children of the freed gob are checked to see if they have
any other parents (which have not themselves been freed); an orphaned
child which is not itself held is also freed.  The syntax of free-gob
is as follows:
@begin(format)

	( @b(free-gob) gob )

		parameters: 
			gob:  gob to be freed.

		returns: Nil.
@end(format)

@section[Attributes]

Attributes are data other than geometrical data which is used in rendering. 
Each gob can have associated with it a list of attribute-value pairs;  this
list is stored in the gob's ':attr' slot.  Attributes are inherited down the
geometry DAG, in such a way that a value for a given attribute on a gob which
is a close parent of a geometrical primitive (see below) will supersede a value
assigned that attribute farther up the DAG.

A gob's attributes are stored in an association list, of the sort searched by
the Common Lisp 'assoc' function.  Association lists are of the form:
@begin(programexample)
( (attr1 . val1) (attr2 . val2) (attr3 . val3) ... )
@end(programexample)
The following attributes are currently supported by P3D:
@begin(verbatim)

    @b(Attribute)     @b(Value)       @b(Default)           @b(Meaning)

    'color        color	      opaque white	default color below this gob

    'backcull     T or nil    nil		enable backface culling

    'text-height  float	      1.0		character height for capital
						X character

    'text-font    string     "simplex_roman"    font in which text will
                                                be written
    'text-stroke-width-fraction
                  float      0.15               strokes with which text is
                                                drawn will be this
                                                fraction of text-height

    'text-thickness-fraction
                  float      0.1                if the text has thickness
                                                (perpendicular to u and v),
                                                it will be this fraction
                                                of text-height

    'material     material   default-material   material type for this gob
                                                and its children
@end(verbatim)
Any given renderer may ignore some of these attributes.

Because the DAG traversal process may reverse the order of attribute-value
pairs within a gob, the attribute list for a given gob should have only
one occurance of a given attribute.

The following example creates a gob with an attribute list setting a default
color of blue and enabling backface culling:
@begin(programexample)

(setq sample_gob (def-gob
		:attr (list (cons 'color blue)
			   '(backcull . T))
		:children (list old_gob_1 old_gob_2)))
@end(programexample)
@section[Geometrical Primitives]

P3D defines a number of functions which return geometrical primitive gobs. 
These gobs are leaf nodes in the rendering DAG;  they have no children.  These
primitives are rendered with the attributes inherited from their parents,
except in those cases where the vertices of the primitive specify the attribute
explicitly (e.g. color).  The funcions which generate primitives are:

@begin(format)
	( @b(sphere) )

		parameters:  none

		returns:  gob containing a sphere of radius 1, centered at
			the origin

	( @b(cylinder) )

		parameters: none

		returns:  gob containing a cylinder of radius 1, with the
			symmetry axis aligned with the z axis.  The cylinder
			ends are at z= 0.0 and z= 1.0 .

	( @b(torus) bigradius smallradius )

		parameters:
			bigradius:  float specifying major radius of torus
			smallradius:  float specifying minor radius of torus

		returns:  gob containing a torus with the given major and
			minor radii, with the hole in the torus running
			along the z axis.

	( @b(polyline) v1 v2 v3 ... vn )

		parameters:
			v1 ... vn:  vertices to be connected by the polyline.
			There must be at least two vertices.

		returns:  gob containing a polyline connecting the given 
			vertices.  The polyline is not closed; i.e. vn is 
			not connected to v1.

	( @b(polymarker) v1 v2 v3 ... vn )

		parameters:
			v1 ... vn:  vertices at which markers are to be
			drawn.  There must be at least one vertex.

		returns:  gob containing a polymarker, the markers of which
			are at the given vertices.  The markers are of the
			current marker type.  Depending on the renderer,
			the marker drawn may or may not be subject to
			geometric transformations.  For example, a marker
			might be of constant size, or might scale with
			the projection of the model.

	( @b(polygon) v1 v2 v3 ... vn )

		parameters:
			v1 ... vn:  vertices to form the perimiter of the
			polygon.  There must be at least three vertices.

		returns:  gob containing a polygon with the given vertices.
			The polygon is closed automatically; i.e. v1 need not
			equal vn.

		comments:  The appearance of non-planar polygons may vary
			from renderer to renderer.

	( @b(triangle) v1 v2 v3 ... vn )

		parameters:
			v1 ... vn:  vertices to be used to form the triangle
			strip.

		returns:  gob containing a triangle strip composed of the given
			vertices.  The strip is a series of triangles, each
			defined by an adjacent set of three vertices from the
			vertex list.  Thus, the first triangle is composed
			of v1, v2, and v3; the second of v2, v3, and v4; and
			so on.  The 'front' face of a triangle strip is
			determined by the right hand rule for odd numbered
			triangles, and a left hand rule for even numbered
			triangles.

	( @b(mesh) vlist flist )

		parameters:
			vlist: list of vertices included in the mesh
			flist: list of lists of vertices, one list per 
			polygonal facet.

		returns:  gob containing a generalized mesh composed of the
			given facets, which are polygons the corners of which 
			are in the given list of vertices.

		comments:  The vertices in the facet lists should be the same
			vertices as those in the vertex list, not simply
			different vertices at the same coordinate location.
			For example, if v1, v2, v3, and v4 are vertices, the
			following is a valid mesh:

			(mesh (list v1 v2 v3 v4)
				(list (list v1 v2 v3) (list v2 v3 v4) ) )

	( @b(bezier) v1 v2 v3 ... v16 ) 

		parameters:
			v1 to v16: The control vertices of a 4 by 4 bicubic
			Bezier patch.

		returns:  gob containing a 4 by 4 bicubic Bezier patch, 
			defined by the control vertices as follows:
@begin(verbatim)

			*-------*--------*--------*
			|v1     |v2      |v3      |v4
			|       |        |        |
			|       |        |        |
			*-------*--------*--------*
			|v5     |v6      |v7      |v8
			|       |        |        |
			|       |        |        |
			*-------*--------*--------*
			|v9     |v10     |v11     |v12
			|       |        |        |
			|       |        |        |
			*-------*--------*--------*
			 v13     v14      v15      v16

@end(verbatim)
			The 'up' surface of the patch is defined by the
			right hand rule circulating in the v1-v4-v16-v13
			order, so that the above illustration's 'up'
			direction is into the page.

		comments:  The patch passes through the corner vertices;
			the other vertices define the shape of the spline
			surface.  The treatment of color and normal
			information associated with the vertices is
			renderer dependent.

	( @b(text) point uvec vvec string )

		parameters:
			point:  point in 3D space at which the text is to
			be written.
			uvec: vector in 3D space specifying the u axis
			of the plane in which the text is to be drawn
			vvec: vector in 3D space specifying the v axis
			of the plane in which the text is to be drawn
			string:  character string to be written.

		returns:  gob containing a written character string.  The
			writing plane and other attributes are determined
			by the inherited values of the appropriate 'text-' 
			attributes.  The text is drawn starting at the given
			point, in a plane the orientation of which is given
			by the u and v vectors.
@end(format)

@section[Lights]

P3D provides two function which return light sources.  These light sources are
gobs which serves as a leaf node of a DAG;  they have no children.  Light
sources can provide directional illumination, or can provide ambient background
light which illuminates all surfaces regardless of orientation.
The functions which produce these lights are:
@begin(format)

	( @b(light) location lightcolor )

		parameters:
			location:  point in 3D space at which to place the
			light source.  (Note that some renderers will retreat
			the light to infinity along the line from the origin
			of the current coordinate system through the given 
			point).
			lightcolor:  color of the light to be emitted from
			this source.

		returns:  gob containing a directional light source specified
			by the given parameters.

	( @b(ambient) lightcolor )

		parameters:
			lightcolor:  color of the light to be emitted from
			this source.

		returns:  gob containing an ambient light source of the given
			color.

@end(format)
For example, the following defines a white light source at the coordinates
(1, 0, 0) and binds it to the symbol 'new_light'.
@begin(programexample)
(setq new_light (light (make-point :x 1.0 )
		       (make-color :r 1.0 :g 1.0 :b 1.0)))
@end(programexample)

@section[Rendering]

Once a geometry DAG and a DAG containing light sources has been constructed,
and a camera structure has been defined, the P3D 'snap' function can be
used to initiate rendering.  (It is possible for a single dag containing
both geometrical primitives and light sources to be used for both the
lighting and geometry DAGs).  The snap function has the following
format:
@begin(format)

	( @b(snap) object lights acamera )

		parameters:
			object:  gob containing the model to be rendered
			lights:  gob containing light sources
			acamera:  camera struct defining the field of view

		returns: 'T

@end(format)
The exact action of 'snap' will depend on the renderer being used.  For
example, a slow renderer might render the single given frame, while a
dynamic renderer might render the object continuously, allowing the point
of view to be changed via a knob box, until control was returned to the
P3D interpreter.

@section[Predefined Symbols]

P3D predefines some symbols for commonly used constructs for convenience's
sake.  The predefined symbols include the following:
@begin(verbatim)
  @b(Symbol)             @b(Meaning)               @b(Definition)

  origin             point at origin       (make-point :x 0.0 :y 0.0 :z 0.0) 

  x-vec              x unit vector         (make-vector :x 1.0)

  y-vec              y unit vector         (make-vector :y 1.0)

  z-vec              z unit vector         (make-vector :z 1.0)

  white	             color white           (make-color :r 1.0 :g 1.0 :b 1.0)

  black	             color black           (make-color :r 0.0 :g 0.0 :b 0.0)

  red                color red             (make-color :r 1.0 :g 0.0 :b 0.0)

  green	             color green           (make-color :r 0.0 :g 1.0 :b 0.0)

  blue	             color blue            (make-color :r 0.0 :g 0.0 :b 1.0)

  yellow             color yellow          (make-color :r 0.0 :g 1.0 :b 1.0)

  cyan               color cyan            (make-color :r 1.0 :g 0.0 :b 1.0)

  magenta            color magenta         (make-color :r 1.0 :g 1.0 :b 0.0)

  null-transform     identity transform    (make-identity)

  default-material   default material      (def-material :ka 0.8 :kd 0.8
                                                :ks 0.3 :exp 30.0
                                                :reflect 0.1 :refract 1.0
                                                :energy black)

  dull-material      material type with    (def-material :ka 0.9 :kd 0.9
                     low reflectivity           :ks 0.1 :exp 5.0
                                                :reflect 0.1 :refract 1.0
                                                :energy black)

  shiny-material     glossy material       (def-material :ka 0.5 :kd 0.5
                     type                       :ks 0.5 :exp 50.0
                                                :reflect 0.3 :refract 1.0
                                                :energy black)

  metallic-material  metallic material     (def-material :ka 0.1 :kd 0.1
                     type, like chrome          :ks 0.9 :exp 100.0
                                                :reflect 0.7 :refract 1.0
                                                :energy black)

  matte-material     material type with    (def-material :ka 1.0 :kd 1.0
                     no reflectivity            :ks 0.0 :exp 0.0
                                                :reflect 0.0 :refract 1.0
                                                :energy black)

  aluminum-material  aluminum              (def-material :ka 0.25 :kd 0.25
                                                :ks 0.75 :exp 6.0
                                                :reflect 0.75 :refract 1.0
                                                :energy black)

@end(verbatim)
