/*
  File:	wff3x3.c
  Authors: K.R. Sloan
  Last Modified: 31 October 1988
  Purpose: convolves a wff image with a 3x3 template,
           plus a constant (DC) value.

           The image is treated as if it were embedded in a 
           1 pixel border of half-intensity values.  This avoids
           one type of edge effect, but introduces another. 

           Template (and DC) values are scaled integers
           (positive or negative.)  The scale factor is 2048.  So,
            2048 == 1.0, 1024 == 0.5, -512 == -0.25, etc.
           
           It is strongly recommended that the individual magnitudes not
           be too large, and that the sum of the values in the template
           (plus the DC value) lie in the range [0,2048] (i.e., [0.0,1.0])
           
 */

#include <stdio.h>
#include <wff.h>
#include <sys/file.h>
#include <sys/errno.h>

#define ShiftFactor 11
#define ScaleFactor 2048

#define ScanLineLength (wffMaxCoordinate+2)

static char RoutineName[] = "wff3x3";

int
wff3x3(fdIn, fdOut, Template, DC)
 FILE *fdIn, *fdOut;
 int Template[3][3]; 
 int DC;
 {
  FrameBufferType *FBin, *FBout;
  int Bottom, Left, Top, Right, BandsPerPixel, passed;
  char WhatBands[10];
  int BitsPerBand;
  char Name[NameLength], Value[ValueLength];
  int x, y, n;
  int j, k, l, b;
  static unsigned short A[ScanLineLength],
                        B[ScanLineLength], 
                        C[ScanLineLength],
                        NewLine[ScanLineLength];
  unsigned short *LastLine, *ThisLine, *NextLine, *t, *p, *q, *r;
  int dc,t00,t01,t02,t10,t11,t12,t20,t21,t22,value;
  int MaxPixelValue, width;
  unsigned short Gray;

  FBin  = (FrameBufferType *)0;
  FBout = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FBin))           {                        return; }
  if (FAILURE == PassImageIn(fdIn, FBin)) { (void)CloseFB(&FBin);  return; }

  if (FAILURE == GetBounds(FBin, &Bottom, &Left, &Top, &Right)) 
   { (void)CloseFB(FBin); return 1; }
  if (FAILURE == GetColorSystem(FBin, WhatBands, &BitsPerBand))
   { (void)CloseFB(FBin); return 1; }
  if (FAILURE == GetDescriptor(FBin,"BitsPerBand",Value))
   { (void)CloseFB(FBin); return 1; }
  BitsPerBand = atoi(Value);
  BandsPerPixel = strlen(WhatBands);

  if (FAILURE == OpenFB(&FBout)) { (void)CloseFB(&FBin);  return 1; }

  /*  Copy over existing NV pairs - watch for "X-PassedBy" */
  passed = 0;
  for (n=0;;n++)
   {
    GetDescriptorN(FBin, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      passed = 1;
     }
    SetDescriptor(FBout, Name, Value);
   }

  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");
    strcpy(Value,RoutineName);
    SetDescriptor(FBout, Name, Value);
   }

  /* Header operations over, now we can start the output stream */
  if (FAILURE == PassImageOut(fdOut, FBout))
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }

  /* Finally, pass the pixels */

  width = Right-Left+1;
  MaxPixelValue = (1 << BitsPerBand) - 1;
  Gray = MaxPixelValue >> 1;
  dc  = DC*MaxPixelValue;
  t00 = Template[0][0];
  t01 = Template[0][1];
  t02 = Template[0][2];
  t10 = Template[1][0];
  t11 = Template[1][1];
  t12 = Template[1][2];
  t20 = Template[2][0];
  t21 = Template[2][1];
  t22 = Template[2][2];

  LastLine = &A[0]; ThisLine = &B[0]; NextLine = &C[0];
  /* load up the three lines line with gray */
  p = LastLine; q = ThisLine; r = NextLine;
  for(x=0;x<width;x++)
   for (b=0;b<BandsPerPixel;b++)
    {
     *p++ = Gray; *q++ = Gray; *r++ = Gray;
    }

  /* preload the first line */
  r = NextLine+BandsPerPixel;
  if (FAILURE == NextNPixelsIn(FBin, width, r)) return 1;

  /* step through destination, one pixel at a time */
  for (y = Bottom; y <= Top; y++)
   {
    /* move forward one line */
    t = LastLine; LastLine = ThisLine; ThisLine = NextLine; NextLine = t;

    p = LastLine+BandsPerPixel;
    q = ThisLine+BandsPerPixel;
    r = NextLine+BandsPerPixel;
    if (y<Top)
     {
      if (FAILURE == NextNPixelsIn(FBin, width, r)) return 1; 
     }
    else
     {  /* last line - pad with gray */
      for(x=0;x<width;x++)
       for (b=0;b<BandsPerPixel;b++) *r++ = Gray;
     }
    /* filter the line */
    {
     register unsigned short *pj, *pk, *pl;
     register unsigned short *qj, *qk, *ql;
     register unsigned short *rj, *rk, *rl;
     unsigned short *out;

     pj = p-BandsPerPixel; pk = p; pl = p+BandsPerPixel;
     qj = q-BandsPerPixel; qk = q; ql = q+BandsPerPixel;
     rj = r-BandsPerPixel; rk = r; rl = r+BandsPerPixel;
     out = NewLine+BandsPerPixel;
     for (x = Left; x <= Right; x++)
      for (b=0;b<BandsPerPixel; b++)
       {
        value = dc;
        value +=   (*pj++)*t00 + (*pk++)*t01 + (*pl++)*t02;
        value +=   (*qj++)*t10 + (*qk++)*t11 + (*ql++)*t12;
        value +=   (*rj++)*t20 + (*rk++)*t21 + (*rl++)*t22;

        value >>= ShiftFactor;
        if      (value < 0)             value = 0;
        else if (value > MaxPixelValue) value = MaxPixelValue;
        *out++ = (unsigned short) value;
       }
    }
    p = NewLine+BandsPerPixel;
    if (FAILURE == NextNPixelsOut(FBout,width, p)) return 1;
   }

  (void)CloseFB(&FBin);
  if (FAILURE == CloseFB(&FBout)) return 1;

  return 0;
 }
