/*
  File: wff2tiff.c
  Author: K.R. Sloan
  Last Modified: 19 December 1991
  Purpose: convert .wff file to tiff
  
  Note: maximum bitspersample = 16

 */

#include <stdio.h>
#include <sys/time.h>
#include "wff.h"
#include "tiffio.h"
#include "tiff.h"

#define TRUE (-1)
#define FALSE (0)

int VERBOSE = -1;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr,"usage is:\n\t%s [-h][-v][-f tiff] <wff\n", RoutineName);
  exit(-1);
 }

static void FatalError(s)
 char *s;
 {
  fprintf(stderr,"%s: FatalError(%s)\n",RoutineName,s);
  exit(1);
 }

static void Pass(fdIn, TIFFFileName)
 FILE *fdIn;
 char *TIFFFileName;
 {
  FrameBufferType *FrameBuffer;
  int Left, Bottom, Top, Right, BitsPerBand;
  int BandsPerPixel;
  char WhatBands[10], Name[NameLength], Value[ValueLength];
  double AspectRatio;
  int width, height, depth, bitspersample;
  unsigned short *WFFScanLine;
  int WFFScanLineLength;
  unsigned char *TIFFScanLine;
  int TIFFScanLineLength;
  int i,c,x,y,b,row;
  struct tm  *ct;
  struct timeval tv;
  static char *datetime = "1990:01:01 12:00:00";
  TIFF       *tif;
  long        rowsperstrip;
  short       photometric;
  int         bpsl;

  FrameBuffer = (FrameBufferType *)0;
  if (FAILURE == OpenFB(&FrameBuffer))
   FatalError("OpenFB failed");
  if (FAILURE == ReadImage(fdIn, FrameBuffer))
   FatalError("ReadPassImage failed");
  if (FAILURE == GetBounds(FrameBuffer, &Bottom, &Left, &Top, &Right))
   FatalError("GetBounds failed");
  if (FAILURE == GetColorSystem(FrameBuffer, WhatBands, &BitsPerBand))
   FatalError("GetColorSystem failed");

  /*     do this right, next pass   */
  AspectRatio = 1.0;
  /*                                */

  if      (0 == strcmp(WhatBands,"I"))   BandsPerPixel = 1;
  else if (0 == strcmp(WhatBands,"RGB")) BandsPerPixel = 3;
  else FatalError("Sorry, we only do I and RGB format");

  width = Right - Left + 1;
  height = Top - Bottom + 1;

  if (VERBOSE)
   fprintf(stderr,"%s: BandsPerPixel = %d; BitsPerBand = %d\n",
     RoutineName, BandsPerPixel, BitsPerBand);

  bitspersample = BitsPerBand;
  if (bitspersample <= 8) depth = 8; else depth = 16;
  
  WFFScanLineLength = width*BandsPerPixel*(sizeof (unsigned short));
  WFFScanLine = (unsigned short *) malloc(WFFScanLineLength);
  if ((unsigned short *)0 == WFFScanLine)
   FatalError("no memory for WFFScanLine");

  TIFFScanLineLength = width*BandsPerPixel*(sizeof (unsigned char));
  TIFFScanLine = (unsigned char *) malloc(TIFFScanLineLength);
  if ((unsigned char *)0 == TIFFScanLine)
   FatalError("no memory for TIFFScanLine");

  gettimeofday(&tv, (struct timezone *)0);
  ct = localtime(&tv.tv_sec);
  sprintf(datetime, "19%02d:%02d:%02d %02d:%02d:%02d",
          ct->tm_year, ct->tm_mon + 1, ct->tm_mday,
          ct->tm_hour, ct->tm_min, ct->tm_sec);

  tif = TIFFOpen(TIFFFileName, "w");
  if ((TIFF *)0 == tif) FatalError("error opening TIFF file");

  switch (BandsPerPixel)
   {
    case 1: photometric = PHOTOMETRIC_MINISBLACK; break;
    case 3: photometric = PHOTOMETRIC_RGB;        break;
    default: FatalError("Sorry, we only do 1 or 3 BandsPerPixel");
   }

  bpsl = ((depth * width + 15) >> 3) & ~1;
  rowsperstrip = (8 * 1024) / bpsl;

  /*   rowsperstrip = height; */

  TIFFSetField(tif, TIFFTAG_SUBFILETYPE, 0);
  TIFFSetField(tif, TIFFTAG_IMAGELENGTH, height);
  TIFFSetField(tif, TIFFTAG_IMAGEWIDTH, width);
  TIFFSetField(tif, TIFFTAG_SAMPLESPERPIXEL, BandsPerPixel);
  TIFFSetField(tif, TIFFTAG_BITSPERSAMPLE, bitspersample);
  TIFFSetField(tif, TIFFTAG_ROWSPERSTRIP, rowsperstrip);
  TIFFSetField(tif, TIFFTAG_STRIPBYTECOUNTS, height / rowsperstrip);
  TIFFSetField(tif, TIFFTAG_XRESOLUTION, 1.0);
  TIFFSetField(tif, TIFFTAG_YRESOLUTION, AspectRatio);
  TIFFSetField(tif, TIFFTAG_RESOLUTIONUNIT, RESUNIT_NONE);
  TIFFSetField(tif, TIFFTAG_PLANARCONFIG, PLANARCONFIG_CONTIG);
  TIFFSetField(tif, TIFFTAG_PHOTOMETRIC, photometric);
  TIFFSetField(tif, TIFFTAG_ORIENTATION, ORIENTATION_TOPLEFT);
  TIFFSetField(tif, TIFFTAG_COMPRESSION, COMPRESSION_NONE);
  TIFFSetField(tif, TIFFTAG_IMAGEDESCRIPTION, "converted from .wff");
  TIFFSetField(tif, TIFFTAG_DATETIME, datetime);

  for (y=Top;y>=Bottom;y--)
   {
    if (FAILURE == GetBlock(FrameBuffer,y,Left,y,Right,WFFScanLine))
     FatalError("GetBlock failed");
    i = 0;  c = 0;
    if (8 == BitsPerBand)
     for(x=Left;x<=Right;x++)
      {
       for(b=0;b<BandsPerPixel;b++)
        {  
         register unsigned int all;
         all = (unsigned int)WFFScanLine[c++];
         TIFFScanLine[i++] = (unsigned char) all;
        }
      }
    else if (8 > BitsPerBand)
     for(x=Left;x<=Right;x++)
      {
       for(b=0;b<BandsPerPixel;b++)
        {  /* convert a small sample from BitsPerBand to 8 bits */
         register unsigned int all, j, sample;
         sample = ((unsigned int)WFFScanLine[c++]) << (8-BitsPerBand);
         for(all=0, j=0; j<8; j += BitsPerBand)
          all = (all >> BitsPerBand) | sample;
         TIFFScanLine[i++] = (unsigned char) all;
        }
      }
    else if (16 == BitsPerBand)
     for(x=Left;x<=Right;x++)
      {
       for(b=0;b<BandsPerPixel;b++)
        {  
         register unsigned int all;
         all = (unsigned int)WFFScanLine[c++];
         if (FILLORDER_MSB2LSB == tif->tif_fillorder)
          {
           TIFFScanLine[i++] = (unsigned char) (all >> 8);
           TIFFScanLine[i++] = (unsigned char) (all & 0xff);
          }
         else 
          {
           TIFFScanLine[i++] = (unsigned char) (all & 0xff);
           TIFFScanLine[i++] = (unsigned char) (all >> 8);
          }
        }
      }
    else
     for(x=Left;x<=Right;x++)
      {
       for(b=0;b<BandsPerPixel;b++)
        {  /* convert a big sample from BitsPerBand to 16 bits */
         register unsigned int all, j, sample;
         sample = ((unsigned int)WFFScanLine[c++]) << (16-BitsPerBand);
         for(all=0, j=0; j<16; j += BitsPerBand)
          all = (all >> BitsPerBand) | sample;
         if (FILLORDER_MSB2LSB == tif->tif_fillorder)
          {
           TIFFScanLine[i++] = (unsigned char) (all >> 8);
           TIFFScanLine[i++] = (unsigned char) (all & 0xff);
          }
         else 
          {
           TIFFScanLine[i++] = (unsigned char) (all & 0xff);
           TIFFScanLine[i++] = (unsigned char) (all >> 8);
          }
        }
      }
    row = Top-y;
    if (0 > TIFFWriteScanline(tif, TIFFScanLine,row,0))
     FatalError("TIFFWriteScanline failed");
   }

  TIFFFlushData(tif);
  TIFFClose(tif);
  CloseFB(&FrameBuffer);
  free(WFFScanLine); free(TIFFScanLine);
 }

int main(argc, argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  char *TIFFFileName;

  TIFFFileName = "wff.tiff";
  RoutineName = argv[ArgsParsed++];
  while (ArgsParsed < argc)
   {
    if ('-' != argv[ArgsParsed][0]) { usage(); exit(-1);}
    switch (argv[ArgsParsed++][1])
     {
      case 'f': if (ArgsParsed >= argc)  { usage(); exit(-1);}
                TIFFFileName = argv[ArgsParsed++];
                break;
      case 'v': VERBOSE = -1; break;
      default:
      case 'h':  usage(); exit(-1);
     }
   }

  Pass(stdin, TIFFFileName);
  
  exit(0);
 }
